<?php
session_start();
require_once '../config.php';
require_once '../includes/enhanced_access_control_fixed.php';

// Check if user is logged in and is a teacher
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'teacher') {
    header('Location: ../login.php');
    exit;
}

require_once '../includes/teacher_header.php';

error_log("Teacher user_id in session: " . ($_SESSION['user_id'] ?? 'not set'));
error_log("Teacher role in session: " . ($_SESSION['role'] ?? 'not set'));

// Get teacher's payslips - teacher may have multiple roles
try {
    $stmt = $pdo->prepare("
        SELECT p.*, pr.payroll_month, pr.payroll_year, pr.processed_at,
               CONCAT(s.first_name, ' ', s.last_name) as teacher_name,
               s.staff_id as staff_id_display
        FROM payslips p
        INNER JOIN payroll_runs pr ON p.payroll_run_id = pr.id
        LEFT JOIN staff s ON p.staff_id = s.staff_id
        WHERE s.id = ?
        ORDER BY pr.payroll_year DESC, pr.payroll_month DESC
    ");
    $stmt->execute([$_SESSION['user_id']]);
    $payslips = $stmt->fetchAll(PDO::FETCH_ASSOC);
    error_log("Number of payslips fetched: " . count($payslips));
} catch (PDOException $e) {
    $payslips = [];
    $error = "Error loading payslips: " . $e->getMessage();
    error_log("PDOException: " . $e->getMessage());
}

// Get school settings for payslip display
$schoolSettings = $pdo->query("SELECT school_name, address, phone, email, logo_path FROM school_settings LIMIT 1")->fetch(PDO::FETCH_ASSOC) ?? [];
?>

<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header">
                    <h3 class="card-title">
                        <i class="fas fa-money-check-alt"></i> My Payslips
                    </h3>
                </div>
                <div class="card-body">
                    <?php if (isset($error)): ?>
                        <div class="alert alert-danger">
                            <i class="fas fa-exclamation-triangle"></i> <?php echo htmlspecialchars($error); ?>
                        </div>
                    <?php endif; ?>

                    <?php if (empty($payslips)): ?>
                        <div class="alert alert-info">
                            <i class="fas fa-info-circle"></i> No payslips available yet.
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-striped table-hover">
                                <thead>
                                    <tr>
                                        <th>Payroll Period</th>
                                        <th>Basic Salary</th>
                                        <th>Total Allowances</th>
                                        <th>Total Deductions</th>
                                        <th>Net Pay</th>
                                        <th>Status</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($payslips as $payslip): ?>
                                        <tr>
                                            <td>
                                                <?php echo htmlspecialchars($payslip['payroll_month'] . ' ' . $payslip['payroll_year']); ?>
                                            </td>
                                            <td>₵<?php echo number_format($payslip['basic_salary'], 2); ?></td>
                                            <td>₵<?php echo number_format($payslip['total_allowances'], 2); ?></td>
                                            <td>₵<?php echo number_format($payslip['total_deductions'], 2); ?></td>
                                            <td>
                                                <strong>₵<?php echo number_format($payslip['net_salary'], 2); ?></strong>
                                            </td>
                                            <td>
                                                <span class="badge badge-success">Paid</span>
                                            </td>
                                            <td>
                                                <button class="btn btn-sm btn-primary print-payslip" data-payslip-id="<?php echo $payslip['id']; ?>">
                                                    <i class="fas fa-print"></i> Print Payslip
                                                </button>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>



<script>
$(document).ready(function() {
    $('.print-payslip').on('click', function() {
        var payslipId = $(this).data('payslip-id');

        // Load payslip details via AJAX and print directly
        $.ajax({
            url: '../api/get_payslip_details.php',
            method: 'GET',
            dataType: 'json',
            data: { id: payslipId },
            success: function(response) {
                console.log('AJAX success response:', response);
                if (response.success) {
                    printPayslip(response.payslip);
                } else {
                    alert('Error: ' + response.message);
                }
            },
            error: function(xhr, status, error) {
                console.error('AJAX error:', status, error);
                console.error('Response text:', xhr.responseText);
                alert('Error loading payslip details. Please try again.');
            }
        });
    });
});

// Pass school settings from PHP to JavaScript
const schoolSettings = <?php echo json_encode($schoolSettings); ?>;

function printPayslip(payslip) {
    let data = {};
    try {
        data = payslip.payslip_data ? JSON.parse(payslip.payslip_data) : {};
    } catch (e) {
        console.error('Error parsing payslip_data JSON:', e);
        data = {};
    }

    // Use school settings from PHP
    const schoolName = schoolSettings.school_name || 'School Name';
    const schoolAddress = schoolSettings.address || 'School Address';
    const schoolPhone = schoolSettings.phone || 'School Phone';
    const schoolEmail = schoolSettings.email || 'School Email';

    const printWindow = window.open('', '_blank');
    if (!printWindow) {
        alert('Please allow popups for this website to print payslips.');
        return;
    }

    const html = `
<!DOCTYPE html>
<html>
<head>
<title>Payslip - ${payslip.teacher_name}</title>
<style>
body { font-family: Arial, sans-serif; margin: 20px; }
.header { text-align: center; margin-bottom: 20px; border-bottom: 2px solid #333; padding-bottom: 15px; }
.header h3 { margin: 0; color: #333; }
.header p { margin: 5px 0; color: #666; }
.payslip-info { text-align: center; margin-bottom: 20px; }
.payslip-info h4 { margin: 0; }
.payslip-info h5 { margin: 5px 0; }
table { width: 100%; border-collapse: collapse; margin-bottom: 20px; }
th, td { padding: 8px; border: 1px solid #ddd; text-align: left; }
th { background: #f8f9fa; }
.total-row { background: #e8f5e8; font-weight: bold; }
.deduction-row { background: #ffeaea; font-weight: bold; }
.net-row { background: #e8f0ff; font-weight: bold; font-size: 1.1em; }
.attendance { margin-top: 20px; padding: 15px; background: #f8f9fa; border-radius: 8px; }
.right-align { text-align: right; }
.print-button { text-align: center; margin-bottom: 20px; }
.print-button button { padding: 10px 20px; background: #007bff; color: white; border: none; border-radius: 5px; cursor: pointer; font-size: 16px; }
.print-button button:hover { background: #0056b3; }
@media print { .print-button { display: none; } }
</style>
</head>
<body>
<div class="print-button">
<button onclick="window.print();"><i class="fas fa-print"></i> Print Payslip</button>
</div>
<div class="header">
<h3>${schoolName}</h3>
<p>${schoolAddress}</p>
<p>Phone: ${schoolPhone} | Email: ${schoolEmail}</p>
</div>
<div class="payslip-info">
<h4>EMPLOYEE PAYSLIP</h4>
<h5>${payslip.teacher_name}</h5>
<p>Pay Period: ${payslip.payroll_month} ${payslip.payroll_year}</p>
</div>
<table>
<tr><th colspan="2">Earnings</th></tr>
<tr><td>Basic Salary</td><td class="right-align">GH₵ ${parseFloat(payslip.basic_salary).toFixed(2)}</td></tr>
<tr><td>Housing Allowance</td><td class="right-align">GH₵ ${((data.allowances && data.allowances.housing) || 0).toFixed(2)}</td></tr>
<tr><td>Transport Allowance</td><td class="right-align">GH₵ ${((data.allowances && data.allowances.transport) || 0).toFixed(2)}</td></tr>
<tr><td>Utility Allowance</td><td class="right-align">GH₵ ${((data.allowances && data.allowances.utility) || 0).toFixed(2)}</td></tr>
<tr><td>Other Allowances</td><td class="right-align">GH₵ ${((data.allowances && data.allowances.other) || 0).toFixed(2)}</td></tr>
<tr><td>Overtime</td><td class="right-align">GH₵ ${parseFloat(payslip.overtime_amount).toFixed(2)}</td></tr>
<tr class="total-row"><td>Gross Salary</td><td class="right-align">GH₵ ${(parseFloat(payslip.gross_salary) + parseFloat(payslip.overtime_amount)).toFixed(2)}</td></tr>
</table>
<table>
<tr><th colspan="2">Deductions</th></tr>
<tr><td>Income Tax (PAYE)</td><td class="right-align">GH₵ ${parseFloat(payslip.tax_amount).toFixed(2)}</td></tr>
<tr><td>SSNIT (5.5%)</td><td class="right-align">GH₵ ${parseFloat(payslip.ssnit_employee).toFixed(2)}</td></tr>
<tr><td>Leave Deductions</td><td class="right-align">GH₵ ${parseFloat(payslip.leave_deductions).toFixed(2)}</td></tr>
<tr class="deduction-row"><td>Total Deductions</td><td class="right-align">GH₵ ${parseFloat(payslip.total_deductions).toFixed(2)}</td></tr>
</table>
<table>
<tr class="net-row">
<td>Net Salary</td>
<td class="right-align">GH₵ ${parseFloat(payslip.net_salary).toFixed(2)}</td>
</tr>
</table>
<div class="attendance">
<h5>Attendance Summary</h5>
<p>Working Days: ${payslip.working_days} | Absent Days: ${payslip.absent_days} | Leave Days: ${payslip.leave_days} | Overtime Hours: ${payslip.overtime_hours}</p>
</div>
</body>
</html>`;

    printWindow.document.write(html);
    printWindow.document.close();

    // Automatically trigger print dialog
    setTimeout(() => {
        printWindow.print();
    }, 100);
}
</script>

<?php require_once '../includes/teacher_footer.php'; ?>
