<?php
require_once '../config.php';

if (!isset($_SESSION['role']) || $_SESSION['role'] != 'teacher') {
    header('Location: ../login.php');
    exit;
}

// Get parameters
$student_id = $_GET['student_id'] ?? null;
$selected_class = $_GET['class'] ?? null;
$selected_term_id = $_GET['term_id'] ?? null;
$selected_academic_year_id = $_GET['academic_year_id'] ?? null;

if (!$student_id) {
    die("Student ID is required.");
}

// Fetch school settings
$school_name = 'Professional School Name';
$school_logo_url = '';
try {
    $stmt = $pdo->prepare("SELECT school_name, logo_path FROM school_settings LIMIT 1");
    $stmt->execute();
    $school_info = $stmt->fetch();
    if ($school_info) {
        $school_name = htmlspecialchars($school_info['school_name']);
        $logo_path = $school_info['logo_path'];
        if ($logo_path && !preg_match('/^https?:\/\//', $logo_path) && !preg_match('/^\//', $logo_path)) {
            $school_logo_url = '../' . $logo_path;
        } else {
            $school_logo_url = $logo_path;
        }
        $school_logo_url = htmlspecialchars($school_logo_url);
    }
} catch (Exception $e) {
    // Use default if error
}

// Get teacher's ID
$user_id = $_SESSION['user_id'];
$stmt = $pdo->prepare("SELECT id FROM staff WHERE id = ? AND role = 'teaching'");
$stmt->execute([$user_id]);
$teacher = $stmt->fetch();

if (!$teacher) {
    die("Teacher record not found. Please contact your administrator.");
}

$teacher_id = $teacher['id'];

// Fetch student name
$stmt = $pdo->prepare("SELECT CONCAT(first_name, ' ', COALESCE(middle_name, ''), ' ', last_name) as name FROM students WHERE id = ?");
$stmt->execute([$student_id]);
$student = $stmt->fetch();
if (!$student) {
    die("Student not found.");
}
$student_name = htmlspecialchars($student['name']);

// Get detailed student results with filters
$query = "
    SELECT
        g.converted_class_mark,
        g.converted_exam_mark,
        g.converted_final_mark,
        g.comments,
        g.graded_at,
        t.name as term_name,
        ay.year as academic_year,
        sub.name as subject_name,
        sub.code as subject_code,
        c.name as course_name
    FROM enrollments e
    JOIN courses c ON e.course_id = c.id
    JOIN teacher_courses tc ON c.id = tc.course_id
    INNER JOIN grades g ON e.id = g.enrollment_id
    LEFT JOIN terms t ON g.term_id = t.id
    LEFT JOIN academic_years ay ON g.academic_year_id = ay.id
    LEFT JOIN subjects sub ON c.subject_id = sub.id
    WHERE e.student_id = ? AND tc.teacher_id = ?
";

$params = [$student_id, $teacher_id];

if ($selected_class) {
    $query .= " AND (SELECT class FROM students WHERE id = e.student_id) = ?";
    $params[] = $selected_class;
}
if ($selected_term_id) {
    $query .= " AND g.term_id = ?";
    $params[] = $selected_term_id;
}
if ($selected_academic_year_id) {
    $query .= " AND g.academic_year_id = ?";
    $params[] = $selected_academic_year_id;
}

$query .= "
    ORDER BY ay.year DESC, t.name DESC, g.graded_at DESC
";

$stmt = $pdo->prepare($query);
$stmt->execute($params);
$results = $stmt->fetchAll();

// Group results by academic year and term
$results_by_term = [];
foreach ($results as $result) {
    $term_key = ($result['academic_year'] ? $result['academic_year'] . ' - ' : '') . ($result['term_name'] ?: 'No Term');
    $results_by_term[$term_key][] = $result;
}

// Get class, term, academic year names for display
$class_name = $selected_class ?: 'All Classes';
$term_name = 'All Terms';
$academic_year_name = 'All Academic Years';

if ($selected_term_id) {
    $stmt = $pdo->prepare("SELECT name FROM terms WHERE id = ?");
    $stmt->execute([$selected_term_id]);
    $term = $stmt->fetch();
    $term_name = $term ? htmlspecialchars($term['name']) : 'N/A';
}

if ($selected_academic_year_id) {
    $stmt = $pdo->prepare("SELECT year FROM academic_years WHERE id = ?");
    $stmt->execute([$selected_academic_year_id]);
    $ay = $stmt->fetch();
    $academic_year_name = $ay ? htmlspecialchars($ay['year']) : 'N/A';
}

?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Student Report Card</title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            margin: 20px;
            color: #333;
            background: white;
        }

        .report-container {
            max-width: 800px;
            margin: auto;
            background: white;
            padding: 20px;
            border: 2px solid #333;
        }

        .header {
            text-align: center;
            margin-bottom: 30px;
            border-bottom: 2px solid #333;
            padding-bottom: 20px;
        }

        .school-logo {
            max-height: 80px;
            margin-bottom: 10px;
        }

        .report-title {
            font-size: 1.5em;
            color: #667eea;
            font-weight: 600;
            margin: 10px 0;
        }

        .student-info {
            text-align: center;
            margin-bottom: 20px;
            font-size: 1.1em;
        }

        .info-section {
            display: flex;
            justify-content: space-between;
            margin-bottom: 20px;
            font-size: 1em;
            color: #555;
        }

        .info-item {
            flex: 1;
            padding: 0 10px;
            text-align: center;
        }

        table {
            width: 100%;
            border-collapse: collapse;
            font-size: 0.95em;
            margin-top: 20px;
        }

        th, td {
            border: 1px solid #ccc;
            padding: 10px 12px;
            text-align: left;
        }

        th {
            background: #f0f0f0;
            font-weight: 700;
            text-transform: uppercase;
            letter-spacing: 1px;
            font-size: 0.8em;
        }

        tr:nth-child(even) {
            background: #f9f9f9;
        }

        .comments-cell {
            max-width: 200px;
            word-wrap: break-word;
        }

        .footer {
            margin-top: 30px;
            text-align: center;
            font-size: 0.9em;
            color: #888;
            border-top: 1px solid #ccc;
            padding-top: 10px;
        }

        @media print {
            body {
                margin: 0;
                background: white;
            }
            .report-container {
                border: none;
                padding: 10px;
            }
            .no-print {
                display: none;
            }
        }
    </style>
</head>
<body>
    <div class="report-container">
        <div class="header">
            <?php if ($school_logo_url): ?>
                <img src="<?php echo $school_logo_url; ?>" alt="School Logo" class="school-logo" />
            <?php endif; ?>
            <h2 class="report-title">Student Report Card</h2>
        </div>

        <div class="student-info">
            <strong><?php echo $student_name; ?></strong>
        </div>

        <div class="info-section">
            <div class="info-item"><strong>Class:</strong> <?php echo $class_name; ?></div>
            <div class="info-item"><strong>Term/Semester:</strong> <?php echo $term_name; ?></div>
            <div class="info-item"><strong>Academic Year:</strong> <?php echo $academic_year_name; ?></div>
        </div>

        <?php if (!empty($results)): ?>
            <?php foreach ($results_by_term as $term_name => $term_results): ?>
                <h3 style="color: #667eea; margin-top: 30px; margin-bottom: 10px;"><?php echo htmlspecialchars($term_name); ?></h3>
                <table>
                    <thead>
                        <tr>
                            <th>Subject</th>
                            <th>Class Mark</th>
                            <th>Exam Mark</th>
                            <th>Final Mark</th>
                            <th>Comments</th>
                            <th>Date</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($term_results as $result): ?>
                            <tr>
                                <td>
                                    <?php
                                        $course = trim((string)($result['course_name'] ?? ''));
                                        $subject = trim((string)($result['subject_name'] ?? ''));
                                        $code = trim((string)($result['subject_code'] ?? ''));

                                        if ($course && $subject) {
                                            echo htmlspecialchars($course . ' — ' . $subject);
                                        } elseif ($course) {
                                            echo htmlspecialchars($course);
                                        } elseif ($subject) {
                                            echo htmlspecialchars($subject);
                                        } else {
                                            echo 'N/A';
                                        }

                                        if ($code) {
                                            echo ' <small>' . htmlspecialchars($code) . '</small>';
                                        }
                                    ?>
                                </td>
                                <td><?php echo htmlspecialchars($result['converted_class_mark'] ?? 'N/A'); ?></td>
                                <td><?php echo htmlspecialchars($result['converted_exam_mark'] ?? 'N/A'); ?></td>
                                <td><?php echo htmlspecialchars($result['converted_final_mark'] ?? 'N/A'); ?></td>
                                <td class="comments-cell"><?php echo htmlspecialchars($result['comments'] ?? ''); ?></td>
                                <td><?php echo $result['graded_at'] ? date('M d, Y', strtotime($result['graded_at'])) : 'N/A'; ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endforeach; ?>
        <?php else: ?>
            <p>No results found for the selected filters.</p>
        <?php endif; ?>

        <div class="footer">
            Generated on <?php echo date('F d, Y'); ?>
        </div>
    </div>

    <script>
        window.onload = function() {
            window.print();
        };
    </script>
</body>
</html>
