<?php
require_once '../config.php';
require_once '../includes/user_activity_logger.php';
require_once '../includes/csrf_protection.php';
require_once '../includes/input_validation.php';
require_once '../includes/password_security.php';

$error = '';

if (isset($_GET['error']) && $_GET['error'] == 'access_denied') {
    $error = 'Access denied. Please log in to access the teacher dashboard.';
}

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    // Validate CSRF token
    if (!CSRFProtection::validateRequest()) {
        $error = 'Security validation failed. Please refresh the page and try again.';
    } else {
        // Sanitize and validate input
        $username = InputValidation::getPostData('username', 'string');
        $password = InputValidation::getPostData('password', 'string');

        // Validate required fields
        $validationErrors = InputValidation::validateRequired($_POST, ['username', 'password']);
        if (!empty($validationErrors)) {
            $error = implode('<br>', $validationErrors);
        } elseif (empty($username) || empty($password)) {
            $error = 'Please fill in all fields.';
        } else {
            // Check user credentials in staff_accounts table
            $stmt = $pdo->prepare("
                SELECT su.id, su.password, s.role, s.first_name, s.last_name
                FROM staff_accounts su
                LEFT JOIN staff s ON su.id = s.user_id
                WHERE su.username = ? AND su.status = 'active'
            ");
            $stmt->execute([$username]);
            $user = $stmt->fetch();

            if ($user && PasswordSecurity::verifyPassword($password, $user['password'])) {
                // Check if user is allowed to login through teacher portal
                if ($user['role'] == 'teaching') {
                    // Set session
                    $_SESSION['user_id'] = $user['id'];
                    $_SESSION['username'] = $username;
                    $_SESSION['role'] = 'teacher';
                    $_SESSION['first_name'] = $user['first_name'] ?? '';
                    $_SESSION['last_name'] = $user['last_name'] ?? '';

                    // Log successful login
                    UserActivityLogger::logActivity($user['id'], 'login', "Teacher portal login successful from IP: " . $_SERVER['REMOTE_ADDR']);

                    // Redirect to teacher dashboard
                    header('Location: teacher_dashboard.php');
                    exit;
                } else {
                    // Log failed login attempt for unauthorized user
                    UserActivityLogger::logActivity(null, 'login_failed', "Unauthorized user attempted teacher portal login: $username (role: {$user['role']}) from IP: " . $_SERVER['REMOTE_ADDR']);
                    $error = 'Access denied. This portal is for teachers only.';
                }
            } else {
                // Log failed login attempt
                UserActivityLogger::logActivity(null, 'login_failed', "Failed teacher portal login attempt for username: $username from IP: " . $_SERVER['REMOTE_ADDR']);
                $error = 'Invalid username or password.';
            }
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Teacher Login - School Management System</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            line-height: 1.6;
            color: #2c3e50;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            margin: 0;
            overflow-x: hidden;
        }

        body::before {
            content: '';
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 100 100"><defs><pattern id="grain" width="100" height="100" patternUnits="userSpaceOnUse"><circle cx="25" cy="25" r="1" fill="%23ffffff" opacity="0.03"/><circle cx="75" cy="75" r="1" fill="%23ffffff" opacity="0.03"/><circle cx="50" cy="10" r="0.5" fill="%23ffffff" opacity="0.02"/></pattern></defs><rect width="100" height="100" fill="url(%23grain)"/></svg>');
            pointer-events: none;
            z-index: -1;
        }

        .top-bar {
            background-color: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            box-shadow: 0 2px 20px rgba(0,0,0,0.1);
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            z-index: 1000;
            padding: 15px 0;
            border-bottom: 1px solid rgba(255, 255, 255, 0.2);
        }

        .top-bar .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 20px;
            display: flex;
            justify-content: center;
            align-items: center;
        }

        .top-bar .logo {
            font-size: 1.5em;
            font-weight: 700;
            color: #667eea;
            display: flex;
            align-items: center;
            text-shadow: 0 1px 2px rgba(0,0,0,0.1);
        }

        .top-bar .logo i {
            margin-right: 10px;
            font-size: 1.8em;
        }

        .top-bar .nav-links {
            display: flex;
            gap: 30px;
            align-items: center;
        }

        .top-bar .nav-links a {
            color: #5a6c7d;
            text-decoration: none;
            font-weight: 500;
            display: flex;
            align-items: center;
            transition: all 0.3s ease;
            padding: 8px 12px;
            border-radius: 6px;
        }

        .top-bar .nav-links a:hover {
            color: #667eea;
            background-color: rgba(102, 126, 234, 0.1);
            transform: translateY(-1px);
        }

        .top-bar .nav-links a i {
            margin-right: 8px;
        }

        .main-content {
            padding-top: 100px;
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding-bottom: 40px;
        }

        .login-container {
            background-color: rgba(255, 255, 255, 0.98);
            backdrop-filter: blur(20px);
            padding: 50px;
            border-radius: 20px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.15), 0 0 0 1px rgba(255, 255, 255, 0.2);
            width: 100%;
            max-width: 480px;
            position: relative;
            overflow: hidden;
            animation: slideIn 0.6s ease-out;
        }

        .login-container::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 4px;
            background: linear-gradient(90deg, #667eea, #764ba2);
        }

        @keyframes slideIn {
            from {
                opacity: 0;
                transform: translateY(30px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .login-header {
            text-align: center;
            margin-bottom: 40px;
            position: relative;
        }

        .login-header .icon {
            font-size: 4em;
            color: #667eea;
            margin-bottom: 15px;
            text-shadow: 0 2px 10px rgba(102, 126, 234, 0.3);
            animation: float 3s ease-in-out infinite;
        }

        @keyframes float {
            0%, 100% { transform: translateY(0px); }
            50% { transform: translateY(-10px); }
        }

        .login-header h1 {
            color: #2c3e50;
            margin-bottom: 10px;
            font-size: 2.8em;
            font-weight: 700;
            letter-spacing: -0.5px;
        }

        .login-header p {
            color: #7f8c8d;
            font-size: 1.1em;
            font-weight: 400;
        }

        .form-group {
            margin-bottom: 25px;
            position: relative;
        }

        .form-group .input-icon {
            position: absolute;
            left: 15px;
            top: 50%;
            transform: translateY(-50%);
            color: #667eea;
            font-size: 18px;
            z-index: 2;
            transition: color 0.3s ease;
        }

        .form-group label {
            display: block;
            margin-bottom: 8px;
            color: #34495e;
            font-weight: 600;
            font-size: 0.95em;
            letter-spacing: 0.5px;
            text-transform: uppercase;
        }

        .form-group input {
            width: 100%;
            padding: 15px 15px 15px 50px;
            border: 2px solid #e1e8ed;
            border-radius: 12px;
            font-size: 16px;
            transition: all 0.3s ease;
            background-color: #f8f9fa;
            color: #2c3e50;
        }

        .form-group input:focus {
            outline: none;
            border-color: #667eea;
            background-color: #fff;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
            transform: translateY(-2px);
        }

        .form-group input:focus + .input-icon {
            color: #764ba2;
        }

        .btn {
            display: inline-block;
            padding: 16px 40px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: #fff;
            text-decoration: none;
            border: none;
            border-radius: 12px;
            cursor: pointer;
            font-size: 16px;
            font-weight: 600;
            width: 100%;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
            text-transform: uppercase;
            letter-spacing: 1px;
        }

        .btn::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
            transition: left 0.5s;
        }

        .btn:hover::before {
            left: 100%;
        }

        .btn:hover {
            transform: translateY(-3px);
            box-shadow: 0 10px 30px rgba(102, 126, 234, 0.4);
        }

        .btn .btn-icon {
            margin-right: 10px;
        }

        .error-message {
            padding: 15px;
            margin-bottom: 25px;
            border-radius: 10px;
            background: linear-gradient(135deg, #ff9a9e 0%, #fecfef 100%);
            color: #721c24;
            border: 1px solid rgba(255, 193, 193, 0.5);
            font-weight: 500;
            animation: shake 0.5s ease-in-out;
        }

        @keyframes shake {
            0%, 100% { transform: translateX(0); }
            25% { transform: translateX(-5px); }
            75% { transform: translateX(5px); }
        }

        .back-link {
            text-align: center;
            margin-top: 25px;
        }

        .back-link a {
            color: #667eea;
            text-decoration: none;
            font-weight: 600;
            transition: all 0.3s ease;
            padding: 8px 16px;
            border-radius: 6px;
        }

        .back-link a:hover {
            color: #764ba2;
            background-color: rgba(102, 126, 234, 0.1);
            transform: translateY(-1px);
        }

        .demo-info {
            margin-top: 25px;
            padding: 20px;
            background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
            border-radius: 12px;
            border-left: 4px solid #667eea;
            color: #fff;
        }

        .demo-info h4 {
            margin-bottom: 12px;
            color: #fff;
            font-weight: 600;
        }

        .demo-info p {
            margin-bottom: 8px;
            font-size: 0.9em;
            opacity: 0.9;
        }

        .portal-links {
            margin-top: 25px;
            padding: 20px;
            background-color: rgba(248, 249, 250, 0.8);
            border-radius: 12px;
            border: 1px solid rgba(222, 226, 230, 0.5);
            backdrop-filter: blur(10px);
        }

        .portal-links h4 {
            margin-bottom: 15px;
            color: #667eea;
            text-align: center;
            font-weight: 600;
        }

        .portal-links .links {
            display: flex;
            gap: 12px;
            justify-content: center;
            flex-wrap: wrap;
        }

        .portal-links .links a {
            color: #667eea;
            text-decoration: none;
            font-weight: 500;
            padding: 8px 16px;
            border: 2px solid #667eea;
            border-radius: 8px;
            transition: all 0.3s ease;
            font-size: 0.9em;
        }

        .portal-links .links a:hover {
            background-color: #667eea;
            color: #fff;
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.3);
        }

        /* Responsive */
        @media (max-width: 768px) {
            .top-bar {
                padding: 12px 0;
            }

            .top-bar .container {
                padding: 0 15px;
            }

            .top-bar .logo {
                font-size: 1.3em;
            }

            .top-bar .logo i {
                font-size: 1.6em;
            }

            .top-bar .nav-links {
                gap: 20px;
            }

            .top-bar .nav-links a span {
                display: none;
            }

            .main-content {
                padding-top: 90px;
            }

            .login-container {
                margin: 20px;
                padding: 35px 25px;
            }

            .login-header h1 {
                font-size: 2.2em;
            }

            .login-header .icon {
                font-size: 3.2em;
            }
        }

        @media (max-width: 480px) {
            .top-bar {
                padding: 10px 0;
            }

            .login-container {
                margin: 15px;
                padding: 30px 20px;
            }

            .login-header h1 {
                font-size: 1.8em;
            }

            .login-header .icon {
                font-size: 2.8em;
            }

            .top-bar .logo {
                font-size: 1.1em;
            }

            .top-bar .logo i {
                font-size: 1.4em;
            }

            .top-bar .nav-links {
                gap: 15px;
            }

            .form-group input {
                padding: 12px 12px 12px 45px;
            }

            .form-group .input-icon {
                left: 12px;
                font-size: 16px;
            }
        }
    </style>
</head>
<body>
    <!-- Top Navigation Bar -->
    <div class="top-bar">
        <div class="container">
            <div class="logo">
                <i class="fas fa-chalkboard-teacher"></i>
                <span>Four Rivers International School</span>
            </div>
            <div class="nav-links">
            </div>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <div class="login-container">
            <div class="login-header">
                <div class="icon"><i class="fas fa-chalkboard-teacher"></i></div>
                <h1>Teacher Portal</h1>
                <p>Login for Teachers</p>
            </div>

        <?php if ($error): ?>
            <div class="error-message"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>

        <form method="post" action="index.php">
            <?php echo CSRFProtection::getHiddenInput(); ?>
            <div class="form-group">
                <label for="username">Username:</label>
                <span class="input-icon" style="margin-top: 13px;"><i class="fas fa-user"></i></span>
                <input type="text" id="username" name="username" required autofocus>
            </div>

            <div class="form-group">
                <label for="password">Password:</label>
                <span class="input-icon" style="margin-top: 13px;"><i class="fas fa-lock"></i></span>
                <input type="password" id="password" name="password" required>
            </div>

            <button type="submit" class="btn"><span class="btn-icon"><i class="fas fa-sign-in-alt"></i></span>Login to Teacher Panel</button>
        </form>


        </div>
    </div>

    <script src="../js/script.js"></script>
</body>
</html>
