<?php
require_once '../config.php';

if (!isset($_SESSION['user_id']) || $_SESSION['role'] != 'teacher') {
    header('Location: ../login.php');
    exit;
}

$user_id = $_SESSION['user_id'];

// Get teacher id from staff table (since teachers are now in staff table)
$stmt = $pdo->prepare("SELECT id FROM staff WHERE user_id = ? AND role = 'teaching'");
$stmt->execute([$user_id]);
$teacher = $stmt->fetch();

if (!$teacher) {
    die("Teacher not found.");
}

$teacher_id = $teacher['id'];

// Get active academic year and term
$active_academic_year = $pdo->query("SELECT id, year FROM academic_years WHERE is_active = 1 LIMIT 1")->fetch();
$active_term = $pdo->query("SELECT id, name FROM terms WHERE is_active = 1 LIMIT 1")->fetch();

try {
    // Fetch classes assigned to the teacher
    $classes_stmt = $pdo->prepare("SELECT DISTINCT c.id, c.name FROM classes c JOIN teacher_classes tc ON c.id = tc.class_id WHERE tc.teacher_id = ?");
    $classes_stmt->execute([$teacher_id]);
    $classes = $classes_stmt->fetchAll();
} catch (PDOException $e) {
    error_log("Error fetching classes for teacher $teacher_id: " . $e->getMessage());
    $classes = [];
}

$selected_class_id = $_GET['class_id'] ?? null;
$selected_date = $_GET['date'] ?? date('Y-m-d');

$students = [];
$attendance_records = [];

if ($selected_class_id) {
    // Get the class name
    $class_stmt = $pdo->prepare("SELECT name FROM classes WHERE id = ?");
    $class_stmt->execute([$selected_class_id]);
    $class = $class_stmt->fetch();
    $class_name = $class['name'];

    // Fetch students in the selected class
    $students_stmt = $pdo->prepare("SELECT id, CONCAT(first_name, ' ', COALESCE(middle_name, ''), ' ', last_name) as first_name, '' as last_name FROM students WHERE class = ? ORDER BY first_name, last_name");
    $students_stmt->execute([$class_name]);
    $students = $students_stmt->fetchAll();

    // Fetch attendance records for the selected date and class
    $attendance_stmt = $pdo->prepare("SELECT * FROM attendance WHERE class_id = ? AND date = ?");
    $attendance_stmt->execute([$selected_class_id, $selected_date]);
    $attendance_records = $attendance_stmt->fetchAll(PDO::FETCH_UNIQUE | PDO::FETCH_ASSOC);

    // If attendance records exist for the selected date, keep them active for editing
    if (count($attendance_records) > 0) {
        // No additional action needed, form will show existing records for editing
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $class_id = $_POST['class_id'];
    $date = $_POST['date'];
    $attendance_data = $_POST['attendance'] ?? [];

    // Delete existing attendance for the date and class
    $delete_stmt = $pdo->prepare("DELETE FROM attendance WHERE class_id = ? AND date = ?");
    $delete_stmt->execute([$class_id, $date]);

    // Insert new attendance records with academic year and term
    $insert_stmt = $pdo->prepare("INSERT INTO attendance (student_id, class_id, date, status, marked_by, academic_year_id, term_id) VALUES (?, ?, ?, ?, ?, ?, ?)");
    foreach ($attendance_data as $student_id => $status) {
        $insert_stmt->execute([$student_id, $class_id, $date, $status, $teacher_id, $active_academic_year['id'], $active_term['id']]);
    }

    header("Location: attendance.php?class_id=$class_id&date=$date&success=1");
    exit;
}

include '../includes/teacher_header.php';
?>

<div class="attendance-container">
    <h1>📅 Attendance Management</h1>

    <?php if (isset($_GET['success']) && $_GET['success'] == 1): ?>
        <div class="success-message">Attendance saved successfully!</div>
    <?php endif; ?>

    <form method="GET" class="filter-form">
        <label for="class_id">Select Class:</label>
        <select name="class_id" id="class_id" required onchange="this.form.submit()">
            <option value="">-- Select Class --</option>
            <?php foreach ($classes as $class): ?>
                <option value="<?php echo $class['id']; ?>" <?php if ($class['id'] == $selected_class_id) echo 'selected'; ?>>
                    <?php echo htmlspecialchars($class['name']); ?>
                </option>
            <?php endforeach; ?>
        </select>

        <label for="date">Select Date:</label>
        <input type="date" name="date" id="date" value="<?php echo htmlspecialchars($selected_date); ?>" required onchange="this.form.submit()">
    </form>

    <?php if ($selected_class_id): ?>
        <form method="POST" class="attendance-form">
            <input type="hidden" name="class_id" value="<?php echo htmlspecialchars($selected_class_id); ?>">
            <input type="hidden" name="date" value="<?php echo htmlspecialchars($selected_date); ?>">

            <table class="attendance-table">
                <thead>
                    <tr>
                        <th>Student Name</th>
                        <th>Present</th>
                        <th>Absent</th>
                        <th>Late</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($students as $student): 
                        $status = $attendance_records[$student['id']]['status'] ?? '';
                    ?>
                    <tr>
                        <td><?php echo htmlspecialchars($student['first_name'] . ' ' . $student['last_name']); ?></td>
                        <td><input type="radio" name="attendance[<?php echo $student['id']; ?>]" value="present" <?php if ($status === 'present') echo 'checked'; ?> required></td>
                        <td><input type="radio" name="attendance[<?php echo $student['id']; ?>]" value="absent" <?php if ($status === 'absent') echo 'checked'; ?>></td>
                        <td><input type="radio" name="attendance[<?php echo $student['id']; ?>]" value="late" <?php if ($status === 'late') echo 'checked'; ?>></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>

            <button type="submit" class="btn-save">Save Attendance</button>
        </form>
    <?php endif; ?>
</div>

<style>
.attendance-container {
    max-width: 900px;
    margin: 20px auto;
    padding: 20px;
    background: white;
    border-radius: 12px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.05);
}

.filter-form {
    display: flex;
    gap: 20px;
    align-items: center;
    margin-bottom: 20px;
}

.filter-form label {
    font-weight: 600;
    color: #2c3e50;
}

.filter-form select,
.filter-form input[type="date"] {
    padding: 8px 12px;
    border-radius: 6px;
    border: 1px solid #ccc;
    font-size: 1em;
}

.attendance-table {
    width: 100%;
    border-collapse: collapse;
    margin-bottom: 20px;
}

.attendance-table th,
.attendance-table td {
    border: 1px solid #ddd;
    padding: 12px 15px;
    text-align: center;
}

.attendance-table th {
    background-color: #f8f9fa;
    font-weight: 700;
    color: #2c3e50;
}

.btn-save {
    background-color: #3498db;
    color: white;
    padding: 12px 25px;
    border: none;
    border-radius: 8px;
    font-size: 1.1em;
    cursor: pointer;
    transition: background-color 0.3s ease;
}

.btn-save:hover {
    background-color: #2980b9;
}

.success-message {
    background-color: #d4edda;
    color: #155724;
    padding: 12px;
    border: 1px solid #c3e6cb;
    border-radius: 6px;
    margin-bottom: 20px;
    font-weight: 600;
}
</style>

<?php
include '../includes/teacher_footer.php';
?>
