<?php
ob_start();
require_once '../config.php';

if (!isset($_SESSION['user_id']) || $_SESSION['role'] != 'student') {
    header('Location: ../login.php');
    exit;
}

$user_id = $_SESSION['user_id'];

// Debug: Log session and user info
error_log("View Grades - Session user_id: " . ($user_id ?? 'NULL'));
error_log("View Grades - Session role: " . ($_SESSION['role'] ?? 'NULL'));

// Fetch student information
$student_info = $pdo->prepare("SELECT s.*, s.class as class_name FROM students s WHERE s.user_id = ?");
$student_info->execute([$user_id]);
$student = $student_info->fetch();

if (!$student) {
    // Handle case where student not found
    error_log("View Grades - Student not found for user_id: $user_id");
    echo "Student not found.";
    exit;
}

$student_id = $student['id']; // Use students.id for further queries
$class_name = $student['class_name'] ?? 'N/A';

// Get class level_id to determine if preschool
$level_query = $pdo->prepare("SELECT level_id FROM classes WHERE name = ?");
$level_query->execute([$class_name]);
$level_info = $level_query->fetch();
$level_id = $level_info['level_id'] ?? 1;
$is_preschool = ($level_id == 3);

error_log("View Grades - Student ID: $student_id, Name: " . trim($student['first_name'] . ' ' . ($student['middle_name'] ?? '') . ' ' . $student['last_name']));

// Get filter parameters
$selected_academic_year = $_GET['academic_year'] ?? '';
$selected_term = $_GET['term'] ?? '';

// Get available academic years and terms for filters
$academic_years_query = $pdo->query("SELECT DISTINCT year FROM academic_years ORDER BY year DESC");
$available_academic_years = $academic_years_query->fetchAll(PDO::FETCH_COLUMN);

$terms_query = $pdo->query("SELECT id, name FROM terms ORDER BY id");
$available_terms = $terms_query->fetchAll(PDO::FETCH_ASSOC);

// Build query with filters based on level
if ($is_preschool) {
    // Query for preschool grades
    $query = "
        SELECT
            pg.exercise1,
            pg.homework1,
            pg.exam_score,
            pg.final_score,
            pg.grade,
            pg.comments,
            pg.created_at as graded_at,
            pg.term_id,
            pg.academic_year_id,
            pg.enrollment_id,
            s.name as subject_name,
            s.code as subject_code,
            c.name as course_name,
            CONCAT(t.first_name, ' ', t.last_name) as teacher_name,
            COALESCE(cl.name, st.class) as class_name,
            ay.year as academic_year,
            tm.name as term_name
        FROM preschool_grades pg
        JOIN enrollments e ON pg.enrollment_id = e.id
        JOIN courses c ON e.course_id = c.id
        LEFT JOIN students st ON e.student_id = st.id
        LEFT JOIN classes cl ON e.class_id = cl.id
        LEFT JOIN subjects s ON c.subject_id = s.id
        LEFT JOIN staff t ON c.teacher_id = t.id
        LEFT JOIN academic_years ay ON pg.academic_year_id = ay.id
        LEFT JOIN terms tm ON pg.term_id = tm.id
        WHERE e.student_id = ?
    ";

    $params = [$student_id];

    if (!empty($selected_academic_year)) {
        $query .= " AND ay.year = ?";
        $params[] = $selected_academic_year;
    }

    if (!empty($selected_term)) {
        $query .= " AND pg.term_id = ?";
        $params[] = $selected_term;
    }

    $query .= " ORDER BY pg.enrollment_id, pg.created_at DESC";
} else {
    // Query for regular grades
    $query = "
        SELECT
            g.converted_class_mark,
            g.converted_exam_mark,
            g.converted_final_mark,
            g.assessment_total,
            g.exam_score,
            g.final_score,
            g.comments,
            g.graded_at,
            g.term_id,
            g.academic_year_id,
            g.enrollment_id,
            s.name as subject_name,
            s.code as subject_code,
            c.name as course_name,
            CONCAT(t.first_name, ' ', t.last_name) as teacher_name,
            COALESCE(cl.name, st.class) as class_name,
            ay.year as academic_year,
            tm.name as term_name
        FROM grades g
        JOIN enrollments e ON g.enrollment_id = e.id
        JOIN courses c ON e.course_id = c.id
        LEFT JOIN students st ON e.student_id = st.id
        LEFT JOIN classes cl ON e.class_id = cl.id
        LEFT JOIN subjects s ON c.subject_id = s.id
        LEFT JOIN staff t ON c.teacher_id = t.id
        LEFT JOIN academic_years ay ON g.academic_year_id = ay.id
        LEFT JOIN terms tm ON g.term_id = tm.id
        WHERE e.student_id = ?
    ";

    $params = [$student_id];

    if (!empty($selected_academic_year)) {
        $query .= " AND ay.year = ?";
        $params[] = $selected_academic_year;
    }

    if (!empty($selected_term)) {
        $query .= " AND g.term_id = ?";
        $params[] = $selected_term;
    }

    $query .= " ORDER BY g.enrollment_id, g.graded_at DESC";
}

try {
    $grades_query = $pdo->prepare($query);
    $grades_query->execute($params);
    $all_grades = $grades_query->fetchAll();

    // Process to get only the latest grade per enrollment
    $grades = [];
    $processed_enrollments = [];
    foreach ($all_grades as $grade) {
        $enrollment_id = $grade['enrollment_id'];
        if (!in_array($enrollment_id, $processed_enrollments)) {
            $grades[] = $grade;
            $processed_enrollments[] = $enrollment_id;
        }
    }

    // Log successful query execution
    error_log("Grades query executed successfully for student ID: $student_id");
    error_log("Number of grades found: " . count($grades));

    // Debug: Show grades data temporarily
    if (!empty($grades)) {
        error_log("Sample grade data: " . json_encode($grades[0]));
    }

} catch (PDOException $e) {
    // Log the error
    error_log("Error executing grades query for student ID $student_id: " . $e->getMessage());
    $grades = [];

    // Display the actual error for debugging
    echo "<div class='error-message'>Unable to load grades at this time. Error: " . htmlspecialchars($e->getMessage()) . "</div>";
}

include '../includes/student_header.php';
?>



<div class="grades-container">
    <div class="page-header">
        <h1>📊 My Grades</h1>
        <p>Track your academic performance and progress</p>
    </div>



    <!-- Filters and Print Button -->
    <div class="filters-section">
        <form method="GET" class="filters-form">
            <div class="filter-group">
                <label for="academic_year">Academic Year:</label>
                <select name="academic_year" id="academic_year">
                    <option value="">All Years</option>
                    <?php foreach ($available_academic_years as $year): ?>
                        <option value="<?php echo htmlspecialchars((string) ($year ?? '')); ?>" <?php echo $selected_academic_year == $year ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars((string) ($year ?? '')); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div class="filter-group">
                <label for="term">Term:</label>
                <select name="term" id="term">
                    <option value="">All Terms</option>
                    <?php foreach ($available_terms as $term): ?>
                        <option value="<?php echo htmlspecialchars($term['id']); ?>" <?php echo $selected_term == $term['id'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($term['name']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div class="filter-actions">
                <button type="submit" class="filter-btn">🔍 Filter</button>
                <a href="view_grades.php" class="clear-btn">🗑️ Clear</a>
                <button type="button" onclick="window.open('print_results.php?academic_year=<?php echo urlencode($selected_academic_year); ?>&term=<?php echo urlencode($selected_term); ?>', '_blank')" class="print-btn">🖨️ Print</button>
            </div>
        </form>
    </div>

    <!-- Grades Table -->
    <?php if (!empty($grades)): ?>
        <?php
        // Calculate overall cumulative statistics
        $total_grades_overall = count($grades);
        $total_obtained_overall = 0;
        foreach ($grades as $grade) {
            $total_obtained_overall += $is_preschool ? ($grade['final_score'] ?? 0) : ($grade['converted_final_mark'] ?? 0);
        }
        $total_expected_overall = $total_grades_overall * 100;
        $cumulative_average = $total_expected_overall > 0 ? ($total_obtained_overall / $total_expected_overall) * 100 : 0;

        $grades_by_group = [];
        foreach ($grades as $grade) {
            $key = ($grade['class_name'] ?? 'N/A') . ' - ' . ($grade['academic_year'] ?? 'N/A') . ' - ' . ($grade['term_name'] ?? 'N/A');
            $grades_by_group[$key][] = $grade;
        }
        foreach ($grades_by_group as $group_key => $group_grades): ?>
            <div class="grades-section">
                <h3><?php echo htmlspecialchars($group_key); ?> Grades</h3>
                <div class="grades-table-container">
                    <table class="grades-table">
                        <thead>
                            <tr>
                        <th><strong>Course Subject</strong></th>
                        <?php if ($is_preschool): ?>
                            <th><strong>Exercise</strong></th>
                            <th><strong>Homework</strong></th>
                            <th><strong>Exam Score</strong></th>
                            <th><strong>Final Score</strong></th>
                            <th><strong>Grade</strong></th>
                        <?php else: ?>
                            <th><strong>Class Mark</strong></th>
                            <th><strong>Exam Mark</strong></th>
                            <th><strong>Final Mark</strong></th>
                        <?php endif; ?>
                        <th><strong>Comments</strong></th>
                        <th><strong>Date</strong></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($group_grades as $grade): ?>
                                <tr>
                                    <td>
                                        <div class="subject-info">
                                            <?php if ($is_preschool): ?>
                                                <?php echo htmlspecialchars($grade['course_name'] ?? ''); ?>
                                                <small><?php echo htmlspecialchars($grade['subject_code'] ?? ''); ?></small>
                                            <?php else: ?>
                                                <?php echo htmlspecialchars($grade['course_name'] ?? ''); ?> <?php echo htmlspecialchars($grade['subject_name'] ?? ''); ?>
                                                <small><?php echo htmlspecialchars($grade['subject_code'] ?? ''); ?></small>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                    <?php if ($is_preschool): ?>
                                        <td><?php echo htmlspecialchars($grade['exercise1'] ?? ''); ?></td>
                                        <td><?php echo htmlspecialchars($grade['homework1'] ?? ''); ?></td>
                                        <td><?php echo htmlspecialchars($grade['exam_score'] ?? ''); ?></td>
                                        <td><?php echo htmlspecialchars($grade['final_score'] ?? ''); ?></td>
                                        <td><?php echo htmlspecialchars((string) ($grade['grade'] ?? '')); ?></td>
                                    <?php else: ?>
                                        <td><?php echo htmlspecialchars($grade['converted_class_mark'] ?? ''); ?></td>
                                        <td><?php echo htmlspecialchars($grade['converted_exam_mark'] ?? ''); ?></td>
                                        <td><?php echo htmlspecialchars($grade['converted_final_mark'] ?? ''); ?></td>
                                    <?php endif; ?>
                                    <td><?php echo htmlspecialchars($grade['comments'] ?? ''); ?></td>
                                    <td><?php echo date('M d, Y', strtotime($grade['graded_at'])); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <!-- Performance Summary for this term -->
                <?php
                // Calculate performance statistics for this group
                $total_grades_group = count($group_grades);
                $total_expected_group = $total_grades_group * 100;
                $total_marks_obtained_group = 0;
                $subject_scores = [];
                $last_graded_group = null;

                foreach ($group_grades as $grade) {
                    $final_mark = $is_preschool ? ($grade['final_score'] ?? 0) : ($grade['converted_final_mark'] ?? 0);
                    $total_marks_obtained_group += $final_mark;
                    $subject = !empty($grade['subject_name']) ? $grade['subject_name'] : $grade['course_name'];
                    if (!isset($subject_scores[$subject])) {
                        $subject_scores[$subject] = [];
                    }
                    $subject_scores[$subject][] = $final_mark;
                    if ($last_graded_group === null || strtotime($grade['graded_at']) > strtotime($last_graded_group)) {
                        $last_graded_group = $grade['graded_at'];
                    }
                }

                $average_score_group = $total_grades_group > 0 ? $total_marks_obtained_group / $total_grades_group : 0;

                // Calculate subject averages for strengths and weaknesses
                $subject_averages = [];
                foreach ($subject_scores as $subject => $scores) {
                    $subject_averages[$subject] = array_sum($scores) / count($scores);
                }
                arsort($subject_averages); // Sort by score descending
                $strength_subject = !empty($subject_averages) ? array_key_first($subject_averages) : 'N/A';
                $weakness_subject = !empty($subject_averages) ? array_key_last($subject_averages) : 'N/A';

                // Determine performance level
                if ($average_score_group >= 90) {
                    $performance_level_group = 'Excellent';
                    $performance_class_group = 'excellent';
                } elseif ($average_score_group >= 80) {
                    $performance_level_group = 'Very Good';
                    $performance_class_group = 'very-good';
                } elseif ($average_score_group >= 70) {
                    $performance_level_group = 'Good';
                    $performance_class_group = 'good';
                } elseif ($average_score_group >= 60) {
                    $performance_level_group = 'Satisfactory';
                    $performance_class_group = 'satisfactory';
                } else {
                    $performance_level_group = 'Needs Improvement';
                    $performance_class_group = 'needs-improvement';
                }
                ?>

                <div class="performance-summary">
                    <h4>📊 Term Performance Summary</h4>
                    <div class="stats-grid">
                        <div class="stat-card">
                            <h5>Total Expected Marks</h5>
                            <p class="value"><?php echo $total_expected_group; ?></p>
                        </div>
                        <div class="stat-card">
                            <h5>Total Marks Obtained</h5>
                            <p class="value"><?php echo $total_marks_obtained_group; ?></p>
                        </div>
                        <div class="stat-card">
                            <h5>Average Score</h5>
                            <p class="value"><?php echo number_format($average_score_group, 1); ?>%</p>
                        </div>
                        <div class="stat-card">
                            <h5>Cumulative Average</h5>
                            <p class="value"><?php echo number_format($cumulative_average, 1); ?>%</p>
                        </div>
                        <div class="stat-card">
                            <h5>Area of Strength</h5>
                            <p class="value"><?php echo htmlspecialchars($strength_subject); ?></p>
                        </div>
                        <div class="stat-card">
                            <h5>Area of Weakness</h5>
                            <p class="value"><?php echo htmlspecialchars($weakness_subject); ?></p>
                        </div>
                    </div>
                </div>
            </div>
        <?php endforeach; ?>
    <?php endif; ?>

    <?php if (empty($grades)): ?>
        <div class="no-grades">
            <div class="no-grades-icon">📝</div>
            <h3>No Grades Available</h3>
            <p>You haven't received any grades yet. Grades will appear here once your teachers submit them.</p>
        </div>
    <?php endif; ?>

    <!-- Performance Summary Section -->
    <?php if (!empty($grades)): ?>
        <?php
        // Calculate performance statistics
        $total_grades = count($grades);
        $total_marks_obtained = 0;
        $total_expected_marks = $total_grades * 100; // Assuming 100 is max per grade
        $last_graded = null;

        foreach ($grades as $grade) {
            $final_mark = $is_preschool ? ($grade['final_score'] ?? 0) : ($grade['converted_final_mark'] ?? 0);
            $total_marks_obtained += $final_mark;
            if ($last_graded === null || strtotime($grade['graded_at']) > strtotime($last_graded)) {
                $last_graded = $grade['graded_at'];
            }
        }

        $average_score = $total_grades > 0 ? $total_marks_obtained / $total_grades : 0;

        // Determine performance level
        if ($average_score >= 90) {
            $performance_level = 'Excellent';
            $performance_class = 'excellent';
        } elseif ($average_score >= 80) {
            $performance_level = 'Very Good';
            $performance_class = 'very-good';
        } elseif ($average_score >= 70) {
            $performance_level = 'Good';
            $performance_class = 'good';
        } elseif ($average_score >= 60) {
            $performance_level = 'Satisfactory';
            $performance_class = 'satisfactory';
        } else {
            $performance_level = 'Needs Improvement';
            $performance_class = 'needs-improvement';
        }
        ?>

        <div class="performance-summary">
            <h3>📊 Performance Summary</h3>
            <div class="stats-grid">
                <div class="stat-card">
                    <h4>Total Grades</h4>
                    <p class="value"><?php echo $total_grades; ?></p>
                </div>
                <div class="stat-card">
                    <h4>Average Score</h4>
                    <p class="value"><?php echo number_format($average_score, 1); ?>%</p>
                </div>
                <div class="stat-card">
                    <h4 class="<?php echo $performance_class; ?>">Performance Level</h4>
                    <p class="value <?php echo $performance_class; ?>"><?php echo $performance_level; ?></p>
                </div>
                <div class="stat-card">
                    <h4>Last Graded</h4>
                    <p class="value"><?php echo $last_graded ? date('M d, Y', strtotime($last_graded)) : 'Never'; ?></p>
                </div>
            </div>
        </div>
    <?php endif; ?>
</div>

<?php
function getGradeClass($grade) {
    if ($grade >= 90) return 'excellent';
    if ($grade >= 80) return 'good';
    if ($grade >= 70) return 'average';
    if ($grade >= 60) return 'below-average';
    return 'poor';
}

function getGradeStatus($grade) {
    if ($grade >= 90) return 'Excellent';
    if ($grade >= 80) return 'Good';
    if ($grade >= 70) return 'Satisfactory';
    if ($grade >= 60) return 'Needs Improvement';
    return 'Poor';
}

function getGradeStatusClass($grade) {
    if ($grade >= 90) return 'excellent';
    if ($grade >= 80) return 'good';
    if ($grade >= 70) return 'satisfactory';
    if ($grade >= 60) return 'needs-improvement';
    return 'poor';
}
?>

<style>
.grades-container {
    padding: 20px;
}

.page-header h1 {
    margin-bottom: 5px;
    color: #2c3e50;
}

.page-header p {
    color: #7f8c8d;
    margin-bottom: 30px;
}

.filters-section {
    background: #fff;
    border-radius: 12px;
    padding: 20px;
    margin-bottom: 30px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.05);
    border: 1px solid #e9ecef;
}

.filters-form {
    display: flex;
    flex-wrap: wrap;
    gap: 20px;
    align-items: end;
}

.filter-group {
    display: flex;
    flex-direction: column;
    min-width: 200px;
}

.filter-group label {
    font-weight: 600;
    color: #2c3e50;
    margin-bottom: 8px;
    font-size: 0.9em;
}

.filter-group select {
    padding: 10px 12px;
    border: 2px solid #e9ecef;
    border-radius: 8px;
    font-size: 1em;
    background: #fff;
    transition: border-color 0.3s ease;
}

.filter-group select:focus {
    outline: none;
    border-color: #3498db;
    box-shadow: 0 0 0 3px rgba(52, 152, 219, 0.1);
}

.filter-actions {
    display: flex;
    gap: 10px;
    align-items: center;
}

.filter-btn, .clear-btn, .print-btn {
    padding: 10px 20px;
    border: none;
    border-radius: 8px;
    font-size: 0.9em;
    font-weight: 600;
    cursor: pointer;
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    gap: 5px;
    transition: all 0.3s ease;
}

.filter-btn {
    background: #3498db;
    color: white;
}

.filter-btn:hover {
    background: #2980b9;
    transform: translateY(-1px);
}

.clear-btn {
    background: #95a5a6;
    color: white;
}

.clear-btn:hover {
    background: #7f8c8d;
    transform: translateY(-1px);
}

.print-btn {
    background: #27ae60;
    color: white;
}

.print-btn:hover {
    background: #229954;
    transform: translateY(-1px);
}



.grades-section {
    background: #fff;
    border-radius: 12px;
    padding: 25px;
    margin-bottom: 25px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.05);
}

.grades-section h3 {
    margin: 0 0 20px 0;
    color: #2c3e50;
    border-bottom: 2px solid #3498db;
    padding-bottom: 10px;
}

.grades-table-container {
    overflow-x: auto;
}

.grades-table {
    width: 100%;
    border-collapse: collapse;
    margin: 0;
}

.grades-table th {
    background: #f8f9fa;
    color: #2c3e50;
    padding: 15px;
    text-align: left;
    font-weight: 500;
    border-bottom: 2px solid #e9ecef;
}

.grades-table td {
    padding: 15px;
    border-bottom: 1px solid #e9ecef;
    vertical-align: middle;
}

.subject-info strong {
    display: block;
    color: #2c3e50;
}

.subject-info small {
    color: #7f8c8d;
    font-size: 0.85em;
}

.grade-score .score {
    font-size: 1.2em;
    font-weight: bold;
    padding: 5px 10px;
    border-radius: 20px;
    display: inline-block;
}

.grade-score .score.excellent {
    background: #d4edda;
    color: #155724;
}

.grade-score .score.good {
    background: #d1ecf1;
    color: #0c5460;
}

.grade-score .score.average {
    background: #fff3cd;
    color: #856404;
}

.grade-score .score.below-average {
    background: #f8d7da;
    color: #721c24;
}

.grade-score .score.poor {
    background: #f5c6cb;
    color: #721c24;
}

.grade-status {
    padding: 4px 8px;
    border-radius: 12px;
    font-size: 0.85em;
    font-weight: 500;
}

.grade-status.excellent {
    background: #d4edda;
    color: #155724;
}

.grade-status.good {
    background: #d1ecf1;
    color: #0c5460;
}

.grade-status.satisfactory {
    background: #fff3cd;
    color: #856404;
}

.grade-status.needs-improvement {
    background: #f8d7da;
    color: #721c24;
}

.grade-status.poor {
    background: #f5c6cb;
    color: #721c24;
}

.no-grades {
    text-align: center;
    background: #fff;
    border-radius: 12px;
    padding: 60px 20px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.05);
}

.no-grades-icon {
    font-size: 4em;
    margin-bottom: 20px;
    opacity: 0.5;
}

.no-grades h3 {
    margin: 0 0 10px 0;
    color: #2c3e50;
}

.no-grades p {
    margin: 0;
    color: #7f8c8d;
    max-width: 400px;
    margin: 0 auto;
}

/* Mobile Responsive */
@media (max-width: 768px) {
    .grades-container {
        padding: 15px;
    }

    .page-header h1 {
        font-size: 1.8em;
    }



    .grades-section {
        padding: 20px;
    }

    .grades-table-container {
        margin: 0 -10px;
    }

    .grades-table th,
    .grades-table td {
        padding: 10px 8px;
        font-size: 0.9em;
    }

    .subject-info strong {
        font-size: 0.95em;
    }

    .grade-score .score {
        font-size: 1em;
        padding: 4px 8px;
    }

    .no-grades {
        padding: 40px 15px;
    }

    .no-grades-icon {
        font-size: 3em;
    }
}

@media (max-width: 480px) {
    .grades-container {
        padding: 10px;
    }

    .page-header h1 {
        font-size: 1.5em;
    }



    .grades-section {
        padding: 15px;
    }

    .grades-table th,
    .grades-table td {
        padding: 8px 6px;
        font-size: 0.85em;
    }

    .grades-table th:nth-child(5),
    .grades-table td:nth-child(5) {
        display: none;
    }

    .no-grades {
        padding: 30px 10px;
    }

    .no-grades h3 {
        font-size: 1.2em;
    }
}

/* Performance Summary Styles */
.performance-summary {
    background: #f8f9fa;
    border-radius: 8px;
    padding: 20px;
    margin-top: 20px;
    border: 1px solid #e9ecef;
}

.performance-summary h3 {
    margin: 0 0 20px 0;
    color: #2c3e50;
    border-bottom: 2px solid #3498db;
    padding-bottom: 10px;
    font-size: 1.1em;
}

.performance-summary h4 {
    margin: 0 0 15px 0;
    color: #2c3e50;
    font-size: 1em;
    font-weight: 600;
}

.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 15px;
}

.stat-card {
    background: #fff;
    border-radius: 6px;
    padding: 15px;
    text-align: center;
    border: 1px solid #e9ecef;
    transition: transform 0.3s ease, box-shadow 0.3s ease;
}

.stat-card:hover {
    transform: translateY(-1px);
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
}

.stat-card h4 {
    margin: 0 0 8px 0;
    color: #2c3e50;
    font-size: 0.85em;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.stat-card h5 {
    margin: 0 0 8px 0;
    color: #2c3e50;
    font-size: 0.8em;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.stat-card .value {
    font-size: 1.5em;
    font-weight: bold;
    color: #3498db;
    margin: 0;
}

.stat-card .value.excellent {
    color: #27ae60;
}

.stat-card .value.very-good {
    color: #2ecc71;
}

.stat-card .value.good {
    color: #f39c12;
}

.stat-card .value.satisfactory {
    color: #e67e22;
}

.stat-card .value.needs-improvement {
    color: #e74c3c;
}

.stat-card h4.excellent,
.stat-card h5.excellent {
    color: #27ae60;
}

.stat-card h4.very-good,
.stat-card h5.very-good {
    color: #2ecc71;
}

.stat-card h4.good,
.stat-card h5.good {
    color: #f39c12;
}

.stat-card h4.satisfactory,
.stat-card h5.satisfactory {
    color: #e67e22;
}

.stat-card h4.needs-improvement,
.stat-card h5.needs-improvement {
    color: #e74c3c;
}

/* Mobile responsive for performance summary */
@media (max-width: 768px) {
    .performance-summary {
        padding: 15px;
        margin-top: 15px;
    }

    .stats-grid {
        grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
        gap: 12px;
    }

    .stat-card {
        padding: 12px;
    }

    .stat-card .value {
        font-size: 1.3em;
    }
}

@media (max-width: 480px) {
    .performance-summary {
        padding: 12px;
    }

    .stats-grid {
        grid-template-columns: 1fr;
        gap: 8px;
    }

    .stat-card {
        padding: 10px;
    }

    .stat-card .value {
        font-size: 1.2em;
    }
}


</style>

<?php
include '../includes/student_footer.php';
?>
