<?php
require_once '../config.php';

if (!isset($_SESSION['user_id']) || $_SESSION['role'] != 'student') {
    header('Location: ../login.php');
    exit;
}

$username = $_SESSION['username'];
$user_id = $_SESSION['user_id'];

// Fetch student information
$student_info = $pdo->prepare("SELECT * FROM students WHERE user_id = ?");
$student_info->execute([$user_id]);
$student = $student_info->fetch();

if (!$student) {
    // Handle case where student not found
    echo "Student not found.";
    exit;
}

$student_id = $student['id']; // Use students.id for further queries

// Get student's grades statistics
$grades_stats = $pdo->prepare("
    SELECT
        COUNT(g.grade) as total_grades,
        AVG(g.grade) as average_grade,
        MAX(g.grade) as highest_grade,
        MIN(g.grade) as lowest_grade
    FROM enrollments e
    LEFT JOIN grades g ON e.id = g.enrollment_id
    WHERE e.student_id = ?
");
$grades_stats->execute([$student_id]);
$grades = $grades_stats->fetch();

// Get student's courses
$courses = $pdo->prepare("
    SELECT c.*, s.name as subject_name, s.code as subject_code
    FROM enrollments e
    JOIN courses c ON e.course_id = c.id
    LEFT JOIN subjects s ON c.subject_id = s.id
    WHERE e.student_id = ?
");
$courses->execute([$student_id]);
$student_courses = $courses->fetchAll();

// Get student's fee information
$fees_info = $pdo->prepare("
    SELECT
        COUNT(*) as total_fees,
        SUM(CASE WHEN paid = TRUE THEN amount ELSE 0 END) as paid_amount,
        SUM(CASE WHEN paid = FALSE THEN amount ELSE 0 END) as outstanding_amount,
        SUM(amount) as total_amount
    FROM student_fees
    WHERE student_id = ?
");
$fees_info->execute([$student_id]);
$fees = $fees_info->fetch();



include '../includes/student_header.php';
?>

<div class="dashboard-container">
    <!-- Welcome Header -->
    <div class="welcome-section">
        <div class="welcome-content">
            <h1><i class="fas fa-graduation-cap"></i> Welcome back, <?php echo htmlspecialchars(trim($student['first_name'] . ' ' . ($student['middle_name'] ?? '') . ' ' . $student['last_name'])); ?>!</h1>
            <p>Track your academic journey and stay on top of your studies</p>
        </div>

    </div>

    <!-- Main Dashboard Grid -->
    <div class="dashboard-grid">
        <!-- Student Profile Card -->
        <div class="dashboard-card profile-card">
            <div class="card-header">
                <i class="fas fa-user-circle"></i>
                <h3>Student Profile</h3>
            </div>
            <div class="card-content">
                <div class="profile-info">
                    <div class="profile-avatar">
                        <i class="fas fa-user-graduate"></i>
                    </div>
                    <div class="profile-details">
                        <h4><?php echo htmlspecialchars(trim($student['first_name'] . ' ' . ($student['middle_name'] ?? '') . ' ' . $student['last_name'])); ?></h4>
                        <p><i class="fas fa-id-card"></i> <?php echo htmlspecialchars($student['student_id'] ?? 'N/A'); ?></p>
                        <p><i class="fas fa-school"></i> <?php echo htmlspecialchars($student['class']); ?></p>
                        <p><i class="fas fa-calendar-alt"></i> <?php echo htmlspecialchars($student['academic_year']); ?></p>
                    </div>
                </div>
            </div>
        </div>

        <div class="dashboard-card announcement-card">
            <div class="card-header">
                <i class="fas fa-bullhorn"></i>
                <h3>Academy News</h3>
            </div>
            <div class="card-content">
                <div class="announcement-content">
                    <div class="announcement-icon">
                        <i class="fas fa-graduation-cap"></i>
                    </div>
                    <div class="announcement-text">
                        <h4>Welcome to Great Victory Academy</h4>
                        <p>We're committed to providing quality education and nurturing your potential for a bright future.</p>
                        <div class="announcement-meta">
                            <span><i class="fas fa-calendar"></i> <?php echo date('M d, Y'); ?></span>
                        </div>
                    </div>
                </div>
            </div>
        </div>

 

        <!-- Fee Status -->
        <div class="dashboard-card fees-card <?php echo ($fees['outstanding_amount'] ?? 0) > 0 ? 'warning' : 'success'; ?>">
            <div class="card-header">
                <i class="fas fa-money-bill-wave"></i>
                <h3>Fee Status</h3>
            </div>
            <div class="card-content">
                <div class="fee-summary">
                    <div class="fee-item">
                        <span class="fee-label">Total Fees:</span>
                        <span class="fee-amount">GH₵<?php echo number_format($fees['total_amount'] ?? 0, 2); ?></span>
                    </div>
                    <div class="fee-item">
                        <span class="fee-label">Paid:</span>
                        <span class="fee-amount paid">GH₵<?php echo number_format($fees['paid_amount'] ?? 0, 2); ?></span>
                    </div>
                    <div class="fee-item">
                        <span class="fee-label">Outstanding:</span>
                        <span class="fee-amount <?php echo ($fees['outstanding_amount'] ?? 0) > 0 ? 'outstanding' : ''; ?>">
                            GH₵<?php echo number_format($fees['outstanding_amount'] ?? 0, 2); ?>
                        </span>
                    </div>
                </div>
                <?php if (($fees['outstanding_amount'] ?? 0) > 0): ?>
                <div class="fee-alert">
                    <i class="fas fa-exclamation-triangle"></i>
                    <span>This is a friendly reminder that your school fee account has an outstanding balance. We kindly request payment to avoid any interruption to academic participation or school services.</span>
                </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Academic Performance -->
        <div class="dashboard-card performance-card">
            <div class="card-header">
                <i class="fas fa-chart-line"></i>
                <h3>Academic Performance</h3>
            </div>
            <div class="card-content">
                <div class="performance-stats">
                    <div class="stat-item">
                        <div class="stat-circle" style="background: conic-gradient(#4CAF50 0% <?php echo $grades['average_grade'] ?? 0; ?>%, #e0e0e0 <?php echo $grades['average_grade'] ?? 0; ?>% 100%)">
                            <div class="stat-inner">
                                <span class="stat-value"><?php echo number_format($grades['average_grade'] ?? 0, 1); ?>%</span>
                                <span class="stat-label">Average</span>
                            </div>
                        </div>
                    </div>
                    <div class="stat-details">
                        <div class="detail-row">
                            <span>Highest Grade:</span>
                            <span class="highlight"><?php echo $grades['highest_grade'] ?? 'N/A'; ?></span>
                        </div>
                        <div class="detail-row">
                            <span>Lowest Grade:</span>
                            <span class="highlight"><?php echo $grades['lowest_grade'] ?? 'N/A'; ?></span>
                        </div>
                    </div>
                </div>
                <div class="progress-bar">
                    <div class="progress-fill" style="width: <?php echo min(100, $grades['average_grade'] ?? 0); ?>%"></div>
                </div>
            </div>
        </div>

        <!-- Quick Actions -->
        <div class="dashboard-card actions-card">
            <div class="card-header">
                <i class="fas fa-bolt"></i>
                <h3>Quick Actions</h3>
            </div>
            <div class="card-content">
                <div class="actions-grid">
                    <a href="view_grades.php" class="action-btn">
                        <i class="fas fa-chart-bar"></i>
                        <span>View Grades</span>
                    </a>
                    <a href="view_profile.php" class="action-btn">
                        <i class="fas fa-user-edit"></i>
                        <span>Edit Profile</span>
                    </a>
                    <a href="#" class="action-btn">
                        <i class="fas fa-calendar-alt"></i>
                        <span>Class Schedule</span>
                    </a>
                    <a href="#" class="action-btn">
                        <i class="fas fa-credit-card"></i>
                        <span>Pay Fees</span>
                    </a>
                    <a href="#" class="action-btn">
                        <i class="fas fa-envelope"></i>
                        <span>Messages</span>
                    </a>
                    <a href="#" class="action-btn">
                        <i class="fas fa-cog"></i>
                        <span>Settings</span>
                    </a>
                </div>
            </div>
        </div>

        <!-- Academy News/Announcement -->

    </div>
</div>

<style>
/* Modern Dashboard Styles */
:root {
    --primary-color: #4CAF50;
    --secondary-color: #2196F3;
    --accent-color: #FF9800;
    --warning-color: #FF5722;
    --success-color: #4CAF50;
    --text-primary: #2c3e50;
    --text-secondary: #7f8c8d;
    --bg-light: #f8f9fa;
    --bg-white: #ffffff;
    --border-color: #e9ecef;
    --shadow: 0 2px 10px rgba(0,0,0,0.1);
    --shadow-hover: 0 4px 20px rgba(0,0,0,0.15);
    --border-radius: 12px;
    --transition: all 0.3s ease;
}

.dashboard-container {
    max-width: 1400px;
    margin: 0 auto;
    padding: 20px;
    background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
    min-height: 100vh;
}

/* Welcome Section */
.welcome-section {
    background: linear-gradient(135deg, var(--secondary-color) 0%, #1976D2 100%);
    color: white;
    border-radius: var(--border-radius);
    padding: 30px;
    margin-bottom: 30px;
    display: flex;
    justify-content: space-between;
    align-items: center;
    box-shadow: var(--shadow);
}

.welcome-content h1 {
    margin: 0 0 10px 0;
    font-size: 2.2em;
    font-weight: 600;
}

.welcome-content h1 i {
    margin-right: 15px;
    color: #BBDEFB;
}

.welcome-content p {
    margin: 0;
    opacity: 0.9;
    font-size: 1.1em;
}

.welcome-stats {
    display: flex;
    gap: 30px;
}

.stat-mini {
    text-align: center;
    background: rgba(255,255,255,0.1);
    padding: 15px;
    border-radius: 10px;
    backdrop-filter: blur(10px);
}

.stat-number {
    display: block;
    font-size: 1.8em;
    font-weight: bold;
    margin-bottom: 5px;
}

.stat-label {
    font-size: 0.9em;
    opacity: 0.8;
}

/* Dashboard Grid */
.dashboard-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(400px, 1fr));
    gap: 25px;
}

/* Dashboard Cards */
.dashboard-card {
    background: var(--bg-white);
    border-radius: var(--border-radius);
    box-shadow: var(--shadow);
    overflow: hidden;
    transition: var(--transition);
    border: 1px solid var(--border-color);
}

.dashboard-card:hover {
    transform: translateY(-5px);
    box-shadow: var(--shadow-hover);
}

.card-header {
    background: linear-gradient(135deg, #667eea 0%, #1976D2 100%);
    color: white;
    padding: 20px;
    display: flex;
    align-items: center;
    gap: 15px;
}

.card-header i {
    font-size: 1.5em;
    opacity: 0.9;
}

.card-header h3 {
    margin: 0;
    font-size: 1.3em;
    font-weight: 600;
}

.card-content {
    padding: 25px;
}

/* Profile Card */
.profile-info {
    display: flex;
    align-items: center;
    gap: 20px;
}

.profile-avatar {
    font-size: 4em;
    color: var(--secondary-color);
    background: var(--bg-light);
    padding: 20px;
    border-radius: 50%;
    border: 3px solid var(--secondary-color);
}

.profile-details h4 {
    margin: 0 0 15px 0;
    color: var(--text-primary);
    font-size: 1.4em;
}

.profile-details p {
    margin: 8px 0;
    color: var(--text-secondary);
    display: flex;
    align-items: center;
    gap: 10px;
}

.profile-details p i {
    color: var(--secondary-color);
    width: 16px;
}

/* Performance Card */
.performance-stats {
    display: flex;
    align-items: center;
    gap: 30px;
    margin-bottom: 20px;
}

.stat-item {
    flex: 1;
}

.stat-circle {
    width: 120px;
    height: 120px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    position: relative;
    margin: 0 auto 15px auto;
}

.stat-inner {
    width: 90px;
    height: 90px;
    border-radius: 50%;
    background: white;
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    box-shadow: inset 0 0 10px rgba(0,0,0,0.1);
}

.stat-value {
    font-size: 1.4em;
    font-weight: bold;
    color: var(--text-primary);
}

.stat-label {
    font-size: 0.8em;
    color: var(--text-secondary);
    margin-top: 2px;
}

.stat-details {
    flex: 1;
}

.detail-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 8px 0;
    border-bottom: 1px solid var(--border-color);
}

.detail-row:last-child {
    border-bottom: none;
}

.detail-row span:first-child {
    color: var(--text-secondary);
    font-weight: 500;
}

.highlight {
    color: var(--primary-color);
    font-weight: bold;
}

.progress-bar {
    width: 100%;
    height: 8px;
    background: var(--bg-light);
    border-radius: 4px;
    overflow: hidden;
    margin-top: 15px;
}

.progress-fill {
    height: 100%;
    background: linear-gradient(90deg, var(--primary-color) 0%, #66BB6A 100%);
    border-radius: 4px;
    transition: width 0.3s ease;
}

/* Fees Card */
.fees-card.warning {
    border-left: 4px solid var(--warning-color);
}

.fees-card.success {
    border-left: 4px solid var(--success-color);
}

.fee-summary {
    margin-bottom: 20px;
}

.fee-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 12px 0;
    border-bottom: 1px solid var(--border-color);
}

.fee-item:last-child {
    border-bottom: none;
}

.fee-label {
    font-weight: 500;
    color: var(--text-primary);
}

.fee-amount {
    font-weight: bold;
    color: var(--text-primary);
}

.fee-amount.paid {
    color: var(--success-color);
}

.fee-amount.outstanding {
    color: var(--warning-color);
}

.fee-alert {
    background: #FFF3E0;
    border: 1px solid #FFCC02;
    border-radius: 8px;
    padding: 15px;
    display: flex;
    align-items: center;
    gap: 12px;
    color: #E65100;
}

.fee-alert i {
    color: var(--warning-color);
    font-size: 1.2em;
}

/* Courses Card */
.courses-list {
    display: flex;
    flex-direction: column;
    gap: 15px;
}

.course-item {
    display: flex;
    align-items: center;
    gap: 15px;
    padding: 15px;
    background: var(--bg-light);
    border-radius: 8px;
    border: 1px solid var(--border-color);
    transition: var(--transition);
}

.course-item:hover {
    background: #e8f5e8;
    border-color: var(--primary-color);
}

.course-icon {
    font-size: 2em;
    color: var(--primary-color);
    background: white;
    padding: 12px;
    border-radius: 50%;
    border: 2px solid var(--primary-color);
}

.course-info {
    flex: 1;
}

.course-info h5 {
    margin: 0 0 5px 0;
    color: var(--text-primary);
    font-size: 1.1em;
}

.course-info p {
    margin: 0;
    color: var(--text-secondary);
    font-size: 0.9em;
}

.course-progress {
    text-align: center;
    min-width: 80px;
}

.progress-bar.small {
    width: 60px;
    height: 6px;
    margin: 0 auto 5px auto;
}

.course-progress span {
    font-size: 0.8em;
    color: var(--text-secondary);
    font-weight: 500;
}

.no-courses {
    text-align: center;
    color: var(--text-secondary);
    padding: 40px 20px;
}

.no-courses i {
    font-size: 3em;
    margin-bottom: 15px;
    opacity: 0.5;
}

/* Actions Card */
.actions-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(140px, 1fr));
    gap: 15px;
}

.action-btn {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 10px;
    padding: 20px 15px;
    background: var(--bg-light);
    border: 2px solid var(--border-color);
    border-radius: 10px;
    text-decoration: none;
    color: var(--text-primary);
    transition: var(--transition);
    text-align: center;
}

.action-btn:hover {
    border-color: var(--secondary-color);
    background: #E3F2FD;
    color: var(--secondary-color);
    transform: translateY(-2px);
}

.action-btn i {
    font-size: 2em;
    color: var(--secondary-color);
}

.action-btn span {
    font-size: 0.9em;
    font-weight: 500;
}

/* Announcement Card */
.announcement-content {
    display: flex;
    align-items: flex-start;
    gap: 20px;
}

.announcement-icon {
    font-size: 3em;
    color: var(--primary-color);
    background: var(--bg-light);
    padding: 20px;
    border-radius: 50%;
    flex-shrink: 0;
}

.announcement-text h4 {
    margin: 0 0 10px 0;
    color: var(--text-primary);
    font-size: 1.2em;
}

.announcement-text p {
    margin: 0 0 15px 0;
    color: var(--text-secondary);
    line-height: 1.6;
}

.announcement-meta {
    color: var(--text-secondary);
    font-size: 0.9em;
}

.announcement-meta span {
    display: flex;
    align-items: center;
    gap: 5px;
}

/* Responsive Design */
@media (max-width: 1200px) {
    .dashboard-grid {
        grid-template-columns: repeat(auto-fit, minmax(350px, 1fr));
        gap: 20px;
    }

    .welcome-section {
        flex-direction: column;
        text-align: center;
        gap: 20px;
    }

    .welcome-stats {
        justify-content: center;
    }
}

@media (max-width: 992px) {
    .dashboard-grid {
        grid-template-columns: repeat(auto-fit, minmax(320px, 1fr));
        gap: 18px;
    }

    .performance-stats {
        flex-direction: column;
        gap: 20px;
        text-align: center;
    }

    .stat-details {
        width: 100%;
    }

    .actions-grid {
        grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
        gap: 12px;
    }

    .action-btn {
        padding: 15px 10px;
    }

    .action-btn i {
        font-size: 1.6em;
    }

    .action-btn span {
        font-size: 0.85em;
    }
}

@media (max-width: 768px) {
    .dashboard-container {
        padding: 15px;
    }

    .dashboard-grid {
        grid-template-columns: 1fr;
        gap: 15px;
    }

    .welcome-section {
        padding: 20px;
        margin-bottom: 20px;
        margin-top: 100px;
    }

    .welcome-content h1 {
        font-size: 1.8em;
    }

    .welcome-content p {
        font-size: 1em;
    }

    .card-header {
        padding: 15px;
    }

    .card-header h3 {
        font-size: 1.1em;
    }

    .card-content {
        padding: 20px;
    }

    .profile-info {
        flex-direction: column;
        text-align: center;
        gap: 15px;
    }

    .profile-avatar {
        font-size: 3em;
        padding: 15px;
    }

    .profile-details h4 {
        font-size: 1.2em;
    }

    .announcement-content {
        flex-direction: column;
        text-align: center;
        gap: 15px;
    }

    .announcement-icon {
        font-size: 2.5em;
        padding: 15px;
    }

    .performance-stats {
        gap: 15px;
    }

    .stat-circle {
        width: 100px;
        height: 100px;
    }

    .stat-inner {
        width: 75px;
        height: 75px;
    }

    .stat-value {
        font-size: 1.2em;
    }

    .actions-grid {
        grid-template-columns: repeat(2, 1fr);
        gap: 10px;
    }

    .action-btn {
        padding: 12px 8px;
    }

    .action-btn i {
        font-size: 1.4em;
    }

    .action-btn span {
        font-size: 0.8em;
    }

    .fee-alert {
        flex-direction: column;
        text-align: center;
        gap: 8px;
    }

    .fee-alert i {
        align-self: center;
    }
}

@media (max-width: 576px) {
    .dashboard-container {
        padding: 10px;
    }

    .welcome-section {
        padding: 15px;
        margin-bottom: 15px;
    }

    .welcome-content h1 {
        font-size: 1.5em;
    }

    .welcome-stats {
        flex-direction: column;
        gap: 15px;
    }

    .stat-mini {
        width: 100%;
    }

    .dashboard-grid {
        gap: 12px;
    }

    .card-content {
        padding: 15px;
    }

    .profile-avatar {
        font-size: 2.5em;
        padding: 12px;
    }

    .profile-details h4 {
        font-size: 1.1em;
    }

    .profile-details p {
        font-size: 0.9em;
        justify-content: center;
    }

    .announcement-icon {
        font-size: 2em;
        padding: 12px;
    }

    .announcement-text h4 {
        font-size: 1.1em;
    }

    .stat-circle {
        width: 80px;
        height: 80px;
    }

    .stat-inner {
        width: 60px;
        height: 60px;
    }

    .stat-value {
        font-size: 1em;
    }

    .actions-grid {
        grid-template-columns: repeat(2, 1fr);
        gap: 8px;
    }

    .action-btn {
        padding: 10px 6px;
    }

    .action-btn i {
        font-size: 1.2em;
    }

    .action-btn span {
        font-size: 0.75em;
    }

    .fee-item {
        flex-direction: column;
        align-items: flex-start;
        gap: 5px;
    }

    .fee-amount {
        font-size: 1.1em;
    }
}

@media (max-width: 480px) {
    .welcome-content h1 {
        font-size: 1.3em;
    }

    .welcome-content p {
        font-size: 0.9em;
    }

    .card-header {
        padding: 12px;
    }

    .card-header h3 {
        font-size: 1em;
    }

    .card-content {
        padding: 12px;
    }

    .profile-details h4 {
        font-size: 1em;
    }

    .announcement-text h4 {
        font-size: 1em;
    }

    .stat-circle {
        width: 70px;
        height: 70px;
    }

    .stat-inner {
        width: 50px;
        height: 50px;
    }

    .stat-value {
        font-size: 0.9em;
    }

    .actions-grid {
        grid-template-columns: 1fr;
        gap: 6px;
    }

    .action-btn {
        flex-direction: row;
        justify-content: flex-start;
        padding: 12px;
        gap: 12px;
    }

    .action-btn i {
        font-size: 1.4em;
        flex-shrink: 0;
    }

    .action-btn span {
        font-size: 0.85em;
        text-align: left;
    }
}

<?php
include '../includes/student_footer.php';
?>

<script>
// Override the mobile menu functionality for dashboard - runs after all other scripts
document.addEventListener('DOMContentLoaded', function() {
    // Longer delay to ensure all other scripts have run, including the 1000ms delay in mobile_menu.js
    setTimeout(function() {
        const mobileMenuToggle = document.getElementById('mobile-menu-toggle');
        const sidebar = document.querySelector('.student-sidebar');

        if (mobileMenuToggle && sidebar) {
            // Force remove any existing event listeners by replacing the element completely
            const toggleContainer = mobileMenuToggle.parentNode;
            const newToggle = mobileMenuToggle.cloneNode(true);
            toggleContainer.replaceChild(newToggle, mobileMenuToggle);

            // Add fresh event listener with high priority
            newToggle.addEventListener('click', function(e) {
                e.preventDefault();
                e.stopImmediatePropagation(); // Stop all other listeners
                sidebar.classList.toggle('open');

                // Prevent body scroll when sidebar is open
                if (sidebar.classList.contains('open')) {
                    document.body.style.overflow = 'hidden';
                } else {
                    document.body.style.overflow = '';
                }
            }, true); // Use capture phase

            // Close sidebar when clicking outside
            document.addEventListener('click', function(e) {
                if (sidebar.classList.contains('open') && !sidebar.contains(e.target) && !newToggle.contains(e.target)) {
                    sidebar.classList.remove('open');
                    document.body.style.overflow = '';
                }
            }, true); // Use capture phase

            // Close sidebar on escape key
            document.addEventListener('keydown', function(e) {
                if (e.key === 'Escape' && sidebar.classList.contains('open')) {
                    sidebar.classList.remove('open');
                    document.body.style.overflow = '';
                }
            }, true); // Use capture phase
        }
    }, 1200); // Delay longer than mobile_menu.js (1000ms + buffer)
});
</script>
