<?php
require_once '../config.php';

if (!isset($_SESSION['user_id']) || $_SESSION['role'] != 'student') {
    header('Location: ../login.php');
    exit;
}

$user_id = $_SESSION['user_id'];

// Fetch student information
$student_info = $pdo->prepare("SELECT * FROM students WHERE user_id = ?");
$student_info->execute([$user_id]);
$student = $student_info->fetch();

if (!$student) {
    // Handle case where student not found
    echo "Student not found.";
    exit;
}

$student_id = $student['id']; // Use students.id for further queries

// Get filter parameters
$selected_academic_year = $_GET['academic_year'] ?? '';
$selected_term = $_GET['term'] ?? '';

// Fetch student info for header
$student_name = '';
$class_name = '';
$school_name = '';
$school_logo_url = '';

try {
    // Fetch school settings with correct column names
    $stmt = $pdo->prepare("SELECT school_name, logo_path, address, phone, email FROM school_settings LIMIT 1");
    $stmt->execute();
    $school_info = $stmt->fetch();
    if ($school_info) {
        $school_name = htmlspecialchars($school_info['school_name']);
        $school_address = htmlspecialchars($school_info['address'] ?? '');
        $school_phone = htmlspecialchars($school_info['phone'] ?? '');
        $school_email = htmlspecialchars($school_info['email'] ?? '');
        $logo_path = $school_info['logo_path'];
        // Adjust logo path for the student directory
        if ($logo_path && !preg_match('/^https?:\/\//', $logo_path) && !preg_match('/^\//', $logo_path)) {
            $school_logo_url = '../' . $logo_path;
        } else {
            $school_logo_url = $logo_path;
        }
        $school_logo_url = htmlspecialchars($school_logo_url);
    }
} catch (Exception $e) {
    // Log error or handle as needed
}

try {
    // Fetch student info directly from students table by id
    $stmt = $pdo->prepare("SELECT CONCAT(first_name, ' ', COALESCE(middle_name, ''), ' ', last_name) as name, class FROM students WHERE id = ?");
    $stmt->execute([$student_id]);
    $student_info = $stmt->fetch();
    if ($student_info) {
        $student_name = htmlspecialchars($student_info['name']);
        $class_name = htmlspecialchars($student_info['class']);
    }
} catch (Exception $e) {
    // Log error or handle as needed
}

// Get available academic years and terms for filters (if needed)
$academic_years_query = $pdo->query("SELECT DISTINCT year FROM academic_years ORDER BY year DESC");
$available_academic_years = $academic_years_query->fetchAll(PDO::FETCH_COLUMN);

$terms_query = $pdo->query("SELECT DISTINCT id FROM terms ORDER BY id");
$available_terms = $terms_query->fetchAll(PDO::FETCH_COLUMN);

// Build query with filters
$query = "
    SELECT
        g.converted_class_mark,
        g.converted_exam_mark,
        g.converted_final_mark,
        g.assessment_total,
        g.exam_score,
        g.final_score,
        g.comments,
        g.graded_at,
        g.term_id,
        g.academic_year_id,
        g.grade,
        s.name as subject_name,
        s.code as subject_code,
        c.name as course_name,
        CONCAT(t.first_name, ' ', t.last_name) as teacher_name,
        cl.name as class_name,
        ay.year as academic_year,
        tm.name as term_name
    FROM grades g
    JOIN enrollments e ON g.enrollment_id = e.id
    JOIN courses c ON e.course_id = c.id
    LEFT JOIN students st ON e.student_id = st.id
    LEFT JOIN classes cl ON e.class_id = cl.id
    LEFT JOIN subjects s ON c.subject_id = s.id
    LEFT JOIN staff t ON c.teacher_id = t.id
    LEFT JOIN academic_years ay ON g.academic_year_id = ay.id
    LEFT JOIN terms tm ON g.term_id = tm.id
    WHERE e.student_id = ?
    AND g.graded_at = (SELECT MAX(graded_at) FROM grades WHERE enrollment_id = e.id)
";

$params = [$student_id];

if (!empty($selected_academic_year)) {
    $query .= " AND ay.year = ?";
    $params[] = $selected_academic_year;
}

if (!empty($selected_term)) {
    $query .= " AND g.term_id = ?";
    $params[] = $selected_term;
}

// Query for filtered grades
$query .= " ORDER BY g.graded_at DESC";

try {
    $grades_query = $pdo->prepare($query);
    $grades_query->execute($params);
    $grades = $grades_query->fetchAll();

    // Log successful query execution
    error_log("Grades query executed successfully for student ID: $student_id");

} catch (PDOException $e) {
    // Log the error
    error_log("Error executing grades query for student ID $student_id: " . $e->getMessage());
    $grades = [];

    // You could also display a user-friendly error message
    echo "<div class='error-message'>Unable to load grades at this time. Please try again later.</div>";
}

// Query for all grades to calculate cumulative average
$all_grades_query = "
    SELECT g.converted_final_mark
    FROM grades g
    JOIN enrollments e ON g.enrollment_id = e.id
    WHERE e.student_id = ?
    AND g.graded_at = (SELECT MAX(graded_at) FROM grades WHERE enrollment_id = e.id)
";

try {
    $all_grades_stmt = $pdo->prepare($all_grades_query);
    $all_grades_stmt->execute([$student_id]);
    $all_grades = $all_grades_stmt->fetchAll(PDO::FETCH_COLUMN);

    // Calculate cumulative average across all grades
    $cumulative_average = 0;
    if (!empty($all_grades)) {
        $total_marks = array_sum($all_grades);
        $total_count = count($all_grades);
        $cumulative_average = $total_count > 0 ? ($total_marks / $total_count) : 0;
    }
} catch (PDOException $e) {
    $cumulative_average = 0;
    error_log("Error calculating cumulative average: " . $e->getMessage());
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Print Grades</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            font-size: 12px;
            line-height: 1.4;
            color: #000;
            background: #fff;
            margin: 0;
            padding: 20px;
        }

        .print-header {
            text-align: center;
            margin-bottom: 30px;
            border: 2px solid #2c3e50;
            padding: 20px;
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            border-radius: 10px;
            box-shadow: 0 4px 6px rgba(0,0,0,0.1);
        }

        .header-top {
            display: flex;
            align-items: center;
            justify-content: center;
            margin-bottom: 20px;
        }

        .school-logo {
            max-height: 80px;
            margin-right: 20px;
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.2);
        }

        .school-info h1 {
            margin: 0;
            font-size: 28px;
            font-weight: bold;
            color: #2c3e50;
            text-shadow: 1px 1px 2px rgba(0,0,0,0.1);
        }

        .school-tagline {
            margin: 5px 0 0 0;
            font-size: 14px;
            color: #7f8c8d;
            font-style: italic;
        }

        .header-details {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 10px;
            margin-bottom: 20px;
            background: #fff;
            padding: 15px;
            border-radius: 8px;
            border: 1px solid #dee2e6;
        }

        .detail-row {
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .label {
            font-weight: bold;
            color: #495057;
            font-size: 12px;
        }

        .value {
            color: #2c3e50;
            font-size: 12px;
        }

        .report-title {
            border-top: 2px solid #2c3e50;
            padding-top: 15px;
        }

        .report-title h2 {
            margin: 0;
            font-size: 20px;
            font-weight: bold;
            color: #2c3e50;
            text-transform: uppercase;
            letter-spacing: 1px;
        }

        .grades-section {
            margin-bottom: 20px;
            page-break-inside: avoid;
        }

        .grades-section h3 {
            font-size: 16px;
            margin: 0 0 15px 0;
            padding: 10px;
            background: #f0f0f0;
            border-bottom: 1px solid #ccc;
            color: #000;
        }

        .grades-table {
            width: 100%;
            border-collapse: collapse;
            margin: 0;
            font-size: 11px;
        }

        .grades-table th {
            background: #f0f0f0;
            color: #000;
            padding: 8px 5px;
            border: 1px solid #ccc;
            font-weight: normal;
            text-align: left;
        }

        .grades-table td {
            padding: 6px 5px;
            border: 1px solid #ccc;
            vertical-align: top;
        }

        .subject-info strong {
            font-size: 11px;
            color: #000;
        }

        .subject-info small {
            font-size: 10px;
            color: #666;
        }

        .no-grades {
            text-align: center;
            padding: 40px 20px;
        }

        .no-grades h3 {
            margin: 0 0 10px 0;
            color: #000;
        }

        .no-grades p {
            margin: 0;
            color: #666;
        }

        @page {
            margin: 0.5in;
            size: A4;
        }

        .performance-summary {
            background: #f8f9fa;
            border-radius: 8px;
            padding: 15px;
            margin-top: 15px;
            border: 1px solid #e9ecef;
            page-break-inside: avoid;
        }

        .performance-summary h4 {
            margin: 0 0 15px 0;
            color: #2c3e50;
            font-size: 14px;
            font-weight: 600;
        }

        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 10px;
        }

        .stat-card {
            background: #fff;
            border-radius: 6px;
            padding: 10px;
            text-align: center;
            border: 1px solid #e9ecef;
        }

        .stat-card h5 {
            margin: 0 0 5px 0;
            color: #2c3e50;
            font-size: 10px;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .stat-card .value {
            font-size: 12px;
            font-weight: bold;
            color: #3498db;
            margin: 0;
        }

        @page {
            margin: 0.5in;
            size: A4;
        }

        @media print {
            body {
                padding: 0;
            }
        }
    </style>
</head>
<body>
    <div class="print-header">
        <div class="header-top">
            <img src="<?php echo $school_logo_url; ?>" alt="School Logo" class="school-logo" />
            <div class="school-info">
                <h1><?php echo $school_name; ?></h1>
                <p class="school-tagline">Academic Excellence Center</p>
            </div>
        </div>
        <div class="header-details">
            <div class="detail-row">
                <span class="label">Student Name:</span>
                <span class="value"><?php echo $student_name; ?></span>
            </div>
            <div class="detail-row">
                <span class="label">Class:</span>
                <span class="value"><?php echo $class_name; ?></span>
            </div>
            <div class="detail-row">
                <span class="label">Academic Year:</span>
                <span class="value"><?php echo $selected_academic_year ?: 'All Years'; ?></span>
            </div>
            <div class="detail-row">
                <span class="label">Report Date:</span>
                <span class="value"><?php echo date('F d, Y'); ?></span>
            </div>
        </div>
        <div class="report-title">
            <h2>Student Grade Report</h2>
        </div>
    </div>

    <?php if (!empty($grades)): ?>
        <?php
        $grades_by_group = [];
        foreach ($grades as $grade) {
            $key = ($grade['class_name'] ?? 'N/A') . ' - ' . ($grade['academic_year'] ?? 'N/A') . ' - ' . ($grade['term_name'] ?? 'N/A');
            $grades_by_group[$key][] = $grade;
        }
        foreach ($grades_by_group as $group_key => $group_grades): ?>
            <div class="grades-section">
                <h3><?php echo htmlspecialchars($group_key); ?> Grades</h3>
                <table class="grades-table">
                    <thead>
                        <tr>
                            <th>Course Subject</th>
                            <th>Class Mark</th>
                            <th>Exam Mark</th>
                            <th>Final Mark</th>
                            <th>Grade</th>
                            <th>Date</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($group_grades as $grade): ?>
                            <tr>
                                <td>
                                    <div class="subject-info">
                                        <strong><?php echo htmlspecialchars($grade['course_name']); ?> <?php echo htmlspecialchars($grade['subject_name']); ?></strong>
                                        <small><?php echo htmlspecialchars($grade['subject_code']); ?></small>
                                    </div>
                                </td>
                                <td><?php echo htmlspecialchars($grade['converted_class_mark'] ?? $grade['assessment_total']); ?></td>
                                <td><?php echo htmlspecialchars($grade['converted_exam_mark'] ?? $grade['exam_score']); ?></td>
                                <td><?php echo htmlspecialchars($grade['converted_final_mark'] ?? $grade['final_score']); ?></td>
                                <td><?php echo htmlspecialchars($grade['grade']); ?></td>
                                <td><?php echo date('M d, Y', strtotime($grade['graded_at'])); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>

                <!-- Performance Summary for this term -->
                <?php
                // Calculate performance statistics for this group
                $total_grades_group = count($group_grades);
                $total_expected_group = $total_grades_group * 100;
                $total_marks_obtained_group = 0;
                $subject_scores = [];
                $last_graded_group = null;

                foreach ($group_grades as $grade) {
                    $final_mark = $grade['converted_final_mark'] ?? 0;
                    $total_marks_obtained_group += $final_mark;
                    $subject = !empty($grade['subject_name']) ? $grade['subject_name'] : $grade['course_name'];
                    if (!isset($subject_scores[$subject])) {
                        $subject_scores[$subject] = [];
                    }
                    $subject_scores[$subject][] = $final_mark;
                    if ($last_graded_group === null || strtotime($grade['graded_at']) > strtotime($last_graded_group)) {
                        $last_graded_group = $grade['graded_at'];
                    }
                }

                $average_score_group = $total_grades_group > 0 ? $total_marks_obtained_group / $total_grades_group : 0;

                // Calculate subject averages for strengths and weaknesses
                $subject_averages = [];
                foreach ($subject_scores as $subject => $scores) {
                    $subject_averages[$subject] = array_sum($scores) / count($scores);
                }
                arsort($subject_averages); // Sort by score descending
                $strength_subject = !empty($subject_averages) ? array_key_first($subject_averages) : 'N/A';
                $weakness_subject = !empty($subject_averages) ? array_key_last($subject_averages) : 'N/A';

                // Determine performance level
                if ($average_score_group >= 90) {
                    $performance_level_group = 'Excellent';
                    $performance_class_group = 'excellent';
                } elseif ($average_score_group >= 80) {
                    $performance_level_group = 'Very Good';
                    $performance_class_group = 'very-good';
                } elseif ($average_score_group >= 70) {
                    $performance_level_group = 'Good';
                    $performance_class_group = 'good';
                } elseif ($average_score_group >= 60) {
                    $performance_level_group = 'Satisfactory';
                    $performance_class_group = 'satisfactory';
                } else {
                    $performance_level_group = 'Needs Improvement';
                    $performance_class_group = 'needs-improvement';
                }
                ?>

                <div class="performance-summary">
                    <h4>📊 Term Performance Summary</h4>
                    <div class="stats-grid">
                        <div class="stat-card">
                            <h5>Total Expected Marks</h5>
                            <p class="value"><?php echo $total_expected_group; ?></p>
                        </div>
                        <div class="stat-card">
                            <h5>Total Marks Obtained</h5>
                            <p class="value"><?php echo $total_marks_obtained_group; ?></p>
                        </div>
                        <div class="stat-card">
                            <h5>Average Score</h5>
                            <p class="value"><?php echo number_format($average_score_group, 1); ?>%</p>
                        </div>
                        <div class="stat-card">
                            <h5>Cumulative Average</h5>
                            <p class="value"><?php echo number_format($cumulative_average, 1); ?>%</p>
                        </div>
                        <div class="stat-card">
                            <h5>Area of Strength</h5>
                            <p class="value"><?php echo htmlspecialchars($strength_subject); ?></p>
                        </div>
                        <div class="stat-card">
                            <h5>Area of Weakness</h5>
                            <p class="value"><?php echo htmlspecialchars($weakness_subject); ?></p>
                        </div>
                    </div>
                </div>
            </div>
        <?php endforeach; ?>
    <?php else: ?>
        <div class="no-grades">
            <h3>No Grades Available</h3>
            <p>You haven't received any grades yet. Grades will appear here once your teachers submit them.</p>
        </div>
    <?php endif; ?>

    <script>
        window.onload = function() {
            window.print();
        };
    </script>
</body>
</html>
