<?php
require_once '../config.php';

// Check if user is logged in and is admin
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: ../login.php');
    exit();
}

$message = '';
$message_type = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['add_book'])) {
        $title = trim($_POST['title']);
        $author = trim($_POST['author']);
        $isbn = trim($_POST['isbn']);
        $category_id = intval($_POST['category_id']);
        $publisher = trim($_POST['publisher']);
        $publication_year = intval($_POST['publication_year']);
        $total_copies = intval($_POST['total_copies']);
        $location = trim($_POST['location']);
        $description = trim($_POST['description']);

        try {
            $stmt = $pdo->prepare("INSERT INTO library_books (title, author, isbn, category_id, publisher, publication_year, total_copies, available_copies, location, description, added_by) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
            $stmt->execute([$title, $author, $isbn, $category_id, $publisher, $publication_year, $total_copies, $total_copies, $location, $description, $_SESSION['user_id']]);
            $message = "Book added successfully!";
            $message_type = "success";
        } catch (PDOException $e) {
            $message = "Error adding book: " . $e->getMessage();
            $message_type = "error";
        }
    }
}

// Fetch categories
try {
    $categories_stmt = $pdo->query("SELECT * FROM library_categories ORDER BY name");
    $categories = $categories_stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $categories = [];
}

// Fetch books
try {
    $books_stmt = $pdo->query("SELECT lb.*, lc.name as category_name FROM library_books lb LEFT JOIN library_categories lc ON lb.category_id = lc.id ORDER BY lb.title");
    $books = $books_stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $books = [];
}

$page_title = "Library Management - Books";
include '../includes/library_header.php';
?>

<style>
/* Page Header */
.container-fluid h2 {
    color: #8e44ad;
    font-weight: 700;
    margin-bottom: 30px;
    font-size: 2em;
}

/* Alert Messages */
.alert {
    border-radius: 10px;
    border: none;
    box-shadow: 0 4px 6px rgba(0,0,0,0.1);
}

.alert-success {
    background-color: #d4edda;
    color: #155724;
}

.alert-danger {
    background-color: #f8d7da;
    color: #721c24;
}

/* Add Book Button */
.btn-primary {
    background: linear-gradient(135deg, #8e44ad 0%, #9b59b6 100%);
    border: none;
    border-radius: 25px;
    padding: 10px 20px;
    font-weight: 600;
    transition: all 0.3s ease;
    box-shadow: 0 4px 15px rgba(142, 68, 173, 0.3);
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(142, 68, 173, 0.4);
    background: linear-gradient(135deg, #9b59b6 0%, #8e44ad 100%);
}

.btn-primary i {
    margin-right: 8px;
}

/* Table Styles */
.table-responsive {
    border-radius: 10px;
    overflow: hidden;
    box-shadow: 0 4px 6px rgba(0,0,0,0.1);
    margin-top: 20px;
}

.library-books-table {
    margin-bottom: 0;
    background: white;
}

.library-books-table thead th {
    background: linear-gradient(135deg, #8e44ad 0%, #9b59b6 100%);
    color: white;
    font-weight: 600;
    border: none;
    padding: 15px;
    text-transform: uppercase;
    font-size: 0.85em;
    letter-spacing: 0.5px;
}

.library-books-table tbody tr {
    transition: all 0.3s ease;
}

.library-books-table tbody tr:hover {
    background-color: #f8f9ff;
    transform: scale(1.01);
}

.library-books-table tbody td {
    padding: 15px;
    border-bottom: 1px solid #f0f0f0;
    vertical-align: middle;
}

.library-books-table tbody tr:last-child td {
    border-bottom: none;
}

/* Modal Styles */
.modal-content {
    border-radius: 15px;
    border: none;
    box-shadow: 0 20px 40px rgba(0,0,0,0.15);
}

.modal-header {
    background: linear-gradient(135deg, #8e44ad 0%, #9b59b6 100%);
    color: white;
    border-radius: 15px 15px 0 0;
    border-bottom: none;
    padding: 20px 25px;
}

.modal-header .close {
    color: white;
    opacity: 0.8;
    font-size: 1.5em;
}

.modal-header .close:hover {
    opacity: 1;
}

.modal-title {
    font-weight: 700;
    font-size: 1.3em;
}

.modal-body {
    padding: 25px;
}

.modal-footer {
    border-top: 1px solid #e9ecef;
    padding: 20px 25px;
    border-radius: 0 0 15px 15px;
}

/* Form Styles */
.form-group label {
    font-weight: 600;
    color: #2c3e50;
    margin-bottom: 8px;
    font-size: 0.9em;
}

.form-control {
    border: 2px solid #e9ecef;
    border-radius: 8px;
    padding: 12px 15px;
    font-size: 0.95em;
    transition: all 0.3s ease;
    background: #f8f9fa;
}

.form-control:focus {
    border-color: #8e44ad;
    box-shadow: 0 0 0 3px rgba(142, 68, 173, 0.1);
    background: white;
}

.form-control:invalid {
    border-color: #e74c3c;
}

/* Select dropdown vertical alignment */
select.form-control {
    line-height: normal;
    vertical-align: middle;
    display: flex;
    align-items: center;
}

/* Secondary Button */
.btn-secondary {
    background: #6c757d;
    border: none;
    border-radius: 25px;
    padding: 10px 20px;
    font-weight: 600;
    transition: all 0.3s ease;
}

.btn-secondary:hover {
    background: #5a6268;
    transform: translateY(-1px);
}

/* Responsive Design */
@media (max-width: 768px) {
    .container-fluid h2 {
        font-size: 1.5em;
        margin-bottom: 20px;
    }

    .table-responsive {
        margin-top: 15px;
    }

    .library-books-table thead th,
    .library-books-table tbody td {
        padding: 10px;
        font-size: 0.85em;
    }

    .modal-dialog {
        margin: 10px;
    }

    .modal-body {
        padding: 20px;
    }

    .form-row .form-group {
        margin-bottom: 15px;
    }
}

@media (max-width: 480px) {
    .container-fluid {
        padding: 15px;
    }

    .btn-primary {
        width: 100%;
        margin-bottom: 15px;
    }

    .modal-body {
        padding: 15px;
    }

    .form-row {
        flex-direction: column;
    }

    .form-row .form-group {
        margin-bottom: 15px;
    }
}

/* Loading Animation */
@keyframes fadeIn {
    from {
        opacity: 0;
        transform: translateY(20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.container-fluid {
    animation: fadeIn 0.5s ease forwards;
}

/* Custom Scrollbar */
.table-responsive::-webkit-scrollbar {
    height: 8px;
}

.table-responsive::-webkit-scrollbar-track {
    background: #f1f1f1;
    border-radius: 4px;
}

.table-responsive::-webkit-scrollbar-thumb {
    background: linear-gradient(135deg, #8e44ad, #9b59b6);
    border-radius: 4px;
}

.table-responsive::-webkit-scrollbar-thumb:hover {
    background: linear-gradient(135deg, #9b59b6, #8e44ad);
}
</style>

<div class="container-fluid">
    <h2>Library Management - Books</h2>

    <?php if ($message): ?>
        <div class="alert alert-<?php echo $message_type === 'success' ? 'success' : 'danger'; ?> alert-dismissible fade show" role="alert">
            <?php echo htmlspecialchars($message); ?>
            <button type="button" class="close" data-dismiss="alert">
                <span>&times;</span>
            </button>
        </div>
    <?php endif; ?>

    <button class="btn btn-primary mb-3" data-toggle="modal" data-target="#addBookModal">
        <i class="fas fa-plus"></i> Add New Book
    </button>

    <div class="table-responsive">
            <table class="table table-striped table-hover library-books-table">
                <thead>
                    <tr>
                        <th>Title</th>
                        <th>Author</th>
                        <th>ISBN</th>
                        <th>Category</th>
                        <th>Publisher</th>
                        <th>Year</th>
                        <th>Total Copies</th>
                        <th>Available Copies</th>
                        <th>Location</th>
                    </tr>
                </thead>
            <tbody>
                <?php foreach ($books as $book): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($book['title']); ?></td>
                        <td><?php echo htmlspecialchars($book['author']); ?></td>
                        <td><?php echo htmlspecialchars($book['isbn']); ?></td>
                        <td><?php echo htmlspecialchars($book['category_name']); ?></td>
                        <td><?php echo htmlspecialchars($book['publisher']); ?></td>
                        <td><?php echo htmlspecialchars($book['publication_year']); ?></td>
                        <td><?php echo htmlspecialchars($book['total_copies']); ?></td>
                        <td><?php echo htmlspecialchars($book['available_copies']); ?></td>
                        <td><?php echo htmlspecialchars($book['location']); ?></td>
                    </tr>
                <?php endforeach; ?>
                <?php if (empty($books)): ?>
                    <tr>
                        <td colspan="9" class="text-center">No books found.</td>
                    </tr>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<!-- Add Book Modal -->
<div class="modal fade" id="addBookModal" tabindex="-1" role="dialog">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <form method="POST">
                <div class="modal-header">
                    <h5 class="modal-title">Add New Book</h5>
                    <button type="button" class="close" data-dismiss="modal">
                        <span>&times;</span>
                    </button>
                </div>
                <div class="modal-body">
                    <div class="form-row">
                        <div class="form-group col-md-6">
                            <label for="title">Title *</label>
                            <input type="text" class="form-control" id="title" name="title" required>
                        </div>
                        <div class="form-group col-md-6">
                            <label for="author">Author *</label>
                            <input type="text" class="form-control" id="author" name="author" required>
                        </div>
                    </div>
                    <div class="form-row">
                        <div class="form-group col-md-4">
                            <label for="isbn">ISBN</label>
                            <input type="text" class="form-control" id="isbn" name="isbn">
                        </div>
                        <div class="form-group col-md-4">
                            <label for="category_id">Category *</label>
                            <select class="form-control" id="category_id" name="category_id" required>
                                <option value="">Select Category</option>
                                <?php foreach ($categories as $category): ?>
                                    <option value="<?php echo $category['id']; ?>"><?php echo htmlspecialchars($category['name']); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="form-group col-md-4">
                            <label for="publisher">Publisher</label>
                            <input type="text" class="form-control" id="publisher" name="publisher">
                        </div>
                    </div>
                    <div class="form-row">
                        <div class="form-group col-md-4">
                            <label for="publication_year">Publication Year</label>
                            <input type="number" class="form-control" id="publication_year" name="publication_year" min="1900" max="<?php echo date('Y'); ?>">
                        </div>
                        <div class="form-group col-md-4">
                            <label for="total_copies">Total Copies *</label>
                            <input type="number" class="form-control" id="total_copies" name="total_copies" min="1" required>
                        </div>
                        <div class="form-group col-md-4">
                            <label for="location">Location</label>
                            <input type="text" class="form-control" id="location" name="location">
                        </div>
                    </div>
                    <div class="form-group">
                        <label for="description">Description</label>
                        <textarea class="form-control" id="description" name="description" rows="3"></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="submit" name="add_book" class="btn btn-primary">Add Book</button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php include '../includes/library_footer.php'; ?>
