<?php
require_once '../config.php';

// Check if user is logged in and is librarian or admin
if (!isset($_SESSION['user_id']) || !in_array($_SESSION['role'], ['librarian', 'admin'])) {
    header('Location: ../login.php');
    exit();
}

$message = '';
$message_type = '';

// Fetch overdue borrowings with fines
try {
    // Load library fine per day setting
    require_once 'settings.php';

    $stmt = $pdo->prepare("
        SELECT lb.id, lb.user_id, lb.book_id, lb.due_date, lb.returned_at, lb.status,
               bk.title, bk.author,
               u.username,
               DATEDIFF(CURDATE(), lb.due_date) as days_overdue,
               (DATEDIFF(CURDATE(), lb.due_date) * :fine_per_day) as fine_amount
        FROM library_borrowings lb
        JOIN library_books bk ON lb.book_id = bk.id
        JOIN users u ON lb.user_id = u.id
        WHERE lb.status = 'overdue' AND lb.due_date < CURDATE()
        ORDER BY lb.due_date ASC
    ");
    $stmt->execute(['fine_per_day' => $library_fine_per_day]);
    $overdue_fines = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $overdue_fines = [];
    $message = "Error loading overdue fines: " . $e->getMessage();
    $message_type = "error";
}

// Handle fine payment
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['pay_fine'])) {
    $borrowing_id = intval($_POST['borrowing_id']);
    $payment_amount = floatval($_POST['payment_amount']);

    try {
        // Validate borrowing exists and is overdue
        $stmt = $pdo->prepare("SELECT * FROM library_borrowings WHERE id = ? AND status = 'overdue'");
        $stmt->execute([$borrowing_id]);
        $borrowing = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$borrowing) {
            $message = "Invalid borrowing record or not overdue.";
            $message_type = "error";
        } elseif ($payment_amount <= 0) {
            $message = "Payment amount must be positive.";
            $message_type = "error";
        } else {
            // Record payment in library_fines_payments table (create if not exists)
            $pdo->beginTransaction();

            // Create fines payments table if not exists
            $pdo->exec("
                CREATE TABLE IF NOT EXISTS library_fines_payments (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    borrowing_id INT NOT NULL,
                    payment_amount DECIMAL(10,2) NOT NULL,
                    payment_date DATE NOT NULL,
                    FOREIGN KEY (borrowing_id) REFERENCES library_borrowings(id) ON DELETE CASCADE
                )
            ");

            $insert_payment = $pdo->prepare("INSERT INTO library_fines_payments (borrowing_id, payment_amount, payment_date) VALUES (?, ?, CURDATE())");
            $insert_payment->execute([$borrowing_id, $payment_amount]);

            // Optionally, update borrowing status if fine fully paid (not implemented here)

            $pdo->commit();

            $message = "Fine payment recorded successfully.";
            $message_type = "success";

            // Reload overdue fines
            $stmt->execute(['fine_per_day' => $library_fine_per_day]);
            $overdue_fines = $stmt->fetchAll(PDO::FETCH_ASSOC);
        }
    } catch (PDOException $e) {
        $pdo->rollBack();
        $message = "Error recording payment: " . $e->getMessage();
        $message_type = "error";
    }
}

$page_title = "Collect Fines - Library";
include '../includes/library_header.php';
?>

<div class="container-fluid">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h2><i class="fas fa-dollar-sign"></i> Collect Fines</h2>
        <div class="text-muted">
            Total Overdue Fines: <span class="badge badge-danger"><?php echo count($overdue_fines); ?></span>
        </div>
    </div>

    <?php if ($message): ?>
        <div class="alert alert-<?php echo $message_type === 'success' ? 'success' : 'danger'; ?> alert-dismissible fade show" role="alert">
            <?php echo htmlspecialchars($message); ?>
            <button type="button" class="close" data-dismiss="alert">
                <span>&times;</span>
            </button>
        </div>
    <?php endif; ?>

    <?php if (empty($overdue_fines)): ?>
        <div class="alert alert-success">
            <i class="fas fa-check-circle"></i> No overdue fines at this time.
        </div>
    <?php else: ?>
        <div class="table-responsive">
            <table class="table table-striped table-hover library-fines-table">
                <thead>
                    <tr>
                        <th>Book Title</th>
                        <th>Author</th>
                        <th>Borrower</th>
                        <th>Due Date</th>
                        <th>Days Overdue</th>
                        <th>Fine Amount</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($overdue_fines as $fine): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($fine['title']); ?></td>
                            <td><?php echo htmlspecialchars($fine['author']); ?></td>
                            <td><?php echo htmlspecialchars($fine['username']); ?></td>
                            <td><?php echo date('M d, Y', strtotime($fine['due_date'])); ?></td>
                            <td><?php echo $fine['days_overdue']; ?> days</td>
                            <td>₵<?php echo number_format($fine['fine_amount'], 2); ?></td>
                            <td>
                                <button class="btn btn-sm btn-primary" onclick="payFine(<?php echo $fine['id']; ?>, '<?php echo htmlspecialchars(addslashes($fine['title'])); ?>', <?php echo $fine['fine_amount']; ?>)">
                                    <i class="fas fa-credit-card"></i> Pay Fine
                                </button>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php endif; ?>
</div>

<!-- Pay Fine Modal -->
<div class="modal fade" id="payFineModal" tabindex="-1" role="dialog" aria-labelledby="payFineModalLabel" aria-hidden="true">
    <div class="modal-dialog" role="document">
        <form method="POST" class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="payFineModalLabel">Pay Fine</h5>
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <p>Pay fine for book: <strong id="fine-book-title"></strong></p>
                <input type="hidden" name="borrowing_id" id="fine_borrowing_id" value="">
                <div class="form-group">
                    <label for="payment_amount">Payment Amount (₵)</label>
                    <input type="number" step="0.01" min="0.01" max="9999" class="form-control" name="payment_amount" id="payment_amount" required>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                <button type="submit" name="pay_fine" class="btn btn-primary">Submit Payment</button>
            </div>
        </form>
    </div>
</div>

<script>
function payFine(borrowingId, bookTitle, fineAmount) {
    document.getElementById('fine_borrowing_id').value = borrowingId;
    document.getElementById('fine-book-title').textContent = bookTitle;
    document.getElementById('payment_amount').value = fineAmount.toFixed(2);
    $('#payFineModal').modal('show');
}
</script>

<?php include '../includes/library_footer.php'; ?>
