<?php
require_once '../config.php';

if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

if (!isset($_SESSION['user_id'])) {
    header('Location: ../login.php');
    exit;
}

require_once '../includes/enhanced_access_control_fixed.php';

// Check if user has permission to manage staff leaves
if (!hasPermission('staff_leave_management')) {
    header('Location: ../index.php');
    exit;
}

$page_title = 'Manage Staff Leaves - School Management System';

// Handle leave approval/rejection
$message = '';
$message_type = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $leave_id = $_POST['leave_id'] ?? 0;
    $action = $_POST['action'] ?? '';
    $admin_comments = trim($_POST['admin_comments'] ?? '');

    if ($leave_id && in_array($action, ['approve', 'reject'])) {
        try {
            $status = ($action === 'approve') ? 'approved' : 'rejected';

            $stmt = $pdo->prepare("
                UPDATE staff_leave_applications
                SET status = ?, admin_comments = ?, reviewed_by = ?, reviewed_at = NOW()
                WHERE id = ?
            ");

            $stmt->execute([$status, $admin_comments, $_SESSION['user_id'], $leave_id]);

            $message = "Leave application " . ($status === 'approved' ? 'approved' : 'rejected') . " successfully!";
            $message_type = 'success';

        } catch(PDOException $e) {
            $message = 'Error updating leave status: ' . $e->getMessage();
            $message_type = 'error';
        }
    }
}

// Get filter parameters
$status_filter = $_GET['status'] ?? 'all';
$role_filter = $_GET['role'] ?? 'all';
$search = $_GET['search'] ?? '';

// Build query
$query = "
    SELECT sla.*,
           u.username,
           CONCAT(s.first_name, ' ', s.last_name) as staff_name,
           reviewer.username as reviewer_name
    FROM staff_leave_applications sla
    LEFT JOIN users u ON sla.user_id = u.id
    LEFT JOIN (
        SELECT user_id, first_name, last_name FROM teachers
        UNION ALL
        SELECT user_id, first_name, last_name FROM librarians
        UNION ALL
        SELECT user_id, first_name, last_name FROM non_teaching
        UNION ALL
        SELECT user_id, first_name, last_name FROM directors
        UNION ALL
        SELECT user_id, first_name, last_name FROM head_teachers
    ) s ON sla.user_id = s.user_id
    LEFT JOIN users reviewer ON sla.reviewed_by = reviewer.id
    WHERE 1=1
";

$params = [];

if ($status_filter !== 'all') {
    $query .= " AND sla.status = ?";
    $params[] = $status_filter;
}

if ($role_filter !== 'all') {
    $query .= " AND sla.staff_role = ?";
    $params[] = $role_filter;
}

if ($search) {
    $query .= " AND (s.first_name LIKE ? OR s.last_name LIKE ? OR u.username LIKE ? OR sla.reason LIKE ?)";
    $search_param = "%$search%";
    $params = array_merge($params, [$search_param, $search_param, $search_param, $search_param]);
}

$query .= " ORDER BY sla.created_at DESC";

$stmt = $pdo->prepare($query);
$stmt->execute($params);
$leave_applications = $stmt->fetchAll();

// Get statistics
$stats_query = "
    SELECT
        COUNT(*) as total,
        SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
        SUM(CASE WHEN status = 'approved' THEN 1 ELSE 0 END) as approved,
        SUM(CASE WHEN status = 'rejected' THEN 1 ELSE 0 END) as rejected
    FROM staff_leave_applications
";

if ($role_filter !== 'all') {
    $stats_query .= " WHERE staff_role = ?";
    $stats_stmt = $pdo->prepare($stats_query);
    $stats_stmt->execute([$role_filter]);
} else {
    $stats_stmt = $pdo->prepare($stats_query);
    $stats_stmt->execute();
}

$stats = $stats_stmt->fetch();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .leave-card {
            transition: transform 0.2s;
            border: none;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }

        .leave-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 20px rgba(0,0,0,0.15);
        }

        .status-badge {
            padding: 5px 10px;
            border-radius: 20px;
            font-size: 0.8em;
            font-weight: 500;
        }

        .status-pending { background-color: #fff3cd; color: #856404; }
        .status-approved { background-color: #d4edda; color: #155724; }
        .status-rejected { background-color: #f8d7da; color: #721c24; }

        .leave-type-icon {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: bold;
        }

        .leave-type-sick { background-color: #e74c3c; }
        .leave-type-casual { background-color: #3498db; }
        .leave-type-maternity { background-color: #9b59b6; }
        .leave-type-paternity { background-color: #1abc9c; }
        .leave-type-emergency { background-color: #e67e22; }
        .leave-type-vacation { background-color: #27ae60; }
        .leave-type-other { background-color: #95a5a6; }

        .stats-card {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
        }

        .filter-section {
            background-color: #f8f9fa;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
        }

        .action-buttons {
            display: flex;
            gap: 10px;
        }

        .modal-content {
            border-radius: 15px;
        }

        .btn-outline-primary .fa-eye {
            color: white;
        }
    </style>
</head>
<body>
    <?php include '../includes/admin_header.php'; ?>

    <div class="container-fluid">
        <div class="row">

            <main class="col-md-9 ms-sm-2 col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2"><i class="fas fa-calendar-times"></i> Manage Staff Leaves</h1>
                </div>

                <?php if ($message): ?>
                    <div class="alert alert-<?php echo $message_type === 'success' ? 'success' : 'danger'; ?> alert-dismissible fade show" role="alert">
                        <?php echo $message; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <!-- Statistics -->
                <div class="row mb-4">
                    <div class="col-md-3">
                        <div class="stats-card">
                            <div class="d-flex justify-content-between">
                                <div>
                                    <h4><?php echo $stats['total']; ?></h4>
                                    <p class="mb-0">Total Applications</p>
                                </div>
                                <i class="fas fa-calendar-alt fa-2x opacity-75"></i>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stats-card">
                            <div class="d-flex justify-content-between">
                                <div>
                                    <h4><?php echo $stats['pending']; ?></h4>
                                    <p class="mb-0">Pending Review</p>
                                </div>
                                <i class="fas fa-clock fa-2x opacity-75"></i>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stats-card">
                            <div class="d-flex justify-content-between">
                                <div>
                                    <h4><?php echo $stats['approved']; ?></h4>
                                    <p class="mb-0">Approved</p>
                                </div>
                                <i class="fas fa-check-circle fa-2x opacity-75"></i>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="stats-card">
                            <div class="d-flex justify-content-between">
                                <div>
                                    <h4><?php echo $stats['rejected']; ?></h4>
                                    <p class="mb-0">Rejected</p>
                                </div>
                                <i class="fas fa-times-circle fa-2x opacity-75"></i>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Filters -->
                <div class="filter-section">
                    <form method="GET" class="row g-3">
                        <div class="col-md-3">
                            <label for="status" class="form-label">Status</label>
                            <select class="form-select" id="status" name="status">
                                <option value="all" <?php echo $status_filter === 'all' ? 'selected' : ''; ?>>All Status</option>
                                <option value="pending" <?php echo $status_filter === 'pending' ? 'selected' : ''; ?>>Pending</option>
                                <option value="approved" <?php echo $status_filter === 'approved' ? 'selected' : ''; ?>>Approved</option>
                                <option value="rejected" <?php echo $status_filter === 'rejected' ? 'selected' : ''; ?>>Rejected</option>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <label for="role" class="form-label">Staff Role</label>
                            <select class="form-select" id="role" name="role">
                                <option value="all" <?php echo $role_filter === 'all' ? 'selected' : ''; ?>>All Roles</option>
                                <option value="teacher" <?php echo $role_filter === 'teacher' ? 'selected' : ''; ?>>Teacher</option>
                                <option value="librarian" <?php echo $role_filter === 'librarian' ? 'selected' : ''; ?>>Librarian</option>
                                <option value="non_teaching" <?php echo $role_filter === 'non_teaching' ? 'selected' : ''; ?>>Non-Teaching</option>
                                <option value="director" <?php echo $role_filter === 'director' ? 'selected' : ''; ?>>Director</option>
                                <option value="head_teacher" <?php echo $role_filter === 'head_teacher' ? 'selected' : ''; ?>>Head Teacher</option>
                            </select>
                        </div>
                        <div class="col-md-4">
                            <label for="search" class="form-label">Search</label>
                            <input type="text" class="form-control" id="search" name="search"
                                   placeholder="Search by name, username, or reason..."
                                   value="<?php echo htmlspecialchars($search); ?>">
                        </div>
                        <div class="col-md-2 d-flex align-items-end">
                            <button type="submit" class="btn btn-primary me-2">
                                <i class="fas fa-filter"></i> Filter
                            </button>
                            <a href="manage_staff_leaves.php" class="btn btn-secondary">
                                <i class="fas fa-times"></i> Clear
                            </a>
                        </div>
                    </form>
                </div>

                <!-- Leave Applications Table -->
                <?php if (empty($leave_applications)): ?>
                    <div class="text-center py-5">
                        <i class="fas fa-calendar-times fa-3x text-muted mb-3"></i>
                        <h5 class="text-muted">No Leave Applications Found</h5>
                        <p class="text-muted">No leave applications match your current filters.</p>
                    </div>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-striped table-hover">
                            <thead class="table-dark">
                                <tr>
                                    <th>Staff Name</th>
                                    <th>Role</th>
                                    <th>Leave Type</th>
                                    <th>Start Date</th>
                                    <th>End Date</th>
                                    <th>Duration</th>
                                    <th>Status</th>
                                    <th>Applied Date</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($leave_applications as $application): ?>
                                    <tr>
                                        <td>
                                            <div class="d-flex align-items-center">
                                                <div class="leave-type-icon leave-type-<?php echo $application['leave_type']; ?> me-2">
                                                    <?php echo strtoupper(substr($application['leave_type'], 0, 1)); ?>
                                                </div>
                                                <div>
                                                    <strong><?php echo htmlspecialchars($application['staff_name']); ?></strong>
                                                    <br>
                                                    <small class="text-muted"><?php echo htmlspecialchars($application['username']); ?></small>
                                                </div>
                                            </div>
                                        </td>
                                        <td><?php echo ucfirst(str_replace('_', ' ', $application['staff_role'])); ?></td>
                                        <td><?php echo ucfirst($application['leave_type']); ?> Leave</td>
                                        <td><?php echo date('M j, Y', strtotime($application['start_date'])); ?></td>
                                        <td><?php echo date('M j, Y', strtotime($application['end_date'])); ?></td>
                                        <td>
                                            <?php
                                            $start = strtotime($application['start_date']);
                                            $end = strtotime($application['end_date']);
                                            $days = ceil(($end - $start) / (60 * 60 * 24)) + 1;
                                            echo $days . ' day' . ($days > 1 ? 's' : '');
                                            ?>
                                        </td>
                                        <td>
                                            <span class="status-badge status-<?php echo $application['status']; ?>">
                                                <?php echo ucfirst($application['status']); ?>
                                            </span>
                                        </td>
                                        <td><?php echo date('M j, Y', strtotime($application['created_at'])); ?></td>
                                        <td>
                                            <div class="btn-group" role="group">
                                                <button class="btn btn-outline-primary btn-sm" onclick="viewDetails(<?php echo $application['id']; ?>)" title="View Details">
                                                    <i class="fas fa-eye"></i>
                                                </button>
                                                <?php if ($application['status'] === 'pending'): ?>
                                                    <button class="btn btn-success btn-sm" onclick="reviewLeave(<?php echo $application['id']; ?>, 'approve')" title="Approve">
                                                        <i class="fas fa-check"></i>
                                                    </button>
                                                    <button class="btn btn-danger btn-sm" onclick="reviewLeave(<?php echo $application['id']; ?>, 'reject')" title="Reject">
                                                        <i class="fas fa-times"></i>
                                                    </button>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>

                    <!-- Table Footer with Summary -->
                    <div class="mt-3 text-muted">
                        <small>
                            Showing <?php echo count($leave_applications); ?> leave application(s)
                            <?php if ($status_filter !== 'all'): ?>
                                with status "<?php echo ucfirst($status_filter); ?>"
                            <?php endif; ?>
                            <?php if ($role_filter !== 'all'): ?>
                                for role "<?php echo ucfirst(str_replace('_', ' ', $role_filter)); ?>"
                            <?php endif; ?>
                            <?php if ($search): ?>
                                matching "<?php echo htmlspecialchars($search); ?>"
                            <?php endif; ?>
                        </small>
                    </div>
                <?php endif; ?>
            </main>
        </div>
    </div>

    <!-- Review Leave Modal -->
    <div class="modal fade" id="reviewLeaveModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Review Leave Application</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <input type="hidden" id="leave_id" name="leave_id">
                        <input type="hidden" id="action" name="action">

                        <div class="mb-3">
                            <label for="admin_comments" class="form-label">Comments (Optional)</label>
                            <textarea class="form-control" id="admin_comments" name="admin_comments" rows="3"
                                      placeholder="Add any comments for the staff member..."></textarea>
                        </div>

                        <div class="alert alert-info">
                            <strong>Action:</strong> <span id="action_text"></span>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn" id="confirm_button">Confirm</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- View Details Modal -->
    <div class="modal fade" id="viewDetailsModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Leave Application Details</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body" id="leaveDetailsContent">
                    <!-- Content will be loaded here -->
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function reviewLeave(leaveId, action) {
            document.getElementById('leave_id').value = leaveId;
            document.getElementById('action').value = action;

            const actionText = action === 'approve' ? 'Approve this leave application' : 'Reject this leave application';
            document.getElementById('action_text').textContent = actionText;

            const confirmButton = document.getElementById('confirm_button');
            confirmButton.textContent = action === 'approve' ? 'Approve' : 'Reject';
            confirmButton.className = `btn ${action === 'approve' ? 'btn-success' : 'btn-danger'}`;

            new bootstrap.Modal(document.getElementById('reviewLeaveModal')).show();
        }

        function viewDetails(leaveId) {
            // Load leave details via AJAX
            fetch(`../api/get_leave_details.php?id=${leaveId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        const leave = data.leave;
                        const content = `
                            <div class="row">
                                <div class="col-md-6">
                                    <h6>Staff Information</h6>
                                    <p><strong>Name:</strong> ${leave.staff_name || 'N/A'}</p>
                                    <p><strong>Role:</strong> ${leave.staff_role.charAt(0).toUpperCase() + leave.staff_role.slice(1).replace('_', ' ')}</p>
                                    <p><strong>Username:</strong> ${leave.username}</p>
                                </div>
                                <div class="col-md-6">
                                    <h6>Leave Information</h6>
                                    <p><strong>Type:</strong> ${leave.leave_type.charAt(0).toUpperCase() + leave.leave_type.slice(1)} Leave</p>
                                    <p><strong>Start Date:</strong> ${new Date(leave.start_date).toLocaleDateString()}</p>
                                    <p><strong>End Date:</strong> ${new Date(leave.end_date).toLocaleDateString()}</p>
                                    <p><strong>Status:</strong>
                                        <span class="badge bg-${leave.status === 'approved' ? 'success' : leave.status === 'rejected' ? 'danger' : 'warning'}">
                                            ${leave.status.charAt(0).toUpperCase() + leave.status.slice(1)}
                                        </span>
                                    </p>
                                </div>
                                <div class="col-12 mt-3">
                                    <h6>Additional Details</h6>
                                    <p><strong>Reason:</strong> ${leave.reason}</p>
                                    ${leave.emergency_contact ? `<p><strong>Emergency Contact:</strong> ${leave.emergency_contact}</p>` : ''}
                                    <p><strong>Applied On:</strong> ${new Date(leave.created_at).toLocaleDateString()}</p>
                                    ${leave.admin_comments ? `<p><strong>Admin Comments:</strong> ${leave.admin_comments}</p>` : ''}
                                    ${leave.reviewer_name ? `<p><strong>Reviewed By:</strong> ${leave.reviewer_name} on ${new Date(leave.reviewed_at).toLocaleDateString()}</p>` : ''}
                                </div>
                            </div>
                        `;
                        document.getElementById('leaveDetailsContent').innerHTML = content;
                        new bootstrap.Modal(document.getElementById('viewDetailsModal')).show();
                    } else {
                        alert('Error loading leave details');
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Error loading leave details');
                });
        }
    </script>
</body>
</html>
