<?php
require_once '../config.php';

// Check if user is logged in and is admin
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: ../login.php');
    exit();
}

$message = '';
$message_type = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['add_item'])) {
        $name = trim($_POST['name']);
        $category_id = intval($_POST['category_id']);
        $description = trim($_POST['description']);
        $unit = trim($_POST['unit']);
        $min_stock = intval($_POST['min_stock']);
        $max_stock = intval($_POST['max_stock']);
        $location = trim($_POST['location']);
        $supplier = trim($_POST['supplier']);
        $cost_price = floatval($_POST['cost_price']);
        $selling_price = floatval($_POST['selling_price']);
        $status = $_POST['status'];

        try {
            $stmt = $pdo->prepare("INSERT INTO inventory_items (name, category_id, description, unit, minimum_stock, maximum_stock, location, supplier, cost_price, selling_price, status, added_by) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
            $stmt->execute([$name, $category_id, $description, $unit, $min_stock, $max_stock, $location, $supplier, $cost_price, $selling_price, $status, $_SESSION['user_id']]);
            $message = "Inventory item added successfully!";
            $message_type = "success";
        } catch (PDOException $e) {
            $message = "Error adding item: " . $e->getMessage();
            $message_type = "error";
        }
    }
}

// Fetch categories
try {
    $categories_stmt = $pdo->query("SELECT * FROM inventory_categories ORDER BY name");
    $categories = $categories_stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $categories = [];
}

// Fetch items
try {
    $items_stmt = $pdo->query("SELECT ii.*, ic.name as category_name FROM inventory_items ii LEFT JOIN inventory_categories ic ON ii.category_id = ic.id ORDER BY ii.name");
    $items = $items_stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $items = [];
}

$page_title = "Inventory Management - Items";
include '../includes/admin_header.php';
?>

<div class="container-fluid">
    <h2>Inventory Management - Items</h2>

    <?php if ($message): ?>
        <div class="alert alert-<?php echo $message_type === 'success' ? 'success' : 'danger'; ?> alert-dismissible fade show" role="alert">
            <?php echo htmlspecialchars($message); ?>
            <button type="button" class="close" data-dismiss="alert">
                <span>&times;</span>
            </button>
        </div>
    <?php endif; ?>

    <button class="btn btn-facilities mb-3" data-toggle="modal" data-target="#addItemModal">
        <i class="fas fa-plus"></i> Add New Item
    </button>

    <div class="table-responsive">
        <table class="table table-striped table-hover facilities-table">
            <thead class="thead-dark">
                <tr>
                    <th>Name</th>
                    <th>Category</th>
                    <th>Unit</th>
                    <th>Min Stock</th>
                    <th>Max Stock</th>
                    <th>Location</th>
                    <th>Supplier</th>
                    <th>Cost Price</th>
                    <th>Selling Price</th>
                    <th>Status</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($items as $item): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($item['name']); ?></td>
                        <td><?php echo htmlspecialchars($item['category_name']); ?></td>
                        <td><?php echo htmlspecialchars($item['unit']); ?></td>
                        <td><?php echo htmlspecialchars($item['minimum_stock']); ?></td>
                        <td><?php echo htmlspecialchars($item['maximum_stock']); ?></td>
                        <td><?php echo htmlspecialchars($item['location']); ?></td>
                        <td><?php echo htmlspecialchars($item['supplier']); ?></td>
                        <td><?php echo htmlspecialchars($item['cost_price']); ?></td>
                        <td><?php echo htmlspecialchars($item['selling_price']); ?></td>
                        <td><?php echo htmlspecialchars($item['status']); ?></td>
                    </tr>
                <?php endforeach; ?>
                <?php if (empty($items)): ?>
                    <tr>
                        <td colspan="10" class="text-center">No inventory items found.</td>
                    </tr>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<!-- Add Item Modal -->
<div class="modal fade" id="addItemModal" tabindex="-1" role="dialog">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <form method="POST">
                <div class="modal-header">
                    <h5 class="modal-title">Add New Inventory Item</h5>
                    <button type="button" class="close" data-dismiss="modal">
                        <span>&times;</span>
                    </button>
                </div>
                <div class="modal-body">
                    <div class="form-row">
                        <div class="form-group col-md-6">
                            <label for="name">Name *</label>
                            <input type="text" class="form-control" id="name" name="name" required>
                        </div>
                        <div class="form-group col-md-6">
                            <label for="category_id">Category *</label>
                            <select class="form-control" id="category_id" name="category_id" required>
                                <option value="">Select Category</option>
                                <?php foreach ($categories as $category): ?>
                                    <option value="<?php echo $category['id']; ?>"><?php echo htmlspecialchars($category['name']); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>
                    <div class="form-group">
                        <label for="description">Description</label>
                        <textarea class="form-control" id="description" name="description" rows="3"></textarea>
                    </div>
                    <div class="form-row">
                        <div class="form-group col-md-4">
                            <label for="unit">Unit</label>
                            <input type="text" class="form-control" id="unit" name="unit" value="pieces">
                        </div>
                        <div class="form-group col-md-4">
                            <label for="min_stock">Minimum Stock</label>
                            <input type="number" class="form-control" id="min_stock" name="min_stock" min="0" value="0">
                        </div>
                        <div class="form-group col-md-4">
                            <label for="max_stock">Maximum Stock</label>
                            <input type="number" class="form-control" id="max_stock" name="max_stock" min="0" value="0">
                        </div>
                    </div>
                    <div class="form-row">
                        <div class="form-group col-md-6">
                            <label for="location">Location</label>
                            <input type="text" class="form-control" id="location" name="location">
                        </div>
                        <div class="form-group col-md-6">
                            <label for="supplier">Supplier</label>
                            <input type="text" class="form-control" id="supplier" name="supplier">
                        </div>
                    </div>
                    <div class="form-row">
                        <div class="form-group col-md-6">
                            <label for="cost_price">Cost Price</label>
                            <input type="number" step="0.01" class="form-control" id="cost_price" name="cost_price" min="0">
                        </div>
                        <div class="form-group col-md-6">
                            <label for="selling_price">Selling Price</label>
                            <input type="number" step="0.01" class="form-control" id="selling_price" name="selling_price" min="0">
                        </div>
                    </div>
                    <div class="form-group">
                        <label for="status">Status</label>
                        <select class="form-control" id="status" name="status">
                            <option value="active" selected>Active</option>
                            <option value="inactive">Inactive</option>
                            <option value="discontinued">Discontinued</option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="submit" name="add_item" class="btn btn-primary">Add Item</button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php include '../includes/admin_footer.php'; ?>
