<?php
require_once '../config.php';
require_once '../includes/user_activity_logger.php';
require_once '../includes/csrf_protection.php';
require_once '../includes/input_validation.php';
require_once '../includes/password_security.php';

$error = '';

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    // Validate CSRF token
    if (!CSRFProtection::validateRequest()) {
        $error = 'Security validation failed. Please refresh the page and try again.';
    } else {
        // Sanitize and validate input
        $username = InputValidation::getPostData('username', 'string');
        $password = InputValidation::getPostData('password', 'string');

        // Validate required fields
        $validationErrors = InputValidation::validateRequired($_POST, ['username', 'password']);
        if (!empty($validationErrors)) {
            $error = implode('<br>', $validationErrors);
        } elseif (empty($username) || empty($password)) {
            $error = 'Please fill in all fields.';
        } else {
            // Check user credentials
            $stmt = $pdo->prepare("SELECT id, password, role FROM admins WHERE username = ?");
            $stmt->execute([$username]);
            $user = $stmt->fetch();

            if ($user && PasswordSecurity::verifyPassword($password, $user['password'])) {
                // Check if user is allowed to login through admin portal
                if ($user['role'] == 'admin' || $user['role'] == 'super_admin' || $user['role'] == 'head_teacher') {
                    // Set session
                    $_SESSION['user_id'] = $user['id'];
                    $_SESSION['username'] = $username;
                    $_SESSION['role'] = $user['role'];

                    // Log successful login
                    UserActivityLogger::logActivity($user['id'], 'login', "Admin portal login successful from IP: " . $_SERVER['REMOTE_ADDR']);

                    // Redirect based on role
                    if ($user['role'] == 'admin' || $user['role'] == 'super_admin') {
                        header('Location: admin_dashboard.php');
                    } elseif ($user['role'] == 'head_teacher') {
                        header('Location: headteacher_dashboard.php');
                    }
                    exit;
                } else {
                    // Log failed login attempt for unauthorized user
                    UserActivityLogger::logActivity(null, 'login_failed', "Unauthorized user attempted admin portal login: $username (role: {$user['role']}) from IP: " . $_SERVER['REMOTE_ADDR']);
                    $error = 'Access denied. This portal is for administrators and head teachers only.';
                }
            } else {
                // Log failed login attempt
                UserActivityLogger::logActivity(null, 'login_failed', "Failed admin portal login attempt for username: $username from IP: " . $_SERVER['REMOTE_ADDR']);
                $error = 'Invalid username or password.';
            }
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Login - School Management System</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            color: #333;
            background: linear-gradient(135deg, #2c3e50 0%, #34495e 100%);
            min-height: 100vh;
            margin: 0;
        }

        .top-bar {
            background-color: #fff;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            z-index: 1000;
            padding: 15px 0;
        }

        .top-bar .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 20px;
            display: flex;
            justify-content: center;
            align-items: center;
        }

        .top-bar .logo {
            font-size: 1.5em;
            font-weight: bold;
            color: #2c3e50;
            display: flex;
            align-items: center;
        }

        .top-bar .logo i {
            margin-right: 10px;
            font-size: 1.8em;
        }

        .top-bar .nav-links {
            display: flex;
            gap: 30px;
            align-items: center;
        }

        .top-bar .nav-links a {
            color: #666;
            text-decoration: none;
            font-weight: 500;
            display: flex;
            align-items: center;
            transition: color 0.3s ease;
        }

        .top-bar .nav-links a:hover {
            color: #2c3e50;
        }

        .top-bar .nav-links a i {
            margin-right: 8px;
        }

        .main-content {
            padding-top: 80px;
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .login-container {
            background-color: #fff;
            padding: 40px;
            border-radius: 10px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.3);
            width: 100%;
            max-width: 400px;
        }

        .login-header {
            text-align: center;
            margin-bottom: 30px;
        }

        .login-header .icon {
            font-size: 3em;
            color: #2c3e50;
            margin-bottom: 10px;
        }

        .login-header h1 {
            color: #2c3e50;
            margin-bottom: 10px;
            font-size: 2.5em;
        }

        .login-header p {
            color: #666;
            font-size: 1.1em;
        }

        .form-group {
            margin-bottom: 20px;
            position: relative;
        }

        .form-group .input-icon {
            position: absolute;
            left: 12px;
            top: 50%;
            transform: translateY(-50%);
            padding-top: 5px;
            color: #2c3e50;
            font-size: 16px;
            z-index: 1;
        }

        .form-group label {
            display: block;
            margin-bottom: 5px;
            color: #333;
            font-weight: 500;
        }

        .form-group input {
            width: 100%;
            padding: 12px 12px 12px 40px;
            border: 2px solid #ddd;
            border-radius: 5px;
            font-size: 16px;
            transition: border-color 0.3s ease;
        }

        .form-group input:focus {
            outline: none;
            border-color: #2c3e50;
        }

        .btn {
            display: inline-block;
            padding: 12px 30px;
            background: linear-gradient(135deg, #2c3e50 0%, #34495e 100%);
            color: #fff;
            text-decoration: none;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            font-size: 16px;
            font-weight: 500;
            width: 100%;
            transition: transform 0.2s ease;
            position: relative;
        }

        .btn .btn-icon {
            margin-right: 8px;
        }

        .btn:hover {
            transform: translateY(-2px);
        }

        .error-message {
            padding: 12px;
            margin-bottom: 20px;
            border-radius: 5px;
            background-color: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }

        .back-link {
            text-align: center;
            margin-top: 20px;
        }

        .back-link a {
            color: #2c3e50;
            text-decoration: none;
            font-weight: 500;
        }

        .back-link a:hover {
            text-decoration: underline;
        }

        .demo-info {
            margin-top: 20px;
            padding: 15px;
            background-color: #f8f9ff;
            border-radius: 5px;
            border-left: 4px solid #2c3e50;
        }

        .demo-info h4 {
            margin-bottom: 10px;
            color: #2c3e50;
        }

        .demo-info p {
            margin-bottom: 5px;
            font-size: 0.9em;
            color: #666;
        }

        .portal-links {
            margin-top: 20px;
            padding: 15px;
            background-color: #f8f9fa;
            border-radius: 5px;
            border: 1px solid #dee2e6;
        }

        .portal-links h4 {
            margin-bottom: 10px;
            color: #2c3e50;
            text-align: center;
        }

        .portal-links .links {
            display: flex;
            gap: 10px;
            justify-content: center;
            flex-wrap: wrap;
        }

        .portal-links .links a {
            color: #2c3e50;
            text-decoration: none;
            font-weight: 500;
            padding: 5px 10px;
            border: 1px solid #2c3e50;
            border-radius: 3px;
            transition: all 0.3s ease;
        }

        .portal-links .links a:hover {
            background-color: #2c3e50;
            color: #fff;
        }

        /* Responsive */
        @media (max-width: 768px) {
            .top-bar {
                padding: 10px 0;
            }

            .top-bar .container {
                padding: 0 15px;
            }

            .top-bar .logo {
                font-size: 1.2em;
            }

            .top-bar .logo i {
                font-size: 1.5em;
            }

            .top-bar .nav-links {
                gap: 20px;
            }

            .top-bar .nav-links a span {
                display: none;
            }
        }

        @media (max-width: 480px) {
            .top-bar {
                padding: 8px 0;
            }

            .login-container {
                margin: 20px;
                padding: 30px 20px;
            }

            .login-header h1 {
                font-size: 2em;
            }

            .top-bar .logo {
                font-size: 1em;
            }

            .top-bar .logo i {
                font-size: 1.3em;
            }

            .top-bar .nav-links {
                gap: 15px;
            }
        }
    </style>
</head>
<body>
    <!-- Top Navigation Bar -->
    <div class="top-bar">
        <div class="container">
            <div class="logo">
                <i class="fas fa-user-shield"></i>
                <span>Elitek School Management System</span>
            </div>
            <div class="nav-links">
                <!-- Removed all links as per request -->
            </div>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <div class="login-container">
            <div class="login-header">
                <div class="icon"><i class="fas fa-user-shield"></i></div>
                <h1>Admin Portal</h1>
                <p>Login for Administrators and Head Teachers</p>
            </div>

        <?php if ($error): ?>
            <div class="error-message"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>

        <form method="post" action="index.php">
            <?php echo CSRFProtection::getHiddenInput(); ?>
            <div class="form-group">
                <label for="username">Username:</label>
                <span class="input-icon" style="margin-top: 13px;"><i class="fas fa-user"></i></span>
                <input type="text" id="username" name="username" required autofocus>
            </div>

            <div class="form-group">
                <label for="password">Password:</label>
                <span class="input-icon" style="margin-top: 13px;"><i class="fas fa-lock"></i></span>
                <input type="password" id="password" name="password" required>
            </div>

            <button type="submit" class="btn"><span class="btn-icon"><i class="fas fa-sign-in-alt"></i></span>Login to Admin Panel</button>
        </form>


        </div>
    </div>

    <script src="../js/script.js"></script>
</body>
</html>
