<?php
require_once '../config.php';
require_once '../includes/access_control.php';

if (!isset($_SESSION['user_id']) || $_SESSION['role'] != 'head_teacher') {
    header('Location: ../login.php');
    exit;
}

$user_id = $_SESSION['user_id'];

// Fetch head teacher information
$stmt = $pdo->prepare("SELECT *, CONCAT(first_name, ' ', last_name) as name FROM head_teachers WHERE user_id = ?");
$stmt->execute([$user_id]);
$head_teacher = $stmt->fetch();

// If no head teacher profile found, get user information as fallback
if (!$head_teacher) {
    $stmt = $pdo->prepare("SELECT username as name FROM users WHERE id = ?");
    $stmt->execute([$user_id]);
    $user_info = $stmt->fetch();
    $head_teacher = [
        'name' => $user_info ? $user_info['name'] : $_SESSION['username'],
        'first_name' => $_SESSION['username'],
        'last_name' => '',
        'email' => '',
        'phone' => '',
        'profile_image' => null
    ];
}

// Handle new discussion group creation
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['create_group'])) {
    $group_name = trim($_POST['group_name']);
    $description = trim($_POST['description']);
    $selected_teachers = $_POST['teachers'] ?? [];

    if (!empty($group_name) && !empty($selected_teachers)) {
        // Insert discussion group
        $insert_stmt = $pdo->prepare("
            INSERT INTO discussion_groups (name, description, created_by, created_at)
            VALUES (?, ?, ?, NOW())
        ");
        $insert_stmt->execute([$group_name, $description, $user_id]);
        $group_id = $pdo->lastInsertId();

        // Add participants (head teacher + selected teachers)
        $participants = array_merge([$user_id], $selected_teachers);
        foreach ($participants as $participant_id) {
            $pdo->prepare("
                INSERT INTO discussion_participants (discussion_group_id, user_id, joined_at)
                VALUES (?, ?, NOW())
            ")->execute([$group_id, $participant_id]);
        }

        $message = 'Discussion group created successfully!';
        $message_type = 'success';
    } else {
        $message = 'Please provide a group name and select at least one teacher.';
        $message_type = 'error';
    }
}

// Handle adding teachers to existing group
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['add_to_group'])) {
    $group_id = $_POST['group_id'];
    $selected_teachers = $_POST['teachers'] ?? [];

    if (!empty($selected_teachers)) {
        foreach ($selected_teachers as $teacher_id) {
            // Check if already a participant
            $stmt = $pdo->prepare("SELECT id FROM discussion_participants WHERE discussion_group_id = ? AND user_id = ?");
            $stmt->execute([$group_id, $teacher_id]);

            if ($stmt->rowCount() == 0) {
                $pdo->prepare("
                    INSERT INTO discussion_participants (discussion_group_id, user_id, joined_at)
                    VALUES (?, ?, NOW())
                ")->execute([$group_id, $teacher_id]);
            }
        }

        $message = 'Teachers added to the discussion group successfully!';
        $message_type = 'success';
    } else {
        $message = 'Please select at least one teacher to add.';
        $message_type = 'error';
    }
}

// Get all discussion groups created by this head teacher
$stmt = $pdo->prepare("
    SELECT dg.*, COUNT(DISTINCT dp.user_id) as participant_count
    FROM discussion_groups dg
    LEFT JOIN discussion_participants dp ON dg.id = dp.discussion_group_id
    WHERE dg.created_by = ?
    GROUP BY dg.id
    ORDER BY dg.created_at DESC
");
$stmt->execute([$user_id]);
$discussion_groups = $stmt->fetchAll();

// Get all teachers for selection
$stmt = $pdo->query("SELECT *, CONCAT(first_name, ' ', last_name) as name FROM staff WHERE role = 'teaching' ORDER BY first_name, last_name");
$all_teachers = $stmt->fetchAll();

// Handle new message submission
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['send_message'])) {
    $group_id = $_POST['group_id'];
    $message = trim($_POST['message']);

    if (!empty($message) && !empty($group_id)) {
        // Check if user is participant in the group
        $stmt = $pdo->prepare("SELECT id FROM discussion_participants WHERE discussion_group_id = ? AND user_id = ? AND is_active = TRUE");
        $stmt->execute([$group_id, $user_id]);

        if ($stmt->rowCount() > 0) {
            // Insert message
            $insert_stmt = $pdo->prepare("
                INSERT INTO discussion_messages (discussion_group_id, user_id, message, message_type, created_at)
                VALUES (?, ?, ?, 'text', NOW())
            ");
            $insert_stmt->execute([$group_id, $user_id, $message]);
            $message_id = $pdo->lastInsertId();

            // Update group updated_at
            $pdo->prepare("UPDATE discussion_groups SET updated_at = NOW() WHERE id = ?")->execute([$group_id]);

            // Check if this is an AJAX request
            if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest') {
                // Return JSON response for AJAX
                header('Content-Type: application/json');
                echo json_encode([
                    'success' => true,
                    'message_id' => $message_id,
                    'message' => $message,
                    'timestamp' => date('Y-m-d H:i:s')
                ]);
                exit;
            } else {
                // Regular form submission - redirect
                header("Location: headteacher_discussion.php?group=" . $group_id . "&chat=1");
                exit;
            }
        } else {
            // User is not a participant
            if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest') {
                header('Content-Type: application/json');
                echo json_encode([
                    'success' => false,
                    'error' => 'You are not a participant in this discussion group.'
                ]);
                exit;
            } else {
                $message = 'You are not a participant in this discussion group.';
                $message_type = 'error';
            }
        }
    } else {
        // Empty message
        if (isset($_SERVER['HTTP_X_REQUESTED_WITH']) && $_SERVER['HTTP_X_REQUESTED_WITH'] === 'XMLHttpRequest') {
            header('Content-Type: application/json');
            echo json_encode([
                'success' => false,
                'error' => 'Message cannot be empty.'
            ]);
            exit;
        } else {
            $message = 'Message cannot be empty.';
            $message_type = 'error';
        }
    }
}

// Get selected group for adding teachers
$selected_group_id = $_GET['group'] ?? null;
$selected_group = null;
$current_participants = [];
$messages = [];
$show_chat = isset($_GET['chat']);

if ($selected_group_id) {
    $stmt = $pdo->prepare("SELECT * FROM discussion_groups WHERE id = ? AND created_by = ?");
    $stmt->execute([$selected_group_id, $user_id]);
    $selected_group = $stmt->fetch();

    if ($selected_group) {
        // Get current participants
        $stmt = $pdo->prepare("
            SELECT t.*, CONCAT(t.first_name, ' ', t.last_name) as name
            FROM discussion_participants dp
            JOIN teachers t ON dp.user_id = t.user_id
            WHERE dp.discussion_group_id = ? AND dp.is_active = TRUE
            ORDER BY t.first_name, t.last_name
        ");
        $stmt->execute([$selected_group_id]);
        $current_participants = $stmt->fetchAll();

        // Get messages if chat is selected
        if ($show_chat) {
            $stmt = $pdo->prepare("
                SELECT dm.*, CONCAT(t.first_name, ' ', t.last_name) as sender_name, t.profile_image
                FROM discussion_messages dm
                LEFT JOIN teachers t ON dm.user_id = t.user_id
                WHERE dm.discussion_group_id = ? AND dm.is_deleted = FALSE
                ORDER BY dm.created_at ASC
            ");
            $stmt->execute([$selected_group_id]);
            $messages = $stmt->fetchAll();
        }
    }
}

include '../includes/admin_header.php';
?>

<div class="headteacher-discussion-container">
    <div class="discussion-header">
        <h1>👨‍💼 Staff Discussion Management</h1>
        <p class="header-subtitle">Manage discussion groups and connect with your teaching staff</p>
    </div>

    <?php if (isset($message)): ?>
        <div class="message <?php echo $message_type; ?>">
            <?php echo htmlspecialchars($message); ?>
        </div>
    <?php endif; ?>

    <div class="discussion-management">
        <!-- Create New Group Section - Hidden when chat is active -->
        <?php if (!$show_chat): ?>
        <div class="create-group-section">
            <div class="section-header">
                <h2>📝 Create New Discussion Group</h2>
                <button id="toggleCreateGroup" class="btn-primary">
                    <span class="btn-icon">➕</span>
                    Create Group
                </button>
            </div>

            <div id="createGroupForm" class="create-group-form" style="display: none;">
                <form method="POST" action="headteacher_discussion.php" class="group-form">
                    <div class="form-row">
                        <div class="form-group">
                            <label for="group_name">Group Name *</label>
                            <input type="text" id="group_name" name="group_name" required
                                   placeholder="e.g., Mathematics Department, Grade 1 Teachers">
                        </div>
                    </div>

                    <div class="form-group">
                        <label for="description">Description</label>
                        <textarea id="description" name="description" rows="3"
                                  placeholder="Brief description of the group's purpose..."></textarea>
                    </div>

                    <div class="form-group">
                        <label>Select Teachers to Add *</label>
                        <div class="teachers-selection">
                            <div class="select-all-container">
                                <label class="checkbox-container">
                                    <input type="checkbox" id="selectAllTeachers">
                                    <span class="checkmark"></span>
                                    Select All Teachers
                                </label>
                            </div>
                            <div class="teachers-grid">
                                <?php foreach ($all_teachers as $teacher): ?>
                                    <label class="teacher-checkbox">
                                        <input type="checkbox" name="teachers[]" value="<?php echo $teacher['user_id']; ?>"
                                               class="teacher-checkbox-input">
                                        <div class="teacher-info">
                                            <div class="teacher-name"><?php echo htmlspecialchars($teacher['name']); ?></div>
                                            <div class="teacher-details">
                                                <?php echo htmlspecialchars($teacher['email'] ?? 'No email'); ?> |
                                                <?php echo htmlspecialchars($teacher['phone'] ?? 'No phone'); ?>
                                            </div>
                                        </div>
                                    </label>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </div>

                    <div class="form-actions">
                        <button type="button" id="cancelCreateGroup" class="btn-secondary">Cancel</button>
                        <button type="submit" name="create_group" class="btn-primary">Create Group</button>
                    </div>
                </form>
            </div>
        </div>
        <?php endif; ?>

        <!-- Add Teachers to Existing Group - Hidden when chat is active -->
        <?php if ($selected_group && !$show_chat): ?>
            <div class="add-teachers-section">
                <div class="section-header">
                    <h2>👥 Add Teachers to "<?php echo htmlspecialchars($selected_group['name']); ?>"</h2>
                    <a href="headteacher_discussion.php" class="btn-secondary">Back to Groups</a>
                </div>

                <div class="current-participants">
                    <h3>Current Participants (<?php echo count($current_participants); ?>)</h3>
                    <div class="participants-tags">
                        <?php foreach ($current_participants as $participant): ?>
                            <span class="participant-tag">
                                <?php echo htmlspecialchars($participant['name']); ?>
                                <?php if ($participant['user_id'] == $user_id): ?>
                                    <span class="tag-role">(You)</span>
                                <?php endif; ?>
                            </span>
                        <?php endforeach; ?>
                    </div>
                </div>

                <form method="POST" action="headteacher_discussion.php?group=<?php echo $selected_group_id; ?>" class="add-teachers-form">
                    <input type="hidden" name="group_id" value="<?php echo $selected_group_id; ?>">

                    <div class="form-group">
                        <label>Select Additional Teachers</label>
                        <div class="teachers-selection">
                            <div class="select-all-container">
                                <label class="checkbox-container">
                                    <input type="checkbox" id="selectAllAddTeachers">
                                    <span class="checkmark"></span>
                                    Select All Available Teachers
                                </label>
                            </div>
                            <div class="teachers-grid">
                                <?php
                                $current_participant_ids = array_column($current_participants, 'user_id');
                                foreach ($all_teachers as $teacher):
                                    if (!in_array($teacher['user_id'], $current_participant_ids)):
                                ?>
                                    <label class="teacher-checkbox">
                                        <input type="checkbox" name="teachers[]" value="<?php echo $teacher['user_id']; ?>"
                                               class="teacher-checkbox-input">
                                        <div class="teacher-info">
                                            <div class="teacher-name"><?php echo htmlspecialchars($teacher['name']); ?></div>
                                            <div class="teacher-details">
                                                <?php echo htmlspecialchars($teacher['email'] ?? 'No email'); ?> |
                                                <?php echo htmlspecialchars($teacher['phone'] ?? 'No phone'); ?>
                                            </div>
                                        </div>
                                    </label>
                                <?php
                                    endif;
                                endforeach;
                                ?>
                            </div>
                        </div>
                    </div>

                    <div class="form-actions">
                        <button type="submit" name="add_to_group" class="btn-primary">Add Selected Teachers</button>
                    </div>
                </form>
            </div>
        <?php endif; ?>

        <!-- Chat Interface -->
        <?php if ($selected_group && $show_chat): ?>
            <div class="chat-section">
                <div class="section-header">
                    <h2>💬 Chat - <?php echo htmlspecialchars($selected_group['name']); ?></h2>
                    <div class="chat-actions">
                        <a href="headteacher_discussion.php?group=<?php echo $selected_group_id; ?>" class="btn-secondary">Back to Manage</a>
                        <a href="headteacher_discussion.php" class="btn-secondary">All Groups</a>
                    </div>
                </div>

                <div class="chat-container">
                    <div class="chat-participants">
                        <h4>👥 Participants (<?php echo count($current_participants); ?>)</h4>
                        <div class="participants-list">
                            <?php foreach ($current_participants as $participant): ?>
                                <div class="participant-item">
                                    <div class="participant-avatar">
                                        <?php echo strtoupper(substr($participant['name'], 0, 1)); ?>
                                    </div>
                                    <div class="participant-info">
                                        <div class="participant-name"><?php echo htmlspecialchars($participant['name']); ?></div>
                                        <?php if ($participant['user_id'] == $user_id): ?>
                                            <div class="participant-role">(You - Head Teacher)</div>
                                        <?php else: ?>
                                            <div class="participant-role">Teacher</div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>

                    <div class="chat-messages">
                        <div class="messages-container" id="messagesContainer">
                            <?php if (empty($messages)): ?>
                                <div class="no-messages">
                                    <div class="no-messages-icon">💬</div>
                                    <h3>No Messages Yet</h3>
                                    <p>Start the conversation by sending the first message!</p>
                                </div>
                            <?php else: ?>
                                <?php
                                $last_message_id = 0;
                                foreach ($messages as $message):
                                    $last_message_id = max($last_message_id, $message['id']);
                                ?>
                                    <div class="message-item <?php echo $message['user_id'] == $user_id ? 'own-message' : 'other-message'; ?>"
                                         data-message-id="<?php echo $message['id']; ?>">
                                        <div class="message-header">
                                            <div class="message-sender">
                                                <?php echo htmlspecialchars($message['sender_name']); ?>
                                                <?php if ($message['user_id'] == $user_id): ?>
                                                    <span class="sender-role">(You)</span>
                                                <?php endif; ?>
                                            </div>
                                            <div class="message-time">
                                                <?php echo date('M d, H:i', strtotime($message['created_at'])); ?>
                                            </div>
                                        </div>
                                        <div class="message-content">
                                            <?php echo htmlspecialchars($message['message']); ?>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </div>

                        <div class="message-form">
                            <form method="POST" action="headteacher_discussion.php?group=<?php echo $selected_group_id; ?>&chat=1" class="send-message-form">
                                <input type="hidden" name="group_id" value="<?php echo $selected_group_id; ?>">
                                <div class="message-input-group">
                                    <textarea name="message" id="messageInput" rows="3" required
                                              placeholder="Type your message here..." maxlength="1000"></textarea>
                                    <button type="submit" name="send_message" class="btn-send" id="sendButton">
                                        <span class="send-icon">📤</span>
                                        Send
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <!-- Discussion Groups List -->
        <div class="groups-list-section">
            <div class="section-header">
                <h2>📚 Your Discussion Groups</h2>
                <div class="groups-count"><?php echo count($discussion_groups); ?> groups created</div>
            </div>

            <?php if (empty($discussion_groups)): ?>
                <div class="no-groups">
                    <div class="no-groups-icon">💬</div>
                    <h3>No Discussion Groups Yet</h3>
                    <p>Create your first discussion group to start connecting with teachers.</p>
                </div>
            <?php else: ?>
                <div class="groups-grid">
                    <?php foreach ($discussion_groups as $group): ?>
                        <div class="group-card">
                            <div class="group-card-header">
                                <div class="group-title">
                                    <h3><?php echo htmlspecialchars($group['name']); ?></h3>
                                    <div class="group-description">
                                        <?php echo htmlspecialchars($group['description'] ?? 'No description'); ?>
                                    </div>
                                </div>
                                <div class="group-actions">
                                    <a href="headteacher_discussion.php?group=<?php echo $group['id']; ?>"
                                       class="btn-action btn-add-teachers" title="Add Teachers">
                                        <span class="btn-icon">👥</span>
                                        <span class="btn-text">Add Teachers</span>
                                    </a>
                                    <a href="headteacher_discussion.php?group=<?php echo $group['id']; ?>&chat=1"
                                       class="btn-action btn-chat" title="Open Chat">
                                        <span class="btn-icon">💬</span>
                                        <span class="btn-text">Chat</span>
                                    </a>
                                </div>
                            </div>

                            <div class="group-card-body">
                                <div class="group-stats">
                                    <div class="stat-item">
                                        <span class="stat-label">👥 Participants:</span>
                                        <span class="stat-value"><?php echo $group['participant_count']; ?></span>
                                    </div>
                                    <div class="stat-item">
                                        <span class="stat-label">📅 Created:</span>
                                        <span class="stat-value">
                                            <?php echo date('M d, Y', strtotime($group['created_at'])); ?>
                                        </span>
                                    </div>
                                </div>

                                <div class="group-participants-preview">
                                    <div class="participants-avatars">
                                        <?php
                                        $stmt = $pdo->prepare("
                                            SELECT CONCAT(t.first_name, ' ', t.last_name) as name
                                            FROM discussion_participants dp
                                            JOIN teachers t ON dp.user_id = t.user_id
                                            WHERE dp.discussion_group_id = ? AND dp.is_active = TRUE
                                            ORDER BY t.first_name, t.last_name
                                            LIMIT 5
                                        ");
                                        $stmt->execute([$group['id']]);
                                        $participants = $stmt->fetchAll();

                                        foreach ($participants as $participant):
                                        ?>
                                            <div class="participant-avatar-small"
                                                 title="<?php echo htmlspecialchars($participant['name']); ?>">
                                                <?php echo strtoupper(substr($participant['name'], 0, 1)); ?>
                                            </div>
                                        <?php endforeach; ?>

                                        <?php if ($group['participant_count'] > 5): ?>
                                            <div class="participant-avatar-small more-count"
                                                 title="And <?php echo $group['participant_count'] - 5; ?> more">
                                                +<?php echo $group['participant_count'] - 5; ?>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<style>
.headteacher-discussion-container {
    max-width: 1400px;
    margin: 0 auto;
    padding: 20px;
}

.discussion-header {
    text-align: center;
    margin-bottom: 40px;
    padding-bottom: 30px;
    border-bottom: 2px solid #e9ecef;
}

.discussion-header h1 {
    margin: 0 0 10px 0;
    font-size: 2.8em;
    font-weight: 700;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.header-subtitle {
    margin: 0;
    font-size: 1.3em;
    color: #6c757d;
    font-weight: 300;
}

.message {
    padding: 15px 20px;
    border-radius: 8px;
    margin-bottom: 20px;
    font-weight: 500;
    border-left: 4px solid;
}

.message.success {
    background-color: #d4edda;
    color: #155724;
    border-left-color: #28a745;
}

.message.error {
    background-color: #f8d7da;
    color: #721c24;
    border-left-color: #dc3545;
}

.discussion-management {
    display: grid;
    gap: 30px;
}

.section-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 25px;
    padding-bottom: 15px;
    border-bottom: 2px solid #e9ecef;
}

.section-header h2 {
    margin: 0;
    color: #2c3e50;
    font-size: 1.8em;
    font-weight: 600;
}

.groups-count {
    color: #6c757d;
    font-size: 1em;
    font-weight: 500;
}

/* Create Group Section */
.create-group-form {
    background: #fff;
    border-radius: 15px;
    padding: 30px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.1);
    border: 2px solid #e9ecef;
}

.group-form .form-row {
    display: grid;
    grid-template-columns: 1fr;
    gap: 20px;
    margin-bottom: 20px;
}

.group-form .form-group {
    display: flex;
    flex-direction: column;
}

.group-form label {
    font-weight: 600;
    color: #495057;
    margin-bottom: 8px;
    font-size: 1em;
}

.group-form input,
.group-form textarea {
    padding: 12px 15px;
    border: 2px solid #e9ecef;
    border-radius: 8px;
    font-size: 1em;
    transition: all 0.3s ease;
    background: #f8f9fa;
}

.group-form input:focus,
.group-form textarea:focus {
    outline: none;
    border-color: #667eea;
    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
    background: #fff;
}

.group-form textarea {
    resize: vertical;
    min-height: 80px;
}

/* Teachers Selection */
.teachers-selection {
    border: 2px solid #e9ecef;
    border-radius: 8px;
    padding: 15px;
    background: #f8f9fa;
}

.select-all-container {
    margin-bottom: 15px;
    padding-bottom: 10px;
    border-bottom: 1px solid #dee2e6;
}

.checkbox-container {
    display: flex;
    align-items: center;
    gap: 8px;
    cursor: pointer;
    font-weight: 500;
}

.checkbox-container input[type="checkbox"] {
    width: 18px;
    height: 18px;
    accent-color: #667eea;
}

.teachers-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
    gap: 12px;
    max-height: 300px;
    overflow-y: auto;
}

.teacher-checkbox {
    display: flex;
    align-items: center;
    gap: 12px;
    padding: 10px;
    background: white;
    border-radius: 8px;
    cursor: pointer;
    transition: all 0.3s ease;
    border: 2px solid transparent;
}

.teacher-checkbox:hover {
    border-color: #667eea;
    background: #f8f9ff;
}

.teacher-checkbox-input {
    width: 18px;
    height: 18px;
    accent-color: #667eea;
}

.teacher-info {
    flex: 1;
}

.teacher-name {
    font-weight: 600;
    color: #495057;
    margin-bottom: 2px;
}

.teacher-details {
    font-size: 0.85em;
    color: #6c757d;
}

.form-actions {
    display: flex;
    justify-content: flex-end;
    gap: 15px;
    margin-top: 30px;
    padding-top: 20px;
    border-top: 2px solid #e9ecef;
}

.btn-primary {
    padding: 12px 25px;
    background: linear-gradient(135deg, #667eea, #764ba2);
    color: white;
    border: none;
    border-radius: 8px;
    font-size: 1em;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    gap: 8px;
    text-decoration: none;
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(102, 126, 234, 0.4);
}

.btn-secondary {
    padding: 12px 25px;
    background: #6c757d;
    color: white;
    border: none;
    border-radius: 8px;
    font-size: 1em;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    text-decoration: none;
}

.btn-secondary:hover {
    background: #5a6268;
    transform: translateY(-1px);
}

.btn-small {
    padding: 8px 16px;
    font-size: 0.85em;
}

/* Enhanced Action Buttons */
.btn-action {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 10px 16px;
    font-size: 0.95em;
    font-weight: 600;
    border-radius: 8px;
    text-decoration: none;
    transition: all 0.3s ease;
    border: 2px solid transparent;
    min-width: 120px;
    justify-content: center;
    box-shadow: 0 3px 12px rgba(0,0,0,0.1);
}

.btn-action:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(0,0,0,0.15);
    text-decoration: none;
}

.btn-add-teachers {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border-color: #667eea;
}

.btn-add-teachers:hover {
    background: linear-gradient(135deg, #5a67d8 0%, #6c5b7b 100%);
    box-shadow: 0 8px 25px rgba(102, 126, 234, 0.4);
}

.btn-chat {
    background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
    color: white;
    border-color: #28a745;
}

.btn-chat:hover {
    background: linear-gradient(135deg, #218838 0%, #1dd1a1 100%);
    box-shadow: 0 8px 25px rgba(40, 167, 69, 0.4);
}

.btn-icon {
    font-size: 1.2em;
    display: flex;
    align-items: center;
    justify-content: center;
}

.btn-text {
    font-weight: 600;
    letter-spacing: 0.5px;
}

/* Add Teachers Section */
.add-teachers-section {
    background: #fff;
    border-radius: 15px;
    padding: 30px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.1);
    border: 2px solid #e9ecef;
}

.current-participants {
    margin-bottom: 25px;
}

.current-participants h3 {
    margin: 0 0 15px 0;
    color: #495057;
    font-size: 1.3em;
}

.participants-tags {
    display: flex;
    flex-wrap: wrap;
    gap: 8px;
}

.participant-tag {
    background: linear-gradient(135deg, #667eea, #764ba2);
    color: white;
    padding: 6px 12px;
    border-radius: 20px;
    font-size: 0.9em;
    font-weight: 500;
}

.tag-role {
    opacity: 0.8;
    font-size: 0.85em;
}

/* Groups List */
.groups-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(400px, 1fr));
    gap: 20px;
}

.group-card {
    background: #fff;
    border-radius: 12px;
    border: 2px solid #e9ecef;
    box-shadow: 0 4px 15px rgba(0,0,0,0.08);
    transition: transform 0.3s ease, box-shadow 0.3s ease;
    overflow: hidden;
}

.group-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(0,0,0,0.12);
}



.group-card-header {
    padding: 20px;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    gap: 15px;
}

.group-title h3 {
    margin: 0 0 8px 0;
    font-size: 1.4em;
    font-weight: 600;
}

.group-description {
    opacity: 0.9;
    font-size: 0.95em;
    line-height: 1.4;
}

.group-card-body {
    padding: 20px;
}

.group-stats {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 15px;
    margin-bottom: 20px;
}

.stat-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.stat-label {
    color: #6c757d;
    font-weight: 500;
}

.stat-value {
    color: #495057;
    font-weight: 600;
}

.group-participants-preview {
    margin-top: 15px;
}

.participants-avatars {
    display: flex;
    gap: 8px;
    align-items: center;
}

.participant-avatar-small {
    width: 35px;
    height: 35px;
    border-radius: 50%;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
    font-size: 0.85em;
    border: 2px solid white;
}

.participant-avatar-small.more-count {
    background: #6c757d;
    font-size: 0.75em;
}

.no-groups {
    text-align: center;
    padding: 60px 20px;
    color: #6c757d;
    background: #fff;
    border-radius: 15px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.1);
}

.no-groups-icon {
    font-size: 4em;
    margin-bottom: 20px;
    opacity: 0.5;
}

.no-groups h3 {
    margin: 0 0 15px 0;
    color: #495057;
    font-size: 1.5em;
}

/* Responsive Design */
@media (max-width: 768px) {
    .headteacher-discussion-container {
        padding: 10px;
    }

    .discussion-header h1 {
        font-size: 2.2em;
    }

    .section-header {
        flex-direction: column;
        align-items: flex-start;
        gap: 15px;
    }

    .form-actions {
        flex-direction: column;
    }

    .btn-primary,
    .btn-secondary {
        width: 100%;
        justify-content: center;
    }

    .teachers-grid {
        grid-template-columns: 1fr;
    }

    .groups-grid {
        grid-template-columns: 1fr;
    }

    .group-card-header {
        flex-direction: column;
        align-items: stretch;
    }

    .group-actions {
        align-self: flex-end;
    }
}

@media (max-width: 480px) {
    .discussion-header h1 {
        font-size: 1.8em;
    }

    .section-header h2 {
        font-size: 1.5em;
    }

    .create-group-form,
    .add-teachers-section,
    .chat-section {
        padding: 20px;
    }
}

/* Chat Interface Styles */
.chat-section {
    background: #fff;
    border-radius: 15px;
    padding: 30px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.1);
    border: 2px solid #e9ecef;
}

.chat-actions {
    display: flex;
    gap: 10px;
}

.chat-container {
    display: grid;
    grid-template-columns: 300px 1fr;
    gap: 30px;
    margin-top: 20px;
}

.chat-participants {
    background: #f8f9fa;
    border-radius: 12px;
    padding: 20px;
    border: 2px solid #e9ecef;
    height: fit-content;
}

.chat-participants h4 {
    margin: 0 0 15px 0;
    color: #495057;
    font-size: 1.2em;
    font-weight: 600;
}

.participants-list {
    display: flex;
    flex-direction: column;
    gap: 12px;
}

.participant-item {
    display: flex;
    align-items: center;
    gap: 12px;
    padding: 10px;
    background: white;
    border-radius: 8px;
    border: 2px solid transparent;
    transition: all 0.3s ease;
}

.participant-item:hover {
    border-color: #667eea;
    background: #f8f9ff;
}

.participant-avatar {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
    font-size: 1.1em;
}

.participant-info {
    flex: 1;
}

.participant-name {
    font-weight: 600;
    color: #495057;
    margin-bottom: 2px;
}

.participant-role {
    font-size: 0.85em;
    color: #6c757d;
}

.chat-messages {
    display: flex;
    flex-direction: column;
    height: 600px;
    background: #fff;
    border-radius: 12px;
    border: 2px solid #e9ecef;
    overflow: hidden;
}

.messages-container {
    flex: 1;
    padding: 20px;
    overflow-y: auto;
    background: #f8f9fa;
}

.message-item {
    margin-bottom: 20px;
    padding: 15px;
    border-radius: 12px;
    background: white;
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    max-width: 80%;
}

.message-item.own-message {
    margin-left: auto;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
}

.message-item.other-message {
    margin-right: auto;
    background: white;
    border: 2px solid #e9ecef;
}

.message-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 8px;
}

.message-sender {
    font-weight: 600;
    font-size: 0.95em;
}

.sender-role {
    opacity: 0.8;
    font-weight: normal;
}

.message-time {
    font-size: 0.8em;
    opacity: 0.7;
}

.message-content {
    line-height: 1.5;
    word-wrap: break-word;
}

.no-messages {
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    height: 100%;
    color: #6c757d;
    text-align: center;
}

.no-messages-icon {
    font-size: 4em;
    margin-bottom: 20px;
    opacity: 0.5;
}

.no-messages h3 {
    margin: 0 0 10px 0;
    color: #495057;
    font-size: 1.3em;
}

.message-form {
    padding: 20px;
    background: white;
    border-top: 2px solid #e9ecef;
}

.message-input-group {
    display: flex;
    gap: 15px;
    align-items: flex-end;
}

.message-input-group textarea {
    flex: 1;
    padding: 12px 15px;
    border: 2px solid #e9ecef;
    border-radius: 8px;
    font-size: 1em;
    resize: vertical;
    min-height: 20px;
    max-height: 120px;
    background: #f8f9fa;
    transition: all 0.3s ease;
}

.message-input-group textarea:focus {
    outline: none;
    border-color: #667eea;
    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
    background: #fff;
}

.btn-send {
    padding: 12px 20px;
    background: linear-gradient(135deg, #28a745, #20c997);
    color: white;
    border: none;
    border-radius: 8px;
    font-size: 1em;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    gap: 8px;
    min-width: 80px;
    justify-content: center;
}

.btn-send:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(40, 167, 69, 0.4);
}

.send-icon {
    font-size: 1.1em;
}

/* Responsive Chat Design */
@media (max-width: 1024px) {
    .chat-container {
        grid-template-columns: 250px 1fr;
        gap: 20px;
    }
}

@media (max-width: 768px) {
    .chat-container {
        grid-template-columns: 1fr;
        gap: 20px;
    }

    .chat-participants {
        order: 2;
    }

    .chat-messages {
        order: 1;
        height: 400px;
    }

    .message-item {
        max-width: 90%;
    }

    .message-input-group {
        flex-direction: column;
        align-items: stretch;
    }

    .btn-send {
        align-self: flex-end;
        min-width: 100px;
    }
}

@media (max-width: 480px) {
    .chat-section {
        padding: 20px;
    }

    .chat-messages {
        height: 350px;
    }

    .messages-container {
        padding: 15px;
    }

    .message-form {
        padding: 15px;
    }
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const toggleCreateGroup = document.getElementById('toggleCreateGroup');
    const createGroupForm = document.getElementById('createGroupForm');
    const cancelCreateGroup = document.getElementById('cancelCreateGroup');
    const selectAllTeachers = document.getElementById('selectAllTeachers');
    const selectAllAddTeachers = document.getElementById('selectAllAddTeachers');
    const messageInput = document.getElementById('messageInput');
    const sendButton = document.getElementById('sendButton');
    const messagesContainer = document.getElementById('messagesContainer');
    const sendMessageForm = document.querySelector('.send-message-form');

    let eventSource = null;
    let lastMessageId = <?php echo $last_message_id; ?>;

    // Initialize real-time messaging if in chat mode
    <?php if ($selected_group && $show_chat): ?>
    initializeRealTimeChat();
    <?php endif; ?>

    function initializeRealTimeChat() {
        const groupId = <?php echo $selected_group_id; ?>;
        const userId = <?php echo $user_id; ?>;

        // Connect to SSE endpoint
        eventSource = new EventSource(`chat_sse.php?group_id=${groupId}&user_id=${userId}&last_message_id=${lastMessageId}`);

        eventSource.onmessage = function(event) {
            const data = JSON.parse(event.data);

            if (data.type === 'connected') {
                console.log('Connected to real-time chat');
            } else if (data.type === 'new_message') {
                addMessageToChat(data.message);
                lastMessageId = Math.max(lastMessageId, data.message.id);
            } else if (data.type === 'error') {
                console.error('SSE Error:', data.message);
            }
        };

        eventSource.onerror = function(event) {
            console.error('SSE connection error:', event);
        };
    }

    function addMessageToChat(message) {
        // Remove "no messages" placeholder if it exists
        const noMessagesDiv = messagesContainer.querySelector('.no-messages');
        if (noMessagesDiv) {
            noMessagesDiv.remove();
        }

        // Create message element
        const messageDiv = document.createElement('div');
        messageDiv.className = `message-item ${message.user_id == <?php echo $user_id; ?> ? 'own-message' : 'other-message'}`;
        messageDiv.setAttribute('data-message-id', message.id);

        messageDiv.innerHTML = `
            <div class="message-header">
                <div class="message-sender">
                    ${message.sender_name}
                    ${message.user_id == <?php echo $user_id; ?> ? '<span class="sender-role">(You)</span>' : ''}
                </div>
                <div class="message-time">
                    ${formatMessageTime(message.created_at)}
                </div>
            </div>
            <div class="message-content">
                ${message.message}
            </div>
        `;

        // Add to messages container
        messagesContainer.appendChild(messageDiv);

        // Scroll to bottom
        messagesContainer.scrollTop = messagesContainer.scrollHeight;

        // Add a subtle animation
        messageDiv.style.opacity = '0';
        messageDiv.style.transform = 'translateY(20px)';
        setTimeout(() => {
            messageDiv.style.transition = 'all 0.3s ease';
            messageDiv.style.opacity = '1';
            messageDiv.style.transform = 'translateY(0)';
        }, 10);
    }

    function formatMessageTime(datetime) {
        const date = new Date(datetime);
        const now = new Date();
        const diff = now - date;

        if (diff < 60000) { // Less than 1 minute
            return 'Just now';
        } else if (diff < 3600000) { // Less than 1 hour
            const minutes = Math.floor(diff / 60000);
            return `${minutes}m ago`;
        } else if (diff < 86400000) { // Less than 1 day
            const hours = Math.floor(diff / 3600000);
            return `${hours}h ago`;
        } else {
            return date.toLocaleDateString() + ' ' + date.toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'});
        }
    }

    // Handle message sending with AJAX
    if (sendMessageForm) {
        sendMessageForm.addEventListener('submit', function(e) {
            e.preventDefault();

            const formData = new FormData(this);
            const message = messageInput.value.trim();

            if (!message) {
                return;
            }

            // Disable send button temporarily
            sendButton.disabled = true;
            sendButton.innerHTML = '<span class="send-icon">⏳</span> Sending...';

            fetch('headteacher_discussion.php?group=<?php echo $selected_group_id; ?>&chat=1', {
                method: 'POST',
                body: formData,
                headers: {
                    'X-Requested-With': 'XMLHttpRequest'
                }
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Clear input
                    messageInput.value = '';

                    // Add message to chat immediately (optimistic update)
                    addMessageToChat({
                        id: data.message_id,
                        message: message,
                        sender_name: '<?php echo addslashes($head_teacher['name']); ?>',
                        user_id: <?php echo $user_id; ?>,
                        created_at: new Date().toISOString()
                    });

                    // Update last message ID
                    lastMessageId = Math.max(lastMessageId, data.message_id);
                } else {
                    alert('Error sending message: ' + data.error);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error sending message. Please try again.');
            })
            .finally(() => {
                // Re-enable send button
                sendButton.disabled = false;
                sendButton.innerHTML = '<span class="send-icon">📤</span> Send';
            });
        });
    }

    // Auto-resize textarea
    if (messageInput) {
        messageInput.addEventListener('input', function() {
            this.style.height = 'auto';
            this.style.height = (this.scrollHeight) + 'px';
        });
    }

    // Toggle create group form
    toggleCreateGroup.addEventListener('click', function() {
        createGroupForm.style.display = createGroupForm.style.display === 'none' ? 'block' : 'none';
        if (createGroupForm.style.display === 'block') {
            toggleCreateGroup.innerHTML = '<span class="btn-icon">✕</span> Cancel';
        } else {
            toggleCreateGroup.innerHTML = '<span class="btn-icon">➕</span> Create Group';
        }
    });

    cancelCreateGroup.addEventListener('click', function() {
        createGroupForm.style.display = 'none';
        toggleCreateGroup.innerHTML = '<span class="btn-icon">➕</span> Create Group';
    });

    // Select all teachers functionality
    function setupSelectAll(checkbox, containerSelector) {
        checkbox.addEventListener('change', function() {
            const teacherCheckboxes = document.querySelectorAll(containerSelector + ' .teacher-checkbox-input');
            teacherCheckboxes.forEach(cb => {
                cb.checked = this.checked;
            });
        });
    }

    if (selectAllTeachers) {
        setupSelectAll(selectAllTeachers, '.teachers-selection');
    }

    if (selectAllAddTeachers) {
        setupSelectAll(selectAllAddTeachers, '.teachers-selection');
    }

    // Individual teacher checkbox change
    document.querySelectorAll('.teacher-checkbox-input').forEach(checkbox => {
        checkbox.addEventListener('change', function() {
            const container = this.closest('.teachers-selection');
            const selectAll = container.querySelector('input[id*="selectAll"]');
            const allCheckboxes = container.querySelectorAll('.teacher-checkbox-input');
            const checkedBoxes = container.querySelectorAll('.teacher-checkbox-input:checked');

            if (selectAll) {
                selectAll.checked = allCheckboxes.length === checkedBoxes.length;
                selectAll.indeterminate = checkedBoxes.length > 0 && checkedBoxes.length < allCheckboxes.length;
            }
        });
    });

    // Cleanup SSE connection when page is unloaded
    window.addEventListener('beforeunload', function() {
        if (eventSource) {
            eventSource.close();
        }
    });
});
</script>

<?php
include '../includes/admin_footer.php';
?>
