<?php
require_once '../config.php';

if (!isset($_SESSION['user_id'])) {
    header('Location: ../login.php');
    exit;
}

// Check if user is head teacher
if ($_SESSION['role'] != 'head_teacher') {
    header('Location: ../login.php');
    exit;
}

$username = $_SESSION['username'];

// Fetch head teacher information
$stmt = $pdo->prepare("SELECT *, CONCAT(first_name, ' ', last_name) as name FROM staff WHERE user_id = ? AND role = 'head_teacher'");
$stmt->execute([$_SESSION['user_id']]);
$head_teacher = $stmt->fetch();

// If no head teacher profile found, get user information as fallback
if (!$head_teacher) {
    $stmt = $pdo->prepare("SELECT username as name FROM users WHERE id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    $user_info = $stmt->fetch();
    $head_teacher = [
        'name' => $user_info ? $user_info['name'] : $_SESSION['username'],
        'first_name' => $_SESSION['username'],
        'last_name' => '',
        'email' => '',
        'phone' => '',
        'profile_image' => null
    ];
}

// Fetch statistics for head teacher dashboard
$total_students = $pdo->query("SELECT COUNT(*) FROM students")->fetchColumn();
$total_teachers = $pdo->query("SELECT COUNT(*) FROM staff WHERE role = 'teaching'")->fetchColumn();
$total_classes = $pdo->query("SELECT COUNT(*) FROM classes")->fetchColumn();

// Get recent activities
$recent_students = $pdo->query("SELECT CONCAT(s.first_name, ' ', COALESCE(s.middle_name, ''), ' ', s.last_name) as name, u.username, s.created_at FROM students s JOIN users u ON s.user_id = u.id ORDER BY s.created_at DESC LIMIT 3")->fetchAll();
$recent_teachers = $pdo->query("SELECT CONCAT_WS(' ', s.first_name, s.middle_name, s.last_name) AS name, u.username, s.created_at FROM staff s JOIN users u ON s.user_id = u.id WHERE s.role = 'teaching' ORDER BY s.created_at DESC LIMIT 3")->fetchAll();

// Get discussion groups count
$discussion_groups_count = $pdo->prepare("SELECT COUNT(*) FROM discussion_groups WHERE created_by = ?");
$discussion_groups_count->execute([$_SESSION['user_id']]);
$discussion_groups_count = $discussion_groups_count->fetchColumn();

include '../includes/admin_header.php';
?>

<div class="headteacher-dashboard-container">
    <div class="dashboard-header">
        <h1>🏫 Welcome back, <?php echo htmlspecialchars($head_teacher['name']); ?>!</h1>
        <p class="dashboard-subtitle">Head Teacher Dashboard - Manage your academic responsibilities</p>
    </div>

    <!-- Statistics Cards -->
    <div class="stats-grid">
        <div class="stat-card">
            <div class="stat-icon">👨‍🎓</div>
            <div class="stat-content">
                <h3><?php echo $total_students; ?></h3>
                <p>Total Students</p>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-icon">👨‍🏫</div>
            <div class="stat-content">
                <h3><?php echo $total_teachers; ?></h3>
                <p>Total Teachers</p>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-icon">🏫</div>
            <div class="stat-content">
                <h3><?php echo $total_classes; ?></h3>
                <p>Total Classes</p>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-icon">💬</div>
            <div class="stat-content">
                <h3><?php echo $discussion_groups_count; ?></h3>
                <p>Discussion Groups</p>
            </div>
        </div>
    </div>

    <!-- Quick Actions -->
    <div class="quick-actions">
        <h2>Quick Actions</h2>
        <div class="actions-grid">
            <a href="manage_students.php" class="action-card">
                <div class="action-icon">👨‍🎓</div>
                <div class="action-content">
                    <h4>Manage Students</h4>
                    <p>View and manage student information</p>
                </div>
            </a>

            <a href="manage_teachers.php" class="action-card">
                <div class="action-icon">👨‍🏫</div>
                <div class="action-content">
                    <h4>Manage Teachers</h4>
                    <p>View and manage teacher information</p>
                </div>
            </a>

            <a href="view_classes.php" class="action-card">
                <div class="action-icon">🏫</div>
                <div class="action-content">
                    <h4>View Classes</h4>
                    <p>Manage class assignments</p>
                </div>
            </a>

            <a href="headteacher_discussion.php" class="action-card">
                <div class="action-icon">💬</div>
                <div class="action-content">
                    <h4>Staff Discussion</h4>
                    <p>Manage discussion groups with teachers</p>
                </div>
            </a>

            <a href="promote_students.php" class="action-card">
                <div class="action-icon">📈</div>
                <div class="action-content">
                    <h4>Student Promotion</h4>
                    <p>Promote students to next academic level</p>
                </div>
            </a>



            <a href="manage_academic_years.php" class="action-card">
                <div class="action-icon">📅</div>
                <div class="action-content">
                    <h4>Academic Years</h4>
                    <p>Manage academic years and terms</p>
                </div>
            </a>

            <a href="manage_terms.php" class="action-card">
                <div class="action-icon">📋</div>
                <div class="action-content">
                    <h4>Terms/Semesters</h4>
                    <p>Manage academic terms and semesters</p>
                </div>
            </a>
        </div>
    </div>

    <!-- Recent Activity -->
    <div class="recent-activity">
        <div class="activity-section">
            <h3>Recently Added Students</h3>
            <?php if (empty($recent_students)): ?>
                <p>No recent student registrations.</p>
            <?php else: ?>
                <ul class="activity-list">
                    <?php foreach ($recent_students as $student): ?>
                        <li class="activity-item">
                            <div class="activity-icon">👨‍🎓</div>
                            <div class="activity-content">
                                <strong><?php echo htmlspecialchars($student['name']); ?></strong>
                                <small>Registered on <?php echo date('M j, Y', strtotime($student['created_at'])); ?></small>
                            </div>
                        </li>
                    <?php endforeach; ?>
                </ul>
            <?php endif; ?>
        </div>

        <div class="activity-section">
            <h3>Recently Added Teachers</h3>
            <?php if (empty($recent_teachers)): ?>
                <p>No recent teacher registrations.</p>
            <?php else: ?>
                <ul class="activity-list">
                    <?php foreach ($recent_teachers as $teacher): ?>
                        <li class="activity-item">
                            <div class="activity-icon">👨‍🏫</div>
                            <div class="activity-content">
                                <strong><?php echo htmlspecialchars($teacher['name']); ?></strong>
                                <small>Registered on <?php echo date('M j, Y', strtotime($teacher['created_at'])); ?></small>
                            </div>
                        </li>
                    <?php endforeach; ?>
                </ul>
            <?php endif; ?>
        </div>
    </div>
</div>

<style>
.headteacher-dashboard-container {
    max-width: 1400px;
    margin: 0 auto;
    padding: 20px;
}

.dashboard-header {
    text-align: center;
    margin-bottom: 40px;
    padding-bottom: 30px;
    border-bottom: 2px solid #e9ecef;
}

.dashboard-header h1 {
    margin: 0 0 10px 0;
    font-size: 2.8em;
    font-weight: 700;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.dashboard-subtitle {
    margin: 0;
    font-size: 1.3em;
    color: #6c757d;
    font-weight: 300;
}

.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
    margin-bottom: 40px;
}

.stat-card {
    background: #fff;
    border-radius: 12px;
    padding: 25px;
    box-shadow: 0 4px 15px rgba(0,0,0,0.1);
    border: 2px solid #e9ecef;
    display: flex;
    align-items: center;
    gap: 20px;
    transition: transform 0.3s ease, box-shadow 0.3s ease;
}

.stat-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(0,0,0,0.15);
}

.stat-icon {
    font-size: 2.5em;
    opacity: 0.8;
}

.stat-content h3 {
    margin: 0 0 5px 0;
    font-size: 2em;
    font-weight: 700;
    color: #495057;
}

.stat-content p {
    margin: 0;
    color: #6c757d;
    font-size: 1em;
    font-weight: 500;
}

.quick-actions {
    margin-bottom: 40px;
}

.quick-actions h2 {
    margin: 0 0 25px 0;
    color: #2c3e50;
    font-size: 2em;
    font-weight: 600;
    text-align: center;
}

.actions-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    gap: 20px;
}

.action-card {
    background: #fff;
    border-radius: 12px;
    padding: 25px;
    box-shadow: 0 4px 15px rgba(0,0,0,0.1);
    border: 2px solid #e9ecef;
    text-decoration: none;
    color: inherit;
    transition: transform 0.3s ease, box-shadow 0.3s ease;
    display: flex;
    align-items: center;
    gap: 20px;
}

.action-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(0,0,0,0.15);
    text-decoration: none;
    color: inherit;
}

.action-icon {
    font-size: 2.5em;
    opacity: 0.8;
}

.action-content h4 {
    margin: 0 0 8px 0;
    font-size: 1.3em;
    font-weight: 600;
    color: #495057;
}

.action-content p {
    margin: 0;
    color: #6c757d;
    font-size: 1em;
    line-height: 1.4;
}

.recent-activity {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 30px;
}

.activity-section {
    background: #fff;
    border-radius: 12px;
    padding: 25px;
    box-shadow: 0 4px 15px rgba(0,0,0,0.1);
    border: 2px solid #e9ecef;
}

.activity-section h3 {
    margin: 0 0 20px 0;
    color: #495057;
    font-size: 1.4em;
    font-weight: 600;
}

.activity-list {
    list-style: none;
    padding: 0;
    margin: 0;
}

.activity-item {
    display: flex;
    align-items: center;
    gap: 15px;
    padding: 12px 0;
    border-bottom: 1px solid #e9ecef;
}

.activity-item:last-child {
    border-bottom: none;
}

.activity-icon {
    font-size: 1.5em;
    opacity: 0.8;
}

.activity-content {
    flex: 1;
}

.activity-content strong {
    color: #495057;
    font-weight: 600;
}

.activity-content small {
    color: #6c757d;
    font-size: 0.9em;
}

/* Responsive Design */
@media (max-width: 768px) {
    .headteacher-dashboard-container {
        padding: 10px;
    }

    .dashboard-header h1 {
        font-size: 2.2em;
    }

    .stats-grid {
        grid-template-columns: 1fr;
    }

    .actions-grid {
        grid-template-columns: 1fr;
    }

    .recent-activity {
        grid-template-columns: 1fr;
        gap: 20px;
    }

    .action-card {
        flex-direction: column;
        text-align: center;
        gap: 15px;
    }
}

@media (max-width: 480px) {
    .dashboard-header h1 {
        font-size: 1.8em;
    }

    .quick-actions h2 {
        font-size: 1.6em;
    }

    .stat-card,
    .action-card,
    .activity-section {
        padding: 20px;
    }
}
</style>

<?php
include '../includes/admin_footer.php';
?>
