<?php
require_once '../config.php';
require_once '../includes/access_control.php';
require_once '../includes/functions.php';
require_once '../auto_enroll_students.php';

requireFeatureAccess('enrollment_management');

$message = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    // Get form data
    $username = trim($_POST['username']);
    $password = $_POST['password'];
    $confirm_password = $_POST['confirm_password'];

    // Student Information
    $first_name = trim($_POST['first_name']);
    $middle_name = trim($_POST['middle_name']);
    $last_name = trim($_POST['last_name']);
    $dob = $_POST['dob'];
    $gender = $_POST['gender'];
    $place_of_birth_city = trim($_POST['place_of_birth_city']);
    $place_of_birth_country = trim($_POST['place_of_birth_country']);
    $nationality = trim($_POST['nationality']);
    $home_street = trim($_POST['home_street']);
    $home_city = trim($_POST['home_city']);
    $home_state = trim($_POST['home_state']);
    $home_postal = trim($_POST['home_postal']);
    $home_country = trim($_POST['home_country']);
    $student_id_field = trim($_POST['student_id']);
    $previous_school_name = trim($_POST['previous_school_name']);
    $previous_school_address = trim($_POST['previous_school_address']);
    $class = trim($_POST['class']);
    $academic_year = trim($_POST['academic_year']);
    $term_admission = trim($_POST['term_admission']);
    $languages_home = trim($_POST['languages_home']);
    $religion = trim($_POST['religion']);
    $phone = trim($_POST['phone']);

    // Parents Information (array for two parents)
    $parents = [];
    for ($i = 1; $i <= 2; $i++) {
        $parents[] = [
            'full_name' => trim($_POST["parent_{$i}_full_name"] ?? ''),
            'relationship' => trim($_POST["parent_{$i}_relationship"] ?? ''),
            'occupation' => trim($_POST["parent_{$i}_occupation"] ?? ''),
            'employer' => trim($_POST["parent_{$i}_employer"] ?? ''),
            'work_address' => trim($_POST["parent_{$i}_work_address"] ?? ''),
            'mobile_phone' => trim($_POST["parent_{$i}_mobile_phone"] ?? ''),
            'home_phone' => trim($_POST["parent_{$i}_home_phone"] ?? ''),
            'work_phone' => trim($_POST["parent_{$i}_work_phone"] ?? ''),
            'email' => trim($_POST["parent_{$i}_email"] ?? ''),
            'education_attainment' => trim($_POST["parent_{$i}_education_attainment"] ?? ''),
            'marital_status' => trim($_POST["parent_{$i}_marital_status"] ?? ''),
            'authorized_pickup' => $_POST["parent_{$i}_authorized_pickup"] ?? 'No'
        ];
    }

    // Emergency Contact
    $emergency_full_name = trim($_POST['emergency_full_name']);
    $emergency_relationship = trim($_POST['emergency_relationship']);
    $emergency_phone_numbers = trim($_POST['emergency_phone_numbers']);
    $emergency_address = trim($_POST['emergency_address']);
    $emergency_authorization_medical = $_POST['emergency_authorization_medical'] ?? 'No';

    // Validation
    if (empty($username) || empty($password) || empty($first_name) || empty($last_name) || empty($class) || empty($academic_year)) {
        $error = 'Please fill in all required fields.';
    } elseif ($password !== $confirm_password) {
        $error = 'Passwords do not match.';
    } else {
        // Use PasswordSecurity class for validation
        require_once '../includes/password_security.php';
        $password_errors = PasswordSecurity::validatePasswordStrength($password);
        if (!empty($password_errors)) {
            $error = 'Password does not meet requirements: ' . implode(', ', $password_errors);
        } else {
        try {
            // Check if username already exists
            $stmt = $pdo->prepare("SELECT id FROM student_users WHERE username = ?");
            $stmt->execute([$username]);
            if ($stmt->fetch()) {
                $error = 'Username already exists.';
            } else {
                // Handle photo upload
                $photo_path = null;
                if (isset($_FILES['student_photo']) && $_FILES['student_photo']['error'] == UPLOAD_ERR_OK) {
                    $upload_dir = '../uploads/students/';
                    if (!is_dir($upload_dir)) {
                        mkdir($upload_dir, 0755, true);
                    }
                    $file_name = uniqid() . '_' . basename($_FILES['student_photo']['name']);
                    $target_file = $upload_dir . $file_name;
                    $imageFileType = strtolower(pathinfo($target_file, PATHINFO_EXTENSION));

                    // Validate image
                    $check = getimagesize($_FILES['student_photo']['tmp_name']);
                    if ($check !== false && in_array($imageFileType, ['jpg', 'jpeg', 'png', 'gif']) && $_FILES['student_photo']['size'] < 5000000) {
                        if (move_uploaded_file($_FILES['student_photo']['tmp_name'], $target_file)) {
                            $photo_path = 'uploads/students/' . $file_name;
                        } else {
                            $error = 'Failed to upload photo.';
                        }
                    } else {
                        $error = 'Invalid photo file. Must be JPG, JPEG, PNG, GIF and less than 5MB.';
                    }
                }

                if (!$error) {
                    // Generate student ID if not provided
                    if (empty($student_id_field)) {
                        $student_id_field = generate_student_id($pdo);
                    }

                    // Insert student user
                    $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                    $stmt = $pdo->prepare("INSERT INTO student_users (username, password, created_at) VALUES (?, ?, NOW())");
                    $stmt->execute([$username, $hashed_password]);
                    $user_id = $pdo->lastInsertId();

                    // Insert student
                    $stmt = $pdo->prepare("INSERT INTO students (user_id, first_name, middle_name, last_name, dob, gender, place_of_birth_city, place_of_birth_country, nationality, home_street, home_city, home_state, home_postal, home_country, student_id, previous_school_name, previous_school_address, class, academic_year, term_admission, languages_home, religion, phone, photo_path) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
                    $stmt->execute([$user_id, $first_name, $middle_name, $last_name, $dob, $gender, $place_of_birth_city, $place_of_birth_country, $nationality, $home_street, $home_city, $home_state, $home_postal, $home_country, $student_id_field, $previous_school_name, $previous_school_address, $class, $academic_year, $term_admission, $languages_home, $religion, $phone, $photo_path]);

                    $student_id = $pdo->lastInsertId();

                    // Insert parents
                    foreach ($parents as $parent) {
                        if (!empty($parent['full_name'])) {
                            $stmt = $pdo->prepare("INSERT INTO parents (student_id, full_name, relationship, occupation, employer, work_address, mobile_phone, home_phone, work_phone, email, education_attainment, marital_status, authorized_pickup) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
                            $stmt->execute([$student_id, $parent['full_name'], $parent['relationship'], $parent['occupation'], $parent['employer'], $parent['work_address'], $parent['mobile_phone'], $parent['home_phone'], $parent['work_phone'], $parent['email'], $parent['education_attainment'], $parent['marital_status'], $parent['authorized_pickup']]);
                        }
                    }

                    // Insert emergency contact
                    if (!empty($emergency_full_name)) {
                        $stmt = $pdo->prepare("INSERT INTO emergency_contacts (student_id, full_name, relationship, phone_numbers, address, authorization_medical) VALUES (?, ?, ?, ?, ?, ?)");
                        $stmt->execute([$student_id, $emergency_full_name, $emergency_relationship, $emergency_phone_numbers, $emergency_address, $emergency_authorization_medical]);
                    }

                    // Auto-enroll student in all courses for their class
                    $enrollment_result = autoEnrollStudent($student_id, $class);

                    if ($enrollment_result['success']) {
                        $message = 'Student enrolled successfully! ' . $enrollment_result['message'];
                    } else {
                        $message = 'Student enrolled successfully, but auto-enrollment failed: ' . $enrollment_result['message'];
                    }
                }
            }
        } catch (Exception $e) {
            $error = 'An error occurred while enrolling the student: ' . $e->getMessage();
        }
    }
}
}

// Get academic years for dropdown
$stmt = $pdo->query("SELECT year FROM academic_years ORDER BY year DESC");
$academic_years = $stmt->fetchAll(PDO::FETCH_COLUMN);

// Get classes for dropdown
$stmt = $pdo->query("SELECT name FROM classes ORDER BY name");
$classes = $stmt->fetchAll(PDO::FETCH_COLUMN);

include '../includes/admin_header.php';
?>

<style>
    .enrollment-form {
        max-width: 900px;
        margin: 0 auto;
        background: #fff;
        border-radius: 15px;
        box-shadow: 0 10px 30px rgba(0,0,0,0.1);
        overflow: hidden;
    }

    .form-header {
        background: linear-gradient(135deg, #0f6da3ff 0%, #0e6699ff 100%);
        color: white;
        padding: 30px;
        text-align: center;
    }

    .form-header h2 {
        margin: 0;
        font-size: 2.2em;
        font-weight: 700;
    }

    .form-header p {
        margin: 10px 0 0;
        opacity: 0.9;
        font-size: 1.1em;
    }

    .form-sections {
        padding: 40px;
    }

    .form-section {
        margin-bottom: 40px;
        border: 1px solid #e1e8ed;
        border-radius: 12px;
        overflow: hidden;
        transition: all 0.3s ease;
    }

    .form-section:hover {
        box-shadow: 0 5px 15px rgba(0,0,0,0.08);
        border-color: #667eea;
    }

    .section-header {
        background: linear-gradient(135deg, #08678dff 0%, #2382dbff 100%);
        color: white;
        padding: 20px 30px;
        display: flex;
        align-items: center;
        font-size: 1.2em;
        font-weight: 600;
    }

    .section-icon {
        font-size: 1.5em;
        margin-right: 15px;
    }

    .section-content {
        padding: 30px;
        background: #fafbfc;
    }

    .form-row {
        display: flex;
        gap: 20px;
        margin-bottom: 20px;
    }

    .form-group {
        flex: 1;
        position: relative;
    }

    .form-group.full-width {
        flex: none;
        width: 100%;
    }

    .form-group label {
        display: block;
        margin-bottom: 8px;
        font-weight: 600;
        color: #2c3e50;
        font-size: 0.95em;
    }

    .form-group input,
    .form-group select,
    .form-group textarea {
        width: 100%;
        padding: 15px 20px;
        border: 2px solid #e1e8ed;
        border-radius: 8px;
        font-size: 1em;
        transition: all 0.3s ease;
        background: white;
        box-sizing: border-box;
    }

    .form-group input:focus,
    .form-group select:focus,
    .form-group textarea:focus {
        border-color: #667eea;
        box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        outline: none;
    }

    .form-group textarea {
        resize: vertical;
        min-height: 80px;
    }

    .required::after {
        content: ' *';
        color: #e74c3c;
        font-weight: bold;
    }

    .form-actions {
        text-align: center;
        padding: 30px 0;
        border-top: 1px solid #e1e8ed;
    }

    .btn-submit {
        background: linear-gradient(135deg, #667eea 0%, #2382dbff  100%);
        color: white;
        border: none;
        padding: 15px 40px;
        border-radius: 25px;
        font-size: 1.1em;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.3s ease;
        box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
    }

    .btn-submit:hover {
        transform: translateY(-2px);
        box-shadow: 0 6px 20px rgba(102, 126, 234, 0.4);
    }

    .btn-submit:active {
        transform: translateY(0);
    }

    .password-requirements {
        margin-top: 5px;
        font-size: 14px;
    }

    .requirement {
        display: flex;
        justify-content: space-between;
        margin-bottom: 2px;
    }

    .requirement .check {
        font-weight: bold;
    }

    .requirement.met .check {
        color: green;
    }

    .requirement.not-met .check {
        color: red;
    }

    .message, .error-message {
        padding: 15px 20px;
        border-radius: 8px;
        margin-bottom: 20px;
        font-weight: 500;
    }

    .message {
        background: #d4edda;
        color: #155724;
        border: 1px solid #c3e6cb;
    }

    .error-message {
        background: #f8d7da;
        color: #721c24;
        border: 1px solid #f5c6cb;
    }

    @media (max-width: 768px) {
        .form-sections {
            padding: 20px;
        }

        .form-row {
            flex-direction: column;
            gap: 0;
        }

        .section-content {
            padding: 20px;
        }

        .section-header {
            padding: 15px 20px;
            font-size: 1.1em;
        }
    }
</style>

<h2>Enroll New Student</h2>

<?php if ($message): ?>
    <div class="message"><?php echo htmlspecialchars($message); ?></div>
<?php endif; ?>

<?php if ($error): ?>
    <div class="error-message"><?php echo htmlspecialchars($error); ?></div>
<?php endif; ?>

<div class="enrollment-form">
    <div class="form-header">
        <h2>🎓 Student Enrollment</h2>
        <p>Complete the form below to enroll a new student</p>
    </div>

    <form method="post" action="enroll_students.php" enctype="multipart/form-data" class="form-sections">
        <!-- Account Information Section -->
        <div class="form-section">
            <div class="section-header">
                <span class="section-icon">🔐</span>
                Account Information
            </div>
            <div class="section-content">
                <div class="form-row">
                    <div class="form-group">
                        <label for="username" class="required">Username</label>
                        <input type="text" id="username" name="username" required>
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="password" class="required">Password</label>
                        <input type="password" id="password" name="password" required>
                        <div id="password-requirements" class="password-requirements">
                            <div id="req-length" class="requirement">At least 8 characters <span class="check">✗</span></div>
                            <div id="req-uppercase" class="requirement">One uppercase letter <span class="check">✗</span></div>
                            <div id="req-lowercase" class="requirement">One lowercase letter <span class="check">✗</span></div>
                            <div id="req-number" class="requirement">One number <span class="check">✗</span></div>
                            <div id="req-special" class="requirement">One special character <span class="check">✗</span></div>
                        </div>
                    </div>
                    <div class="form-group">
                        <label for="confirm_password" class="required">Confirm Password</label>
                        <input type="password" id="confirm_password" name="confirm_password" required>
                        <div id="confirm-password-match" class="password-requirements">
                            <div id="req-match" class="requirement">Passwords match <span class="check">✗</span></div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Student Information Section -->
        <div class="form-section">
            <div class="section-header">
                <span class="section-icon">🧒</span>
                Student Information
            </div>
            <div class="section-content">
                <!-- Full Name -->
                <div class="form-row">
                    <div class="form-group">
                        <label for="first_name" class="required">First Name</label>
                        <input type="text" id="first_name" name="first_name" required>
                    </div>
                    <div class="form-group">
                        <label for="middle_name">Middle Name</label>
                        <input type="text" id="middle_name" name="middle_name">
                    </div>
                    <div class="form-group">
                        <label for="last_name" class="required">Last Name</label>
                        <input type="text" id="last_name" name="last_name" required>
                    </div>
                </div>

                <!-- Date of Birth and Gender -->
                <div class="form-row">
                    <div class="form-group">
                        <label for="dob">Date of Birth</label>
                        <input type="date" id="dob" name="dob">
                    </div>
                    <div class="form-group">
                        <label for="gender">Gender</label>
                        <select id="gender" name="gender">
                            <option value="">Select Gender</option>
                            <option value="Male">Male</option>
                            <option value="Female">Female</option>
                            <option value="Other">Other</option>
                        </select>
                    </div>
                </div>

                <!-- Place of Birth -->
                <div class="form-row">
                    <div class="form-group">
                        <label for="place_of_birth_city">Place of Birth (City)</label>
                        <input type="text" id="place_of_birth_city" name="place_of_birth_city">
                    </div>
                    <div class="form-group">
                        <label for="place_of_birth_country">Place of Birth (Country)</label>
                        <input type="text" id="place_of_birth_country" name="place_of_birth_country">
                    </div>
                </div>

                <!-- Nationality and Home Address -->
                <div class="form-row">
                    <div class="form-group">
                        <label for="nationality">Nationality / Citizenship</label>
                        <input type="text" id="nationality" name="nationality">
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="home_street">Home Address (Street)</label>
                        <input type="text" id="home_street" name="home_street">
                    </div>
                    <div class="form-group">
                        <label for="home_city">Home Address (City)</label>
                        <input type="text" id="home_city" name="home_city">
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="home_state">Home Address (State/Province)</label>
                        <input type="text" id="home_state" name="home_state">
                    </div>
                    <div class="form-group">
                        <label for="home_postal">Home Address (Postal Code)</label>
                        <input type="text" id="home_postal" name="home_postal">
                    </div>
                    <div class="form-group">
                        <label for="home_country">Home Address (Country)</label>
                        <input type="text" id="home_country" name="home_country">
                    </div>
                </div>

                <!-- Student ID and Previous School -->
                <div class="form-row">
                    <div class="form-group">
                        <label for="student_id">Student ID (if returning student)</label>
                        <input type="text" id="student_id" name="student_id">
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group full-width">
                        <label for="previous_school_name">Previous School Name</label>
                        <input type="text" id="previous_school_name" name="previous_school_name">
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group full-width">
                        <label for="previous_school_address">Previous School Address</label>
                        <textarea id="previous_school_address" name="previous_school_address" rows="2"></textarea>
                    </div>
                </div>

                <!-- Class, Academic Year, Term -->
                <div class="form-row">
                    <div class="form-group">
                        <label for="class" class="required">Grade / Class Applying For</label>
                        <select id="class" name="class" required>
                            <option value="">Select Class/Grade</option>
                            <?php foreach ($classes as $class_name): ?>
                                <option value="<?php echo htmlspecialchars($class_name); ?>"><?php echo htmlspecialchars($class_name); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="academic_year" class="required">Academic Year</label>
                        <select id="academic_year" name="academic_year" required>
                            <option value="">Select Academic Year</option>
                            <?php foreach ($academic_years as $year): ?>
                                <option value="<?php echo htmlspecialchars($year); ?>"><?php echo htmlspecialchars($year); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="term_admission">Academic Year / Term of Admission</label>
                        <input type="text" id="term_admission" name="term_admission">
                    </div>
                </div>

                <!-- Languages, Religion, Phone -->
                <div class="form-row">
                    <div class="form-group">
                        <label for="languages_home">Language(s) Spoken at Home</label>
                        <input type="text" id="languages_home" name="languages_home">
                    </div>
                    <div class="form-group">
                        <label for="religion">Religion / Faith (if applicable)</label>
                        <input type="text" id="religion" name="religion">
                    </div>
                    <div class="form-group">
                        <label for="phone">Phone Number</label>
                        <input type="tel" id="phone" name="phone">
                    </div>
                </div>

                <!-- Student Photo -->
                <div class="form-row">
                    <div class="form-group full-width">
                        <label for="student_photo">Student Photo (passport-size)</label>
                        <input type="file" id="student_photo" name="student_photo" accept="image/*">
                        <small>Accepted formats: JPG, JPEG, PNG, GIF. Max size: 5MB</small>
                    </div>
                </div>
            </div>
        </div>

        <!-- Parent / Guardian Information Section -->
        <div class="form-section">
            <div class="section-header">
                <span class="section-icon">👨‍👩‍👧</span>
                Parent / Guardian Information
            </div>
            <div class="section-content">
                <?php for ($i = 1; $i <= 2; $i++): ?>
                <h4>Parent/Guardian <?php echo $i; ?></h4>
                <div class="form-row">
                    <div class="form-group">
                        <label for="parent_<?php echo $i; ?>_full_name">Full Name</label>
                        <input type="text" id="parent_<?php echo $i; ?>_full_name" name="parent_<?php echo $i; ?>_full_name">
                    </div>
                    <div class="form-group">
                        <label for="parent_<?php echo $i; ?>_relationship">Relationship to Student</label>
                        <select id="parent_<?php echo $i; ?>_relationship" name="parent_<?php echo $i; ?>_relationship">
                            <option value="">Select Relationship</option>
                            <option value="Father">Father</option>
                            <option value="Mother">Mother</option>
                            <option value="Guardian">Guardian</option>
                            <option value="Other">Other</option>
                        </select>
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="parent_<?php echo $i; ?>_occupation">Occupation / Job Title</label>
                        <input type="text" id="parent_<?php echo $i; ?>_occupation" name="parent_<?php echo $i; ?>_occupation">
                    </div>
                    <div class="form-group">
                        <label for="parent_<?php echo $i; ?>_employer">Employer / Company Name</label>
                        <input type="text" id="parent_<?php echo $i; ?>_employer" name="parent_<?php echo $i; ?>_employer">
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group full-width">
                        <label for="parent_<?php echo $i; ?>_work_address">Work Address</label>
                        <textarea id="parent_<?php echo $i; ?>_work_address" name="parent_<?php echo $i; ?>_work_address" rows="2"></textarea>
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="parent_<?php echo $i; ?>_mobile_phone">Contact Numbers (Mobile)</label>
                        <input type="tel" id="parent_<?php echo $i; ?>_mobile_phone" name="parent_<?php echo $i; ?>_mobile_phone">
                    </div>
                    <div class="form-group">
                        <label for="parent_<?php echo $i; ?>_home_phone">Contact Numbers (Home)</label>
                        <input type="tel" id="parent_<?php echo $i; ?>_home_phone" name="parent_<?php echo $i; ?>_home_phone">
                    </div>
                    <div class="form-group">
                        <label for="parent_<?php echo $i; ?>_work_phone">Contact Numbers (Work)</label>
                        <input type="tel" id="parent_<?php echo $i; ?>_work_phone" name="parent_<?php echo $i; ?>_work_phone">
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="parent_<?php echo $i; ?>_email">Email Address</label>
                        <input type="email" id="parent_<?php echo $i; ?>_email" name="parent_<?php echo $i; ?>_email">
                    </div>
                    <div class="form-group">
                        <label for="parent_<?php echo $i; ?>_education_attainment">Highest Educational Attainment</label>
                        <select id="parent_<?php echo $i; ?>_education_attainment" name="parent_<?php echo $i; ?>_education_attainment">
                            <option value="">Select Education</option>
                            <option value="High School">High School</option>
                            <option value="Associate Degree">Associate Degree</option>
                            <option value="Bachelor's Degree">Bachelor's Degree</option>
                            <option value="Master's Degree">Master's Degree</option>
                            <option value="Doctorate">Doctorate</option>
                            <option value="Other">Other</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="parent_<?php echo $i; ?>_marital_status">Marital Status</label>
                        <select id="parent_<?php echo $i; ?>_marital_status" name="parent_<?php echo $i; ?>_marital_status">
                            <option value="">Select Status</option>
                            <option value="Married">Married</option>
                            <option value="Single">Single</option>
                            <option value="Divorced">Divorced</option>
                            <option value="Widowed">Widowed</option>
                            <option value="Separated">Separated</option>
                        </select>
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="parent_<?php echo $i; ?>_authorized_pickup">Authorized to Pick Up Child?</label>
                        <select id="parent_<?php echo $i; ?>_authorized_pickup" name="parent_<?php echo $i; ?>_authorized_pickup">
                            <option value="Yes">Yes</option>
                            <option value="No">No</option>
                        </select>
                    </div>
                </div>
                <hr style="margin: 20px 0;">
                <?php endfor; ?>
            </div>
        </div>

        <!-- Emergency Contact Information Section -->
        <div class="form-section">
            <div class="section-header">
                <span class="section-icon">🏠</span>
                Emergency Contact Information
            </div>
            <div class="section-content">
                <div class="form-row">
                    <div class="form-group">
                        <label for="emergency_full_name">Full Name</label>
                        <input type="text" id="emergency_full_name" name="emergency_full_name">
                    </div>
                    <div class="form-group">
                        <label for="emergency_relationship">Relationship to Student</label>
                        <input type="text" id="emergency_relationship" name="emergency_relationship">
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="emergency_phone_numbers">Phone Number(s)</label>
                        <input type="text" id="emergency_phone_numbers" name="emergency_phone_numbers">
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group full-width">
                        <label for="emergency_address">Address</label>
                        <textarea id="emergency_address" name="emergency_address" rows="2"></textarea>
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="emergency_authorization_medical">Authorization for Medical Treatment</label>
                        <select id="emergency_authorization_medical" name="emergency_authorization_medical">
                            <option value="Yes">Yes</option>
                            <option value="No">No</option>
                        </select>
                    </div>
                </div>
            </div>
        </div>

        <!-- Form Actions -->
        <div class="form-actions">
            <button type="submit" class="btn-submit">🎓 Enroll Student</button>
        </div>
    </form>
</div>

<script>
    // Password strength validation
    function checkPasswordStrength(password) {
        const requirements = {
            length: password.length >= 8,
            uppercase: /[A-Z]/.test(password),
            lowercase: /[a-z]/.test(password),
            number: /\d/.test(password),
            special: /[!@#$%^&*()_+\-=\[\]{};':"\\|,.<>\/?]/.test(password)
        };
        return requirements;
    }

    function updatePasswordRequirements(password) {
        const requirements = checkPasswordStrength(password);
        document.getElementById('req-length').className = requirements.length ? 'requirement met' : 'requirement not-met';
        document.getElementById('req-length').querySelector('.check').textContent = requirements.length ? '✓' : '✗';
        document.getElementById('req-uppercase').className = requirements.uppercase ? 'requirement met' : 'requirement not-met';
        document.getElementById('req-uppercase').querySelector('.check').textContent = requirements.uppercase ? '✓' : '✗';
        document.getElementById('req-lowercase').className = requirements.lowercase ? 'requirement met' : 'requirement not-met';
        document.getElementById('req-lowercase').querySelector('.check').textContent = requirements.lowercase ? '✓' : '✗';
        document.getElementById('req-number').className = requirements.number ? 'requirement met' : 'requirement not-met';
        document.getElementById('req-number').querySelector('.check').textContent = requirements.number ? '✓' : '✗';
        document.getElementById('req-special').className = requirements.special ? 'requirement met' : 'requirement not-met';
        document.getElementById('req-special').querySelector('.check').textContent = requirements.special ? '✓' : '✗';
    }

    function updateConfirmPasswordMatch() {
        const password = document.getElementById('password').value;
        const confirmPassword = document.getElementById('confirm_password').value;
        const match = password === confirmPassword && password !== '';
        document.getElementById('req-match').className = match ? 'requirement met' : 'requirement not-met';
        document.getElementById('req-match').querySelector('.check').textContent = match ? '✓' : '✗';
    }

    // Password input event listener
    document.getElementById('password').addEventListener('input', function() {
        updatePasswordRequirements(this.value);
        updateConfirmPasswordMatch();
    });

    // Password confirmation validation
    document.getElementById('confirm_password').addEventListener('input', function() {
        const password = document.getElementById('password').value;
        const confirmPassword = this.value;

        if (password !== confirmPassword) {
            this.setCustomValidity('Passwords do not match');
        } else {
            this.setCustomValidity('');
        }
        updateConfirmPasswordMatch();
    });

    // Form validation enhancement
    document.querySelector('form').addEventListener('submit', function(e) {
        const password = document.getElementById('password').value;
        const confirmPassword = document.getElementById('confirm_password').value;

        if (password !== confirmPassword) {
            e.preventDefault();
            alert('Passwords do not match. Please try again.');
        }
    });
</script>

<?php
include '../includes/admin_footer.php';
?>
