<?php
require_once '../config.php';
require_once '../includes/access_control.php';
require_once 'asset_audit_functions.php';

// Access control temporarily disabled for asset audit logs
// requireFeatureAccess('accounts_management');

include 'includes/accounting_header.php';

$message = '';
$error = '';

// Get filter parameters
$asset_id = $_GET['asset_id'] ?? '';
$user_id = $_GET['user_id'] ?? '';
$action = $_GET['action'] ?? '';
$date_from = $_GET['date_from'] ?? '';
$date_to = $_GET['date_to'] ?? '';
$page = max(1, intval($_GET['page'] ?? 1));
$limit = 50;
$offset = ($page - 1) * $limit;

// Get audit logs with filters
try {
    $audit_logs = getAssetAuditLogs($pdo, $asset_id, $user_id, $action, $limit);
    $total_count = count($audit_logs); // For now, just use the count of returned logs
    $total_pages = 1; // Simplified pagination for now
} catch (Exception $e) {
    $error = "Error loading audit logs: " . $e->getMessage();
    $audit_logs = [];
    $total_count = 0;
    $total_pages = 0;
}

// Get assets for filter dropdown
try {
    $stmt = $pdo->query("SELECT id, name FROM assets ORDER BY name");
    $assets = $stmt->fetchAll();
} catch (PDOException $e) {
    $assets = [];
}

// Get users for filter dropdown
try {
    $stmt = $pdo->query("SELECT id, username FROM users WHERE role IN ('admin', 'accounts') ORDER BY username");
    $users = $stmt->fetchAll();
} catch (PDOException $e) {
    $users = [];
}

// Get audit statistics
try {
    $stats = getAssetAuditStats($pdo);
} catch (Exception $e) {
    $stats = [
        'total_logs' => 0,
        'today_logs' => 0,
        'unique_users' => 0,
        'unique_assets' => 0
    ];
}
?>

<style>
.audit-logs-container {
    max-width: 1400px;
    margin: 0 auto;
    padding: 0;
    background: transparent;
    min-height: auto;
    box-shadow: none;
}

.audit-header {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: white;
    padding: 30px;
    border-radius: 12px;
    margin-bottom: 30px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.1);
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.audit-header h1 {
    margin: 0;
    font-size: 2.2em;
    font-weight: 300;
}

.audit-header p {
    margin: 5px 0 0 0;
    opacity: 0.9;
    font-size: 1.1em;
}

.audit-content {
    display: grid;
    grid-template-columns: 1fr 320px;
    gap: 30px;
}

.audit-main {
    background: white;
    border-radius: 12px;
    box-shadow: 0 2px 15px rgba(0,0,0,0.08);
    border: 1px solid #e1e5e9;
}

.filters-section {
    background: #f8f9fa;
    padding: 25px;
    border-bottom: 1px solid #dee2e6;
}

.filters-form {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
    gap: 20px;
    align-items: end;
}

.filter-group {
    display: flex;
    flex-direction: column;
}

.filter-group label {
    font-weight: 600;
    margin-bottom: 8px;
    color: #495057;
    font-size: 0.95em;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.filter-group input,
.filter-group select {
    padding: 12px 16px;
    border: 2px solid #2E7D32;
    border-radius: 8px;
    font-size: 14px;
    transition: border-color 0.3s ease;
}

.filter-group input:focus,
.filter-group select:focus {
    outline: none;
    border-color: #4CAF50;
    box-shadow: 0 0 0 3px rgba(76, 175, 80, 0.1);
}

.filter-actions {
    display: flex;
    gap: 12px;
}

.btn {
    padding: 12px 24px;
    border: none;
    border-radius: 8px;
    cursor: pointer;
    font-size: 14px;
    font-weight: 600;
    text-decoration: none;
    display: inline-block;
    transition: all 0.3s ease;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.btn-primary {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(46, 125, 50, 0.3);
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(46, 125, 50, 0.4);
}

.btn-secondary {
    background: #6c757d;
    color: white;
}

.btn-secondary:hover {
    background: #5a6268;
    transform: translateY(-1px);
}

.audit-table {
    width: 100%;
    border-collapse: collapse;
}

.audit-table th,
.audit-table td {
    padding: 16px 20px;
    text-align: left;
    border-bottom: 1px solid #e9ecef;
}

.audit-table th {
    background: #fff;
    font-weight: 700;
    color: #000;
    position: sticky;
    top: 0;
    text-transform: uppercase;
    font-size: 0.85em;
    letter-spacing: 0.5px;
}

.audit-table tr:hover {
    background: #f8f9fa;
}

.audit-action {
    font-weight: 600;
    color: #2c3e50;
}

.audit-user {
    color: #495057;
    font-weight: 500;
}

.audit-asset {
    color: #2c3e50;
    font-weight: 500;
    max-width: 200px;
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
}

.audit-date {
    color: #6c757d;
    font-size: 0.9em;
    font-weight: 500;
}

.audit-ip {
    font-family: 'Courier New', monospace;
    font-size: 0.85em;
    color: #6c757d;
}

.action-badge {
    padding: 4px 12px;
    border-radius: 20px;
    font-size: 0.8em;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    display: inline-block;
}

.action-create {
    background: #d4edda;
    color: #155724;
}

.action-update {
    background: #fff3cd;
    color: #856404;
}

.action-view {
    background: #d1ecf1;
    color: #0c5460;
}

.action-delete {
    background: #f8d7da;
    color: #721c24;
}

.summary-cards {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 20px;
    margin-bottom: 25px;
}

.summary-card {
    background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
    padding: 20px;
    border-radius: 12px;
    text-align: center;
    border: 1px solid #e1e5e9;
    box-shadow: 0 2px 10px rgba(0,0,0,0.05);
    transition: transform 0.3s ease;
}

.summary-card:hover {
    transform: translateY(-3px);
    box-shadow: 0 4px 20px rgba(0,0,0,0.1);
}

.summary-card h3 {
    margin: 0 0 8px 0;
    font-size: 2em;
    color: #2c3e50;
    font-weight: 700;
}

.summary-card p {
    margin: 0;
    color: #6c757d;
    font-size: 0.95em;
    font-weight: 500;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.pagination {
    display: flex;
    justify-content: center;
    gap: 8px;
    margin-top: 25px;
    padding-top: 25px;
    border-top: 1px solid #dee2e6;
}

.pagination a,
.pagination span {
    padding: 10px 16px;
    border: 2px solid #e9ecef;
    border-radius: 8px;
    text-decoration: none;
    color: #2E7D32;
    font-weight: 600;
    transition: all 0.3s ease;
}

.pagination .active {
    background: #2E7D32;
    color: white;
    border-color: #2E7D32;
}

.pagination .disabled {
    color: #6c757d;
    cursor: not-allowed;
}

.message {
    padding: 15px;
    margin-bottom: 20px;
    border-radius: 8px;
    font-weight: 600;
}

.message.success {
    background: #d4edda;
    color: #155724;
    border: 1px solid #c3e6cb;
}

.message.error {
    background: #f8d7da;
    color: #721c24;
    border: 1px solid #f5c6cb;
}

.no-logs {
    text-align: center;
    padding: 50px 20px;
    color: #6c757d;
}

.no-logs h3 {
    margin: 0 0 10px 0;
    color: #495057;
}

.audit-sidebar {
    background: white;
    border-radius: 12px;
    box-shadow: 0 2px 15px rgba(0,0,0,0.08);
    padding: 25px;
    border: 1px solid #e1e5e9;
}

.audit-sidebar h3 {
    margin-top: 0;
    color: #2c3e50;
    margin-bottom: 20px;
}

.quick-actions {
    display: flex;
    flex-direction: column;
    gap: 10px;
}

.quick-actions .btn {
    text-align: center;
    padding: 12px;
}

.changes-details {
    background: #f8f9fa;
    border: 1px solid #dee2e6;
    border-radius: 8px;
    padding: 15px;
    margin-top: 10px;
    font-size: 0.9em;
}

.changes-details h4 {
    margin: 0 0 10px 0;
    color: #2c3e50;
    font-size: 1em;
}

.changes-diff {
    background: #fff;
    border: 1px solid #e9ecef;
    border-radius: 4px;
    padding: 10px;
    font-family: 'Courier New', monospace;
    font-size: 0.85em;
    max-height: 200px;
    overflow-y: auto;
}

.change-added {
    color: #28a745;
    background: #d4edda;
    padding: 2px 4px;
    border-radius: 3px;
}

.change-removed {
    color: #dc3545;
    background: #f8d7da;
    padding: 2px 4px;
    border-radius: 3px;
    text-decoration: line-through;
}
</style>

<div class="audit-logs-container">
    <div class="audit-header">
        <div>
            <h1>📋 Asset Audit Logs</h1>
            <p>Track all asset operations and changes</p>
        </div>
        <div>
            <a href="manage_assets.php" class="btn btn-secondary">🏢 Assets</a>
            <a href="index.php" class="btn btn-secondary">🏠 Dashboard</a>
        </div>
    </div>

    <?php if ($message): ?>
        <div class="message success"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="message error"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <div class="audit-content">
        <div class="audit-main">
            <!-- Summary Cards -->
            <div class="summary-cards">
                <div class="summary-card">
                    <h3><?php echo number_format($stats['total_logs']); ?></h3>
                    <p>Total Audit Logs</p>
                </div>
                <div class="summary-card">
                    <h3><?php echo number_format($stats['today_logs']); ?></h3>
                    <p>Today's Logs</p>
                </div>
                <div class="summary-card">
                    <h3><?php echo number_format($stats['unique_users']); ?></h3>
                    <p>Active Users</p>
                </div>
                <div class="summary-card">
                    <h3><?php echo number_format($stats['unique_assets']); ?></h3>
                    <p>Assets Tracked</p>
                </div>
            </div>

            <!-- Filters -->
            <div class="filters-section">
                <form method="get" class="filters-form">
                    <div class="filter-group">
                        <label for="asset_id">Asset</label>
                        <select name="asset_id" id="asset_id">
                            <option value="">All Assets</option>
                            <?php foreach ($assets as $asset): ?>
                                <option value="<?php echo $asset['id']; ?>" <?php echo $asset_id == $asset['id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($asset['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="filter-group">
                        <label for="user_id">User</label>
                        <select name="user_id" id="user_id">
                            <option value="">All Users</option>
                            <?php foreach ($users as $user): ?>
                                <option value="<?php echo $user['id']; ?>" <?php echo $user_id == $user['id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($user['username']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="filter-group">
                        <label for="action">Action</label>
                        <select name="action" id="action">
                            <option value="">All Actions</option>
                            <option value="create" <?php echo $action == 'create' ? 'selected' : ''; ?>>Create</option>
                            <option value="update" <?php echo $action == 'update' ? 'selected' : ''; ?>>Update</option>
                            <option value="view" <?php echo $action == 'view' ? 'selected' : ''; ?>>View</option>
                            <option value="delete" <?php echo $action == 'delete' ? 'selected' : ''; ?>>Delete</option>
                        </select>
                    </div>

                    <div class="filter-group">
                        <label for="date_from">From Date</label>
                        <input type="date" name="date_from" id="date_from" value="<?php echo htmlspecialchars($date_from); ?>">
                    </div>

                    <div class="filter-group">
                        <label for="date_to">To Date</label>
                        <input type="date" name="date_to" id="date_to" value="<?php echo htmlspecialchars($date_to); ?>">
                    </div>

                    <div class="filter-actions">
                        <button type="submit" class="btn btn-primary">🔍 Filter</button>
                        <a href="view_asset_audit_logs.php" class="btn btn-secondary">🔄 Reset</a>
                    </div>
                </form>
            </div>

            <!-- Audit Logs Table -->
            <?php if (empty($audit_logs)): ?>
                <div class="no-logs">
                    <h3>No audit logs found</h3>
                    <p>Try adjusting your filters or check back later for new activity.</p>
                </div>
            <?php else: ?>
                <div class="table-container">
                    <table class="audit-table">
                        <thead>
                            <tr>
                                <th>Action</th>
                                <th>User</th>
                                <th>Asset</th>
                                <th>Date/Time</th>
                                <th>IP Address</th>
                                <th>Details</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($audit_logs as $log): ?>
                                <tr>
                                    <td>
                                        <span class="action-badge action-<?php echo $log['action']; ?>">
                                            <?php echo ucfirst($log['action']); ?>
                                        </span>
                                    </td>
                                    <td class="audit-user">
                                        <?php echo htmlspecialchars($log['username']); ?>
                                    </td>
                                    <td>
                                        <div class="audit-asset" title="<?php echo htmlspecialchars($log['asset_name']); ?>">
                                            <?php echo htmlspecialchars($log['asset_name']); ?>
                                        </div>
                                    </td>
                                    <td class="audit-date">
                                        <?php echo date('M d, Y H:i:s', strtotime($log['created_at'])); ?>
                                    </td>
                                    <td class="audit-ip">
                                        <?php echo htmlspecialchars($log['ip_address']); ?>
                                    </td>
                                    <td>
                                        <?php if ($log['action'] === 'update' && !empty($log['changes'])): ?>
                                            <button class="btn btn-secondary" onclick="showChanges(<?php echo $log['id']; ?>)">View Changes</button>
                                        <?php else: ?>
                                            <span style="color: #6c757d; font-style: italic;">No details</span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php if ($log['action'] === 'update' && !empty($log['changes'])): ?>
                                <tr id="changes-row-<?php echo $log['id']; ?>" style="display: none;">
                                    <td colspan="6">
                                        <div class="changes-details">
                                            <h4>Changes Made:</h4>
                                            <div class="changes-diff">
                                                <?php
                                                $changes = json_decode($log['changes'], true);
                                                if ($changes) {
                                                    foreach ($changes as $field => $change) {
                                                        if (isset($change['old']) && isset($change['new'])) {
                                                            echo "<div><strong>" . htmlspecialchars($field) . ":</strong> ";
                                                            echo "<span class='change-removed'>" . htmlspecialchars($change['old']) . "</span> → ";
                                                            echo "<span class='change-added'>" . htmlspecialchars($change['new']) . "</span></div>";
                                                        }
                                                    }
                                                }
                                                ?>
                                            </div>
                                        </div>
                                    </td>
                                </tr>
                                <?php endif; ?>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <!-- Pagination -->
                <?php if ($total_pages > 1): ?>
                    <div class="pagination">
                        <?php if ($page > 1): ?>
                            <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page - 1])); ?>">Previous</a>
                        <?php else: ?>
                            <span class="disabled">Previous</span>
                        <?php endif; ?>

                        <?php for ($i = max(1, $page - 2); $i <= min($total_pages, $page + 2); $i++): ?>
                            <?php if ($i == $page): ?>
                                <span class="active"><?php echo $i; ?></span>
                            <?php else: ?>
                                <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $i])); ?>"><?php echo $i; ?></a>
                            <?php endif; ?>
                        <?php endfor; ?>

                        <?php if ($page < $total_pages): ?>
                            <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page + 1])); ?>">Next</a>
                        <?php else: ?>
                            <span class="disabled">Next</span>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
        </div>

        <div class="audit-sidebar">
            <h3>📊 Audit Statistics</h3>
            <div style="margin-bottom: 20px;">
                <p><strong>Total Logs:</strong> <?php echo number_format($stats['total_logs']); ?></p>
                <p><strong>Today's Activity:</strong> <?php echo number_format($stats['today_logs']); ?></p>
                <p><strong>Active Users:</strong> <?php echo number_format($stats['unique_users']); ?></p>
                <p><strong>Assets Tracked:</strong> <?php echo number_format($stats['unique_assets']); ?></p>
            </div>

            <h3 style="margin-top: 30px;">🔗 Quick Actions</h3>
            <div class="quick-actions">
                <a href="manage_assets.php" class="btn btn-primary">Manage Assets</a>
                <a href="add_asset.php" class="btn btn-success">Add New Asset</a>
                <a href="activity_logs.php" class="btn btn-primary">Accountant Logs</a>
                <a href="index.php" class="btn btn-secondary">Dashboard</a>
            </div>
        </div>
    </div>
</div>

<script>
function showChanges(logId) {
    const row = document.getElementById('changes-row-' + logId);
    if (row.style.display === 'none' || row.style.display === '') {
        row.style.display = 'table-row';
    } else {
        row.style.display = 'none';
    }
}
</script>

<?php include 'includes/accounting_footer.php'; ?>
