<?php
require_once '../config.php';
require_once '../includes/accountant_activity_logger.php';
require_once 'purchase_order_functions.php';

// Check permissions - only admins and accounts users can manage invoices
if (!isset($_SESSION['role']) || !in_array($_SESSION['role'], ['admin', 'accounts'])) {
    header('Location: ../login.php');
    exit;
}

$message = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['create_invoice'])) {
        // Create new supplier invoice
        $supplier_id = $_POST['supplier_id'];
        $po_id = $_POST['po_id'] ?: null;
        $invoice_number = trim($_POST['invoice_number']);
        $invoice_date = $_POST['invoice_date'];
        $due_date = $_POST['due_date'];
        $subtotal = floatval($_POST['subtotal']);
        $tax_amount = floatval($_POST['tax_amount']);
        $discount_amount = floatval($_POST['discount_amount']);
        $notes = trim($_POST['notes']);
        $items = $_POST['items'] ?? [];

        if (empty($supplier_id) || empty($invoice_number) || empty($invoice_date)) {
            $error = "Supplier, Invoice Number, and Invoice Date are required.";
        } elseif (empty($items)) {
            $error = "At least one item must be included.";
        } else {
            // Check for duplicate invoice number
            $stmt = $pdo->prepare("SELECT id FROM supplier_invoices WHERE invoice_number = ?");
            $stmt->execute([$invoice_number]);
            if ($stmt->fetch()) {
                $error = "Invoice number '$invoice_number' already exists. Please use a unique invoice number.";
            }
        }

        if (!$error) {
            try {
                $pdo->beginTransaction();

                // Calculate total amount
                $total_amount = $subtotal + $tax_amount - $discount_amount;

                // Insert invoice
                $stmt = $pdo->prepare("
                    INSERT INTO supplier_invoices
                    (invoice_number, supplier_id, purchase_order_id, invoice_date, due_date, subtotal, tax_amount, discount_amount, total_amount, notes, status, created_by)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'pending_approval', ?)
                ");
                $stmt->execute([$invoice_number, $supplier_id, $po_id, $invoice_date, $due_date, $subtotal, $tax_amount, $discount_amount, $total_amount, $notes, $_SESSION['user_id']]);
                $invoice_id = $pdo->lastInsertId();

                // Insert invoice items
                foreach ($items as $item) {
                    $po_item_id = $item['po_item_id'] ?: null;
                    $item_name = trim($item['item_name']);
                    $description = trim($item['description']);
                    $quantity = floatval($item['quantity']);
                    $unit_price = floatval($item['unit_price']);
                    $total_price = $quantity * $unit_price;

                    $stmt = $pdo->prepare("
                        INSERT INTO supplier_invoice_items
                        (supplier_invoice_id, purchase_order_item_id, item_name, description, quantity, unit_price, total_price)
                        VALUES (?, ?, ?, ?, ?, ?, ?)
                    ");
                    $stmt->execute([$invoice_id, $po_item_id, $item_name, $description, $quantity, $unit_price, $total_price]);

                    // Update PO item invoiced quantity if linked to PO
                    if ($po_item_id) {
                        $stmt = $pdo->prepare("
                            UPDATE purchase_order_items
                            SET quantity_invoiced = quantity_invoiced + ?
                            WHERE id = ?
                        ");
                        $stmt->execute([$quantity, $po_item_id]);
                    }
                }

                // Create accounting journal entry
                PurchaseOrderAccounting::createInvoiceJournalEntry($pdo, $invoice_id, $_SESSION['user_id']);

                // Update PO status if linked
                if ($po_id) {
                    PurchaseOrderAccounting::updatePurchaseOrderStatus($pdo, $po_id);
                }

                $pdo->commit();

                // Log activity
                AccountantActivityLogger::logActivity(
                    $_SESSION['user_id'],
                    'supplier_invoice_created',
                    "Created supplier invoice $invoice_number" . ($po_id ? " for PO ID: $po_id" : "")
                );

                $message = "Supplier invoice created successfully!";

            } catch (Exception $e) {
                $pdo->rollBack();
                $error = "Error creating invoice: " . $e->getMessage();
            }
        }
    } elseif (isset($_POST['approve_invoice'])) {
        // Approve invoice
        $invoice_id = $_POST['invoice_id'];

        try {
            $stmt = $pdo->prepare("
                UPDATE supplier_invoices
                SET status = 'pending_payment', approved_by = ?, approved_at = NOW()
                WHERE id = ? AND status = 'pending_approval'
            ");
            $stmt->execute([$_SESSION['user_id'], $invoice_id]);

            if ($stmt->rowCount() > 0) {
                AccountantActivityLogger::logActivity(
                    $_SESSION['user_id'],
                    'supplier_invoice_approved',
                    "Approved supplier invoice ID: $invoice_id"
                );

                $message = "Invoice approved successfully!";
            } else {
                $error = "Could not approve invoice. It may not be in pending status.";
            }
        } catch (Exception $e) {
            $error = "Error approving invoice: " . $e->getMessage();
        }
    }
}

// Get all suppliers for dropdown
try {
    $stmt = $pdo->query("SELECT id, name FROM suppliers WHERE status = 'active' ORDER BY name");
    $suppliers = $stmt->fetchAll();
} catch (PDOException $e) {
    $suppliers = [];
}

// Get all POs that can have invoices (received or partially invoiced)
try {
    $stmt = $pdo->query("
        SELECT po.id, po.po_number, po.supplier_id, s.name AS supplier_name
        FROM purchase_orders po
        JOIN suppliers s ON po.supplier_id = s.id
        WHERE po.status IN ('received', 'partially_received', 'partially_invoiced', 'invoiced')
        ORDER BY po.order_date DESC
    ");
    $invoiceable_pos = $stmt->fetchAll();
} catch (PDOException $e) {
    $invoiceable_pos = [];
}

// Get all supplier invoices
try {
    $stmt = $pdo->query("
        SELECT si.*, s.name AS supplier_name, po.po_number,
               u1.name AS created_by_name, u2.name AS approved_by_name
        FROM supplier_invoices si
        LEFT JOIN suppliers s ON si.supplier_id = s.id
        LEFT JOIN purchase_orders po ON si.purchase_order_id = po.id
        LEFT JOIN users u1 ON si.created_by = u1.id
        LEFT JOIN users u2 ON si.approved_by = u2.id
        ORDER BY si.invoice_date DESC
    ");
    $supplier_invoices = $stmt->fetchAll();
} catch (PDOException $e) {
    $supplier_invoices = [];
}

// Get PO items for AJAX
if (isset($_GET['get_po_items'])) {
    $po_id = $_GET['po_id'];
    try {
        $stmt = $pdo->prepare("
            SELECT poi.id, poi.item_name, poi.description, poi.quantity_ordered, poi.quantity_received, poi.quantity_invoiced,
                   poi.unit_price, poi.total_price,
                   (poi.quantity_received - poi.quantity_invoiced) as remaining_quantity
            FROM purchase_order_items poi
            WHERE poi.purchase_order_id = ? AND (poi.quantity_received - poi.quantity_invoiced) > 0
        ");
        $stmt->execute([$po_id]);
        $items = $stmt->fetchAll();

        header('Content-Type: application/json');
        echo json_encode($items);
        exit;
    } catch (PDOException $e) {
        header('Content-Type: application/json');
        echo json_encode(['error' => $e->getMessage()]);
        exit;
    }
}
?>

<?php include 'includes/accounting_header.php'; ?>

<style>
    .invoice-container {
        max-width: 1400px;
        margin: 0 auto;
        padding: 20px;
        background: #f8f9fa;
        min-height: 100vh;
    }

    .invoice-header {
        background: white;
        padding: 25px;
        border-radius: 10px;
        margin-bottom: 30px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        text-align: center;
    }

    .invoice-content {
        display: grid;
        grid-template-columns: 1fr 2fr;
        gap: 30px;
    }

    .create-invoice-section,
    .invoices-list-section {
        background: white;
        padding: 30px;
        border-radius: 10px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    }

    .section-title {
        color: #2c3e50;
        margin: 0 0 20px 0;
        padding-bottom: 10px;
        border-bottom: 2px solid #e9ecef;
    }

    .form-group {
        margin-bottom: 20px;
    }

    .form-group label {
        display: block;
        font-weight: 600;
        margin-bottom: 5px;
        color: #495057;
    }

    .form-group input,
    .form-group select,
    .form-group textarea {
        width: 100%;
        padding: 10px 12px;
        border: 1px solid #ced4da;
        border-radius: 5px;
        font-size: 14px;
    }

    .form-group textarea {
        resize: vertical;
        min-height: 80px;
    }

    .form-row {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 20px;
    }

    .btn {
        padding: 10px 20px;
        border: none;
        border-radius: 5px;
        cursor: pointer;
        font-size: 14px;
        font-weight: 600;
        text-decoration: none;
        display: inline-block;
        margin-right: 10px;
    }

    .btn-primary {
        background: #007bff;
        color: white;
    }

    .btn-success {
        background: #28a745;
        color: white;
    }

    .btn-warning {
        background: #ffc107;
        color: #212529;
    }

    .btn-secondary {
        background: #6c757d;
        color: white;
    }

    .message {
        padding: 15px;
        border-radius: 5px;
        margin-bottom: 20px;
    }

    .success-message {
        background: #d4edda;
        color: #155724;
        border: 1px solid #c3e6cb;
    }

    .error-message {
        background: #f8d7da;
        color: #721c24;
        border: 1px solid #f5c6cb;
    }

    .invoices-table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 20px;
    }

    .invoices-table th,
    .invoices-table td {
        padding: 12px;
        text-align: left;
        border-bottom: 1px solid #e9ecef;
    }

    .invoices-table th {
        background: #f8f9fa;
        font-weight: 600;
        color: #495057;
    }

    .invoice-number {
        font-weight: bold;
        color: #007bff;
    }

    .status-badge {
        display: inline-block;
        padding: 2px 8px;
        border-radius: 12px;
        font-size: 0.8em;
        font-weight: 600;
    }

    .status-pending_approval { background: #fff3cd; color: #856404; }
    .status-pending_payment { background: #d4edda; color: #155724; }
    .status-paid { background: #cce5ff; color: #004085; }
    .status-overdue { background: #f8d7da; color: #721c24; }

    .items-section {
        margin-top: 20px;
        padding: 20px;
        background: #f8f9fa;
        border-radius: 5px;
    }

    .item-row {
        display: grid;
        grid-template-columns: 2fr 1fr 1fr 1fr 2fr;
        gap: 10px;
        margin-bottom: 10px;
        align-items: center;
    }

    .item-row input,
    .item-row textarea {
        padding: 8px;
        border: 1px solid #ced4da;
        border-radius: 3px;
    }

    .item-header {
        font-weight: 600;
        color: #495057;
        margin-bottom: 10px;
    }

    .totals-section {
        background: #e9ecef;
        padding: 15px;
        border-radius: 5px;
        margin-top: 20px;
    }

    .totals-row {
        display: grid;
        grid-template-columns: 1fr 1fr 1fr 1fr;
        gap: 10px;
        margin-bottom: 10px;
    }

    @media (max-width: 768px) {
        .invoice-content {
            grid-template-columns: 1fr;
        }

        .form-row {
            grid-template-columns: 1fr;
        }

        .item-row {
            grid-template-columns: 1fr;
        }

        .totals-row {
            grid-template-columns: 1fr;
        }
    }
</style>

<div class="invoice-container">
    <div class="invoice-header">
        <h1>📄 Supplier Invoices</h1>
        <p>Record and manage supplier invoices</p>
    </div>

    <?php if ($message): ?>
        <div class="message success-message"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="message error-message"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <div class="invoice-content">
        <div class="create-invoice-section">
            <h2 class="section-title">➕ Create New Invoice</h2>
            <form method="post" id="invoiceForm">
                <div class="form-row">
                    <div class="form-group">
                        <label for="supplier_id">Supplier *</label>
                        <select name="supplier_id" id="supplier_id" required onchange="updatePOOptions()">
                            <option value="">Select Supplier</option>
                            <?php foreach ($suppliers as $supplier): ?>
                                <option value="<?php echo $supplier['id']; ?>"><?php echo htmlspecialchars($supplier['name']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="form-group">
                        <label for="po_id">Purchase Order (Optional)</label>
                        <select name="po_id" id="po_id" onchange="loadPOItems()">
                            <option value="">Select Purchase Order</option>
                            <?php foreach ($invoiceable_pos as $po): ?>
                                <option value="<?php echo $po['id']; ?>" data-supplier="<?php echo $po['supplier_id']; ?>">
                                    <?php echo htmlspecialchars($po['po_number']); ?> - <?php echo htmlspecialchars($po['supplier_name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label for="invoice_number">Invoice Number *</label>
                        <input type="text" name="invoice_number" id="invoice_number" required>
                    </div>

                    <div class="form-group">
                        <label for="invoice_date">Invoice Date *</label>
                        <input type="date" name="invoice_date" id="invoice_date" required>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label for="due_date">Due Date</label>
                        <input type="date" name="due_date" id="due_date">
                    </div>
                </div>

                <div id="itemsSection" class="items-section" style="display: none;">
                    <h3 class="item-header">Invoice Items</h3>
                    <div id="itemsContainer"></div>
                    <button type="button" class="btn btn-secondary" onclick="addManualItem()">Add Manual Item</button>
                </div>

                <div class="totals-section">
                    <h3 class="item-header">Invoice Totals</h3>
                    <div class="totals-row">
                        <div class="form-group">
                            <label for="subtotal">Subtotal (GHS)</label>
                            <input type="number" name="subtotal" id="subtotal" step="0.01" min="0" readonly>
                        </div>
                        <div class="form-group">
                            <label for="tax_amount">Tax Amount (GHS)</label>
                            <input type="number" name="tax_amount" id="tax_amount" step="0.01" min="0" value="0.00">
                        </div>
                        <div class="form-group">
                            <label for="discount_amount">Discount (GHS)</label>
                            <input type="number" name="discount_amount" id="discount_amount" step="0.01" min="0" value="0.00">
                        </div>
                        <div class="form-group">
                            <label for="total_amount">Total Amount (GHS)</label>
                            <input type="number" name="total_amount" id="total_amount" step="0.01" readonly>
                        </div>
                    </div>
                </div>

                <div class="form-group">
                    <label for="notes">Notes</label>
                    <textarea name="notes" id="notes" placeholder="Additional notes..."></textarea>
                </div>

                <button type="submit" name="create_invoice" class="btn btn-primary">Create Invoice</button>
            </form>
        </div>

        <div class="invoices-list-section">
            <h2 class="section-title">📋 All Supplier Invoices</h2>
            <table class="invoices-table">
                <thead>
                    <tr>
                        <th>Invoice #</th>
                        <th>Supplier</th>
                        <th>PO</th>
                        <th>Date</th>
                        <th>Total (GHS)</th>
                        <th>Status</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($supplier_invoices as $invoice): ?>
                        <tr>
                            <td class="invoice-number"><?php echo htmlspecialchars($invoice['invoice_number']); ?></td>
                            <td><?php echo htmlspecialchars($invoice['supplier_name'] ?? 'N/A'); ?></td>
                            <td><?php echo htmlspecialchars($invoice['po_number'] ?? 'N/A'); ?></td>
                            <td><?php echo date('M j, Y', strtotime($invoice['invoice_date'])); ?></td>
                            <td><?php echo number_format($invoice['total_amount'], 2); ?></td>
                            <td>
                                <span class="status-badge status-<?php echo strtolower($invoice['status']); ?>">
                                    <?php echo ucfirst($invoice['status']); ?>
                                </span>
                            </td>
                            <td>
                                <?php if ($invoice['status'] === 'pending_approval'): ?>
                                    <form method="post" style="display: inline;">
                                        <input type="hidden" name="invoice_id" value="<?php echo $invoice['id']; ?>">
                                        <button type="submit" name="approve_invoice" class="btn btn-warning btn-sm">Approve</button>
                                    </form>
                                <?php endif; ?>
                                <button class="btn btn-secondary btn-sm" onclick="viewInvoice(<?php echo $invoice['id']; ?>)">View</button>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<script>
let itemCount = 0;

function updatePOOptions() {
    const supplierId = document.getElementById('supplier_id').value;
    const poSelect = document.getElementById('po_id');

    // Filter PO options by selected supplier
    const options = poSelect.querySelectorAll('option');
    options.forEach(option => {
        if (option.value === '' || option.getAttribute('data-supplier') === supplierId) {
            option.style.display = 'block';
        } else {
            option.style.display = 'none';
        }
    });

    // Reset PO selection
    poSelect.value = '';
    document.getElementById('itemsSection').style.display = 'none';
}

function loadPOItems() {
    const poId = document.getElementById('po_id').value;
    if (!poId) {
        document.getElementById('itemsSection').style.display = 'none';
        return;
    }

    fetch(`?get_po_items=1&po_id=${poId}`)
        .then(response => response.json())
        .then(data => {
            if (data.error) {
                alert('Error loading items: ' + data.error);
                return;
            }

            const container = document.getElementById('itemsContainer');
            container.innerHTML = '';

            if (data.length === 0) {
                container.innerHTML = '<p>No items available for invoicing.</p>';
                document.getElementById('itemsSection').style.display = 'none';
                return;
            }

            data.forEach(item => {
                addPOItem(item);
            });

            document.getElementById('itemsSection').style.display = 'block';
            calculateTotals();
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error loading purchase order items');
        });
}

function addPOItem(item) {
    const container = document.getElementById('itemsContainer');
    const itemDiv = document.createElement('div');
    itemDiv.className = 'item-row';
    itemDiv.innerHTML = `
        <div>
            <input type="text" name="items[${itemCount}][item_name]" value="${item.item_name}" readonly>
            <input type="hidden" name="items[${itemCount}][po_item_id]" value="${item.id}">
        </div>
        <div>
            <textarea name="items[${itemCount}][description]" readonly rows="2">${item.description || ''}</textarea>
        </div>
        <div>
            <input type="number" name="items[${itemCount}][quantity]" value="${item.remaining_quantity}"
                   max="${item.remaining_quantity}" step="0.01" min="0" onchange="calculateTotals()" required>
        </div>
        <div>
            <input type="number" name="items[${itemCount}][unit_price]" value="${item.unit_price}"
                   step="0.01" min="0" onchange="calculateTotals()" required>
        </div>
        <div>
            <input type="number" name="items[${itemCount}][total_price]" readonly step="0.01">
        </div>
    `;
    container.appendChild(itemDiv);
    itemCount++;
}

function addManualItem() {
    const container = document.getElementById('itemsContainer');
    const itemDiv = document.createElement('div');
    itemDiv.className = 'item-row';
    itemDiv.innerHTML = `
        <div>
            <input type="text" name="items[${itemCount}][item_name]" placeholder="Item name" required>
            <input type="hidden" name="items[${itemCount}][po_item_id]" value="">
        </div>
        <div>
            <textarea name="items[${itemCount}][description]" placeholder="Description" rows="2"></textarea>
        </div>
        <div>
            <input type="number" name="items[${itemCount}][quantity]" placeholder="Quantity"
                   step="0.01" min="0" onchange="calculateTotals()" required>
        </div>
        <div>
            <input type="number" name="items[${itemCount}][unit_price]" placeholder="Unit Price"
                   step="0.01" min="0" onchange="calculateTotals()" required>
        </div>
        <div>
            <input type="number" name="items[${itemCount}][total_price]" readonly step="0.01">
        </div>
    `;
    container.appendChild(itemDiv);
    itemCount++;
    document.getElementById('itemsSection').style.display = 'block';
}

function calculateTotals() {
    const itemRows = document.querySelectorAll('.item-row');
    let subtotal = 0;

    itemRows.forEach(row => {
        const quantity = parseFloat(row.querySelector('input[name*="[quantity]"]').value) || 0;
        const unitPrice = parseFloat(row.querySelector('input[name*="[unit_price]"]').value) || 0;
        const totalPrice = quantity * unitPrice;

        row.querySelector('input[name*="[total_price]"]').value = totalPrice.toFixed(2);
        subtotal += totalPrice;
    });

    document.getElementById('subtotal').value = subtotal.toFixed(2);

    const taxAmount = parseFloat(document.getElementById('tax_amount').value) || 0;
    const discountAmount = parseFloat(document.getElementById('discount_amount').value) || 0;
    const total = subtotal + taxAmount - discountAmount;

    document.getElementById('total_amount').value = total.toFixed(2);
}

function viewInvoice(invoiceId) {
    // Open invoice details in a new window or modal
    window.open(`view_supplier_invoice.php?id=${invoiceId}`, '_blank');
}

// Set today's date as default for invoice date
document.addEventListener('DOMContentLoaded', function() {
    const today = new Date().toISOString().split('T')[0];
    document.getElementById('invoice_date').value = today;

    // Set due date to 30 days from today
    const dueDate = new Date();
    dueDate.setDate(dueDate.getDate() + 30);
    document.getElementById('due_date').value = dueDate.toISOString().split('T')[0];

    // Recalculate totals when tax or discount changes
    document.getElementById('tax_amount').addEventListener('input', calculateTotals);
    document.getElementById('discount_amount').addEventListener('input', calculateTotals);
});
</script>

<?php include '../includes/admin_footer.php'; ?>
