<?php
require_once '../config.php';
require_once '../includes/accountant_activity_logger.php';
require_once 'accounting_functions.php';

$message = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $pdo->beginTransaction();

        $stmt = $pdo->prepare("
            INSERT INTO expenses (title, amount, category_id, expense_date, vendor_name, description, created_by)
            VALUES (?, ?, ?, ?, ?, ?, ?)
        ");

        $stmt->execute([
            $_POST['title'],
            $_POST['amount'],
            $_POST['category_id'],
            $_POST['expense_date'],
            $_POST['vendor_name'],
            $_POST['description'],
            $_SESSION['user_id']
        ]);

        $expense_id = $pdo->lastInsertId();

        // Create journal entry for expense liability (accrual basis - expense incurred but not paid)
        try {
            createExpenseLiabilityEntry($pdo, $expense_id, $_POST['amount'], $_POST['expense_date'], $_SESSION['user_id'], false);
        } catch (Exception $e) {
            throw new Exception("Expense recorded but journal entry failed: " . $e->getMessage());
        }

        // Expense status remains 'pending' (default in database) - will be approved then paid

        // Log the expense activity
        AccountantActivityLogger::logActivity(
            $_SESSION['user_id'],
            'expense_recorded',
            "Recorded expense liability of GHC {$_POST['amount']} for {$_POST['title']}",
            $_POST['amount']
        );

        $pdo->commit();
        $message = "Expense recorded successfully!";

        // Clear form data after successful submission
        $_POST = array();

    } catch (PDOException $e) {
        $pdo->rollBack();
        $error = "Error recording expense: " . $e->getMessage();
    }
}

// Get expense categories (both system and user-defined)
try {
    $stmt = $pdo->query("
        SELECT id, name, is_system,
               CASE WHEN is_system = 1 THEN 'System' ELSE 'Custom' END as type
        FROM budget_categories
        WHERE type = 'expense'
        ORDER BY is_system DESC, name ASC
    ");
    $categories = $stmt->fetchAll();
} catch (PDOException $e) {
    $error = "Error loading categories: " . $e->getMessage();
}



// Get recent expenses for reference
try {
    $stmt = $pdo->query("
        SELECT e.*, bc.name as category_name
        FROM expenses e
        LEFT JOIN budget_categories bc ON e.category_id = bc.id
        ORDER BY e.created_at DESC
        LIMIT 5
    ");
    $recent_expenses = $stmt->fetchAll();
} catch (PDOException $e) {
    $recent_expenses = [];
}
?>

<?php include 'includes/accounting_header.php'; ?>

<style>
        .expense-container {
            max-width: 1000px;
            margin: 0 auto;
            padding: 20px;
            background: #f8f9fa;
            min-height: 100vh;
        }

        .expense-header {
            background: white;
            padding: 25px;
            border-radius: 10px;
            margin-bottom: 30px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            text-align: center;
        }

        .expense-header h1 {
            color: #2c3e50;
            margin: 0 0 10px 0;
        }

        .expense-header p {
            color: #7f8c8d;
            margin: 0;
        }

        .expense-content {
            display: grid;
            grid-template-columns: 2fr 1fr;
            gap: 30px;
        }

        .expense-form-section {
            background: white;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }

        .recent-expenses-section {
            background: white;
            padding: 20px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }

        .section-title {
            color: #2c3e50;
            margin: 0 0 20px 0;
            padding-bottom: 10px;
            border-bottom: 2px solid #e9ecef;
        }

        .form-row {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 20px;
            margin-bottom: 20px;
        }

        .form-group {
            display: flex;
            flex-direction: column;
        }

        .form-group.full-width {
            grid-column: 1 / -1;
        }

        .form-group label {
            font-weight: 600;
            margin-bottom: 5px;
            color: #495057;
        }

        .form-group input,
        .form-group select,
        .form-group textarea {
            padding: 10px 12px;
            border: 1px solid #ced4da;
            border-radius: 5px;
            font-size: 14px;
            transition: border-color 0.2s;
        }

        .form-group input:focus,
        .form-group select:focus,
        .form-group textarea:focus {
            outline: none;
            border-color: #007bff;
            box-shadow: 0 0 0 2px rgba(0,123,255,0.25);
        }

        .form-group textarea {
            resize: vertical;
            min-height: 80px;
        }

        .form-actions {
            display: flex;
            gap: 15px;
            margin-top: 30px;
            padding-top: 20px;
            border-top: 1px solid #e9ecef;
        }

        .btn {
            padding: 12px 24px;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            font-size: 14px;
            font-weight: 600;
            text-decoration: none;
            display: inline-block;
            transition: all 0.2s;
        }

        .btn-primary {
            background: #007bff;
            color: white;
        }

        .btn-primary:hover {
            background: #0056b3;
        }

        .btn-secondary {
            background: #6c757d;
            color: white;
        }

        .btn-secondary:hover {
            background: #545b62;
        }

        .success-message {
            background: #d4edda;
            color: #155724;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
            border: 1px solid #c3e6cb;
        }

        .error-message {
            background: #f8d7da;
            color: #721c24;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
            border: 1px solid #f5c6cb;
        }

        .recent-expenses-list {
            max-height: 400px;
            overflow-y: auto;
        }

        .expense-item {
            padding: 15px;
            border: 1px solid #e9ecef;
            border-radius: 5px;
            margin-bottom: 10px;
            background: #f8f9fa;
        }

        .expense-item-header {
            display: flex;
            justify-content: space-between;
            align-items: start;
            margin-bottom: 8px;
        }

        .expense-title {
            font-weight: 600;
            color: #2c3e50;
            margin: 0;
        }

        .expense-amount {
            font-weight: 600;
            color: #e74c3c;
        }

        .expense-meta {
            font-size: 0.9em;
            color: #7f8c8d;
            margin-bottom: 5px;
        }

        .expense-category {
            background: #007bff;
            color: white;
            padding: 2px 8px;
            border-radius: 12px;
            font-size: 0.8em;
            display: inline-block;
        }

        .expense-description {
            font-size: 0.9em;
            color: #6c757d;
            margin-top: 8px;
        }

        .no-expenses {
            text-align: center;
            color: #7f8c8d;
            font-style: italic;
            padding: 20px;
        }

        .status-badge {
            display: inline-block;
            padding: 2px 8px;
            border-radius: 12px;
            font-size: 0.8em;
            font-weight: 600;
            text-transform: uppercase;
            border: 1px solid;
        }

        .status-pending {
            background: #fff3cd;
            color: #856404;
            border-color: #ffeaa7;
        }

        .status-approved {
            background: #d1ecf1;
            color: #0c5460;
            border-color: #bee5eb;
        }

        .status-rejected {
            background: #f8d7da;
            color: #721c24;
            border-color: #f5c6cb;
        }

        .status-paid {
            background: #d4edda;
            color: #155724;
            border-color: #c3e6cb;
        }

        @media (max-width: 768px) {
            .expense-content {
                grid-template-columns: 1fr;
            }

            .form-row {
                grid-template-columns: 1fr;
            }

            .form-actions {
                flex-direction: column;
            }
        }
    </style>

    <div class="expense-container">
        <div class="expense-header">
            <h1>💰 Record New Expense</h1>
            <p>Record school expenses and track spending</p>
        </div>

        <div class="expense-content">
            <div class="expense-form-section">
                <?php if ($message): ?>
                    <div class="success-message"><?php echo htmlspecialchars($message); ?></div>
                <?php endif; ?>

                <?php if ($error): ?>
                    <div class="error-message"><?php echo htmlspecialchars($error); ?></div>
                <?php endif; ?>

                <form method="post" class="expense-form">
                    <div class="form-row">
                        <div class="form-group">
                            <label for="title">Expense Title *</label>
                            <input type="text" name="title" id="title" required
                                   placeholder="e.g., Electricity Bill - January 2024"
                                   value="<?php echo htmlspecialchars($_POST['title'] ?? ''); ?>">
                        </div>

                        <div class="form-group">
                            <label for="amount">Amount (GHC) *</label>
                            <input type="number" name="amount" id="amount" step="0.01" min="0" required
                                   placeholder="0.00"
                                   value="<?php echo htmlspecialchars($_POST['amount'] ?? ''); ?>">
                        </div>
                    </div>

                    <div class="form-row">
                        <div class="form-group">
                            <label for="category_id">Category *</label>
                            <select name="category_id" id="category_id" required>
                                <option value="">Select Category</option>
                                <?php foreach ($categories as $category): ?>
                                    <option value="<?php echo $category['id']; ?>"
                                            <?php echo ($_POST['category_id'] ?? '') == $category['id'] ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($category['name']); ?>
                                        <?php if (!$category['is_system']): ?>
                                            <span style="color: #28a745; font-size: 0.8em;">(Custom)</span>
                                        <?php endif; ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <div class="form-group">
                            <label for="expense_date">Expense Date *</label>
                            <input type="date" name="expense_date" id="expense_date" required
                                   value="<?php echo htmlspecialchars($_POST['expense_date'] ?? date('Y-m-d')); ?>">
                        </div>
                    </div>

                    <div class="form-row">
                        <div class="form-group full-width">
                            <label for="vendor_name">Vendor/Supplier</label>
                            <input type="text" name="vendor_name" id="vendor_name"
                                   placeholder="e.g., Electricity Company of Ghana"
                                   value="<?php echo htmlspecialchars($_POST['vendor_name'] ?? ''); ?>">
                        </div>
                    </div>

                    <div class="form-group full-width">
                        <label for="description">Description</label>
                        <textarea name="description" id="description" rows="3"
                                  placeholder="Additional details about this expense..."><?php echo htmlspecialchars($_POST['description'] ?? ''); ?></textarea>
                    </div>



                    <div class="form-actions">
                        <button type="submit" class="btn btn-primary">💾 Record Expense</button>
                        <a href="expense_list.php" class="btn btn-secondary">📋 View All Expenses</a>
                        <a href="index.php" class="btn btn-secondary">🏠 Back to Dashboard</a>
                    </div>
                </form>
            </div>

            <div class="recent-expenses-section">
                <h2 class="section-title">📊 Recent Expenses</h2>
                <div class="recent-expenses-list">
                    <?php if (empty($recent_expenses)): ?>
                        <div class="no-expenses">
                            No expenses recorded yet.
                        </div>
                    <?php else: ?>
                        <?php foreach ($recent_expenses as $expense): ?>
                            <div class="expense-item">
                                <div class="expense-item-header">
                                    <h4 class="expense-title"><?php echo htmlspecialchars($expense['title']); ?></h4>
                                    <span class="expense-amount">GHC <?php echo number_format($expense['amount'], 2); ?></span>
                                </div>
                                <div class="expense-meta">
                                    <span class="expense-category"><?php echo htmlspecialchars($expense['category_name'] ?? 'Uncategorized'); ?></span>
                                    <span><?php echo date('M d, Y', strtotime($expense['expense_date'])); ?></span>
                                    <?php if ($expense['vendor_name']): ?>
                                        <span>• <?php echo htmlspecialchars($expense['vendor_name']); ?></span>
                                    <?php endif; ?>
                                    <span class="status-badge status-<?php echo $expense['status'] ?? 'pending'; ?>">
                                        <?php echo ucfirst($expense['status'] ?? 'pending'); ?>
                                    </span>
                                </div>
                                <?php if ($expense['description']): ?>
                                    <div class="expense-description">
                                        <?php echo htmlspecialchars($expense['description']); ?>
                                    </div>
                                <?php endif; ?>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>

<?php include '../includes/admin_footer.php'; ?>
