<?php
require_once '../config.php';
require_once '../includes/user_activity_logger.php';
require_once 'accounting_functions.php';

// Handle session messages
$message = '';
$message_type = '';
if (isset($_SESSION['message'])) {
    $message = $_SESSION['message'];
    $message_type = $_SESSION['message_type'];
    unset($_SESSION['message']);
    unset($_SESSION['message_type']);
}

// Handle payroll processing
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['process_payroll'])) {
    $payroll_month = $_POST['payroll_month'];
    $payroll_year = $_POST['payroll_year'];

    try {
        $pdo->beginTransaction();

        // Check if payroll already exists for this month
        $payroll_date = $payroll_year . '-' . $payroll_month . '-01'; // First day of the month
        $stmt = $pdo->prepare("SELECT id FROM payroll_runs WHERE payroll_month = ?");
        $stmt->execute([$payroll_date]);
        $existing_payroll = $stmt->fetch();

        if ($existing_payroll) {
            throw new Exception("Payroll for {$payroll_month}/{$payroll_year} already exists.");
        }

        // Create payroll run
        $stmt = $pdo->prepare("INSERT INTO payroll_runs (payroll_month, payroll_year, status, processed_by, processed_at) VALUES (?, ?, 'draft', ?, NOW())");
        $stmt->execute([$payroll_date, $payroll_year, $_SESSION['user_id']]);
        $payroll_run_id = $pdo->lastInsertId();

        // Update payroll run status to processed after creating payslips
        $stmt = $pdo->prepare("UPDATE payroll_runs SET status = 'processed' WHERE id = ?");
        $stmt->execute([$payroll_run_id]);

        // Get the latest active salary structure per staff member with their allowances
        $stmt = $pdo->query("
            SELECT * FROM (
                SELECT
                    ss.id as structure_id,
                    s.staff_id as staff_id,
                    ss.basic_salary,
                    s.first_name,
                    s.last_name,
                    s.role as staff_type,
                    COALESCE(sa.total_allowances, 0) as total_allowances,
                    ROW_NUMBER() OVER (PARTITION BY ss.staff_id ORDER BY ss.effective_date DESC, ss.id DESC) as rn
                FROM salary_structures ss
                JOIN staff s ON ss.staff_id = s.staff_id
                LEFT JOIN (
                    SELECT salary_structure_id, SUM(amount) as total_allowances
                    FROM staff_allowances
                    GROUP BY salary_structure_id
                ) sa ON ss.id = sa.salary_structure_id
                WHERE ss.is_active = TRUE
            ) t WHERE t.rn = 1
            ORDER BY t.last_name, t.first_name
        ");
        $salary_structures = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Validate that all salary structures have valid staff_ids
        foreach ($salary_structures as $structure) {
            if (empty($structure['staff_id'])) {
                throw new Exception("Invalid staff_id for {$structure['first_name']} {$structure['last_name']}. Please ensure all staff members have valid staff references before processing payroll.");
            }
        }

        // Process each staff member's payroll
        foreach ($salary_structures as $structure) {
            $basic_salary = $structure['basic_salary'];
            $total_allowances = $structure['total_allowances'];
            $gross_salary = $basic_salary + $total_allowances;

            // Calculate deductions
            $ssnit_employee = $basic_salary * 0.055; // 5.5% of basic salary
            $chargeable_income = $gross_salary - $ssnit_employee;
            $paye_tax = calculateGhanaTax($chargeable_income);
            $total_deductions = $ssnit_employee + $paye_tax;
            $net_salary = $gross_salary - $total_deductions;

            // Insert payroll entry
            $stmt = $pdo->prepare("
                INSERT INTO payslips (
                    payroll_run_id, staff_id, staff_type, first_name, last_name, basic_salary, total_allowances, gross_salary,
                    ssnit_employee, tax_amount, total_deductions, net_salary, generated_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
            ");
            $stmt->execute([
                $payroll_run_id,
                $structure['staff_id'],
                $structure['staff_type'],
                $structure['first_name'],
                $structure['last_name'],
                $basic_salary,
                $total_allowances,
                $gross_salary,
                $ssnit_employee,
                $paye_tax,
                $total_deductions,
                $net_salary
            ]);
        }

        // Create accounting entries for payroll expenses
        $total_gross = 0;
        $total_employer_ssnit = 0;
        $total_tier2_pension = 0;
        $total_paye = 0;
        $total_net = 0;

        foreach ($salary_structures as $structure) {
            $basic_salary = $structure['basic_salary'];
            $total_allowances = $structure['total_allowances'];
            $gross_salary = $basic_salary + $total_allowances;
            $ssnit_employee = $basic_salary * 0.055;
            $employer_ssnit = $basic_salary * 0.055; // 5.5% SSNIT employer contribution
            $tier2_pension = $basic_salary * 0.13; // 13% Tier 2 pension employer contribution
            $chargeable_income = $gross_salary - $ssnit_employee;
            $paye_tax = calculateGhanaTax($chargeable_income);
            $net_salary = $gross_salary - $ssnit_employee - $paye_tax;

            $total_gross += $gross_salary;
            $total_employer_ssnit += $employer_ssnit;
            $total_tier2_pension += $tier2_pension;
            $total_paye += $paye_tax;
            $total_net += $net_salary;
        }

        // Update payroll_runs table with SSNIT, PAYE, and Tier 2 pension amounts
        $stmt = $pdo->prepare("UPDATE payroll_runs SET ssnit_amount = ?, paye_amount = ?, tier2_amount = ? WHERE id = ?");
        $stmt->execute([$total_employer_ssnit, $total_paye, $total_tier2_pension, $payroll_run_id]);

        // Create payroll expense journal entry (accrual basis)
        createPayrollExpenseEntry($pdo, $payroll_run_id, $total_gross, $total_employer_ssnit, $total_tier2_pension, $total_paye, $total_net, $payroll_date, $_SESSION['user_id'], false);

        $pdo->commit();

        // Log activity
        UserActivityLogger::logActivity($_SESSION['user_id'], 'process_payroll', "Processed payroll for {$payroll_month}/{$payroll_year}");

        $_SESSION['message'] = "Payroll processed successfully for {$payroll_month}/{$payroll_year}. " . count($salary_structures) . " staff members processed.";
        $_SESSION['message_type'] = 'success';
        header('Location: process_payroll.php');
        exit();

    } catch (Exception $e) {
        $pdo->rollBack();
        $message = 'Failed to process payroll: ' . $e->getMessage();
        $message_type = 'error';
    }
}

// Get current payroll data for display
$current_month = date('m');
$current_year = date('Y');
$current_payroll_date = $current_year . '-' . $current_month . '-01'; // First day of current month

// Check if current month payroll exists
$stmt = $pdo->prepare("SELECT * FROM payroll_runs WHERE payroll_month = ?");
$stmt->execute([$current_payroll_date]);
$current_payroll = $stmt->fetch();

// Get the latest active salary structure per staff member with allowances for display
$stmt = $pdo->query("
    SELECT * FROM (
        SELECT
            ss.id as structure_id,
            ss.staff_id,
            ss.basic_salary,
            ss.effective_date,
            s.first_name,
            s.last_name,
            s.role as staff_type,
            COALESCE(sa.total_allowances, 0) as total_allowances,
            COALESCE(sa.allowance_details, '') as allowance_details,
            ROW_NUMBER() OVER (PARTITION BY ss.staff_id ORDER BY ss.effective_date DESC, ss.id DESC) as rn
        FROM salary_structures ss
        JOIN staff s ON ss.staff_id = s.staff_id
        LEFT JOIN (
            SELECT
                salary_structure_id,
                SUM(amount) as total_allowances,
                GROUP_CONCAT(CONCAT(at.name, ': GH₵', FORMAT(amount, 2)) SEPARATOR ', ') as allowance_details
            FROM staff_allowances sa
            JOIN allowance_types at ON sa.allowance_type_id = at.id
            GROUP BY salary_structure_id
        ) sa ON ss.id = sa.salary_structure_id
        WHERE ss.is_active = TRUE
    ) t WHERE t.rn = 1
    ORDER BY t.last_name, t.first_name
");
$salary_structures = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Calculate totals for summary
$total_staff = count($salary_structures);
$total_basic = 0;
$total_allowances = 0;
$total_gross = 0;
$total_ssnit = 0;
$total_paye = 0;
$total_deductions = 0;
$total_net = 0;
$total_employer_ssnit = 0;
$total_tier2_pension = 0;

foreach ($salary_structures as $structure) {
    $basic = $structure['basic_salary'];
    $allowances = $structure['total_allowances'];
    $gross = $basic + $allowances;

    $ssnit_employee = $basic * 0.055;
    $employer_ssnit = $basic * 0.055;
    $tier2_pension = $basic * 0.13;
    $chargeable_income = $gross - $ssnit_employee;
    $paye_tax = calculateGhanaTax($chargeable_income);
    $deductions = $ssnit_employee + $paye_tax;
    $net = $gross - $deductions;

    $total_basic += $basic;
    $total_allowances += $allowances;
    $total_gross += $gross;
    $total_ssnit += $ssnit_employee;
    $total_paye += $paye_tax;
    $total_deductions += $deductions;
    $total_net += $net;
    $total_employer_ssnit += $employer_ssnit;
    $total_tier2_pension += $tier2_pension;
}

// Calculate total cash outflow: Net Pay (to staff) + PAYE (to GRA) + SSNIT (to SSNIT)
$total_cash_outflow = $total_net + $total_paye + $total_employer_ssnit + $total_tier2_pension;

function calculateGhanaTax($chargeable_income) {
    // GRA 2024 PAYE tax brackets for monthly chargeable income
    $tax = 0;

    if ($chargeable_income <= 490) {
        $tax = 0;
    } elseif ($chargeable_income <= 600) {
        $tax = ($chargeable_income - 490) * 0.05;
    } elseif ($chargeable_income <= 730) {
        $tax = 5.50 + ($chargeable_income - 600) * 0.10;
    } elseif ($chargeable_income <= 3896.67) {
        $tax = 18.50 + ($chargeable_income - 730) * 0.175;
    } elseif ($chargeable_income <= 19896.67) {
        $tax = 572.67 + ($chargeable_income - 3896.67) * 0.25;
    } elseif ($chargeable_income <= 50416.67) {
        $tax = 4572.67 + ($chargeable_income - 19896.67) * 0.30;
    } else {
        $tax = 13728.67 + ($chargeable_income - 50416.67) * 0.35;
    }

    return $tax;
}

include 'includes/accounting_header.php';
?>

<style>
.payroll-container {
    max-width: 1400px;
    margin: 0 auto;
    padding: 20px;
}

.payroll-header {
    background: linear-gradient(135deg, #089045ff 0%, #7b599cff 100%);
    color: white;
    padding: 30px;
    border-radius: 15px;
    margin-bottom: 30px;
    box-shadow: 0 8px 25px rgba(0,0,0,0.15);
}

.payroll-header h1 {
    margin: 0;
    font-size: 2.5em;
    font-weight: 300;
}

.payroll-header p {
    margin: 10px 0 0 0;
    opacity: 0.9;
    font-size: 1.1em;
}

.payroll-summary {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 25px;
    margin-bottom: 30px;
}

.summary-card {
    background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
    padding: 30px;
    border-radius: 16px;
    box-shadow: 0 8px 25px rgba(0,0,0,0.08), 0 4px 10px rgba(0,0,0,0.04);
    border: 1px solid rgba(255,255,255,0.8);
    transition: all 0.4s cubic-bezier(0.175, 0.885, 0.32, 1.275);
    position: relative;
    overflow: hidden;
}

.summary-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 4px;
    background: linear-gradient(90deg, #667eea 0%, #764ba2 100%);
    border-radius: 16px 16px 0 0;
}

.summary-card:hover {
    transform: translateY(-8px) scale(1.02);
    box-shadow: 0 15px 35px rgba(0,0,0,0.12), 0 8px 20px rgba(0,0,0,0.08);
}

.summary-card-icon {
    font-size: 2.5em;
    margin-bottom: 15px;
    display: block;
    opacity: 0.8;
}

.summary-card.staff .summary-card-icon { color: #4f46e5; }
.summary-card.basic .summary-card-icon { color: #059669; }
.summary-card.allowances .summary-card-icon { color: #d97706; }
.summary-card.gross .summary-card-icon { color: #7c3aed; }
.summary-card.deductions .summary-card-icon { color: #dc2626; }
.summary-card.net .summary-card-icon { color: #0891b2; }

.summary-card h3 {
    margin: 0 0 8px 0;
    color: #374151;
    font-size: 0.95em;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 1.2px;
    opacity: 0.9;
}

.summary-card .amount {
    font-size: 2.2em;
    font-weight: 700;
    margin: 0;
    background: linear-gradient(135deg, #1f2937 0%, #374151 100%);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
    line-height: 1.2;
}

.payroll-actions {
    background: white;
    padding: 25px;
    border-radius: 12px;
    box-shadow: 0 4px 15px rgba(0,0,0,0.1);
    margin-bottom: 30px;
}

.payroll-actions h3 {
    margin-top: 0;
    color: #333;
    border-bottom: 2px solid #f0f0f0;
    padding-bottom: 10px;
}

.payroll-form {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 20px;
    align-items: end;
}

.payroll-form .form-group {
    display: flex;
    flex-direction: column;
}

.payroll-form label {
    font-weight: 600;
    margin-bottom: 5px;
    color: #555;
}

.payroll-form select,
.payroll-form input {
    padding: 12px;
    border: 1px solid #ddd;
    border-radius: 6px;
    font-size: 14px;
}

.payroll-table-container {
    background: white;
    border-radius: 12px;
    box-shadow: 0 4px 15px rgba(0,0,0,0.1);
    overflow: hidden;
}

.payroll-table {
    width: 100%;
    border-collapse: collapse;
}

.payroll-table thead th {
    background: linear-gradient(135deg, #0b9131ff 0%, #34b257ff 100%);
    color: white;
    padding: 15px 12px;
    text-align: left;
    font-weight: 600;
    font-size: 0.85em;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    border-bottom: 2px solid #5a67d8;
}

.payroll-table tbody td {
    padding: 12px;
    border-bottom: 1px solid #f0f0f0;
    vertical-align: top;
}

.payroll-table tbody tr:hover {
    background-color: #f8f9ff;
}

.staff-name {
    font-weight: 600;
    color: #333;
}

.staff-role {
    background: #e2e8f0;
    color: #4a5568;
    padding: 2px 8px;
    border-radius: 12px;
    font-size: 0.8em;
    display: inline-block;
    margin-top: 4px;
}

.salary-breakdown {
    font-family: 'Courier New', monospace;
    font-size: 0.9em;
}

.salary-breakdown .amount {
    font-weight: bold;
    color: #2d3748;
}

.deductions {
    background: #fed7d7;
    border-left: 3px solid #e53e3e;
}

.deductions .amount {
    color: #c53030;
}

.net-salary {
    background: #c6f6d5;
    border-left: 3px solid #38a169;
}

.net-salary .amount {
    color: #22543d;
    font-weight: bold;
    font-size: 1.1em;
}

.btn {
    padding: 12px 24px;
    border: none;
    border-radius: 6px;
    cursor: pointer;
    font-size: 14px;
    font-weight: 600;
    text-decoration: none;
    display: inline-block;
    transition: all 0.3s ease;
}

.btn-primary {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 15px rgba(102, 126, 234, 0.4);
}

.btn-success {
    background: linear-gradient(135deg, #48bb78 0%, #38a169 100%);
    color: white;
}

.btn-success:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 15px rgba(72, 187, 120, 0.4);
}

.message {
    padding: 15px;
    border-radius: 6px;
    margin-bottom: 20px;
    font-weight: 500;
}

.message.success {
    background: #c6f6d5;
    color: #22543d;
    border: 1px solid #9ae6b4;
}

.message.error {
    background: #fed7d7;
    color: #742a2a;
    border: 1px solid #feb2b2;
}

.payroll-status {
    background: #fef5e7;
    border: 1px solid #f6e05e;
    color: #744210;
    padding: 15px;
    border-radius: 6px;
    margin-bottom: 20px;
}

.payroll-status h4 {
    margin: 0 0 10px 0;
    color: #744210;
}

.payroll-totals-summary {
    background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
    border: 1px solid rgba(255,255,255,0.8);
    border-radius: 16px;
    padding: 30px;
    margin-bottom: 30px;
    text-align: center;
    box-shadow: 0 8px 25px rgba(0,0,0,0.08), 0 4px 10px rgba(0,0,0,0.04);
    position: relative;
    overflow: hidden;
}

.payroll-totals-summary::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 4px;
    background: linear-gradient(90deg, #28a745 0%, #20c997 100%);
    border-radius: 16px 16px 0 0;
}

.payroll-totals-summary h3 {
    margin: 0 0 15px 0;
    color: #374151;
    font-size: 1.8em;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 1.2px;
    opacity: 0.9;
}

.payroll-totals-summary .totals-text {
    font-size: 2.5em;
    font-weight: 700;
    margin: 0;
    background: linear-gradient(135deg, #1f2937 0%, #374151 100%);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
    line-height: 1.2;
}

@media (max-width: 768px) {
    .payroll-container {
        padding: 10px;
    }

    .payroll-header {
        padding: 20px;
    }

    .payroll-header h1 {
        font-size: 2em;
    }

    .payroll-summary {
        grid-template-columns: 1fr;
    }

    .payroll-form {
        grid-template-columns: 1fr;
    }

    .payroll-table {
        font-size: 0.8em;
    }

    .payroll-table thead th,
    .payroll-table tbody td {
        padding: 8px 6px;
    }
}
</style>

<div class="payroll-container">
    <div class="payroll-header">
        <h1>💰 Payroll Processing</h1>
        <p>Process monthly payroll for all staff members</p>
    </div>

    <?php if ($message): ?>
        <div class="message <?php echo $message_type; ?>"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <!-- Total Payroll Summary -->
    <div class="payroll-totals-summary">
        <h3>💸 Total Cash Outflow</h3>
        <div class="totals-text">
            GH₵ <?php echo number_format($total_cash_outflow, 2); ?>
        </div>
    </div>

    <!-- Payroll Actions -->
    <div class="payroll-actions">
        <h3>Process Payroll</h3>

        <?php if ($current_payroll): ?>
            <div class="payroll-status">
                <h4>📋 Payroll Status for <?php echo date('F Y', strtotime($current_year . '-' . $current_month . '-01')); ?></h4>
                <p>Status: <strong><?php echo ucfirst($current_payroll['status']); ?></strong></p>
                <p>Processed by: <?php echo htmlspecialchars($current_payroll['processed_by']); ?> on <?php echo date('M d, Y H:i', strtotime($current_payroll['processed_at'])); ?></p>

                <?php if ($current_payroll['status'] == 'draft'): ?>
                    <a href="payroll_management.php?id=<?php echo $current_payroll['id']; ?>" class="btn btn-success">Continue Processing</a>
                <?php elseif ($current_payroll['status'] == 'processed'): ?>
                    <a href="view_payslips.php?payroll_id=<?php echo $current_payroll['id']; ?>" class="btn btn-primary">View Payslips</a>
                <?php endif; ?>
            </div>
        <?php endif; ?>

        <h4>Process Payroll for Specific Month</h4>
        <form method="post" action="process_payroll.php" class="payroll-form">
            <div class="form-group">
                <label for="payroll_month">Month:</label>
                <select id="payroll_month" name="payroll_month" required>
                    <?php
                    $months = [
                        '01' => 'January', '02' => 'February', '03' => 'March',
                        '04' => 'April', '05' => 'May', '06' => 'June',
                        '07' => 'July', '08' => 'August', '09' => 'September',
                        '10' => 'October', '11' => 'November', '12' => 'December'
                    ];
                    foreach ($months as $num => $name) {
                        $selected = ($num == $current_month) ? 'selected' : '';
                        echo "<option value='$num' $selected>$name</option>";
                    }
                    ?>
                </select>
            </div>
            <div class="form-group">
                <label for="payroll_year">Year:</label>
                <input type="number" id="payroll_year" name="payroll_year" value="<?php echo $current_year; ?>" required min="2020" max="2030">
            </div>
            <div class="form-group">
                <button type="submit" name="process_payroll" class="btn btn-primary">🚀 Process Payroll</button>
            </div>
        </form>
    </div>

    <!-- Salary Structures Table -->
    <div class="payroll-table-container">
        <table class="payroll-table">
            <thead>
                <tr>
                    <th>Staff Member</th>
                    <th>Basic Salary</th>
                    <th>Allowances</th>
                    <th>Gross Salary</th>
                    <th>SSNIT (5.5%)</th>
                    <th>PAYE Tax</th>
                    <th>Total Deductions</th>
                    <th>Net Salary</th>
                    <th>Effective Date</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($salary_structures as $structure): ?>
                    <?php
                    $basic = $structure['basic_salary'];
                    $allowances = $structure['total_allowances'];
                    $gross = $basic + $allowances;

                    $ssnit_employee = $basic * 0.055;
                    $chargeable_income = $gross - $ssnit_employee;
                    $paye_tax = calculateGhanaTax($chargeable_income);
                    $total_deductions = $ssnit_employee + $paye_tax;
                    $net_salary = $gross - $total_deductions;
                    ?>
                    <tr>
                        <td>
                            <div class="staff-name"><?php echo htmlspecialchars($structure['first_name'] . ' ' . $structure['last_name']); ?></div>
                            <div class="staff-role"><?php echo ucfirst(str_replace('_', ' ', $structure['staff_type'])); ?></div>
                        </td>
                        <td class="salary-breakdown">
                            <div class="amount">GH₵ <?php echo number_format($basic, 2); ?></div>
                        </td>
                        <td class="salary-breakdown">
                            <div class="amount">GH₵ <?php echo number_format($allowances, 2); ?></div>
                            <?php if ($structure['allowance_details']): ?>
                                <small style="color: #666;"><?php echo htmlspecialchars($structure['allowance_details']); ?></small>
                            <?php endif; ?>
                        </td>
                        <td class="salary-breakdown">
                            <div class="amount">GH₵ <?php echo number_format($gross, 2); ?></div>
                        </td>
                        <td class="salary-breakdown deductions">
                            <div class="amount">GH₵ <?php echo number_format($ssnit_employee, 2); ?></div>
                        </td>
                        <td class="salary-breakdown deductions">
                            <div class="amount">GH₵ <?php echo number_format($paye_tax, 2); ?></div>
                            <small style="color: #666;">on GH₵<?php echo number_format($chargeable_income, 2); ?></small>
                        </td>
                        <td class="salary-breakdown deductions">
                            <div class="amount">GH₵ <?php echo number_format($total_deductions, 2); ?></div>
                        </td>
                        <td class="salary-breakdown net-salary">
                            <div class="amount">GH₵ <?php echo number_format($net_salary, 2); ?></div>
                        </td>
                        <td><?php echo date('M d, Y', strtotime($structure['effective_date'])); ?></td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
            <tfoot>
                <tr style="background: #f8f9fa; font-weight: bold;">
                    <td style="padding: 15px 12px; font-weight: bold; color: #333;">TOTALS</td>
                    <td style="padding: 15px 12px; font-weight: bold; color: #059669;">GH₵ <?php echo number_format($total_basic, 2); ?></td>
                    <td style="padding: 15px 12px; font-weight: bold; color: #d97706;">GH₵ <?php echo number_format($total_allowances, 2); ?></td>
                    <td style="padding: 15px 12px; font-weight: bold; color: #7c3aed;">GH₵ <?php echo number_format($total_gross, 2); ?></td>
                    <td style="padding: 15px 12px; font-weight: bold; color: #dc2626;">GH₵ <?php echo number_format($total_ssnit, 2); ?></td>
                    <td style="padding: 15px 12px; font-weight: bold; color: #dc2626;">GH₵ <?php echo number_format($total_paye, 2); ?></td>
                    <td style="padding: 15px 12px; font-weight: bold; color: #dc2626;">GH₵ <?php echo number_format($total_deductions, 2); ?></td>
                    <td style="padding: 15px 12px; font-weight: bold; color: #0891b2;">GH₵ <?php echo number_format($total_net, 2); ?></td>
                    <td style="padding: 15px 12px;"></td>
                </tr>
            </tfoot>
        </table>
    </div>
</div>

<?php
include '../includes/admin_footer.php';
?>
