<?php
require_once '../config.php';

$message = '';
$message_type = '';

// Initialize variables
$salary_count = 0;
$total_teachers = 0;
$recent_payrolls = [];
$current_payroll = null;

// Get payroll statistics
try {
    // Count of staff with salary structures
    $stmt = $pdo->query("SELECT COUNT(DISTINCT staff_id) as staff_with_salary FROM salary_structures WHERE is_active = 1");
    $salary_count = $stmt->fetch()['staff_with_salary'];

    // Total staff from unified staff table
    $stmt = $pdo->query("SELECT COUNT(*) as total_staff FROM staff");
    $total_teachers = $stmt->fetch()['total_staff'];

    // Recent payroll runs
    $stmt = $pdo->query("SELECT * FROM payroll_runs ORDER BY created_at DESC LIMIT 5");
    $recent_payrolls = $stmt->fetchAll();

    // Current month payroll status
    $current_month = date('Y-m-01');
    $stmt = $pdo->prepare("SELECT * FROM payroll_runs WHERE payroll_month = ?");
    $stmt->execute([$current_month]);
    $current_payroll = $stmt->fetch();

} catch (PDOException $e) {
    $message = "Database error: " . $e->getMessage();
    $message_type = 'error';
}
include 'includes/accounting_header.php';
?>

<!DOCTYPE html>
<html lang="en">
<body>

<style>
.payroll-dashboard {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    gap: 20px;
    margin-bottom: 30px;
}

.stat-card {
    background: white;
    border-radius: 12px;
    padding: 25px;
    box-shadow: 0 4px 15px rgba(0,0,0,0.1);
    border-left: 4px solid #3498db;
    transition: transform 0.3s ease;
}

.stat-card:hover {
    transform: translateY(-2px);
}

.stat-card h3 {
    margin: 0 0 10px 0;
    color: #2c3e50;
    font-size: 1.1em;
}

.stat-value {
    font-size: 2.5em;
    font-weight: bold;
    color: #3498db;
    margin-bottom: 5px;
}

.stat-label {
    color: #7f8c8d;
    font-size: 0.9em;
}

.quick-actions {
    background: white;
    border-radius: 12px;
    padding: 25px;
    box-shadow: 0 4px 15px rgba(0,0,0,0.1);
    margin-bottom: 30px;
}

.quick-actions h3 {
    margin-top: 0;
    color: #2c3e50;
}

.action-buttons {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 15px;
}

.action-btn {
    display: flex;
    align-items: center;
    gap: 10px;
    padding: 15px 20px;
    background: linear-gradient(135deg, #3498db, #2980b9);
    color: white;
    text-decoration: none;
    border-radius: 8px;
    font-weight: 600;
    transition: all 0.3s ease;
}

.action-btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(52, 152, 219, 0.3);
}

.recent-payrolls {
    background: white;
    border-radius: 12px;
    padding: 25px;
    box-shadow: 0 4px 15px rgba(0,0,0,0.1);
}

.payroll-table {
    width: 100%;
    border-collapse: collapse;
    margin-top: 15px;
}

.payroll-table th,
.payroll-table td {
    padding: 12px 15px;
    text-align: left;
    border-bottom: 1px solid #ecf0f1;
}

.payroll-table th {
    background: #f8f9fa;
    font-weight: 600;
    color: #2c3e50;
}

.payroll-status {
    padding: 4px 8px;
    border-radius: 12px;
    font-size: 0.8em;
    font-weight: 600;
}

.status-draft { background: #fff3cd; color: #856404; }
.status-processed { background: #d1ecf1; color: #0c5460; }
.status-approved { background: #d4edda; color: #155724; }

.current-month-alert {
    background: linear-gradient(135deg, #f39c12, #e67e22);
    color: white;
    padding: 20px;
    border-radius: 12px;
    margin-bottom: 20px;
    text-align: center;
}

.current-month-alert h3 {
    margin: 0 0 10px 0;
}

.process-btn {
    background: white;
    color: #e67e22;
    border: none;
    padding: 10px 20px;
    border-radius: 6px;
    font-weight: 600;
    cursor: pointer;
    margin-top: 10px;
}
</style>

<h2>💰 Payroll Management System</h2>

<?php if ($message): ?>
    <div class="message <?php echo $message_type; ?>"><?php echo htmlspecialchars($message); ?></div>
<?php endif; ?>

<!-- Current Month Status -->
<?php if (!$current_payroll): ?>
<div class="current-month-alert">
    <h3>📅 <?php echo date('F Y'); ?> Payroll Processing</h3>
    <p>Monthly payroll for <?php echo date('F Y'); ?> has not been processed yet.</p>
    <button class="process-btn" onclick="window.location.href='process_payroll.php'">Process This Month's Payroll</button>
</div>
<?php elseif ($current_payroll['status'] == 'draft'): ?>
<div class="current-month-alert">
    <h3>📝 <?php echo date('F Y'); ?> Payroll (Draft)</h3>
    <p>Payroll for <?php echo date('F Y'); ?> is in draft status and ready for processing.</p>
    <button class="process-btn" onclick="window.location.href='process_payroll.php?id=<?php echo $current_payroll['id']; ?>'">Continue Processing</button>
</div>
<?php elseif ($current_payroll['status'] == 'processed'): ?>
<div class="current-month-alert" style="background: linear-gradient(135deg, #27ae60, #2ecc71);">
    <h3>✅ <?php echo date('F Y'); ?> Payroll Processed</h3>
    <p>Payroll for <?php echo date('F Y'); ?> has been processed and is awaiting approval.</p>
    <button class="process-btn" onclick="window.location.href='view_payslips.php?payroll_id=<?php echo $current_payroll['id']; ?>'">View Payslips</button>
</div>
<?php endif; ?>

<!-- Statistics Dashboard -->
<div class="payroll-dashboard">
    <div class="stat-card">
        <h3>👥 Total Staff</h3>
        <div class="stat-value"><?php echo $total_teachers; ?></div>
        <div class="stat-label">Registered staff</div>
    </div>

    <div class="stat-card">
        <h3>💼 Salary Structures</h3>
        <div class="stat-value"><?php echo $salary_count; ?></div>
        <div class="stat-label">Staff with salary setup</div>
    </div>

    <div class="stat-card">
        <h3>📊 Completion Rate</h3>
        <div class="stat-value"><?php echo $total_teachers > 0 ? round(($salary_count / $total_teachers) * 100) : 0; ?>%</div>
        <div class="stat-label">Salary structures configured</div>
    </div>

    <div class="stat-card">
        <h3>📄 Recent Payrolls</h3>
        <div class="stat-value"><?php echo count($recent_payrolls); ?></div>
        <div class="stat-label">Payroll runs this month</div>
    </div>
</div>

<!-- Quick Actions -->
<div class="quick-actions">
    <h3>⚡ Quick Actions</h3>
    <div class="action-buttons">
        <a href="manage_salary_structures.php" class="action-btn">
            <span>💰</span>
            Manage Salary Structures
        </a>
        <a href="manage_allowances.php" class="action-btn">
            <span>➕</span>
            Setup Allowances
        </a>
        <a href="manage_deductions.php" class="action-btn">
            <span>➖</span>
            Setup Deductions
        </a>
        <a href="manage_tax_rates.php" class="action-btn">
            <span>📊</span>
            Tax Rates Configuration
        </a>
        <a href="process_payroll.php" class="action-btn">
            <span>🔄</span>
            Process Payroll
        </a>
        <a href="view_payslips.php" class="action-btn">
            <span>📋</span>
            View Payslips
        </a>
    </div>
</div>

<!-- Recent Payroll Runs -->
<div class="recent-payrolls">
    <h3>📅 Recent Payroll Runs</h3>

    <?php if (empty($recent_payrolls)): ?>
        <p style="text-align: center; color: #7f8c8d; padding: 40px;">No payroll runs found. Start by processing your first payroll.</p>
    <?php else: ?>
        <table class="payroll-table">
            <thead>
                <tr>
                    <th>Payroll Month</th>
                    <th>Status</th>
                    <th>Total Gross</th>
                    <th>Total Net</th>
                    <th>Processed By</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($recent_payrolls as $payroll): ?>
                    <tr>
                        <td><?php echo date('M Y', strtotime($payroll['payroll_month'])); ?></td>
                        <td>
                            <span class="payroll-status status-<?php echo $payroll['status']; ?>">
                                <?php echo ucfirst($payroll['status']); ?>
                            </span>
                        </td>
                        <td>GH₵ <?php echo number_format($payroll['total_gross'], 2); ?></td>
                        <td>GH₵ <?php echo number_format($payroll['total_net'], 2); ?></td>
                        <td><?php
                            $stmt = $pdo->prepare("SELECT username FROM users WHERE id = ?");
                            $stmt->execute([$payroll['processed_by']]);
                            $user = $stmt->fetch();
                            echo htmlspecialchars($user['username'] ?? 'Unknown');
                        ?></td>
                        <td>
                            <a href="view_payslips.php?payroll_id=<?php echo $payroll['id']; ?>" class="btn btn-small">View</a>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    <?php endif; ?>
</div>

<script>
// Auto-refresh current month status every 30 seconds
setInterval(function() {
    location.reload();
}, 30000);
</script>

        </main>
    </div>
</body>
</html>
