<?php
require_once '../config.php';
require_once '../includes/access_control.php';
require_once 'accounting_functions.php';

ob_start();

if (false) { // Temporarily disable feature access check to remove subscription upgrade message
    requireFeatureAccess('accounts_management');
} else {
    // Do nothing, allow access temporarily
}

$message = '';
$error = '';

// Handle success message from redirect
if (isset($_GET['success'])) {
    $message = "Authority payment processed successfully!";
}

// Handle payment actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    if ($action === 'pay_ssnit') {
        $payroll_run_id = $_POST['payroll_run_id'] ?? 0;
        $payment_amount = floatval($_POST['payment_amount'] ?? 0);
        $payment_method = $_POST['payment_method'] ?? 'cash';
        $comments = $_POST['comments'] ?? '';

        if ($payroll_run_id && $payment_amount > 0) {
            try {
                $pdo->beginTransaction();

                // Get current SSNIT liability and paid amount for this payroll run
                $stmt = $pdo->prepare("
                    SELECT pr.ssnit_amount, pr.ssnit_paid_amount, pr.payroll_month as payroll_run_date
                    FROM payroll_runs pr
                    WHERE pr.id = ?
                ");
                $stmt->execute([$payroll_run_id]);
                $payroll = $stmt->fetch();

                if ($payroll && $payroll['ssnit_amount'] > 0) {
                    $remaining_amount = $payroll['ssnit_amount'] - $payroll['ssnit_paid_amount'];

                    // Validate payment amount doesn't exceed remaining balance
                    if ($payment_amount > $remaining_amount) {
                        throw new Exception("Payment amount cannot exceed remaining balance of GHC " . number_format($remaining_amount, 2));
                    }

                    // Create statutory payment journal entry for SSNIT
                    createStatutoryPaymentEntry($pdo, 'ssnit', $payment_amount, $payment_method, date('Y-m-d'), $_SESSION['user_id'], false);

                    // Update payroll run with new paid amount and status
                    $new_paid_amount = $payroll['ssnit_paid_amount'] + $payment_amount;
                    $new_status = ($new_paid_amount >= $payroll['ssnit_amount']) ? 'paid' : 'partially_paid';
                    $is_fully_paid = ($new_paid_amount >= $payroll['ssnit_amount']) ? 1 : 0;

                    $stmt = $pdo->prepare("
                        UPDATE payroll_runs
                        SET ssnit_paid_amount = ?, ssnit_status = ?, ssnit_paid = ?, ssnit_paid_at = NOW(), ssnit_paid_by = ?, ssnit_payment_method = ?, ssnit_payment_comments = ?
                        WHERE id = ?
                    ");
                    $stmt->execute([$new_paid_amount, $new_status, $is_fully_paid, $_SESSION['user_id'], $payment_method, $comments, $payroll_run_id]);
                }

                $pdo->commit();
                $message = "SSNIT payment of GHC " . number_format($payment_amount, 2) . " processed successfully!";
                header("Location: pay_authorities.php?success=1");
                exit;
            } catch (PDOException $e) {
                if ($pdo->inTransaction()) {
                    $pdo->rollBack();
                }
                $error = "Error processing SSNIT payment: " . $e->getMessage();
            } catch (Exception $e) {
                $error = $e->getMessage();
            }
        }
    } elseif ($action === 'pay_paye') {
        $payroll_run_id = $_POST['payroll_run_id'] ?? 0;
        $payment_amount = floatval($_POST['payment_amount'] ?? 0);
        $payment_method = $_POST['payment_method'] ?? 'cash';
        $comments = $_POST['comments'] ?? '';

        if ($payroll_run_id && $payment_amount > 0) {
            try {
                $pdo->beginTransaction();

                // Get current PAYE liability and paid amount for this payroll run
                $stmt = $pdo->prepare("
                    SELECT pr.paye_amount, pr.paye_paid_amount, pr.payroll_month as payroll_run_date
                    FROM payroll_runs pr
                    WHERE pr.id = ?
                ");
                $stmt->execute([$payroll_run_id]);
                $payroll = $stmt->fetch();

                if ($payroll && $payroll['paye_amount'] > 0) {
                    $remaining_amount = $payroll['paye_amount'] - $payroll['paye_paid_amount'];

                    // Validate payment amount doesn't exceed remaining balance
                    if ($payment_amount > $remaining_amount) {
                        throw new Exception("Payment amount cannot exceed remaining balance of GHC " . number_format($remaining_amount, 2));
                    }

                    // Create statutory payment journal entry for PAYE
                    createStatutoryPaymentEntry($pdo, 'paye', $payment_amount, $payment_method, date('Y-m-d'), $_SESSION['user_id'], false);

                    // Update payroll run with new paid amount and status
                    $new_paid_amount = $payroll['paye_paid_amount'] + $payment_amount;
                    $new_status = ($new_paid_amount >= $payroll['paye_amount']) ? 'paid' : 'partially_paid';
                    $is_fully_paid = ($new_paid_amount >= $payroll['paye_amount']) ? 1 : 0;

                    $stmt = $pdo->prepare("
                        UPDATE payroll_runs
                        SET paye_paid_amount = ?, paye_status = ?, paye_paid = ?, paye_paid_at = NOW(), paye_paid_by = ?, paye_payment_method = ?, paye_payment_comments = ?
                        WHERE id = ?
                    ");
                    $stmt->execute([$new_paid_amount, $new_status, $is_fully_paid, $_SESSION['user_id'], $payment_method, $comments, $payroll_run_id]);
                }

                $pdo->commit();
                $message = "PAYE payment of GHC " . number_format($payment_amount, 2) . " processed successfully!";
                header("Location: pay_authorities.php?success=1");
                exit;
            } catch (PDOException $e) {
                if ($pdo->inTransaction()) {
                    $pdo->rollBack();
                }
                $error = "Error processing PAYE payment: " . $e->getMessage();
            } catch (Exception $e) {
                $error = $e->getMessage();
            }
        }
    } elseif ($action === 'pay_tier2') {
        $payroll_run_id = $_POST['payroll_run_id'] ?? 0;
        $payment_amount = floatval($_POST['payment_amount'] ?? 0);
        $payment_method = $_POST['payment_method'] ?? 'cash';
        $comments = $_POST['comments'] ?? '';

        if ($payroll_run_id && $payment_amount > 0) {
            try {
                $pdo->beginTransaction();

                // Get current Tier 2 liability and paid amount for this payroll run
                $stmt = $pdo->prepare("
                    SELECT pr.tier2_amount, pr.tier2_paid_amount, pr.payroll_month as payroll_run_date
                    FROM payroll_runs pr
                    WHERE pr.id = ?
                ");
                $stmt->execute([$payroll_run_id]);
                $payroll = $stmt->fetch();

                if ($payroll && $payroll['tier2_amount'] > 0) {
                    $remaining_amount = $payroll['tier2_amount'] - $payroll['tier2_paid_amount'];

                    // Validate payment amount doesn't exceed remaining balance
                    if ($payment_amount > $remaining_amount) {
                        throw new Exception("Payment amount cannot exceed remaining balance of GHC " . number_format($remaining_amount, 2));
                    }

                    // Create statutory payment journal entry for Tier 2
                    createStatutoryPaymentEntry($pdo, 'tier2', $payment_amount, $payment_method, date('Y-m-d'), $_SESSION['user_id'], false);

                    // Update payroll run with new paid amount and status
                    $new_paid_amount = $payroll['tier2_paid_amount'] + $payment_amount;
                    $new_status = ($new_paid_amount >= $payroll['tier2_amount']) ? 'paid' : 'partially_paid';
                    $is_fully_paid = ($new_paid_amount >= $payroll['tier2_amount']) ? 1 : 0;

                    $stmt = $pdo->prepare("
                        UPDATE payroll_runs
                        SET tier2_paid_amount = ?, tier2_status = ?, tier2_paid = ?, tier2_paid_at = NOW(), tier2_paid_by = ?, tier2_payment_method = ?, tier2_payment_comments = ?
                        WHERE id = ?
                    ");
                    $stmt->execute([$new_paid_amount, $new_status, $is_fully_paid, $_SESSION['user_id'], $payment_method, $comments, $payroll_run_id]);
                }

                $pdo->commit();
                $message = "Tier 2 pension payment of GHC " . number_format($payment_amount, 2) . " processed successfully!";
                header("Location: pay_authorities.php?success=1");
                exit;
            } catch (PDOException $e) {
                if ($pdo->inTransaction()) {
                    $pdo->rollBack();
                }
                $error = "Error processing Tier 2 payment: " . $e->getMessage();
            } catch (Exception $e) {
                $error = $e->getMessage();
            }
        }
    }
}

include 'includes/accounting_header.php';

// Get filter parameters
$status_filter = $_GET['status'] ?? 'pending';
$authority_filter = $_GET['authority'] ?? '';
$payroll_run_filter = $_GET['payroll_run'] ?? '';
$date_from = $_GET['date_from'] ?? '';
$date_to = $_GET['date_to'] ?? '';
$sort_by = $_GET['sort_by'] ?? 'payroll_month';
$sort_order = $_GET['sort_order'] ?? 'DESC';

// Validate sort_by parameter to prevent SQL injection and invalid columns
$allowed_sort_columns = ['payroll_month', 'ssnit_amount', 'paye_amount', 'tier2_amount', 'ssnit_paid', 'paye_paid', 'tier2_paid', 'ssnit_paid_at', 'paye_paid_at', 'tier2_paid_at', 'ssnit_paid_amount', 'paye_paid_amount', 'tier2_paid_amount', 'ssnit_status', 'paye_status', 'tier2_status'];
if (!in_array($sort_by, $allowed_sort_columns)) {
    $sort_by = 'payroll_month';
}

// Get payroll runs for filter
try {
    $stmt = $pdo->prepare("SELECT id, payroll_month FROM payroll_runs ORDER BY payroll_month DESC");
    $stmt->execute();
    $payroll_runs = $stmt->fetchAll();
} catch (PDOException $e) {
    $payroll_runs = [];
}

// Build query for SSNIT liabilities
$ssnit_where_conditions = [];
$ssnit_params = [];

// Always include SSNIT amount > 0 condition
$ssnit_where_conditions[] = "pr.ssnit_amount > 0";

if (!empty($status_filter)) {
    if ($status_filter === 'pending') {
        $ssnit_where_conditions[] = "pr.ssnit_paid = 0";
    } elseif ($status_filter === 'paid') {
        $ssnit_where_conditions[] = "pr.ssnit_paid = 1";
    }
}



if (!empty($payroll_run_filter)) {
    $ssnit_where_conditions[] = "pr.id = ?";
    $ssnit_params[] = $payroll_run_filter;
}

if (!empty($date_from)) {
    $ssnit_where_conditions[] = "pr.payroll_month >= ?";
    $ssnit_params[] = $date_from;
}

if (!empty($date_to)) {
    $ssnit_where_conditions[] = "pr.payroll_month <= ?";
    $ssnit_params[] = $date_to;
}

$ssnit_where_clause = !empty($ssnit_where_conditions) ? "WHERE " . implode(" AND ", $ssnit_where_conditions) : "";

// Build query for PAYE liabilities
$paye_where_conditions = [];
$paye_params = [];

// Always include PAYE amount > 0 condition
$paye_where_conditions[] = "pr.paye_amount > 0";

if (!empty($status_filter)) {
    if ($status_filter === 'pending') {
        $paye_where_conditions[] = "pr.paye_paid = 0";
    } elseif ($status_filter === 'paid') {
        $paye_where_conditions[] = "pr.paye_paid = 1";
    }
}

if (!empty($date_from)) {
    $paye_where_conditions[] = "pr.payroll_month >= ?";
    $paye_params[] = $date_from;
}

if (!empty($date_to)) {
    $paye_where_conditions[] = "pr.payroll_month <= ?";
    $paye_params[] = $date_to;
}

$paye_where_clause = !empty($paye_where_conditions) ? "WHERE " . implode(" AND ", $paye_where_conditions) : "";

// Build query for Tier 2 liabilities
$tier2_where_conditions = [];
$tier2_params = [];

// Always include Tier 2 amount > 0 condition
$tier2_where_conditions[] = "pr.tier2_amount > 0";

if (!empty($status_filter)) {
    if ($status_filter === 'pending') {
        $tier2_where_conditions[] = "pr.tier2_paid = 0";
    } elseif ($status_filter === 'paid') {
        $tier2_where_conditions[] = "pr.tier2_paid = 1";
    }
}

if (!empty($payroll_run_filter)) {
    $tier2_where_conditions[] = "pr.id = ?";
    $tier2_params[] = $payroll_run_filter;
}

if (!empty($date_from)) {
    $tier2_where_conditions[] = "pr.payroll_month >= ?";
    $tier2_params[] = $date_from;
}

if (!empty($date_to)) {
    $tier2_where_conditions[] = "pr.payroll_month <= ?";
    $tier2_params[] = $date_to;
}

$tier2_where_clause = !empty($tier2_where_conditions) ? "WHERE " . implode(" AND ", $tier2_where_conditions) : "";

try {
    // Get SSNIT liabilities
    $stmt = $pdo->prepare("
        SELECT pr.*, u.username as paid_by_name, pr.ssnit_paid_at, pr.ssnit_payment_comments, pr.ssnit_status, pr.paye_status
        FROM payroll_runs pr
        LEFT JOIN users u ON pr.ssnit_paid_by = u.id
        $ssnit_where_clause
        ORDER BY $sort_by $sort_order
    ");
    $stmt->execute($ssnit_params);
    $ssnit_liabilities = $stmt->fetchAll();

    // Get PAYE liabilities
    $stmt = $pdo->prepare("
        SELECT pr.*, u.username as paid_by_name, pr.paye_paid_at, pr.paye_payment_comments, pr.ssnit_status, pr.paye_status
        FROM payroll_runs pr
        LEFT JOIN users u ON pr.paye_paid_by = u.id
        $paye_where_clause
        ORDER BY $sort_by $sort_order
    ");
    $stmt->execute($paye_params);
    $paye_liabilities = $stmt->fetchAll();

    // Get status summary for SSNIT
    $stmt = $pdo->prepare("
        SELECT
            SUM(CASE WHEN ssnit_paid = 0 THEN 1 ELSE 0 END) as pending_count,
            SUM(CASE WHEN ssnit_paid = 1 THEN 1 ELSE 0 END) as paid_count,
            COALESCE(SUM(CASE WHEN ssnit_paid = 0 THEN ssnit_amount ELSE 0 END), 0) as pending_amount,
            COALESCE(SUM(CASE WHEN ssnit_paid = 1 THEN ssnit_amount ELSE 0 END), 0) as paid_amount
        FROM payroll_runs
        WHERE ssnit_amount > 0
    ");
    $stmt->execute();
    $ssnit_summary = $stmt->fetch();

    // Get status summary for PAYE
    $stmt = $pdo->prepare("
        SELECT
            SUM(CASE WHEN paye_paid = 0 THEN 1 ELSE 0 END) as pending_count,
            SUM(CASE WHEN paye_paid = 1 THEN 1 ELSE 0 END) as paid_count,
            COALESCE(SUM(CASE WHEN paye_paid = 0 THEN paye_amount ELSE 0 END), 0) as pending_amount,
            COALESCE(SUM(CASE WHEN paye_paid = 1 THEN paye_amount ELSE 0 END), 0) as paid_amount
        FROM payroll_runs
        WHERE paye_amount > 0
    ");
    $stmt->execute();
    $paye_summary = $stmt->fetch();

    // Get Tier 2 liabilities
    $stmt = $pdo->prepare("
        SELECT pr.*, u.username as paid_by_name, pr.tier2_paid_at, pr.tier2_payment_comments, pr.ssnit_status, pr.paye_status, pr.tier2_status
        FROM payroll_runs pr
        LEFT JOIN users u ON pr.tier2_paid_by = u.id
        $tier2_where_clause
        ORDER BY $sort_by $sort_order
    ");
    $stmt->execute($tier2_params);
    $tier2_liabilities = $stmt->fetchAll();

    // Get status summary for Tier 2
    $stmt = $pdo->prepare("
        SELECT
            SUM(CASE WHEN tier2_paid = 0 THEN 1 ELSE 0 END) as pending_count,
            SUM(CASE WHEN tier2_paid = 1 THEN 1 ELSE 0 END) as paid_count,
            COALESCE(SUM(CASE WHEN tier2_paid = 0 THEN tier2_amount ELSE 0 END), 0) as pending_amount,
            COALESCE(SUM(CASE WHEN tier2_paid = 1 THEN tier2_amount ELSE 0 END), 0) as paid_amount
        FROM payroll_runs
        WHERE tier2_amount > 0
    ");
    $stmt->execute();
    $tier2_summary = $stmt->fetch();

} catch (PDOException $e) {
    $error = "Error loading authority payments: " . $e->getMessage();
    $ssnit_liabilities = [];
    $paye_liabilities = [];
    $tier2_liabilities = [];
    $ssnit_summary = ['pending_count' => 0, 'paid_count' => 0, 'pending_amount' => 0, 'paid_amount' => 0];
    $paye_summary = ['pending_count' => 0, 'paid_count' => 0, 'pending_amount' => 0, 'paid_amount' => 0];
    $tier2_summary = ['pending_count' => 0, 'paid_count' => 0, 'pending_amount' => 0, 'paid_amount' => 0];
}
?>

<style>
.pay-authorities-container {
    max-width: 1400px;
    margin: 0 auto;
    padding: 0;
    background: transparent;
    min-height: auto;
    box-shadow: none;
}

.authority-header {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: white;
    padding: 30px;
    border-radius: 12px;
    margin-bottom: 30px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.1);
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.authority-header h1 {
    margin: 0;
    font-size: 2.2em;
    font-weight: 300;
}

.authority-header p {
    margin: 5px 0 0 0;
    opacity: 0.9;
    font-size: 1.1em;
}

.authority-content {
    display: grid;
    grid-template-columns: 1fr 320px;
    gap: 30px;
}

.authority-main {
    background: white;
    border-radius: 12px;
    box-shadow: 0 2px 15px rgba(0,0,0,0.08);
    border: 1px solid #e1e5e9;
}

.filters-section {
    background: #f8f9fa;
    padding: 25px;
    border-bottom: 1px solid #dee2e6;
}

.filters-form {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
    gap: 20px;
    align-items: end;
}

.filter-group {
    display: flex;
    flex-direction: column;
}

.filter-group label {
    font-weight: 600;
    margin-bottom: 8px;
    color: #495057;
    font-size: 0.95em;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.filter-group input,
.filter-group select {
    padding: 12px 16px;
    border: 2px solid #2E7D32;
    border-radius: 8px;
    font-size: 14px;
    transition: border-color 0.3s ease;
}

.filter-group input:focus,
.filter-group select:focus {
    outline: none;
    border-color: #4CAF50;
    box-shadow: 0 0 0 3px rgba(76, 175, 80, 0.1);
}

.filter-actions {
    display: flex;
    gap: 12px;
}

.btn {
    padding: 12px 24px;
    border: none;
    border-radius: 8px;
    cursor: pointer;
    font-size: 14px;
    font-weight: 600;
    text-decoration: none;
    display: inline-block;
    transition: all 0.3s ease;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.btn-primary {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(46, 125, 50, 0.3);
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(46, 125, 50, 0.4);
}

.btn-secondary {
    background: #6c757d;
    color: white;
}

.btn-secondary:hover {
    background: #5a6268;
    transform: translateY(-1px);
}

.btn-success {
    background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(40, 167, 69, 0.3);
}

.btn-success:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(40, 167, 69, 0.4);
}

.btn-danger {
    background: linear-gradient(135deg, #dc3545 0%, #fd7e14 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(220, 53, 69, 0.3);
}

.btn-danger:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(220, 53, 69, 0.4);
}

.authority-tabs {
    display: flex;
    border-bottom: 2px solid #dee2e6;
}

.tab-button {
    padding: 15px 30px;
    background: none;
    border: none;
    border-bottom: 3px solid transparent;
    cursor: pointer;
    font-size: 1em;
    font-weight: 600;
    color: #6c757d;
    transition: all 0.3s ease;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.tab-button.active {
    color: #2E7D32;
    border-bottom-color: #2E7D32;
}

.tab-content {
    display: none;
    padding: 25px;
}

.tab-content.active {
    display: block;
}

.authorities-table {
    width: 100%;
    border-collapse: collapse;
}

.authorities-table th,
.authorities-table td {
    padding: 16px 20px;
    text-align: left;
    border-bottom: 1px solid #e9ecef;
}

.authorities-table th {
    background: #fff;
    font-weight: 700;
    color: #000;
    position: sticky;
    top: 0;
    text-transform: uppercase;
    font-size: 0.85em;
    letter-spacing: 0.5px;
}

.authorities-table tr:hover {
    background: #f8f9fa;
}

.payroll-run {
    font-weight: 600;
    color: #2c3e50;
}

.authority-amount {
    font-weight: 700;
    color: #e74c3c;
    font-size: 1.1em;
}

.payroll-date {
    color: #6c757d;
    font-size: 0.9em;
    font-weight: 500;
}

.status-badge {
    padding: 4px 12px;
    border-radius: 20px;
    font-size: 0.8em;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    display: inline-block;
}

.status-pending {
    background: #fff3cd;
    color: #856404;
    border: 1px solid #ffeaa7;
}

.status-paid {
    background: #d4edda;
    color: #155724;
    border: 1px solid #c3e6cb;
}

.actions-cell {
    white-space: nowrap;
}

.action-btn {
    padding: 6px 12px;
    border: none;
    border-radius: 6px;
    cursor: pointer;
    font-size: 0.85em;
    margin-right: 8px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    transition: all 0.3s ease;
}

.pay-btn {
    background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
    color: white;
}

.pay-btn:hover {
    transform: translateY(-1px);
    box-shadow: 0 4px 10px rgba(40, 167, 69, 0.3);
}

.summary-cards {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 20px;
    margin-bottom: 25px;
}

.summary-card {
    background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
    padding: 20px;
    border-radius: 12px;
    text-align: center;
    border: 1px solid #e1e5e9;
    box-shadow: 0 2px 10px rgba(0,0,0,0.05);
    transition: transform 0.3s ease;
}

.summary-card:hover {
    transform: translateY(-3px);
    box-shadow: 0 4px 20px rgba(0,0,0,0.1);
}

.summary-card h3 {
    margin: 0 0 8px 0;
    font-size: 2em;
    color: #2c3e50;
    font-weight: 700;
}

.summary-card p {
    margin: 0;
    color: #6c757d;
    font-size: 0.95em;
    font-weight: 500;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.message {
    padding: 15px;
    margin-bottom: 20px;
    border-radius: 8px;
    font-weight: 600;
}

.message.success {
    background: #d4edda;
    color: #155724;
    border: 1px solid #c3e6cb;
}

.message.error {
    background: #f8d7da;
    color: #721c24;
    border: 1px solid #f5c6cb;
}

.no-authorities {
    text-align: center;
    padding: 50px 20px;
    color: #6c757d;
}

.no-authorities h3 {
    margin: 0 0 10px 0;
    color: #495057;
}

.authority-sidebar {
    background: white;
    border-radius: 12px;
    box-shadow: 0 2px 15px rgba(0,0,0,0.08);
    padding: 25px;
    border: 1px solid #e1e5e9;
}

.authority-sidebar h3 {
    margin-top: 0;
    color: #2c3e50;
    margin-bottom: 20px;
}

.quick-actions {
    display: flex;
    flex-direction: column;
    gap: 10px;
}

.quick-actions .btn {
    text-align: center;
    padding: 12px;
}

/* Modal Styles */
.modal {
    display: none;
    position: fixed;
    z-index: 1000;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0,0,0,0.5);
}

.modal-content {
    background-color: #fefefe;
    margin: 5% auto;
    padding: 0;
    border: 1px solid #888;
    width: 90%;
    max-width: 500px;
    max-height: 90vh;
    border-radius: 12px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.15);
    overflow-y: auto;
}

.modal-header {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: white;
    padding: 20px;
    border-radius: 12px 12px 0 0;
}

.modal-header h2 {
    margin: 0;
    font-size: 1.5em;
    font-weight: 300;
}

.modal-body {
    padding: 20px;
}

.modal-footer {
    padding: 20px;
    background-color: #f8f9fa;
    border-radius: 0 0 12px 12px;
    display: flex;
    justify-content: flex-end;
    gap: 10px;
}

.form-group {
    margin-bottom: 20px;
}

.form-group label {
    display: block;
    margin-bottom: 8px;
    font-weight: 600;
    color: #495057;
}

.form-group textarea {
    width: 100%;
    padding: 12px;
    border: 2px solid #dee2e6;
    border-radius: 8px;
    font-size: 14px;
    resize: vertical;
    min-height: 80px;
}

.form-group textarea:focus {
    outline: none;
    border-color: #2E7D32;
    box-shadow: 0 0 0 3px rgba(46, 125, 50, 0.1);
}

.form-group select {
    width: 100%;
    padding: 12px;
    border: 2px solid #dee2e6;
    border-radius: 8px;
    font-size: 14px;
}

.form-group select:focus {
    outline: none;
    border-color: #2E7D32;
    box-shadow: 0 0 0 3px rgba(46, 125, 50, 0.1);
}

.authority-details {
    background: #f8f9fa;
    padding: 15px;
    border-radius: 8px;
    margin-bottom: 20px;
}

.authority-details h4 {
    margin: 0 0 10px 0;
    color: #2c3e50;
    font-size: 1.1em;
}

.authority-details p {
    margin: 5px 0;
    font-size: 0.9em;
    color: #495057;
}

.authority-details .amount {
    font-weight: 700;
    color: #e74c3c;
    font-size: 1.1em;
}
</style>

<div class="pay-authorities-container">
    <div class="authority-header">
        <div>
            <h1>🏛️ Pay Authorities</h1>
            <p>Process payments to statutory authorities (SSNIT, PAYE & Tier 2 Pension)</p>
        </div>
        <div>
            <a href="process_payroll.php" class="btn btn-success">📊 Process Payroll</a>
            <a href="pay_salaries.php" class="btn btn-primary">💰 Pay Salaries</a>
            <a href="accounting_dashboard.php" class="btn btn-secondary">🏠 Dashboard</a>
        </div>
    </div>

    <?php if ($message): ?>
        <div class="message success"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="message error"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <div class="authority-content">
        <div class="authority-main">
            <!-- Filters -->
            <div class="filters-section">
                <form method="get" class="filters-form">
                    <div class="filter-group">
                        <label for="status">Status</label>
                        <select name="status" id="status">
                            <option value="">All Statuses</option>
                            <option value="pending" <?php echo $status_filter === 'pending' ? 'selected' : ''; ?>>Pending</option>
                            <option value="paid" <?php echo $status_filter === 'paid' ? 'selected' : ''; ?>>Paid</option>
                        </select>
                    </div>

                    <div class="filter-group">
                        <label for="authority">Authority</label>
                        <select name="authority" id="authority">
                            <option value="">All Authorities</option>
                            <option value="ssnit" <?php echo $authority_filter === 'ssnit' ? 'selected' : ''; ?>>SSNIT</option>
                            <option value="paye" <?php echo $authority_filter === 'paye' ? 'selected' : ''; ?>>PAYE</option>
                            <option value="tier2" <?php echo $authority_filter === 'tier2' ? 'selected' : ''; ?>>Tier 2 Pension</option>
                        </select>
                    </div>

                    <div class="filter-group">
                        <label for="payroll_run">Payroll Run</label>
                        <select name="payroll_run" id="payroll_run">
                            <option value="">All Payroll Runs</option>
                            <?php foreach ($payroll_runs as $run): ?>
                                <option value="<?php echo $run['id']; ?>" <?php echo $payroll_run_filter == $run['id'] ? 'selected' : ''; ?>>
                                    Run #<?php echo $run['id']; ?> - <?php echo date('M Y', strtotime($run['payroll_month'])); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="filter-actions">
                        <button type="submit" class="btn btn-primary">🔍 Filter</button>
                        <a href="pay_authorities.php" class="btn btn-secondary">🔄 Reset</a>
                    </div>
                </form>
            </div>

            <!-- Authority Tabs -->
            <div class="authority-tabs">
                <button class="tab-button active" onclick="showTab('ssnit')">SSNIT Contributions</button>
                <button class="tab-button" onclick="showTab('paye')">PAYE Tax</button>
                <button class="tab-button" onclick="showTab('tier2')">Tier 2 Pension</button>
            </div>

            <!-- SSNIT Tab -->
            <div id="ssnit-tab" class="tab-content active">
                <!-- Summary Cards for SSNIT -->
                <div class="summary-cards">
                    <div class="summary-card">
                        <h3><?php echo $ssnit_summary['pending_count']; ?></h3>
                        <p>Pending SSNIT</p>
                    </div>
                    <div class="summary-card">
                        <h3><?php echo $ssnit_summary['paid_count']; ?></h3>
                        <p>Paid SSNIT</p>
                    </div>
                </div>

                <?php if (empty($ssnit_liabilities)): ?>
                    <div class="no-authorities">
                        <h3>No SSNIT liabilities found</h3>
                        <p>Try adjusting your filters or <a href="process_payroll.php">process a new payroll run</a>.</p>
                    </div>
                <?php else: ?>
                    <div class="table-container">
                        <table class="authorities-table">
                            <thead>
                                <tr>
                                    <th>Payroll Run</th>
                                    <th>Amount</th>
                                    <th>Run Date</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($ssnit_liabilities as $liability): ?>
                                    <tr>
                                        <td class="payroll-run">
                                            Run #<?php echo $liability['id']; ?>
                                        </td>
                                        <td class="authority-amount">
                                            GHC <?php echo number_format($liability['ssnit_amount'], 2); ?>
                                        </td>
                                        <td class="payroll-date">
                                            <?php echo date('M d, Y', strtotime($liability['payroll_month'])); ?>
                                        </td>
                                        <td>
                                            <span class="status-badge status-<?php echo strtolower(str_replace('_', '-', $liability['ssnit_status'])); ?>">
                                                <?php echo ucfirst(str_replace('_', ' ', $liability['ssnit_status'])); ?>
                                            </span>
                                            <?php if ($liability['paid_by_name']): ?>
                                                <div style="font-size: 0.7em; color: #6c757d; margin-top: 2px;">
                                                    by <?php echo htmlspecialchars($liability['paid_by_name']); ?>
                                                </div>
                                            <?php endif; ?>
                                        </td>
                                        <td class="actions-cell">
                                            <?php if ($liability['ssnit_status'] !== 'paid'): ?>
                                                <button class="action-btn pay-btn" onclick="openPaymentModal('ssnit', <?php echo $liability['id']; ?>, <?php echo $liability['ssnit_amount'] - $liability['ssnit_paid_amount']; ?>)">
                                                    💰 Pay SSNIT
                                                </button>
                                            <?php else: ?>
                                                <button class="action-btn" style="background: #6c757d; cursor: not-allowed;" disabled>
                                                    Paid
                                                </button>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>

            <!-- PAYE Tab -->
            <div id="paye-tab" class="tab-content">
                <!-- Summary Cards for PAYE -->
                <div class="summary-cards">
                    <div class="summary-card">
                        <h3><?php echo $paye_summary['pending_count']; ?></h3>
                        <p>Pending PAYE</p>
                    </div>
                    <div class="summary-card">
                        <h3><?php echo $paye_summary['paid_count']; ?></h3>
                        <p>Paid PAYE</p>
                    </div>
                </div>

                <?php if (empty($paye_liabilities)): ?>
                    <div class="no-authorities">
                        <h3>No PAYE liabilities found</h3>
                        <p>Try adjusting your filters or <a href="process_payroll.php">process a new payroll run</a>.</p>
                    </div>
                <?php else: ?>
                    <div class="table-container">
                        <table class="authorities-table">
                            <thead>
                                <tr>
                                    <th>Payroll Run</th>
                                    <th>Amount</th>
                                    <th>Run Date</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($paye_liabilities as $liability): ?>
                                    <tr>
                                        <td class="payroll-run">
                                            Run #<?php echo $liability['id']; ?>
                                        </td>
                                        <td class="authority-amount">
                                            GHC <?php echo number_format($liability['paye_amount'], 2); ?>
                                        </td>
                                        <td class="payroll-date">
                                            <?php echo date('M d, Y', strtotime($liability['payroll_month'])); ?>
                                        </td>
                                        <td>
                                            <span class="status-badge status-<?php echo strtolower(str_replace('_', '-', $liability['paye_status'])); ?>">
                                                <?php echo ucfirst(str_replace('_', ' ', $liability['paye_status'])); ?>
                                            </span>
                                            <?php if ($liability['paid_by_name']): ?>
                                                <div style="font-size: 0.7em; color: #6c757d; margin-top: 2px;">
                                                    by <?php echo htmlspecialchars($liability['paid_by_name']); ?>
                                                </div>
                                            <?php endif; ?>
                                        </td>
                                        <td class="actions-cell">
                                            <?php if ($liability['paye_status'] !== 'paid'): ?>
                                                <button class="action-btn pay-btn" onclick="openPaymentModal('paye', <?php echo $liability['id']; ?>, <?php echo $liability['paye_amount'] - $liability['paye_paid_amount']; ?>)">
                                                    💰 Pay PAYE
                                                </button>
                                            <?php else: ?>
                                                <button class="action-btn" style="background: #6c757d; cursor: not-allowed;" disabled>
                                                    Paid
                                                </button>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>

            <!-- Tier 2 Tab -->
            <div id="tier2-tab" class="tab-content">
                <!-- Summary Cards for Tier 2 -->
                <div class="summary-cards">
                    <div class="summary-card">
                        <h3><?php echo $tier2_summary['pending_count']; ?></h3>
                        <p>Pending Tier 2</p>
                    </div>
                    <div class="summary-card">
                        <h3><?php echo $tier2_summary['paid_count']; ?></h3>
                        <p>Paid Tier 2</p>
                    </div>
                </div>

                <?php if (empty($tier2_liabilities)): ?>
                    <div class="no-authorities">
                        <h3>No Tier 2 pension liabilities found</h3>
                        <p>Try adjusting your filters or <a href="process_payroll.php">process a new payroll run</a>.</p>
                    </div>
                <?php else: ?>
                    <div class="table-container">
                        <table class="authorities-table">
                            <thead>
                                <tr>
                                    <th>Payroll Run</th>
                                    <th>Amount</th>
                                    <th>Run Date</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($tier2_liabilities as $liability): ?>
                                    <tr>
                                        <td class="payroll-run">
                                            Run #<?php echo $liability['id']; ?>
                                        </td>
                                        <td class="authority-amount">
                                            GHC <?php echo number_format($liability['tier2_amount'], 2); ?>
                                        </td>
                                        <td class="payroll-date">
                                            <?php echo date('M d, Y', strtotime($liability['payroll_month'])); ?>
                                        </td>
                                        <td>
                                            <span class="status-badge status-<?php echo strtolower(str_replace('_', '-', $liability['tier2_status'])); ?>">
                                                <?php echo ucfirst(str_replace('_', ' ', $liability['tier2_status'])); ?>
                                            </span>
                                            <?php if ($liability['paid_by_name']): ?>
                                                <div style="font-size: 0.7em; color: #6c757d; margin-top: 2px;">
                                                    by <?php echo htmlspecialchars($liability['paid_by_name']); ?>
                                                </div>
                                            <?php endif; ?>
                                        </td>
                                        <td class="actions-cell">
                                            <?php if ($liability['tier2_status'] !== 'paid'): ?>
                                                <button class="action-btn pay-btn" onclick="openPaymentModal('tier2', <?php echo $liability['id']; ?>, <?php echo $liability['tier2_amount'] - $liability['tier2_paid_amount']; ?>)">
                                                    💰 Pay Tier 2
                                                </button>
                                            <?php else: ?>
                                                <button class="action-btn" style="background: #6c757d; cursor: not-allowed;" disabled>
                                                    Paid
                                                </button>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>

        <div class="authority-sidebar">
            <h3>📊 Payment Summary</h3>
            <div style="max-height: 300px; overflow-y: auto;">
                <!-- SSNIT Summary -->
                <div style="margin-bottom: 20px; padding-bottom: 20px; border-bottom: 1px solid #f0f0f0;">
                    <h4 style="margin: 0 0 15px 0; color: #2E7D32;">SSNIT Contributions</h4>
                    <div style='display: flex; justify-content: space-between; padding: 8px 0; border-bottom: 1px solid #f0f0f0;'>
                        <div>
                            <strong style='color: #ffc107'>Pending</strong>
                            <div style='font-size: 0.8em; color: #7f8c8d;'><?php echo $ssnit_summary['pending_count']; ?> payroll runs</div>
                        </div>
                        <div style='text-align: right;'>
                            <div style='font-weight: 600; color: #e74c3c;'>GHC <?php echo number_format($ssnit_summary['pending_amount'], 2); ?></div>
                        </div>
                    </div>
                    <div style='display: flex; justify-content: space-between; padding: 8px 0;'>
                        <div>
                            <strong style='color: #28a745'>Paid</strong>
                            <div style='font-size: 0.8em; color: #7f8c8d;'><?php echo $ssnit_summary['paid_count']; ?> payroll runs</div>
                        </div>
                        <div style='text-align: right;'>
                            <div style='font-weight: 600; color: #e74c3c;'>GHC <?php echo number_format($ssnit_summary['paid_amount'], 2); ?></div>
                        </div>
                    </div>
                </div>

                <!-- PAYE Summary -->
                <div style="margin-bottom: 20px; padding-bottom: 20px; border-bottom: 1px solid #f0f0f0;">
                    <h4 style="margin: 0 0 15px 0; color: #2E7D32;">PAYE Tax</h4>
                    <div style='display: flex; justify-content: space-between; padding: 8px 0; border-bottom: 1px solid #f0f0f0;'>
                        <div>
                            <strong style='color: #ffc107'>Pending</strong>
                            <div style='font-size: 0.8em; color: #7f8c8d;'><?php echo $paye_summary['pending_count']; ?> payroll runs</div>
                        </div>
                        <div style='text-align: right;'>
                            <div style='font-weight: 600; color: #e74c3c;'>GHC <?php echo number_format($paye_summary['pending_amount'], 2); ?></div>
                        </div>
                    </div>
                    <div style='display: flex; justify-content: space-between; padding: 8px 0;'>
                        <div>
                            <strong style='color: #28a745'>Paid</strong>
                            <div style='font-size: 0.8em; color: #7f8c8d;'><?php echo $paye_summary['paid_count']; ?> payroll runs</div>
                        </div>
                        <div style='text-align: right;'>
                            <div style='font-weight: 600; color: #e74c3c;'>GHC <?php echo number_format($paye_summary['paid_amount'], 2); ?></div>
                        </div>
                    </div>
                </div>

                <!-- Tier 2 Summary -->
                <div>
                    <h4 style="margin: 0 0 15px 0; color: #2E7D32;">Tier 2 Pension</h4>
                    <div style='display: flex; justify-content: space-between; padding: 8px 0; border-bottom: 1px solid #f0f0f0;'>
                        <div>
                            <strong style='color: #ffc107'>Pending</strong>
                            <div style='font-size: 0.8em; color: #7f8c8d;'><?php echo $tier2_summary['pending_count']; ?> payroll runs</div>
                        </div>
                        <div style='text-align: right;'>
                            <div style='font-weight: 600; color: #e74c3c;'>GHC <?php echo number_format($tier2_summary['pending_amount'], 2); ?></div>
                        </div>
                    </div>
                    <div style='display: flex; justify-content: space-between; padding: 8px 0;'>
                        <div>
                            <strong style='color: #28a745'>Paid</strong>
                            <div style='font-size: 0.8em; color: #7f8c8d;'><?php echo $tier2_summary['paid_count']; ?> payroll runs</div>
                        </div>
                        <div style='text-align: right;'>
                            <div style='font-weight: 600; color: #e74c3c;'>GHC <?php echo number_format($tier2_summary['paid_amount'], 2); ?></div>
                        </div>
                    </div>
                </div>
            </div>

            <h3 style="margin-top: 30px;">🔗 Quick Actions</h3>
            <div class="quick-actions">
                <a href="process_payroll.php" class="btn btn-success">Process New Payroll</a>
                <a href="pay_salaries.php" class="btn btn-primary">Pay Salaries</a>
                <a href="balance_sheet_report.php" class="btn btn-primary">View Balance Sheet</a>
                <a href="profit_loss_report.php" class="btn btn-primary">View P&L Report</a>
            </div>
        </div>
    </div>
</div>

<!-- Payment Modal -->
<div id="paymentModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h2 id="modalTitle">💰 Pay Authority</h2>
        </div>
        <div class="modal-body">
            <div class="authority-details">
                <h4 id="authorityType"></h4>
                <p><strong>Total Amount:</strong> <span id="authorityAmount" class="amount"></span></p>
                <p><strong>Remaining Balance:</strong> <span id="remainingBalance" class="amount"></span></p>
                <p><strong>Action:</strong> <span id="paymentAction"></span></p>
            </div>

            <form id="paymentForm" method="post">
                <input type="hidden" name="payroll_run_id" id="modalPayrollRunId">
                <input type="hidden" name="action" id="modalAction">
                <input type="hidden" name="remaining_balance" id="modalRemainingBalance">

                <div class="form-group">
                    <label for="payment_amount">Payment Amount (GHC)</label>
                    <input type="number" name="payment_amount" id="payment_amount" step="0.01" min="0.01" required>
                    <small style="color: #6c757d; font-size: 0.85em;">Maximum: <span id="maxAmount"></span></small>
                </div>

                <div class="form-group">
                    <label for="payment_method">Payment Method</label>
                    <select name="payment_method" id="payment_method" required>
                        <option value="cash">Cash</option>
                        <option value="bank_transfer">Bank Transfer</option>
                        <option value="cheque">Cheque</option>
                        <option value="mobile_money">Mobile Money</option>
                    </select>
                </div>

                <div class="form-group">
                    <label for="comments">Comments (Optional)</label>
                    <textarea name="comments" id="comments" placeholder="Add any comments about this payment..."></textarea>
                </div>

                <div class="form-group" style="text-align: right; margin-top: 20px;">
                    <button type="button" class="btn btn-secondary" onclick="closeModal()" style="margin-right: 10px;">Cancel</button>
                    <button type="submit" class="btn btn-success">Process Payment</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function showTab(tabName) {
    // Hide all tabs
    const tabs = document.querySelectorAll('.tab-content');
    tabs.forEach(tab => tab.classList.remove('active'));

    // Remove active class from all buttons
    const buttons = document.querySelectorAll('.tab-button');
    buttons.forEach(button => button.classList.remove('active'));

    // Show selected tab
    document.getElementById(tabName + '-tab').classList.add('active');

    // Add active class to clicked button
    event.target.classList.add('active');
}

function openPaymentModal(authority, payrollRunId, amount) {
    document.getElementById('authorityType').textContent = authority.toUpperCase() + ' Payment';
    document.getElementById('authorityAmount').textContent = 'GHC ' + amount.toFixed(2);
    document.getElementById('paymentAction').textContent = 'Pay ' + authority.toUpperCase();
    document.getElementById('modalPayrollRunId').value = payrollRunId;
    document.getElementById('modalAction').value = 'pay_' + authority;
    document.getElementById('modalTitle').textContent = '💰 Pay ' + authority.toUpperCase();

    document.getElementById('paymentModal').style.display = 'block';
}

function closeModal() {
    document.getElementById('paymentModal').style.display = 'none';
}

// Close modal when clicking outside
window.onclick = function(event) {
    const modal = document.getElementById('paymentModal');
    if (event.target == modal) {
        modal.style.display = 'none';
    }
}
</script>
