<?php
require_once '../config.php';
require_once '../includes/access_control.php';

// Access control temporarily disabled for incomes management page
// requireFeatureAccess('accounts_management');

include 'includes/accounting_header.php';



// Get filter parameters
$search = $_GET['search'] ?? '';
$category_filter = $_GET['category'] ?? '';
$year_filter = $_GET['year'] ?? '';
$page = max(1, intval($_GET['page'] ?? 1));
$per_page = 10;
$offset = ($page - 1) * $per_page;

// Build query
$query = "
    SELECT i.*, bc.name as category_name, ay.year as year_name
    FROM incomes i
    LEFT JOIN budget_categories bc ON i.category_id = bc.id
    LEFT JOIN academic_years ay ON i.academic_year_id = ay.id
    WHERE 1=1
";

$params = [];

if ($search) {
    $query .= " AND (i.source LIKE ? OR i.description LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

if ($category_filter) {
    $query .= " AND i.category_id = ?";
    $params[] = $category_filter;
}

if ($year_filter) {
    $query .= " AND i.academic_year_id = ?";
    $params[] = $year_filter;
}

$query .= " ORDER BY i.income_date DESC, i.created_at DESC";

// Get total count for pagination
$count_query = str_replace("SELECT i.*, bc.name as category_name, ay.year as year_name", "SELECT COUNT(*)", $query);
$count_stmt = $pdo->prepare($count_query);
$count_stmt->execute($params);
$total_records = $count_stmt->fetchColumn();
$total_pages = ceil($total_records / $per_page);

// Get records
$query .= " LIMIT $per_page OFFSET $offset";

$stmt = $pdo->prepare($query);
$stmt->execute($params);
$incomes = $stmt->fetchAll();

// Get categories for filter dropdown
$categories_stmt = $pdo->query("SELECT id, name FROM budget_categories WHERE type = 'income' ORDER BY name");
$categories = $categories_stmt->fetchAll();

// Get academic years for filter dropdown
$years_stmt = $pdo->query("SELECT id, year FROM academic_years ORDER BY year DESC");
$academic_years = $years_stmt->fetchAll();

// Calculate totals
$total_amount = 0;
foreach ($incomes as $income) {
    $total_amount += $income['amount'];
}
?>

<style>
.manage-incomes-container {
    max-width: 1200px;
    margin: 0 auto;
    padding: 0;
    background: transparent;
    min-height: auto;
    box-shadow: none;
}

.incomes-header {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: white;
    padding: 30px;
    border-radius: 12px;
    margin-bottom: 30px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.1);
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.incomes-header h1 {
    margin: 0;
    font-size: 2.2em;
    font-weight: 300;
}

.incomes-header p {
    margin: 5px 0 0 0;
    opacity: 0.9;
    font-size: 1.1em;
}

.filters-section {
    background: white;
    border-radius: 12px;
    box-shadow: 0 2px 15px rgba(0,0,0,0.08);
    padding: 25px;
    margin-bottom: 25px;
    border: 1px solid #e1e5e9;
}

.filters-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
    align-items: end;
}

.filter-group {
    display: flex;
    flex-direction: column;
}

.filter-group label {
    font-weight: 600;
    margin-bottom: 8px;
    color: #495057;
    font-size: 0.9em;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.filter-group input,
.filter-group select {
    padding: 10px 12px;
    border: 2px solid #2E7D32;
    border-radius: 6px;
    font-size: 14px;
    transition: border-color 0.3s ease;
}

.filter-group input:focus,
.filter-group select:focus {
    outline: none;
    border-color: #4CAF50;
}

.btn {
    padding: 10px 20px;
    border: none;
    border-radius: 6px;
    cursor: pointer;
    font-size: 14px;
    font-weight: 600;
    text-decoration: none;
    display: inline-block;
    transition: all 0.3s ease;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.btn-primary {
    background: #2E7D32;
    color: white;
}

.btn-secondary {
    background: #6c757d;
    color: white;
}

.btn-success {
    background: linear-gradient(135deg, #4CAF50 0%, #2E7D32 100%);
    color: white;
    border: 2px solid white;
}

.btn-danger {
    background: #2E7D32;
    color: white;
}

.btn-sm {
    padding: 6px 12px;
    font-size: 12px;
}

.table-container {
    background: white;
    border-radius: 12px;
    box-shadow: 0 2px 15px rgba(0,0,0,0.08);
    overflow: hidden;
    border: 1px solid #e1e5e9;
}

.table {
    width: 100%;
    border-collapse: collapse;
}

.table th,
.table td {
    padding: 15px;
    text-align: left;
    border-bottom: 1px solid #dee2e6;
}

.table th {
    background: #f8f9fa;
    font-weight: 600;
    color: #495057;
    text-transform: uppercase;
    font-size: 0.85em;
    letter-spacing: 0.5px;
}

.table tbody tr:hover {
    background: #f8f9fa;
}

.amount-cell {
    font-weight: 600;
    color: #2E7D32;
}

.date-cell {
    color: #6c757d;
}

.actions-cell {
    white-space: nowrap;
}

.pagination {
    display: flex;
    justify-content: center;
    align-items: center;
    gap: 10px;
    margin-top: 30px;
    padding: 20px;
}

.pagination a,
.pagination span {
    padding: 8px 12px;
    border: 1px solid #dee2e6;
    border-radius: 4px;
    text-decoration: none;
    color: #495057;
    transition: all 0.3s ease;
}

.pagination a:hover,
.pagination .current {
    background: #2E7D32;
    color: white;
    border-color: #2E7D32;
}

.summary-card {
    background: linear-gradient(135deg, #FFF3E0 0%, #FFE0B2 100%);
    border-radius: 12px;
    padding: 25px;
    margin-bottom: 25px;
    border: 1px solid #FFCC80;
    display: flex;
    align-items: center;
    gap: 20px;
}

.summary-card .icon {
    font-size: 2.5em;
    color: #FF9800;
}

.summary-card .content h3 {
    margin: 0 0 5px 0;
    color: #2E7D32;
    font-size: 1.8em;
    font-weight: 300;
}

.summary-card .content p {
    margin: 0;
    color: #2E7D32;
    font-weight: 600;
}

.message {
    padding: 15px;
    margin-bottom: 20px;
    border-radius: 8px;
    font-weight: 600;
}

.message.success {
    background: #d4edda;
    color: #155724;
    border: 1px solid #c3e6cb;
}

.message.error {
    background: #f8d7da;
    color: #721c24;
    border: 1px solid #f5c6cb;
}

.no-records {
    text-align: center;
    padding: 50px;
    color: #6c757d;
    font-style: italic;
}
</style>

<div class="manage-incomes-container">
    <div class="incomes-header">
        <div>
            <h1>💰 Manage Incomes</h1>
            <p>View and manage all income records</p>
        </div>
        <div>
            <a href="add_income.php" class="btn btn-success">➕ Add Income</a>
        </div>
    </div>

    <?php if (isset($message)): ?>
        <div class="message success"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <?php if (isset($error)): ?>
        <div class="message error"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <!-- Summary Card -->
    <div class="summary-card">
        <div class="icon">💵</div>
        <div class="content">
            <h3>GHC <?php echo number_format($total_amount, 2); ?></h3>
            <p>Total Income (Current Page)</p>
        </div>
    </div>

    <!-- Filters -->
    <div class="filters-section">
        <form method="get" class="filters-grid">
            <div class="filter-group">
                <label for="search">Search</label>
                <input type="text" id="search" name="search" value="<?php echo htmlspecialchars($search); ?>"
                       placeholder="Search by source or description">
            </div>

            <div class="filter-group">
                <label for="category">Category</label>
                <select id="category" name="category">
                    <option value="">All Categories</option>
                    <?php foreach ($categories as $category): ?>
                        <option value="<?php echo $category['id']; ?>" <?php echo $category_filter == $category['id'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($category['name']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div class="filter-group">
                <label for="year">Academic Year</label>
                <select id="year" name="year">
                    <option value="">All Years</option>
                    <?php foreach ($academic_years as $year): ?>
                        <option value="<?php echo $year['id']; ?>" <?php echo $year_filter == $year['id'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($year['year']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div class="filter-group">
                <button type="submit" class="btn btn-primary">🔍 Filter</button>
                <a href="manage_incomes.php" class="btn btn-secondary" style="margin-left: 10px;">🗑️ Clear</a>
            </div>
        </form>
    </div>

    <!-- Table -->
    <div class="table-container">
        <table class="table">
            <thead>
                <tr>
                    <th>Source</th>
                    <th>Amount</th>
                    <th>Date</th>
                    <th>Category</th>
                    <th>Academic Year</th>
                    <th>Description</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($incomes)): ?>
                    <tr>
                        <td colspan="7" class="no-records">
                            No income records found. <a href="add_income.php">Add the first income record</a>.
                        </td>
                    </tr>
                <?php else: ?>
                    <?php foreach ($incomes as $income): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($income['source']); ?></td>
                            <td class="amount-cell">GHC <?php echo number_format($income['amount'], 2); ?></td>
                            <td class="date-cell"><?php echo date('M d, Y', strtotime($income['income_date'])); ?></td>
                            <td><?php echo htmlspecialchars($income['category_name'] ?: 'Uncategorized'); ?></td>
                            <td><?php echo htmlspecialchars($income['year_name'] ?: 'N/A'); ?></td>
                          
                            <td><?php echo htmlspecialchars(substr($income['description'], 0, 50)) . (strlen($income['description']) > 50 ? '...' : ''); ?></td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>

    <!-- Pagination -->
    <?php if ($total_pages > 1): ?>
        <div class="pagination">
            <?php if ($page > 1): ?>
                <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page - 1])); ?>">&laquo; Previous</a>
            <?php endif; ?>

            <?php for ($i = max(1, $page - 2); $i <= min($total_pages, $page + 2); $i++): ?>
                <?php if ($i == $page): ?>
                    <span class="current"><?php echo $i; ?></span>
                <?php else: ?>
                    <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $i])); ?>"><?php echo $i; ?></a>
                <?php endif; ?>
            <?php endfor; ?>

            <?php if ($page < $total_pages): ?>
                <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $page + 1])); ?>">Next &raquo;</a>
            <?php endif; ?>
        </div>
    <?php endif; ?>
</div>

<?php include 'includes/accounting_footer.php'; ?>
