<?php
require_once '../config.php';

$message = '';
$error = '';

// Check for success message from redirect
$message = '';
if (isset($_GET['success'])) {
    $message = htmlspecialchars($_GET['success']);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['add_account'])) {
        // Add new cash account
        $account_name = trim($_POST['account_name']);
        $account_type = $_POST['account_type'];
        $account_number = trim($_POST['account_number']);
        $bank_name = trim($_POST['bank_name']);
        $opening_balance = floatval($_POST['opening_balance']);
        $designation = $_POST['designation'] ?? '';

        if (empty($account_name)) {
            $error = "Account name is required.";
        } elseif (!empty($designation)) {
            // Check if designation is already taken
            $check_stmt = $pdo->prepare("SELECT id FROM cash_accounts WHERE designation = ?");
            $check_stmt->execute([$designation]);
            if ($check_stmt->fetch()) {
                $error = "This designation is already assigned to another account. Each designation can only be used once.";
            }
        }

        if (empty($error)) {
            try {
                $stmt = $pdo->prepare("
                    INSERT INTO cash_accounts (account_name, account_type, account_number, bank_name, opening_balance, current_balance, designation, created_by)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?)
                ");
                $stmt->execute([$account_name, $account_type, $account_number, $bank_name, $opening_balance, $opening_balance, $designation, $_SESSION['user_id'] ?? null]);
                header("Location: " . $_SERVER['PHP_SELF'] . "?success=Cash account added successfully!");
                exit();
            } catch (PDOException $e) {
                $error = "Error adding cash account: " . $e->getMessage();
            }
        }
    } elseif (isset($_POST['update_balance'])) {
        // Update account balance
        $account_id = intval($_POST['account_id']);
        $adjustment_type = $_POST['adjustment_type'];
        $adjustment_amount = floatval($_POST['adjustment_amount']);
        $description = trim($_POST['description']);

        if ($adjustment_amount <= 0) {
            $error = "Adjustment amount must be greater than zero.";
        } else {
            try {
                // Get current balance
                $stmt = $pdo->prepare("SELECT current_balance FROM cash_accounts WHERE id = ?");
                $stmt->execute([$account_id]);
                $current_balance = $stmt->fetch()['current_balance'];

                // Calculate new balance
                $new_balance = $adjustment_type === 'increase' ?
                    $current_balance + $adjustment_amount :
                    $current_balance - $adjustment_amount;

                if ($new_balance < 0 && $adjustment_type === 'decrease') {
                    $error = "Insufficient funds. Current balance: GHC " . number_format($current_balance, 2);
                } else {
                    // Update balance
                    $stmt = $pdo->prepare("UPDATE cash_accounts SET current_balance = ? WHERE id = ?");
                    $stmt->execute([$new_balance, $account_id]);

                    // Log the transaction
                    $stmt = $pdo->prepare("
                        INSERT INTO cash_account_transactions (account_id, transaction_type, amount, description, created_by)
                        VALUES (?, ?, ?, ?, ?)
                    ");
                    $stmt->execute([$account_id, $adjustment_type, $adjustment_amount, $description, $_SESSION['user_id'] ?? null]);

                    header("Location: " . $_SERVER['PHP_SELF'] . "?success=Account balance updated successfully!");
                    exit();
                }
            } catch (PDOException $e) {
                $error = "Error updating balance: " . $e->getMessage();
            }
        }
    } elseif (isset($_POST['update_designation'])) {
        // Update account designation
        $account_id = intval($_POST['account_id']);
        $new_designation = $_POST['new_designation'] ?? '';

        if (!empty($new_designation)) {
            // Check if designation is already taken by another account
            $check_stmt = $pdo->prepare("SELECT id FROM cash_accounts WHERE designation = ? AND id != ?");
            $check_stmt->execute([$new_designation, $account_id]);
            if ($check_stmt->fetch()) {
                $error = "This designation is already assigned to another account. Each designation can only be used once.";
            }
        }

        if (empty($error)) {
            try {
                $stmt = $pdo->prepare("UPDATE cash_accounts SET designation = ? WHERE id = ?");
                $stmt->execute([$new_designation, $account_id]);
                header("Location: " . $_SERVER['PHP_SELF'] . "?success=Account designation updated successfully!");
                exit();
            } catch (PDOException $e) {
                $error = "Error updating designation: " . $e->getMessage();
            }
        }
    } elseif (isset($_POST['update_account'])) {
        // Update account details (excluding designation and balance)
        $account_id = intval($_POST['account_id']);
        $account_name = trim($_POST['account_name']);
        $account_type = $_POST['account_type'];
        $account_number = trim($_POST['account_number']);
        $bank_name = trim($_POST['bank_name']);

        if (empty($account_name)) {
            $error = "Account name is required.";
        }

        if (empty($error)) {
            try {
                $stmt = $pdo->prepare("
                    UPDATE cash_accounts
                    SET account_name = ?, account_type = ?, account_number = ?, bank_name = ?
                    WHERE id = ?
                ");
                $stmt->execute([$account_name, $account_type, $account_number, $bank_name, $account_id]);
                header("Location: " . $_SERVER['PHP_SELF'] . "?success=Account details updated successfully!");
                exit();
            } catch (PDOException $e) {
                $error = "Error updating account details: " . $e->getMessage();
            }
        }
    }
}

// Get all cash accounts
try {
    $stmt = $pdo->query("
        SELECT ca.*,
               COALESCE(SUM(CASE WHEN cat.transaction_type = 'increase' THEN cat.amount ELSE -cat.amount END), 0) as total_adjustments
        FROM cash_accounts ca
        LEFT JOIN cash_account_transactions cat ON ca.id = cat.account_id
        GROUP BY ca.id
        ORDER BY ca.created_at DESC
    ");
    $cash_accounts = $stmt->fetchAll();
} catch (PDOException $e) {
    $error = "Error loading cash accounts: " . $e->getMessage();
    $cash_accounts = [];
}

// Get recent transactions
try {
    $stmt = $pdo->query("
        SELECT cat.*, ca.account_name
        FROM cash_account_transactions cat
        JOIN cash_accounts ca ON cat.account_id = ca.id
        ORDER BY cat.created_at DESC
        LIMIT 10
    ");
    $recent_transactions = $stmt->fetchAll();
} catch (PDOException $e) {
    $recent_transactions = [];
}

include 'includes/accounting_header.php';
?>

<div class="cash-accounts-container">
    <div class="page-header">
        <h1>💰 Cash Account Management</h1>
        <p>Manage your school's cash accounts and track balances</p>
    </div>

    <!-- Toast Notification -->
    <div id="successToast" class="toast-notification">
        <div class="toast-content">
            <span class="toast-icon">✅</span>
            <span id="toastMessage"></span>
        </div>
    </div>

    <?php if ($error): ?>
        <div class="error-message"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <div class="accounts-grid">
        <!-- Add New Account -->
        <div class="account-card add-account-card">
            <h3>➕ Add New Cash Account</h3>
            <form method="post" class="account-form">
                <div class="form-group">
                    <label for="account_name">Account Name *</label>
                    <input type="text" id="account_name" name="account_name" required>
                </div>

                <div class="form-group">
                    <label for="account_type">Account Type</label>
                    <select id="account_type" name="account_type">
                        <option value="bank">Bank Account</option>
                        <option value="cash">Cash</option>
                        <option value="petty_cash">Petty Cash</option>
                        <option value="mobile_money">Mobile Money</option>
                    </select>
                </div>

                <div class="form-group">
                    <label for="account_number">Account Number</label>
                    <input type="text" id="account_number" name="account_number">
                </div>

                <div class="form-group">
                    <label for="bank_name">Bank Name</label>
                    <input type="text" id="bank_name" name="bank_name">
                </div>

                <div class="form-group">
                    <label for="opening_balance">Opening Balance (GHC)</label>
                    <input type="number" id="opening_balance" name="opening_balance" step="0.01" min="0" value="0.00" required>
                </div>

                <div class="form-group">
                    <label for="designation">Account Designation</label>
                    <select id="designation" name="designation">
                        <option value="">No Designation</option>
                        <option value="primary_income">Primary Income Account</option>
                        <option value="primary_expense">Primary Expense Account</option>
                        <option value="secondary_income">Secondary Income Account</option>
                        <option value="secondary_expense">Secondary Expense Account</option>
                        <option value="petty_cash">Petty Cash Account</option>
                        <option value="savings">Savings Account</option>
                    </select>
                </div>

                <button type="submit" name="add_account" class="btn btn-primary">Add Account</button>
            </form>
        </div>

        <!-- Account Summary -->
        <div class="account-card summary-card">
            <h3>📊 Account Summary</h3>
            <div class="summary-stats">
                <div class="stat-item">
                    <span class="stat-label">Total Accounts</span>
                    <span class="stat-value"><?php echo count($cash_accounts); ?></span>
                </div>
                <div class="stat-item">
                    <span class="stat-label">Total Balance</span>
                    <span class="stat-value">GHC <?php echo number_format(array_sum(array_column($cash_accounts, 'current_balance')), 2); ?></span>
                </div>
                <div class="stat-item">
                    <span class="stat-label">Beginning Cash Balance</span>
                    <span class="stat-value">GHC <?php echo number_format(array_sum(array_column($cash_accounts, 'current_balance')), 2); ?></span>
                </div>
            </div>
        </div>
    </div>

    <!-- Existing Accounts -->
    <div class="accounts-section">
        <h2>🏦 Existing Cash Accounts</h2>
        <?php if (empty($cash_accounts)): ?>
            <div class="no-accounts">
                <p>No cash accounts found. Add your first account above.</p>
            </div>
        <?php else: ?>
            <div class="accounts-table-container">
                <table class="accounts-table">
                    <thead>
                        <tr>
                            <th>Account Name</th>
                            <th>Type</th>
                            <th>Designation</th>
                            <th>Account Number</th>
                            <th>Bank</th>
                            <th>Opening Balance</th>
                            <th>Current Balance</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($cash_accounts as $account): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($account['account_name']); ?></td>
                                <td><?php echo ucfirst(str_replace('_', ' ', $account['account_type'])); ?></td>
                                <td><?php echo htmlspecialchars($account['designation'] ? ucfirst(str_replace('_', ' ', $account['designation'])) : '-'); ?></td>
                                <td><?php echo htmlspecialchars($account['account_number'] ?? '-'); ?></td>
                                <td><?php echo htmlspecialchars($account['bank_name'] ?? '-'); ?></td>
                                <td>GHC <?php echo number_format($account['opening_balance'], 2); ?></td>
                                <td class="balance-cell">GHC <?php echo number_format($account['current_balance'], 2); ?></td>
                                <td>
                                    <div class="action-buttons">
                                        <button class="btn btn-sm btn-primary edit-account-btn"
                                                data-account-id="<?php echo $account['id']; ?>"
                                                data-account-name="<?php echo htmlspecialchars($account['account_name']); ?>"
                                                data-account-type="<?php echo htmlspecialchars($account['account_type']); ?>"
                                                data-account-number="<?php echo htmlspecialchars($account['account_number'] ?? ''); ?>"
                                                data-bank-name="<?php echo htmlspecialchars($account['bank_name'] ?? ''); ?>">
                                            Edit Account
                                        </button>
                                        <button class="btn btn-sm btn-info manage-designation-btn"
                                                data-account-id="<?php echo $account['id']; ?>"
                                                data-account-name="<?php echo htmlspecialchars($account['account_name']); ?>"
                                                data-current-designation="<?php echo htmlspecialchars($account['designation'] ?? ''); ?>">
                                            Manage Designation
                                        </button>
                                        <button class="btn btn-sm btn-secondary adjust-balance-btn"
                                                data-account-id="<?php echo $account['id']; ?>"
                                                data-account-name="<?php echo htmlspecialchars($account['account_name']); ?>"
                                                data-current-balance="<?php echo $account['current_balance']; ?>">
                                            Adjust Balance
                                        </button>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>

    <!-- Recent Transactions -->
    <?php if (!empty($recent_transactions)): ?>
        <div class="transactions-section">
            <h2>📝 Recent Transactions</h2>
            <div class="transactions-table-container">
                <table class="transactions-table">
                    <thead>
                        <tr>
                            <th>Date</th>
                            <th>Account</th>
                            <th>Type</th>
                            <th>Amount</th>
                            <th>Description</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($recent_transactions as $transaction): ?>
                            <tr>
                                <td><?php echo date('M d, Y H:i', strtotime($transaction['created_at'])); ?></td>
                                <td><?php echo htmlspecialchars($transaction['account_name']); ?></td>
                                <td>
                                    <span class="transaction-type <?php echo $transaction['transaction_type']; ?>">
                                        <?php echo ucfirst($transaction['transaction_type']); ?>
                                    </span>
                                </td>
                                <td>GHC <?php echo number_format($transaction['amount'], 2); ?></td>
                                <td><?php echo htmlspecialchars($transaction['description'] ?? '-'); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    <?php endif; ?>
</div>

<!-- Adjust Balance Modal -->
<div id="adjustBalanceModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h3>Adjust Account Balance</h3>
            <span class="close">&times;</span>
        </div>
        <form method="post" id="adjustBalanceForm">
            <input type="hidden" name="account_id" id="modal_account_id">

            <div class="form-group">
                <label>Account:</label>
                <span id="modal_account_name"></span>
            </div>

            <div class="form-group">
                <label>Current Balance:</label>
                <span id="modal_current_balance"></span>
            </div>

            <div class="form-group">
                <label for="adjustment_type">Adjustment Type:</label>
                <select name="adjustment_type" id="adjustment_type" required>
                    <option value="increase">Increase Balance</option>
                    <option value="decrease">Decrease Balance</option>
                </select>
            </div>

            <div class="form-group">
                <label for="adjustment_amount">Amount (GHC):</label>
                <input type="number" name="adjustment_amount" id="adjustment_amount" step="0.01" min="0.01" required>
            </div>

            <div class="form-group">
                <label for="description">Description:</label>
                <textarea name="description" id="description" rows="3" placeholder="Reason for adjustment..."></textarea>
            </div>

            <div class="modal-actions">
                <button type="button" class="btn btn-secondary" onclick="closeModal()">Cancel</button>
                <button type="submit" name="update_balance" class="btn btn-primary">Update Balance</button>
            </div>
        </form>
    </div>
</div>

<!-- Manage Designation Modal -->
<div id="manageDesignationModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h3>Manage Account Designation</h3>
            <span class="close" onclick="closeDesignationModal()">&times;</span>
        </div>
        <form method="post" id="manageDesignationForm">
            <input type="hidden" name="account_id" id="designation_account_id">

            <div class="form-group">
                <label>Account:</label>
                <span id="designation_account_name"></span>
            </div>

            <div class="form-group">
                <label>Current Designation:</label>
                <span id="current_designation_display"></span>
            </div>

            <div class="form-group">
                <label for="new_designation">New Designation:</label>
                <select name="new_designation" id="new_designation">
                    <option value="">Remove Designation</option>
                    <option value="primary_income">Primary Income Account</option>
                    <option value="primary_expense">Primary Expense Account</option>
                    <option value="secondary_income">Secondary Income Account</option>
                    <option value="secondary_expense">Secondary Expense Account</option>
                    <option value="petty_cash">Petty Cash Account</option>
                    <option value="savings">Savings Account</option>
                </select>
                <small class="form-help">Note: Each designation can only be assigned to one account. Selecting a designation that is already in use will show an error.</small>
            </div>

            <div class="modal-actions">
                <button type="button" class="btn btn-secondary" onclick="closeDesignationModal()">Cancel</button>
                <button type="submit" name="update_designation" class="btn btn-primary">Update Designation</button>
            </div>
        </form>
    </div>
</div>

<!-- Edit Account Modal -->
<div id="editAccountModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h3>Edit Account Details</h3>
            <span class="close" onclick="closeEditAccountModal()">&times;</span>
        </div>
        <form method="post" id="editAccountForm">
            <input type="hidden" name="account_id" id="edit_account_id">

            <div class="form-group">
                <label for="edit_account_name">Account Name *</label>
                <input type="text" id="edit_account_name" name="account_name" required>
            </div>

            <div class="form-group">
                <label for="edit_account_type">Account Type</label>
                <select id="edit_account_type" name="account_type">
                    <option value="bank">Bank Account</option>
                    <option value="cash">Cash</option>
                    <option value="petty_cash">Petty Cash</option>
                    <option value="mobile_money">Mobile Money</option>
                </select>
            </div>

            <div class="form-group">
                <label for="edit_account_number">Account Number</label>
                <input type="text" id="edit_account_number" name="account_number">
            </div>

            <div class="form-group">
                <label for="edit_bank_name">Bank Name</label>
                <input type="text" id="edit_bank_name" name="bank_name">
            </div>

            <div class="modal-actions">
                <button type="button" class="btn btn-secondary" onclick="closeEditAccountModal()">Cancel</button>
                <button type="submit" name="update_account" class="btn btn-primary">Update Account</button>
            </div>
        </form>
    </div>
</div>

<style>
.cash-accounts-container {
    padding: 20px;
    background: #f8f9fa;
    min-height: 100vh;
}

.page-header {
    background: white;
    padding: 30px;
    border-radius: 10px;
    margin-bottom: 30px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    text-align: center;
}

.page-header h1 {
    color: #2c3e50;
    margin: 0 0 10px 0;
}

.page-header p {
    color: #7f8c8d;
    margin: 0;
}

.accounts-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 20px;
    margin-bottom: 30px;
}

.account-card {
    background: white;
    padding: 25px;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.add-account-card h3,
.summary-card h3 {
    color: #2c3e50;
    margin: 0 0 20px 0;
    padding-bottom: 10px;
    border-bottom: 2px solid #e9ecef;
}

.account-form {
    display: grid;
    gap: 15px;
}

.form-group {
    display: flex;
    flex-direction: column;
}

.form-group label {
    font-weight: 600;
    margin-bottom: 5px;
    color: #495057;
}

.form-group input,
.form-group select,
.form-group textarea {
    padding: 10px;
    border: 1px solid #ced4da;
    border-radius: 4px;
    font-size: 14px;
}

.form-group textarea {
    resize: vertical;
}

.summary-stats {
    display: grid;
    gap: 15px;
}

.stat-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 10px 0;
    border-bottom: 1px solid #e9ecef;
}

.stat-item:last-child {
    border-bottom: none;
}

.stat-label {
    font-weight: 600;
    color: #495057;
}

.stat-value {
    font-size: 1.2em;
    font-weight: 600;
    color: #2c3e50;
}

.accounts-section h2,
.transactions-section h2 {
    color: #2c3e50;
    margin: 40px 0 20px 0;
    padding-bottom: 10px;
    border-bottom: 2px solid #e9ecef;
}

.accounts-table-container,
.transactions-table-container {
    background: white;
    border-radius: 10px;
    overflow: hidden;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.accounts-table,
.transactions-table {
    width: 100%;
    border-collapse: collapse;
}

.accounts-table th,
.accounts-table td,
.transactions-table th,
.transactions-table td {
    padding: 12px 15px;
    text-align: left;
    border-bottom: 1px solid #e9ecef;
}

.accounts-table th,
.transactions-table th {
    background: #f8f9fa;
    font-weight: 600;
    color: #495057;
}

.balance-cell {
    font-weight: 600;
    color: #27ae60;
}

.btn {
    padding: 8px 16px;
    border: none;
    border-radius: 4px;
    cursor: pointer;
    font-size: 14px;
    text-decoration: none;
    display: inline-block;
    transition: background-color 0.2s;
}

.btn-primary {
    background: #007bff;
    color: white;
}

.btn-primary:hover {
    background: #0056b3;
}

.btn-secondary {
    background: #6c757d;
    color: white;
}

.btn-secondary:hover {
    background: #545b62;
}

.btn.btn-sm.btn-primary.edit-account-btn,
.btn.btn-sm.btn-info.manage-designation-btn,
.btn.btn-sm.btn-secondary.adjust-balance-btn {
    padding: 4px 10px !important;
    font-size: 16px !important;
    min-width: 60px !important;
    border-radius: 1px !important;
    line-height: 1 !important;
    border-radius: 5px !important;
}

.no-accounts {
    background: white;
    padding: 40px;
    border-radius: 10px;
    text-align: center;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.no-accounts p {
    color: #7f8c8d;
    font-style: italic;
}

.transaction-type {
    padding: 4px 8px;
    border-radius: 12px;
    font-size: 12px;
    font-weight: 600;
}

.transaction-type.increase {
    background: #d4edda;
    color: #155724;
}

.transaction-type.decrease {
    background: #f8d7da;
    color: #721c24;
}

/* Toast Notification Styles */
.toast-notification {
    position: fixed;
    top: 20px;
    right: 20px;
    background: linear-gradient(135deg, #4CAF50 0%, #45a049 100%);
    color: white;
    padding: 16px 20px;
    border-radius: 12px;
    box-shadow: 0 8px 25px rgba(76, 175, 80, 0.4);
    z-index: 10000;
    display: none;
    min-width: 300px;
    max-width: 400px;
    transform: translateX(100%);
    transition: transform 0.5s ease-out, opacity 0.5s ease-out;
    border: 1px solid rgba(255, 255, 255, 0.2);
    backdrop-filter: blur(10px);
}

.toast-notification.show {
    display: flex;
    align-items: center;
    transform: translateX(0);
    opacity: 1;
}

.toast-notification.hide {
    transform: translateX(100%);
    opacity: 0;
}

.toast-content {
    display: flex;
    align-items: center;
    gap: 12px;
    width: 100%;
}

.toast-icon {
    font-size: 24px;
    flex-shrink: 0;
}

#toastMessage {
    font-weight: 500;
    font-size: 14px;
    line-height: 1.4;
    flex: 1;
}

/* Toast Animation */
@keyframes toastSlideIn {
    from {
        transform: translateX(100%);
        opacity: 0;
    }
    to {
        transform: translateX(0);
        opacity: 1;
    }
}

@keyframes toastSlideOut {
    from {
        transform: translateX(0);
        opacity: 1;
    }
    to {
        transform: translateX(100%);
        opacity: 0;
    }
}

/* Enhanced Modal Styles */
.modal {
    display: none;
    position: fixed;
    z-index: 1000;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    background: linear-gradient(135deg, rgba(0,0,0,0.4) 0%, rgba(0,0,0,0.6) 100%);
    backdrop-filter: blur(5px);
    animation: fadeIn 0.3s ease-out;
}

.modal-content {
    background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
    margin: 5% auto;
    padding: 0;
    border-radius: 20px;
    width: 90%;
    max-width: 550px;
    box-shadow: 0 20px 60px rgba(0,0,0,0.3);
    transform: scale(0.9);
    animation: modalSlideIn 0.4s ease-out forwards;
    border: 1px solid rgba(255,255,255,0.2);
    overflow: hidden;
    max-height: 90vh;
    display: flex;
    flex-direction: column;
}

.modal-header {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: white;
    padding: 25px 30px;
    border-bottom: none;
    display: flex;
    justify-content: space-between;
    align-items: center;
    position: relative;
}

.modal-header::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 100 100"><defs><pattern id="grain" width="100" height="100" patternUnits="userSpaceOnUse"><circle cx="25" cy="25" r="1" fill="rgba(255,255,255,0.1)"/><circle cx="75" cy="75" r="1" fill="rgba(255,255,255,0.1)"/><circle cx="50" cy="50" r="1" fill="rgba(255,255,255,0.1)"/></pattern></defs><rect width="100" height="100" fill="url(%23grain)"/></svg>');
    opacity: 0.3;
}

.modal-header h3 {
    margin: 0;
    color: white;
    font-size: 1.4em;
    font-weight: 600;
    text-shadow: 0 2px 4px rgba(0,0,0,0.2);
    position: relative;
    z-index: 1;
}

.close {
    color: rgba(255,255,255,0.8);
    font-size: 32px;
    font-weight: bold;
    cursor: pointer;
    transition: all 0.3s ease;
    position: relative;
    z-index: 1;
    width: 40px;
    height: 40px;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 50%;
    background: rgba(255,255,255,0.1);
}

.close:hover {
    color: white;
    background: rgba(255,255,255,0.2);
    transform: rotate(90deg);
}

#adjustBalanceForm,
#manageDesignationForm {
    padding: 30px;
    background: white;
    overflow-y: auto;
    flex: 1;
    min-height: 0;
}

.form-group {
    margin-bottom: 25px;
    position: relative;
}

.form-group label {
    display: block;
    font-weight: 600;
    margin-bottom: 8px;
    color: #2c3e50;
    font-size: 0.95em;
    transition: color 0.3s ease;
}

.form-group input:focus + label,
.form-group select:focus + label,
.form-group textarea:focus + label {
    color: #4CAF50;
}

.form-group input,
.form-group select,
.form-group textarea {
    width: 100%;
    padding: 15px 20px;
    border: 2px solid #e9ecef;
    border-radius: 12px;
    font-size: 16px;
    transition: all 0.3s ease;
    background: white;
    box-sizing: border-box;
}

.form-group input:focus,
.form-group select:focus,
.form-group textarea:focus {
    border-color: #4CAF50;
    box-shadow: 0 0 0 3px rgba(76, 175, 80, 0.1);
    outline: none;
    transform: translateY(-2px);
}

.form-group textarea {
    resize: vertical;
    min-height: 80px;
}

.form-help {
    display: block;
    margin-top: 8px;
    font-size: 0.85em;
    color: #6c757d;
    font-style: italic;
}

.modal-actions {
    display: flex;
    gap: 15px;
    justify-content: flex-end;
    margin-top: 30px;
    padding-top: 25px;
    border-top: 1px solid #e9ecef;
}

.btn {
    padding: 12px 24px;
    border: none;
    border-radius: 10px;
    cursor: pointer;
    font-size: 16px;
    font-weight: 600;
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    justify-content: center;
    transition: all 0.3s ease;
    min-width: 120px;
    position: relative;
    overflow: hidden;
}

.btn::before {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
    transition: left 0.5s;
}

.btn:hover::before {
    left: 100%;
}

.btn-primary {
    background: linear-gradient(135deg, #4CAF50 0%, #45a049 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(76, 175, 80, 0.3);
}

.btn-primary:hover {
    background: linear-gradient(135deg, #45a049 0%, #3d8b40 100%);
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(76, 175, 80, 0.4);
}

.btn-secondary {
    background: linear-gradient(135deg, #6c757d 0%, #5a6268 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(108, 117, 125, 0.3);
}

.btn-secondary:hover {
    background: linear-gradient(135deg, #5a6268 0%, #495057 100%);
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(108, 117, 125, 0.4);
}

/* Account info display styling */
#modal_account_name,
#designation_account_name,
#modal_current_balance,
#current_designation_display {
    font-weight: 600;
    color: #2c3e50;
    background: #f8f9fa;
    padding: 10px 15px;
    border-radius: 8px;
    border: 1px solid #e9ecef;
    display: inline-block;
    min-width: 150px;
}

#modal_current_balance {
    color: #27ae60;
    font-size: 1.1em;
}

#current_designation_display {
    color: #007bff;
}

/* Animations */
@keyframes fadeIn {
    from {
        opacity: 0;
    }
    to {
        opacity: 1;
    }
}

@keyframes modalSlideIn {
    from {
        opacity: 0;
        transform: scale(0.9) translateY(-20px);
    }
    to {
        opacity: 1;
        transform: scale(1) translateY(0);
    }
}

@keyframes modalSlideOut {
    from {
        opacity: 1;
        transform: scale(1) translateY(0);
    }
    to {
        opacity: 0;
        transform: scale(0.9) translateY(-20px);
    }
}

/* Responsive enhancements */
@media (max-width: 768px) {
    .modal-content {
        margin: 10% auto;
        width: 95%;
        max-width: none;
    }

    .modal-header {
        padding: 20px;
    }

    .modal-header h3 {
        font-size: 1.2em;
    }

    #adjustBalanceForm,
    #manageDesignationForm {
        padding: 20px;
    }

    .modal-actions {
        flex-direction: column;
    }

    .btn {
        width: 100%;
        margin-bottom: 10px;
    }

    .form-group input,
    .form-group select,
    .form-group textarea {
        font-size: 16px; /* Prevents zoom on iOS */
    }
}


</style>

<script>
// Modal functionality
function openModal() {
    document.getElementById('adjustBalanceModal').style.display = 'block';
}

function closeModal() {
    document.getElementById('adjustBalanceModal').style.display = 'none';
    document.getElementById('adjustBalanceForm').reset();
}

function openDesignationModal() {
    document.getElementById('manageDesignationModal').style.display = 'block';
}

function closeDesignationModal() {
    document.getElementById('manageDesignationModal').style.display = 'none';
    document.getElementById('manageDesignationForm').reset();
}

function openEditAccountModal() {
    document.getElementById('editAccountModal').style.display = 'block';
}

function closeEditAccountModal() {
    document.getElementById('editAccountModal').style.display = 'none';
    document.getElementById('editAccountForm').reset();
}

// Close modal when clicking outside
window.onclick = function(event) {
    const adjustModal = document.getElementById('adjustBalanceModal');
    const designationModal = document.getElementById('manageDesignationModal');
    const editModal = document.getElementById('editAccountModal');
    if (event.target == adjustModal) {
        closeModal();
    }
    if (event.target == designationModal) {
        closeDesignationModal();
    }
    if (event.target == editModal) {
        closeEditAccountModal();
    }
}

// Toast notification functionality
function showToast(message) {
    const toast = document.getElementById('successToast');
    const toastMessage = document.getElementById('toastMessage');

    toastMessage.textContent = message;
    toast.classList.add('show');

    // Auto-hide after 5 seconds
    setTimeout(() => {
        hideToast();
    }, 5000);
}

function hideToast() {
    const toast = document.getElementById('successToast');
    toast.classList.remove('show');
    toast.classList.add('hide');

    setTimeout(() => {
        toast.classList.remove('hide');
    }, 500);
}

// Handle adjust balance button clicks
document.addEventListener('DOMContentLoaded', function() {
    const adjustButtons = document.querySelectorAll('.adjust-balance-btn');

    adjustButtons.forEach(button => {
        button.addEventListener('click', function() {
            const accountId = this.getAttribute('data-account-id');
            const accountName = this.getAttribute('data-account-name');
            const currentBalance = this.getAttribute('data-current-balance');

            document.getElementById('modal_account_id').value = accountId;
            document.getElementById('modal_account_name').textContent = accountName;
            document.getElementById('modal_current_balance').textContent = 'GHC ' + parseFloat(currentBalance).toLocaleString('en-GH', {minimumFractionDigits: 2});

            openModal();
        });
    });

    // Handle manage designation button clicks
    const designationButtons = document.querySelectorAll('.manage-designation-btn');

    designationButtons.forEach(button => {
        button.addEventListener('click', function() {
            const accountId = this.getAttribute('data-account-id');
            const accountName = this.getAttribute('data-account-name');
            const currentDesignation = this.getAttribute('data-current-designation');

            document.getElementById('designation_account_id').value = accountId;
            document.getElementById('designation_account_name').textContent = accountName;
            document.getElementById('current_designation_display').textContent = currentDesignation ? currentDesignation.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase()) : 'None';

            openDesignationModal();
        });
    });

    // Handle edit account button clicks
    const editButtons = document.querySelectorAll('.edit-account-btn');

    editButtons.forEach(button => {
        button.addEventListener('click', function() {
            const accountId = this.getAttribute('data-account-id');
            const accountName = this.getAttribute('data-account-name');
            const accountType = this.getAttribute('data-account-type');
            const accountNumber = this.getAttribute('data-account-number');
            const bankName = this.getAttribute('data-bank-name');

            document.getElementById('edit_account_id').value = accountId;
            document.getElementById('edit_account_name').value = accountName;
            document.getElementById('edit_account_type').value = accountType;
            document.getElementById('edit_account_number').value = accountNumber;
            document.getElementById('edit_bank_name').value = bankName;

            openEditAccountModal();
        });
    });

    // Close modals with X button
    document.querySelectorAll('.close').forEach(closeBtn => {
        closeBtn.addEventListener('click', function() {
            closeModal();
            closeDesignationModal();
            closeEditAccountModal();
        });
    });

    // Show toast notification if success message exists
    <?php if ($message): ?>
        showToast("<?php echo addslashes($message); ?>");
        // Remove the query string from the URL to prevent message on refresh
        if (window.history.replaceState) {
            window.history.replaceState(null, null, window.location.pathname);
        }
    <?php endif; ?>

    // Allow manual closing of toast
    document.getElementById('successToast').addEventListener('click', function() {
        hideToast();
    });
});
</script>

<?php include '../includes/admin_footer.php'; ?>
