<?php
require_once '../config.php';
require_once '../includes/accountant_activity_logger.php';

// Check permissions - only admins and accounts users can manage budget categories
if (!isset($_SESSION['role']) || !in_array($_SESSION['role'], ['admin', 'accounts'])) {
    header('Location: ../login.php');
    exit;
}

$message = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['add_category'])) {
        // Add new category
        $name = trim($_POST['name']);
        $type = $_POST['type'];
        $description = trim($_POST['description']);

        if (empty($name)) {
            $error = "Category name is required.";
        } elseif (!in_array($type, ['income', 'expense'])) {
            $error = "Invalid category type.";
        } else {
            try {
                // Check if category name already exists
                $stmt = $pdo->prepare("SELECT id FROM budget_categories WHERE name = ?");
                $stmt->execute([$name]);
                if ($stmt->fetch()) {
                    $error = "A category with this name already exists.";
                } else {
                    $stmt = $pdo->prepare("
                        INSERT INTO budget_categories (name, type, description, is_system)
                        VALUES (?, ?, ?, FALSE)
                    ");
                    $stmt->execute([$name, $type, $description]);

                    // Log activity
                    AccountantActivityLogger::logActivity(
                        $_SESSION['user_id'],
                        'budget_category_created',
                        "Created new budget category: $name ($type)"
                    );

                    $message = "Category added successfully!";
                }
            } catch (PDOException $e) {
                $error = "Error adding category: " . $e->getMessage();
            }
        }
    } elseif (isset($_POST['edit_category'])) {
        // Edit existing category
        $id = $_POST['category_id'];
        $name = trim($_POST['name']);
        $type = $_POST['type'];
        $description = trim($_POST['description']);

        if (empty($name)) {
            $error = "Category name is required.";
        } elseif (!in_array($type, ['income', 'expense'])) {
            $error = "Invalid category type.";
        } else {
            try {
                // Check if another category with this name exists
                $stmt = $pdo->prepare("SELECT id FROM budget_categories WHERE name = ? AND id != ?");
                $stmt->execute([$name, $id]);
                if ($stmt->fetch()) {
                    $error = "Another category with this name already exists.";
                } else {
                    $stmt = $pdo->prepare("
                        UPDATE budget_categories
                        SET name = ?, type = ?, description = ?
                        WHERE id = ? AND is_system = FALSE
                    ");
                    $stmt->execute([$name, $type, $description, $id]);

                    if ($stmt->rowCount() > 0) {
                        // Log activity
                        AccountantActivityLogger::logActivity(
                            $_SESSION['user_id'],
                            'budget_category_updated',
                            "Updated budget category: $name ($type)"
                        );

                        $message = "Category updated successfully!";
                    } else {
                        $error = "Category not found or cannot be edited.";
                    }
                }
            } catch (PDOException $e) {
                $error = "Error updating category: " . $e->getMessage();
            }
        }
    } elseif (isset($_POST['delete_category'])) {
        // Delete category (only non-system categories)
        $id = $_POST['category_id'];

        try {
            // Check if category is used in expenses or incomes
            $stmt = $pdo->prepare("
                SELECT
                    (SELECT COUNT(*) FROM expenses WHERE category_id = ?) +
                    (SELECT COUNT(*) FROM incomes WHERE category_id = ?) as usage_count
            ");
            $stmt->execute([$id, $id]);
            $usage = $stmt->fetch();

            if ($usage['usage_count'] > 0) {
                $error = "Cannot delete category that is being used in transactions.";
            } else {
                $stmt = $pdo->prepare("DELETE FROM budget_categories WHERE id = ? AND is_system = FALSE");
                $stmt->execute([$id]);

                if ($stmt->rowCount() > 0) {
                    AccountantActivityLogger::logActivity(
                        $_SESSION['user_id'],
                        'budget_category_deleted',
                        "Deleted budget category ID: $id"
                    );

                    $message = "Category deleted successfully!";
                } else {
                    $error = "Category not found or cannot be deleted.";
                }
            }
        } catch (PDOException $e) {
            $error = "Error deleting category: " . $e->getMessage();
        }
    }
}

// Get all categories for display
try {
    $stmt = $pdo->query("
        SELECT bc.*,
               (
                   SELECT COUNT(*) FROM expenses e WHERE e.category_id = bc.id
               ) +
               (
                   SELECT COUNT(*) FROM incomes i WHERE i.category_id = bc.id
               ) as usage_count
        FROM budget_categories bc
        ORDER BY bc.type ASC, bc.is_system DESC, bc.name ASC
    ");
    $categories = $stmt->fetchAll();
} catch (PDOException $e) {
    $error = "Error loading categories: " . $e->getMessage();
    $categories = [];
}
?>

<?php include 'includes/accounting_header.php'; ?>

<style>
    .categories-container {
        max-width: 1400px;
        margin: 0 auto;
        padding: 20px;
        background: #f8f9fa;
        min-height: 100vh;
    }

    .categories-header {
        background: white;
        padding: 25px;
        border-radius: 10px;
        margin-bottom: 30px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        text-align: center;
    }

    .categories-content {
        display: grid;
        grid-template-columns: 1fr 2fr;
        gap: 30px;
    }

    .add-category-section,
    .categories-list-section {
        background: white;
        padding: 30px;
        border-radius: 10px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    }

    .section-title {
        color: #2c3e50;
        margin: 0 0 20px 0;
        padding-bottom: 10px;
        border-bottom: 2px solid #e9ecef;
    }

    .form-group {
        margin-bottom: 20px;
    }

    .form-group label {
        display: block;
        font-weight: 600;
        margin-bottom: 5px;
        color: #495057;
    }

    .form-group input,
    .form-group select,
    .form-group textarea {
        width: 100%;
        padding: 10px 12px;
        border: 1px solid #ced4da;
        border-radius: 5px;
        font-size: 14px;
    }

    .form-group textarea {
        resize: vertical;
        min-height: 80px;
    }

    .btn {
        padding: 10px 20px;
        border: none;
        border-radius: 5px;
        cursor: pointer;
        font-size: 14px;
        font-weight: 600;
        text-decoration: none;
        display: inline-block;
        margin-right: 10px;
    }

    .btn-primary {
        background: #007bff;
        color: white;
    }

    .btn-success {
        background: #28a745;
        color: white;
    }

    .btn-danger {
        background: #dc3545;
        color: white;
    }

    .btn-secondary {
        background: #6c757d;
        color: white;
    }

    .btn-warning {
        background: #ffc107;
        color: #212529;
    }

    .message {
        padding: 15px;
        border-radius: 5px;
        margin-bottom: 20px;
    }

    .success-message {
        background: #d4edda;
        color: #155724;
        border: 1px solid #c3e6cb;
    }

    .error-message {
        background: #f8d7da;
        color: #721c24;
        border: 1px solid #f5c6cb;
    }

    .categories-table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 20px;
    }

    .categories-table th,
    .categories-table td {
        padding: 12px;
        text-align: left;
        border-bottom: 1px solid #e9ecef;
    }

    .categories-table th {
        background: #f8f9fa;
        font-weight: 600;
        color: #495057;
    }

    .category-name {
        font-weight: bold;
        color: #007bff;
    }

    .category-type {
        display: inline-block;
        padding: 2px 8px;
        border-radius: 12px;
        font-size: 0.8em;
        font-weight: 600;
    }

    .type-income {
        background: #d4edda;
        color: #155724;
    }

    .type-expense {
        background: #f8d7da;
        color: #721c24;
    }

    .system-category {
        background: #e9ecef;
        color: #495057;
    }

    .actions-column {
        width: 200px;
    }

    .modal {
        display: none;
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0,0,0,0.5);
        z-index: 1000;
    }

    .modal-content {
        background: white;
        margin: 5% auto;
        padding: 30px;
        border-radius: 10px;
        width: 90%;
        max-width: 600px;
        box-shadow: 0 4px 20px rgba(0,0,0,0.3);
        max-height: 90vh;
        overflow-y: auto;
    }

    .modal-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 20px;
        padding-bottom: 10px;
        border-bottom: 1px solid #e9ecef;
    }

    .modal-title {
        margin: 0;
        color: #2c3e50;
    }

    .close {
        font-size: 24px;
        cursor: pointer;
        color: #6c757d;
    }

    @media (max-width: 768px) {
        .categories-content {
            grid-template-columns: 1fr;
        }
    }
</style>

<div class="categories-container">
    <div class="categories-header">
        <h1>🏷️ Budget Categories Management</h1>
        <p>Manage expense and income categories for your school</p>
    </div>

    <?php if ($message): ?>
        <div class="message success-message"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="message error-message"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <div class="categories-content">
        <div class="add-category-section">
            <h2 class="section-title">➕ Add New Category</h2>
            <form method="post">
                <div class="form-group">
                    <label for="name">Category Name *</label>
                    <input type="text" name="name" id="name" required placeholder="e.g., Office Supplies">
                </div>

                <div class="form-group">
                    <label for="type">Category Type *</label>
                    <select name="type" id="type" required>
                        <option value="">Select Type</option>
                        <option value="expense">Expense</option>
                        <option value="income">Income</option>
                    </select>
                </div>

                <div class="form-group">
                    <label for="description">Description</label>
                    <textarea name="description" id="description" placeholder="Optional description..."></textarea>
                </div>

                <button type="submit" name="add_category" class="btn btn-primary">Add Category</button>
            </form>
        </div>

        <div class="categories-list-section">
            <h2 class="section-title">📋 All Categories</h2>
            <table class="categories-table">
                <thead>
                    <tr>
                        <th>Name</th>
                        <th>Type</th>
                        <th>Status</th>
                        <th>Usage</th>
                        <th class="actions-column">Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($categories as $category): ?>
                        <tr>
                            <td>
                                <span class="category-name"><?php echo htmlspecialchars($category['name']); ?></span>
                                <?php if ($category['description']): ?>
                                    <br><small style="color: #6c757d;"><?php echo htmlspecialchars($category['description']); ?></small>
                                <?php endif; ?>
                            </td>
                            <td>
                                <span class="category-type type-<?php echo $category['type']; ?>">
                                    <?php echo ucfirst($category['type']); ?>
                                </span>
                            </td>
                            <td>
                                <?php if ($category['is_system']): ?>
                                    <span class="category-type system-category">System</span>
                                <?php else: ?>
                                    <span style="color: #28a745;">Custom</span>
                                <?php endif; ?>
                            </td>
                            <td><?php echo $category['usage_count']; ?> transactions</td>
                            <td class="actions-column">
                                <?php if (!$category['is_system']): ?>
                                    <button class="btn btn-secondary btn-sm" onclick="editCategory(<?php echo $category['id']; ?>, '<?php echo addslashes($category['name']); ?>', '<?php echo addslashes($category['type']); ?>', '<?php echo addslashes($category['description'] ?? ''); ?>')">Edit</button>
                                    <button class="btn btn-danger btn-sm" onclick="deleteCategory(<?php echo $category['id']; ?>, '<?php echo addslashes($category['name']); ?>')">Delete</button>
                                <?php else: ?>
                                    <em style="color: #6c757d;">System Category</em>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Edit Category Modal -->
<div id="editModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h2 class="modal-title">Edit Category</h2>
            <span class="close" onclick="closeModal()">&times;</span>
        </div>
        <form method="post">
            <input type="hidden" name="category_id" id="edit_category_id">
            <div class="form-group">
                <label for="edit_name">Category Name *</label>
                <input type="text" name="name" id="edit_name" required>
            </div>
            <div class="form-group">
                <label for="edit_type">Category Type *</label>
                <select name="type" id="edit_type" required>
                    <option value="expense">Expense</option>
                    <option value="income">Income</option>
                </select>
            </div>
            <div class="form-group">
                <label for="edit_description">Description</label>
                <textarea name="description" id="edit_description"></textarea>
            </div>
            <button type="submit" name="edit_category" class="btn btn-success">Update Category</button>
            <button type="button" class="btn btn-secondary" onclick="closeModal()">Cancel</button>
        </form>
    </div>
</div>

<!-- Delete Category Modal -->
<div id="deleteModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h2 class="modal-title">Delete Category</h2>
            <span class="close" onclick="closeModal()">&times;</span>
        </div>
        <p>Are you sure you want to delete the category "<span id="delete_category_name"></span>"?</p>
        <p style="color: #dc3545; font-weight: 600;">This action cannot be undone and will permanently remove the category.</p>
        <form method="post">
            <input type="hidden" name="category_id" id="delete_category_id">
            <button type="submit" name="delete_category" class="btn btn-danger">Yes, Delete Category</button>
            <button type="button" class="btn btn-secondary" onclick="closeModal()">Cancel</button>
        </form>
    </div>
</div>

<script>
function editCategory(id, name, type, description) {
    document.getElementById('edit_category_id').value = id;
    document.getElementById('edit_name').value = name;
    document.getElementById('edit_type').value = type;
    document.getElementById('edit_description').value = description;
    document.getElementById('editModal').style.display = 'block';
}

function deleteCategory(id, name) {
    document.getElementById('delete_category_id').value = id;
    document.getElementById('delete_category_name').textContent = name;
    document.getElementById('deleteModal').style.display = 'block';
}

function closeModal() {
    document.getElementById('editModal').style.display = 'none';
    document.getElementById('deleteModal').style.display = 'none';
}

// Close modal when clicking outside
window.onclick = function(event) {
    if (event.target.className === 'modal') {
        event.target.style.display = 'none';
    }
}
</script>

<?php include '../includes/admin_footer.php'; ?>
