<?php
/**
 * CSRF Protection Utilities
 * Provides Cross-Site Request Forgery protection for forms
 */

class CSRFProtection {
    const TOKEN_NAME = 'csrf_token';
    const TOKEN_LIFETIME = 3600; // 1 hour

    /**
     * Generate a CSRF token
     */
    public static function generateToken() {
        if (!isset($_SESSION[self::TOKEN_NAME])) {
            $_SESSION[self::TOKEN_NAME] = [
                'token' => bin2hex(random_bytes(32)),
                'timestamp' => time()
            ];
        }

        // Check if token needs refresh (expired)
        if (time() - $_SESSION[self::TOKEN_NAME]['timestamp'] > self::TOKEN_LIFETIME) {
            $_SESSION[self::TOKEN_NAME] = [
                'token' => bin2hex(random_bytes(32)),
                'timestamp' => time()
            ];
        }

        return $_SESSION[self::TOKEN_NAME]['token'];
    }

    /**
     * Validate a CSRF token
     */
    public static function validateToken($token) {
        // Check if token exists in session
        if (!isset($_SESSION[self::TOKEN_NAME])) {
            return false;
        }

        $storedToken = $_SESSION[self::TOKEN_NAME];

        // Check if token is expired
        if (time() - $storedToken['timestamp'] > self::TOKEN_LIFETIME) {
            self::clearToken();
            return false;
        }

        // Use hash_equals for timing attack protection
        if (!hash_equals($storedToken['token'], $token)) {
            return false;
        }

        // Token is valid, generate a new one for next request
        self::regenerateToken();
        return true;
    }

    /**
     * Get CSRF token for forms
     */
    public static function getToken() {
        return self::generateToken();
    }

    /**
     * Clear CSRF token
     */
    public static function clearToken() {
        unset($_SESSION[self::TOKEN_NAME]);
    }

    /**
     * Regenerate token after successful validation
     */
    private static function regenerateToken() {
        $_SESSION[self::TOKEN_NAME] = [
            'token' => bin2hex(random_bytes(32)),
            'timestamp' => time()
        ];
    }

    /**
     * Get hidden input field for forms
     */
    public static function getHiddenInput() {
        $token = self::getToken();
        return '<input type="hidden" name="' . self::TOKEN_NAME . '" value="' . htmlspecialchars($token) . '">';
    }

    /**
     * Validate token from POST request
     */
    public static function validateRequest() {
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $token = $_POST[self::TOKEN_NAME] ?? '';
            return self::validateToken($token);
        }
        return true; // GET requests don't need CSRF protection
    }

    /**
     * Middleware function for protecting forms
     */
    public static function protectForm($callback = null) {
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            if (!self::validateRequest()) {
                // CSRF attack detected
                http_response_code(403);
                die('CSRF token validation failed. Please refresh the page and try again.');
            }
        }

        if ($callback && is_callable($callback)) {
            return $callback();
        }
    }
}
?>
