<?php

class AccountantActivityLogger {
    private static $pdo;

    public static function init($pdo) {
        self::$pdo = $pdo;
    }

    /**
     * Log accountant activity
     */
    public static function logActivity($user_id, $action, $description = '', $amount = 0.00, $additional_data = []) {
        if (!self::$pdo) {
            return false;
        }

        try {
            // Get user info
            $stmt = self::$pdo->prepare("SELECT username FROM users WHERE id = ?");
            $stmt->execute([$user_id]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$user) {
                return false;
            }

            // Prepare log data
            $log_data = [
                'user_id' => $user_id,
                'username' => $user['username'],
                'action' => $action,
                'description' => $description,
                'amount' => floatval($amount),
                'created_at' => date('Y-m-d H:i:s')
            ];

            // Add additional data if provided
            if (!empty($additional_data)) {
                $log_data['description'] .= ' | Additional Data: ' . json_encode($additional_data);
            }

            // Insert log
            $stmt = self::$pdo->prepare("
                INSERT INTO accountants_logs
                (user_id, username, action, description, amount, created_at)
                VALUES (?, ?, ?, ?, ?, ?)
            ");

            $stmt->execute([
                $log_data['user_id'],
                $log_data['username'],
                $log_data['action'],
                $log_data['description'],
                $log_data['amount'],
                $log_data['created_at']
            ]);

            return true;

        } catch (PDOException $e) {
            error_log("Accountant Activity Logger Error: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Get accountant activity logs
     */
    public static function getAccountantLogs($user_id = null, $limit = 50, $offset = 0, $date_from = null, $date_to = null, $action_filter = null) {
        if (!self::$pdo) {
            error_log("AccountantActivityLogger: PDO not initialized");
            return [];
        }

        try {
            $where_conditions = [];
            $params = [];

            if ($user_id) {
                $where_conditions[] = "user_id = ?";
                $params[] = $user_id;
            }

            if ($date_from) {
                $where_conditions[] = "created_at >= ?";
                $params[] = $date_from . ' 00:00:00';
            }

            if ($date_to) {
                $where_conditions[] = "created_at <= ?";
                $params[] = $date_to . ' 23:59:59';
            }

            if ($action_filter) {
                $where_conditions[] = "action = ?";
                $params[] = $action_filter;
            }

            $where_clause = !empty($where_conditions) ? "WHERE " . implode(" AND ", $where_conditions) : "";

            $sql = "
                SELECT id, user_id, username, action, description, amount, created_at
                FROM accountants_logs
                {$where_clause}
                ORDER BY created_at DESC
                LIMIT ? OFFSET ?
            ";

            $stmt = self::$pdo->prepare($sql);

            // Bind parameters dynamically
            $paramIndex = 1;
            if ($user_id) {
                $stmt->bindValue($paramIndex++, $user_id, PDO::PARAM_INT);
            }
            if ($date_from) {
                $stmt->bindValue($paramIndex++, $date_from . ' 00:00:00', PDO::PARAM_STR);
            }
            if ($date_to) {
                $stmt->bindValue($paramIndex++, $date_to . ' 23:59:59', PDO::PARAM_STR);
            }
            if ($action_filter) {
                $stmt->bindValue($paramIndex++, $action_filter, PDO::PARAM_STR);
            }
            $stmt->bindValue($paramIndex++, $limit, PDO::PARAM_INT);
            $stmt->bindValue($paramIndex++, $offset, PDO::PARAM_INT);

            $stmt->execute();
            $result = $stmt->fetchAll(PDO::FETCH_ASSOC);

            return $result;

        } catch (PDOException $e) {
            error_log("Get Accountant Logs Error: " . $e->getMessage());
            return [];
        }
    }

    /**
     * Get activity statistics for accountants
     */
    public static function getActivityStats($user_id = null, $days = 30) {
        if (!self::$pdo) {
            return [];
        }

        try {
            $date_condition = $user_id ? "AND user_id = ?" : "";
            $params = $user_id ? [$user_id] : [];

            $stmt = self::$pdo->prepare("
                SELECT
                    action,
                    COUNT(*) as count,
                    SUM(amount) as total_amount,
                    MAX(created_at) as last_activity
                FROM accountants_logs
                WHERE created_at >= DATE_SUB(NOW(), INTERVAL ? DAY)
                {$date_condition}
                GROUP BY action
                ORDER BY count DESC
            ");

            array_unshift($params, $days);
            $stmt->execute($params);

            return $stmt->fetchAll(PDO::FETCH_ASSOC);

        } catch (PDOException $e) {
            error_log("Get Accountant Activity Stats Error: " . $e->getMessage());
            return [];
        }
    }

    /**
     * Get total amount logged for a specific action type within date range
     */
    public static function getTotalAmount($action, $user_id = null, $date_from = null, $date_to = null) {
        if (!self::$pdo) {
            return 0.00;
        }

        try {
            $where_conditions = ["action = ?"];
            $params = [$action];

            if ($user_id) {
                $where_conditions[] = "user_id = ?";
                $params[] = $user_id;
            }

            if ($date_from) {
                $where_conditions[] = "created_at >= ?";
                $params[] = $date_from . ' 00:00:00';
            }

            if ($date_to) {
                $where_conditions[] = "created_at <= ?";
                $params[] = $date_to . ' 23:59:59';
            }

            $where_clause = "WHERE " . implode(" AND ", $where_conditions);

            $stmt = self::$pdo->prepare("
                SELECT SUM(amount) as total_amount
                FROM accountants_logs
                {$where_clause}
            ");

            $stmt->execute($params);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);

            return floatval($result['total_amount'] ?? 0.00);

        } catch (PDOException $e) {
            error_log("Get Total Amount Error: " . $e->getMessage());
            return 0.00;
        }
    }

    /**
     * Clean old logs (keep only last N days)
     */
    public static function cleanOldLogs($days_to_keep = 90) {
        if (!self::$pdo) {
            return false;
        }

        try {
            $stmt = self::$pdo->prepare("
                DELETE FROM accountants_logs
                WHERE created_at < DATE_SUB(NOW(), INTERVAL ? DAY)
            ");

            $stmt->execute([$days_to_keep]);
            return $stmt->rowCount();

        } catch (PDOException $e) {
            error_log("Clean Old Accountant Logs Error: " . $e->getMessage());
            return false;
        }
    }
}

// Initialize the logger with the database connection
if (isset($pdo)) {
    AccountantActivityLogger::init($pdo);
}

// Also ensure the logger is initialized when this file is included
if (isset($pdo)) {
    AccountantActivityLogger::init($pdo);
}
?>
