<?php
require_once '../config.php';

$message = '';
$error = '';

// Get filter parameters
$account_id = $_GET['account_id'] ?? '';
$start_date = $_GET['start_date'] ?? date('Y-m-01');
$end_date = $_GET['end_date'] ?? date('Y-m-t');
$account_type = $_GET['account_type'] ?? 'all';

// Fetch accounts for dropdown
try {
    $stmt = $pdo->query("SELECT id, account_code, account_name, account_type FROM chart_of_accounts ORDER BY account_code ASC");
    $accounts = $stmt->fetchAll();
} catch (PDOException $e) {
    $error = "Error loading accounts: " . $e->getMessage();
    $accounts = [];
}

// Build general ledger query
try {
    $where_conditions = [];
    $params = [];

    // Date range
    $where_conditions[] = "je.transaction_date BETWEEN ? AND ?";
    $params[] = $start_date;
    $params[] = $end_date;

    // Account filter
    if (!empty($account_id)) {
        $where_conditions[] = "jel.account_id = ?";
        $params[] = $account_id;
    }

    // Account type filter
    if ($account_type !== 'all') {
        $where_conditions[] = "coa.account_type = ?";
        $params[] = $account_type;
    }

    $where_clause = implode(' AND ', $where_conditions);

    // Get general ledger entries
    $query = "
        SELECT
            je.id as journal_entry_id,
            je.transaction_date,
            je.description,
            je.reference_number,
            je.source_type,
            jel.account_id,
            jel.debit_amount,
            jel.credit_amount,
            coa.account_code,
            coa.account_name,
            coa.account_type
        FROM journal_entries je
        JOIN journal_entry_lines jel ON je.id = jel.journal_entry_id
        JOIN chart_of_accounts coa ON jel.account_id = coa.id
        WHERE {$where_clause}
        AND je.status = 'posted'
        ORDER BY coa.account_code ASC, je.transaction_date ASC, je.id ASC
    ";

    $stmt = $pdo->prepare($query);
    $stmt->execute($params);
    $ledger_entries = $stmt->fetchAll();

    // Group entries by account and calculate running balances
    $grouped_entries = [];
    $account_balances = [];

    foreach ($ledger_entries as $entry) {
        $account_id = $entry['account_id'];

        if (!isset($grouped_entries[$account_id])) {
            $grouped_entries[$account_id] = [
                'account_code' => $entry['account_code'],
                'account_name' => $entry['account_name'],
                'account_type' => $entry['account_type'],
                'entries' => [],
                'opening_balance' => 0,
                'closing_balance' => 0
            ];

            // Calculate opening balance for this account (before start date)
            $balance_query = "
                SELECT COALESCE(SUM(jel.debit_amount - jel.credit_amount), 0) as opening_balance
                FROM journal_entry_lines jel
                JOIN journal_entries je ON jel.journal_entry_id = je.id
                WHERE jel.account_id = ?
                AND je.transaction_date < ?
                AND je.status = 'posted'
            ";
            $balance_stmt = $pdo->prepare($balance_query);
            $balance_stmt->execute([$account_id, $start_date]);
            $grouped_entries[$account_id]['opening_balance'] = $balance_stmt->fetch()['opening_balance'];
        }

        $grouped_entries[$account_id]['entries'][] = $entry;
    }

    // Calculate running balances for each account
    foreach ($grouped_entries as $account_id => &$account_data) {
        $running_balance = $account_data['opening_balance'];

        foreach ($account_data['entries'] as &$entry) {
            $amount = $entry['debit_amount'] - $entry['credit_amount'];
            $running_balance += $amount;
            $entry['running_balance'] = $running_balance;
        }

        $account_data['closing_balance'] = $running_balance;
    }

} catch (PDOException $e) {
    $error = "Error fetching general ledger data: " . $e->getMessage();
    $grouped_entries = [];
}

include 'includes/accounting_header.php';
?>

<div class="general-ledger-container">
    <div class="page-header">
        <h1>📚 General Ledger</h1>
        <p>Complete transaction history for all accounts</p>
    </div>

    <?php if ($error): ?>
        <div class="error-message"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <!-- Filters -->
    <div class="filters-section">
        <form method="get" class="filters-form">
            <div class="filter-row">
                <div class="filter-group">
                    <label for="account_id">Account:</label>
                    <select name="account_id" id="account_id">
                        <option value="">All Accounts</option>
                        <?php foreach ($accounts as $account): ?>
                            <option value="<?php echo $account['id']; ?>" <?php echo $account_id == $account['id'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($account['account_code'] . ' - ' . $account['account_name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="filter-group">
                    <label for="account_type">Account Type:</label>
                    <select name="account_type" id="account_type">
                        <option value="all" <?php echo $account_type === 'all' ? 'selected' : ''; ?>>All Types</option>
                        <option value="asset" <?php echo $account_type === 'asset' ? 'selected' : ''; ?>>Assets</option>
                        <option value="liability" <?php echo $account_type === 'liability' ? 'selected' : ''; ?>>Liabilities</option>
                        <option value="equity" <?php echo $account_type === 'equity' ? 'selected' : ''; ?>>Equity</option>
                        <option value="revenue" <?php echo $account_type === 'revenue' ? 'selected' : ''; ?>>Revenue</option>
                        <option value="expense" <?php echo $account_type === 'expense' ? 'selected' : ''; ?>>Expenses</option>
                    </select>
                </div>

                <div class="filter-group">
                    <label for="start_date">Start Date:</label>
                    <input type="date" name="start_date" id="start_date" value="<?php echo $start_date; ?>">
                </div>

                <div class="filter-group">
                    <label for="end_date">End Date:</label>
                    <input type="date" name="end_date" id="end_date" value="<?php echo $end_date; ?>">
                </div>

                <div class="filter-group">
                    <button type="submit" class="btn btn-primary">Filter</button>
                    <a href="general_ledger.php" class="btn btn-secondary">Reset</a>
                </div>
            </div>
        </form>
    </div>

    <!-- General Ledger Tables -->
    <?php if (empty($grouped_entries)): ?>
        <div class="no-entries">
            <p>No transactions found for the selected criteria.</p>
        </div>
    <?php else: ?>
        <div class="ledger-tables-container">
            <?php foreach ($grouped_entries as $account_id => $account_data): ?>
                <div class="account-ledger-section">
                    <div class="account-header">
                        <h3><?php echo htmlspecialchars($account_data['account_code'] . ' - ' . $account_data['account_name']); ?></h3>
                        <div class="account-info">
                            <span class="account-type"><?php echo ucfirst($account_data['account_type']); ?></span>
                            <span class="opening-balance">Opening Balance: GHC <?php echo number_format($account_data['opening_balance'], 2); ?></span>
                            <span class="closing-balance">Closing Balance: GHC <?php echo number_format($account_data['closing_balance'], 2); ?></span>
                        </div>
                    </div>

                    <div class="table-responsive">
                        <table class="ledger-table">
                            <thead>
                                <tr>
                                    <th>Date</th>
                                    <th>Reference</th>
                                    <th>Description</th>
                                    <th>Source</th>
                                    <th>Debit (GHC)</th>
                                    <th>Credit (GHC)</th>
                                    <th>Balance (GHC)</th>
                                </tr>
                            </thead>
                            <tbody>
                                <!-- Opening Balance Row -->
                                <tr class="opening-balance-row">
                                    <td><?php echo date('M d, Y', strtotime($start_date)); ?></td>
                                    <td>-</td>
                                    <td><strong>Opening Balance</strong></td>
                                    <td>-</td>
                                    <td>-</td>
                                    <td>-</td>
                                    <td class="balance-cell"><strong><?php echo number_format($account_data['opening_balance'], 2); ?></strong></td>
                                </tr>

                                <?php foreach ($account_data['entries'] as $entry): ?>
                                    <tr>
                                        <td><?php echo date('M d, Y', strtotime($entry['transaction_date'])); ?></td>
                                        <td><?php echo htmlspecialchars($entry['reference_number'] ?? $entry['journal_entry_id']); ?></td>
                                        <td><?php echo htmlspecialchars($entry['description']); ?></td>
                                        <td><?php echo htmlspecialchars($entry['source_type']); ?></td>
                                        <td class="debit-cell"><?php echo $entry['debit_amount'] > 0 ? number_format($entry['debit_amount'], 2) : '-'; ?></td>
                                        <td class="credit-cell"><?php echo $entry['credit_amount'] > 0 ? number_format($entry['credit_amount'], 2) : '-'; ?></td>
                                        <td class="balance-cell"><?php echo number_format($entry['running_balance'], 2); ?></td>
                                    </tr>
                                <?php endforeach; ?>

                                <!-- Closing Balance Row -->
                                <tr class="closing-balance-row">
                                    <td><?php echo date('M d, Y', strtotime($end_date)); ?></td>
                                    <td>-</td>
                                    <td><strong>Closing Balance</strong></td>
                                    <td>-</td>
                                    <td>-</td>
                                    <td>-</td>
                                    <td class="balance-cell"><strong><?php echo number_format($account_data['closing_balance'], 2); ?></strong></td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>

        <!-- Export Actions -->
        <div class="export-section">
            <h4>Export General Ledger</h4>
            <div class="export-buttons">
                <a href="export_general_ledger.php?<?php echo http_build_query($_GET); ?>" class="btn btn-secondary">Export CSV</a>
                <a href="export_general_ledger.php?<?php echo http_build_query(array_merge($_GET, ['format' => 'excel'])); ?>" class="btn btn-success">Export Excel</a>
                <a href="export_general_ledger.php?<?php echo http_build_query(array_merge($_GET, ['format' => 'pdf'])); ?>" class="btn btn-warning">Export PDF</a>
            </div>
        </div>
    <?php endif; ?>
</div>

<style>
.general-ledger-container {
    padding: 20px;
    background: #f8f9fa;
    min-height: 100vh;
}

.page-header {
    background: white;
    padding: 30px;
    border-radius: 10px;
    margin-bottom: 30px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    text-align: center;
}

.page-header h1 {
    color: #2c3e50;
    margin: 0 0 10px 0;
}

.page-header p {
    color: #7f8c8d;
    margin: 0;
}

.filters-section {
    background: white;
    padding: 25px;
    border-radius: 10px;
    margin-bottom: 30px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.filters-form {
    display: flex;
    flex-direction: column;
    gap: 20px;
}

.filter-row {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 20px;
    align-items: end;
}

.filter-group {
    display: flex;
    flex-direction: column;
}

.filter-group label {
    font-weight: 600;
    margin-bottom: 5px;
    color: #495057;
}

.filter-group select,
.filter-group input {
    padding: 10px 12px;
    border: 1px solid #ced4da;
    border-radius: 4px;
    font-size: 14px;
}

.btn {
    padding: 10px 16px;
    border: none;
    border-radius: 4px;
    cursor: pointer;
    font-size: 14px;
    text-decoration: none;
    display: inline-block;
    transition: background-color 0.2s;
}

.btn-primary {
    background: #007bff;
    color: white;
}

.btn-primary:hover {
    background: #0056b3;
}

.btn-secondary {
    background: #6c757d;
    color: white;
}

.btn-secondary:hover {
    background: #545b62;
}

.btn-success {
    background: #28a745;
    color: white;
}

.btn-success:hover {
    background: #1e7e34;
}

.btn-warning {
    background: #ffc107;
    color: #212529;
}

.btn-warning:hover {
    background: #e0a800;
}

.ledger-tables-container {
    display: flex;
    flex-direction: column;
    gap: 30px;
}

.account-ledger-section {
    background: white;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    overflow: hidden;
}

.account-header {
    padding: 20px 25px;
    border-bottom: 1px solid #e9ecef;
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap;
    gap: 10px;
}

.account-header h3 {
    color: #2c3e50;
    margin: 0;
    font-size: 1.2em;
}

.account-info {
    display: flex;
    gap: 20px;
    flex-wrap: wrap;
}

.account-type {
    background: #e9ecef;
    padding: 4px 8px;
    border-radius: 4px;
    font-size: 0.9em;
    font-weight: 600;
    text-transform: uppercase;
}

.opening-balance,
.closing-balance {
    font-weight: 600;
    color: #495057;
}

.table-responsive {
    overflow-x: auto;
}

.ledger-table {
    width: 100%;
    border-collapse: collapse;
}

.ledger-table th,
.ledger-table td {
    padding: 12px 15px;
    text-align: left;
    border-bottom: 1px solid #e9ecef;
}

.ledger-table th {
    background: #f8f9fa;
    font-weight: 600;
    color: #495057;
    position: sticky;
    top: 0;
    z-index: 10;
}

.ledger-table tr:hover {
    background: #f8f9fa;
}

.debit-cell {
    color: #27ae60;
    font-weight: 600;
}

.credit-cell {
    color: #e74c3c;
    font-weight: 600;
}

.balance-cell {
    font-weight: 600;
    color: #2c3e50;
}

.opening-balance-row,
.closing-balance-row {
    background: #e8f5e8;
    font-weight: 600;
}

.opening-balance-row td,
.closing-balance-row td {
    border-top: 2px solid #27ae60;
}

.no-entries {
    background: white;
    padding: 40px;
    border-radius: 10px;
    text-align: center;
    color: #7f8c8d;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.export-section {
    background: white;
    padding: 25px;
    border-radius: 10px;
    margin-top: 30px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    text-align: center;
}

.export-section h4 {
    color: #2c3e50;
    margin: 0 0 20px 0;
}

.export-buttons {
    display: flex;
    gap: 15px;
    justify-content: center;
    flex-wrap: wrap;
}

.error-message {
    background: #f8d7da;
    color: #721c24;
    padding: 15px;
    border-radius: 5px;
    margin-bottom: 20px;
    border: 1px solid #f5c6cb;
}

@media (max-width: 768px) {
    .filter-row {
        grid-template-columns: 1fr;
    }

    .account-header {
        flex-direction: column;
        text-align: center;
    }

    .account-info {
        justify-content: center;
    }

    .ledger-table th,
    .ledger-table td {
        padding: 8px 10px;
        font-size: 14px;
    }

    .export-buttons {
        flex-direction: column;
        align-items: center;
    }
}

@media print {
    .filters-section,
    .export-section,
    .btn {
        display: none !important;
    }

    .general-ledger-container {
        padding: 0;
        background: white;
    }

    .page-header {
        margin-bottom: 20px;
    }

    .account-ledger-section {
        margin-bottom: 20px;
        box-shadow: none;
        border: 1px solid #ddd;
    }
}
</style>

<?php include '../includes/admin_footer.php'; ?>
