<?php
require_once '../config.php';

// Financial dashboard is now accessible to all logged-in users regardless of role or subscription

// Get date filters
$start_date = $_GET['start_date'] ?? date('Y-m-01'); // First day of current month
$end_date = $_GET['end_date'] ?? date('Y-m-t'); // Last day of current month
$report_type = $_GET['report_type'] ?? 'monthly'; // monthly, yearly, custom

// Calculate date range based on report type
if ($report_type === 'yearly') {
    $start_date = date('Y-01-01');
    $end_date = date('Y-12-31');
} elseif ($report_type === 'monthly') {
    $start_date = date('Y-m-01');
    $end_date = date('Y-m-t');
}

// Financial Overview Statistics
try {
    // Total Revenue
    $stmt = $pdo->prepare("
        SELECT COALESCE(SUM(paid_amount), 0) as total_revenue
        FROM student_fees
        WHERE paid_at BETWEEN ? AND ?
    ");
    $stmt->execute([$start_date, $end_date]);
    $total_revenue = $stmt->fetch()['total_revenue'];

    // Total Outstanding
    $stmt = $pdo->prepare("
        SELECT COALESCE(SUM(f.amount - COALESCE(sf.paid_amount, 0)), 0) as total_outstanding
        FROM fees f
        JOIN student_fees sf ON f.id = sf.fee_id
        WHERE sf.due_date <= ?
    ");
    $stmt->execute([$end_date]);
    $total_outstanding = $stmt->fetch()['total_outstanding'];

    // Collection Rate
    $stmt = $pdo->prepare("
        SELECT
            COUNT(CASE WHEN paid = 1 THEN 1 END) as paid_fees,
            COUNT(*) as total_fees
        FROM student_fees
        WHERE due_date BETWEEN ? AND ?
    ");
    $stmt->execute([$start_date, $end_date]);
    $collection_data = $stmt->fetch();
    $collection_rate = $collection_data['total_fees'] > 0 ?
        round(($collection_data['paid_fees'] / $collection_data['total_fees']) * 100, 2) : 0;

    // Monthly Revenue Trend (Last 6 months)
    $monthly_revenue = [];
    for ($i = 5; $i >= 0; $i--) {
        $month_start = date('Y-m-01', strtotime("-{$i} months"));
        $month_end = date('Y-m-t', strtotime("-{$i} months"));
        $month_name = date('M Y', strtotime("-{$i} months"));

        $stmt = $pdo->prepare("
            SELECT COALESCE(SUM(paid_amount), 0) as revenue
            FROM student_fees
            WHERE paid_at BETWEEN ? AND ?
        ");
        $stmt->execute([$month_start, $month_end]);
        $monthly_revenue[] = [
            'month' => $month_name,
            'revenue' => $stmt->fetch()['revenue']
        ];
    }

    // Fee Type Breakdown
    $stmt = $pdo->prepare("
        SELECT
            f.name as fee_name,
            COUNT(sf.id) as assigned_count,
            COUNT(CASE WHEN sf.paid = 1 THEN 1 END) as paid_count,
            COALESCE(SUM(f.amount), 0) as total_amount,
            COALESCE(SUM(sf.paid_amount), 0) as collected_amount
        FROM fees f
        LEFT JOIN student_fees sf ON f.id = sf.fee_id
        GROUP BY f.id, f.name
        ORDER BY total_amount DESC
    ");
    $stmt->execute();
    $fee_breakdown = $stmt->fetchAll();

    // Payment Method Statistics
    $stmt = $pdo->prepare("
        SELECT
            payment_method,
            COUNT(*) as count,
            COALESCE(SUM(paid_amount), 0) as total_amount
        FROM student_fees
        WHERE paid_at BETWEEN ? AND ? AND payment_method IS NOT NULL
        GROUP BY payment_method
        ORDER BY total_amount DESC
    ");
    $stmt->execute([$start_date, $end_date]);
    $payment_methods = $stmt->fetchAll();

    // Top 10 Outstanding Students
    $stmt = $pdo->prepare("
        SELECT
            s.name as student_name,
            s.class,
            COALESCE(SUM(f.amount - sf.paid_amount), 0) as outstanding_amount
        FROM students s
        JOIN student_fees sf ON s.id = sf.student_id
        JOIN fees f ON sf.fee_id = f.id
        WHERE sf.paid = 0 OR sf.paid_amount < f.amount
        GROUP BY s.id, s.name, s.class
        ORDER BY outstanding_amount DESC
        LIMIT 10
    ");
    $stmt->execute();
    $top_outstanding = $stmt->fetchAll();

} catch (PDOException $e) {
    $error_message = "Error fetching financial data: " . $e->getMessage();
}

include 'includes/accounting_header.php';
?>

<div class="financial-dashboard">
    <div class="dashboard-header">
        <h1>📊 Financial Dashboard</h1>
        <div class="dashboard-controls">
            <form method="get" class="date-filter-form">
                <select name="report_type" onchange="this.form.submit()">
                    <option value="monthly" <?php echo $report_type === 'monthly' ? 'selected' : ''; ?>>Monthly Report</option>
                    <option value="yearly" <?php echo $report_type === 'yearly' ? 'selected' : ''; ?>>Yearly Report</option>
                    <option value="custom" <?php echo $report_type === 'custom' ? 'selected' : ''; ?>>Custom Range</option>
                </select>

                <?php if ($report_type === 'custom'): ?>
                    <input type="date" name="start_date" value="<?php echo $start_date; ?>">
                    <input type="date" name="end_date" value="<?php echo $end_date; ?>">
                    <button type="submit" class="btn btn-primary">Filter</button>
                <?php endif; ?>
            </form>
        </div>
    </div>

    <?php if (isset($error_message)): ?>
        <div class="error-message"><?php echo htmlspecialchars($error_message); ?></div>
    <?php endif; ?>

    <!-- Financial Overview Cards -->
    <div class="overview-cards">
        <div class="card revenue-card">
            <div class="card-icon">💰</div>
            <div class="card-content">
                <h3>GHC <?php echo number_format($total_revenue, 2); ?></h3>
                <p>Total Revenue</p>
                <small>Period: <?php echo date('M d, Y', strtotime($start_date)); ?> - <?php echo date('M d, Y', strtotime($end_date)); ?></small>
            </div>
        </div>

        <div class="card outstanding-card">
            <div class="card-icon">⚠️</div>
            <div class="card-content">
                <h3>GHC <?php echo number_format($total_outstanding, 2); ?></h3>
                <p>Outstanding Amount</p>
                <small>Total unpaid fees</small>
            </div>
        </div>

        <div class="card collection-card">
            <div class="card-icon">📈</div>
            <div class="card-content">
                <h3><?php echo $collection_rate; ?>%</h3>
                <p>Collection Rate</p>
                <small>Fees collected vs assigned</small>
            </div>
        </div>

        <div class="card fees-card">
            <div class="card-icon">📋</div>
            <div class="card-content">
                <h3><?php echo count($fee_breakdown); ?></h3>
                <p>Active Fee Types</p>
                <small>Different fee categories</small>
            </div>
        </div>
    </div>

    <!-- Charts Section -->
    <div class="charts-section">
        <div class="chart-container">
            <h3>Revenue Trend (Last 6 Months)</h3>
            <canvas id="revenueChart" width="400" height="200"></canvas>
        </div>

        <div class="chart-container">
            <h3>Fee Collection Status</h3>
            <canvas id="collectionChart" width="400" height="200"></canvas>
        </div>
    </div>

    <!-- Detailed Tables -->
    <div class="details-section">
        <div class="table-container">
            <h3>Fee Type Performance</h3>
            <table class="fee-performance-table">
                <thead>
                    <tr>
                        <th>Fee Type</th>
                        <th>Assigned</th>
                        <th>Collected</th>
                        <th>Collection Rate</th>
                        <th>Total Amount</th>
                        <th>Collected Amount</th>
                        <th>Outstanding</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($fee_breakdown as $fee): ?>
                        <?php
                            $assigned_count = $fee['assigned_count'];
                            $paid_count = $fee['paid_count'];
                            $fee_collection_rate = $assigned_count > 0 ? round(($paid_count / $assigned_count) * 100, 2) : 0;
                            $outstanding = $fee['total_amount'] - $fee['collected_amount'];
                        ?>
                        <tr>
                            <td><?php echo htmlspecialchars($fee['fee_name']); ?></td>
                            <td><?php echo $assigned_count; ?></td>
                            <td><?php echo $paid_count; ?></td>
                            <td>
                                <span class="rate-badge <?php echo $fee_collection_rate >= 80 ? 'high' : ($fee_collection_rate >= 50 ? 'medium' : 'low'); ?>">
                                    <?php echo $fee_collection_rate; ?>%
                                </span>
                            </td>
                            <td>GHC <?php echo number_format($fee['total_amount'], 2); ?></td>
                            <td>GHC <?php echo number_format($fee['collected_amount'], 2); ?></td>
                            <td>GHC <?php echo number_format($outstanding, 2); ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>

        <div class="table-container">
            <h3>Payment Methods</h3>
            <table class="payment-methods-table">
                <thead>
                    <tr>
                        <th>Payment Method</th>
                        <th>Transactions</th>
                        <th>Total Amount</th>
                        <th>Percentage</th>
                    </tr>
                </thead>
                <tbody>
                    <?php
                        $total_payment_amount = array_sum(array_column($payment_methods, 'total_amount'));
                        foreach ($payment_methods as $method):
                            $percentage = $total_payment_amount > 0 ? round(($method['total_amount'] / $total_payment_amount) * 100, 2) : 0;
                    ?>
                        <tr>
                            <td><?php echo htmlspecialchars($method['payment_method']); ?></td>
                            <td><?php echo $method['count']; ?></td>
                            <td>GHC <?php echo number_format($method['total_amount'], 2); ?></td>
                            <td><?php echo $percentage; ?>%</td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>

        <div class="table-container">
            <h3>Top Outstanding Students</h3>
            <table class="outstanding-students-table">
                <thead>
                    <tr>
                        <th>Student Name</th>
                        <th>Class</th>
                        <th>Outstanding Amount</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($top_outstanding as $student): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($student['student_name']); ?></td>
                            <td><?php echo htmlspecialchars($student['class']); ?></td>
                            <td>GHC <?php echo number_format($student['outstanding_amount'], 2); ?></td>
                            <td>
                                <a href="record_payments.php?search=<?php echo urlencode($student['student_name']); ?>" class="btn btn-small">
                                    Record Payment
                                </a>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Export Options -->
    <div class="export-section">
        <h3>Export Reports</h3>
        <div class="export-buttons">
            <a href="export_financial_report.php?type=pdf&start_date=<?php echo $start_date; ?>&end_date=<?php echo $end_date; ?>" class="btn btn-secondary">
                Export as PDF
            </a>
            <a href="export_financial_report.php?type=excel&start_date=<?php echo $start_date; ?>&end_date=<?php echo $end_date; ?>" class="btn btn-secondary">
                Export as Excel
            </a>
            <a href="export_financial_report.php?type=csv&start_date=<?php echo $start_date; ?>&end_date=<?php echo $end_date; ?>" class="btn btn-secondary">
                Export as CSV
            </a>
        </div>
    </div>
</div>

<style>
.financial-dashboard {
    padding: 20px;
    background: #f8f9fa;
    min-height: 100vh;
}

.dashboard-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 30px;
    background: rgba(255, 255, 255, 0.95);
    padding: 20px;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.dashboard-header h1 {
    color: #2c3e50;
    margin: 0;
}

.date-filter-form {
    display: flex;
    gap: 15px;
    align-items: center;
}

.date-filter-form select {
    padding: 10px 15px;
    border: 2px solid #2E7D32;
    border-radius: 5px;
    background: #fff;
    color: #2c3e50;
    font-size: 0.9em;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.3s ease;
    min-width: 150px;
}

.date-filter-form select:focus {
    outline: none;
    border-color: #4CAF50;
    box-shadow: 0 0 0 3px rgba(76, 175, 80, 0.1);
}

.date-filter-form select:hover {
    border-color: #4CAF50;
}

.date-filter-form input[type="date"] {
    padding: 8px 12px;
    border: 2px solid #2E7D32;
    border-radius: 5px;
    background: #fff;
    color: #2c3e50;
    font-size: 0.9em;
    cursor: pointer;
    transition: all 0.3s ease;
}

.date-filter-form input[type="date"]:focus {
    outline: none;
    border-color: #4CAF50;
    box-shadow: 0 0 0 3px rgba(76, 175, 80, 0.1);
}

.overview-cards {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
    margin-bottom: 30px;
}

.card {
    background: linear-gradient(135deg, #ffffff 0%, #f8f9fa 100%);
    padding: 25px;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    text-align: center;
    transition: transform 0.2s;
    position: relative;
    overflow: hidden;
}

.card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 4px;
    background: linear-gradient(90deg, #2E7D32 0%, #4CAF50 100%);
    border-radius: 10px 10px 0 0;
}

.card:hover {
    transform: translateY(-5px);
}

.card-icon {
    font-size: 2.5em;
    margin-bottom: 15px;
}

.card h3 {
    color: #2c3e50;
    font-size: 1.8em;
    margin-bottom: 5px;
}

.card p {
    color: #7f8c8d;
    font-size: 1.1em;
    margin-bottom: 10px;
}

.revenue-card .card-icon { color: #27ae60; }
.outstanding-card .card-icon { color: #e74c3c; }
.collection-card .card-icon { color: #3498db; }
.fees-card .card-icon { color: #9b59b6; }

.btn {
    padding: 10px 20px;
    border: none;
    border-radius: 5px;
    cursor: pointer;
    font-size: 0.9em;
    font-weight: 500;
    text-decoration: none;
    display: inline-block;
    transition: all 0.3s ease;
}

.btn-primary {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: #fff;
}

.btn-primary:hover {
    background: linear-gradient(135deg, #4CAF50 0%, #2E7D32 100%);
    transform: translateY(-2px);
    box-shadow: 0 4px 8px rgba(0,0,0,0.2);
}

.btn-secondary {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: #fff;
}

.btn-secondary:hover {
    background: linear-gradient(135deg, #4CAF50 0%, #2E7D32 100%);
    transform: translateY(-2px);
    box-shadow: 0 4px 8px rgba(0,0,0,0.2);
}

.btn-small {
    padding: 6px 12px;
    font-size: 0.8em;
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: #fff;
}

.btn-small:hover {
    background: linear-gradient(135deg, #4CAF50 0%, #2E7D32 100%);
    transform: translateY(-2px);
    box-shadow: 0 4px 8px rgba(0,0,0,0.2);
}

.charts-section {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(400px, 1fr));
    gap: 30px;
    margin-bottom: 30px;
}

.chart-container {
    background: white;
    padding: 20px;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.chart-container h3 {
    margin-bottom: 20px;
    color: #2c3e50;
    text-align: center;
}

.details-section {
    display: grid;
    grid-template-columns: 1fr;
    gap: 30px;
    margin-bottom: 30px;
}

.table-container {
    background: white;
    padding: 20px;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.table-container h3 {
    margin-bottom: 20px;
    color: #2c3e50;
}

.fee-performance-table,
.payment-methods-table,
.outstanding-students-table {
    width: 100%;
    border-collapse: collapse;
}

.fee-performance-table th,
.payment-methods-table th,
.outstanding-students-table th {
    background: #fff;
    color: #000;
    padding: 12px;
    text-align: left;
    border-bottom: 2px solid #dee2e6;
}

.fee-performance-table td,
.payment-methods-table td,
.outstanding-students-table td {
    padding: 12px;
    border-bottom: 1px solid #dee2e6;
}

.rate-badge {
    padding: 4px 8px;
    border-radius: 12px;
    font-size: 0.8em;
    font-weight: bold;
}

.rate-badge.high { background: #d4edda; color: #155724; }
.rate-badge.medium { background: #fff3cd; color: #856404; }
.rate-badge.low { background: #f8d7da; color: #721c24; }

.export-section {
    background: white;
    padding: 20px;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    text-align: center;
}

.export-buttons {
    display: flex;
    gap: 15px;
    justify-content: center;
    margin-top: 15px;
}

.error-message {
    background: #f8d7da;
    color: #721c24;
    padding: 15px;
    border-radius: 5px;
    margin-bottom: 20px;
    border: 1px solid #f5c6cb;
}

@media (max-width: 768px) {
    .dashboard-header {
        flex-direction: column;
        gap: 20px;
    }

    .date-filter-form {
        flex-wrap: wrap;
    }

    .overview-cards {
        grid-template-columns: 1fr;
    }

    .charts-section {
        grid-template-columns: 1fr;
    }

    .details-section {
        grid-template-columns: 1fr;
    }

    .export-buttons {
        flex-direction: column;
    }
}
</style>

<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Revenue Trend Chart
    const revenueCtx = document.getElementById('revenueChart').getContext('2d');
    const revenueData = <?php echo json_encode(array_column($monthly_revenue, 'revenue')); ?>;
    const revenueLabels = <?php echo json_encode(array_column($monthly_revenue, 'month')); ?>;

    new Chart(revenueCtx, {
        type: 'line',
        data: {
            labels: revenueLabels,
            datasets: [{
                label: 'Revenue (GHC)',
                data: revenueData,
                borderColor: '#3498db',
                backgroundColor: 'rgba(52, 152, 219, 0.1)',
                tension: 0.4,
                fill: true
            }]
        },
        options: {
            responsive: true,
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: {
                        callback: function(value) {
                            return 'GHC ' + value.toLocaleString();
                        }
                    }
                }
            }
        }
    });

    // Collection Status Chart
    const collectionCtx = document.getElementById('collectionChart').getContext('2d');
    const paidFees = <?php echo $collection_data['paid_fees']; ?>;
    const unpaidFees = <?php echo $collection_data['total_fees'] - $collection_data['paid_fees']; ?>;

    new Chart(collectionCtx, {
        type: 'doughnut',
        data: {
            labels: ['Collected', 'Outstanding'],
            datasets: [{
                data: [paidFees, unpaidFees],
                backgroundColor: ['#27ae60', '#e74c3c'],
                borderWidth: 0
            }]
        },
        options: {
            responsive: true,
            plugins: {
                legend: {
                    position: 'bottom'
                }
            }
        }
    });
});
</script>

<?php
include 'includes/accounting_footer.php';
?>
