<?php
/**
 * Fee Payment Accounting Module
 * Handles proper cash account mapping for fee payments based on payment method
 */

require_once '../config.php';
require_once '../includes/accountant_activity_logger.php';
require_once 'payment_methods.php';

/**
 * Process fee payment accounting entries
 * Creates proper journal entries and updates cash account balances
 *
 * @param PDO $pdo Database connection
 * @param int $studentFeeId The student fee ID
 * @param float $paymentAmount The payment amount
 * @param string $paymentMethod The payment method
 * @param string $paymentDate The payment date
 * @param int $userId The user ID processing the payment
 * @param array $studentFeeData Student fee data array
 * @return bool Success status
 */
function processFeePaymentAccounting($pdo, $studentFeeId, $paymentAmount, $paymentMethod, $paymentDate, $userId, $studentFeeData) {
    try {
        $pdo->beginTransaction();

        // Get the correct cash account for this payment method
        $cashAccountName = getFeePaymentCashAccount($paymentMethod);

        // Get cash account from cash_accounts table
        $cashAccountStmt = $pdo->prepare("SELECT id FROM cash_accounts WHERE account_name = ? LIMIT 1");
        $cashAccountStmt->execute([$cashAccountName]);
        $cashAccount = $cashAccountStmt->fetch();

        if ($cashAccount) {
            // Record transaction crediting the appropriate cash account
            $transactionStmt = $pdo->prepare("
                INSERT INTO cash_account_transactions
                (account_id, transaction_type, amount, description, created_by)
                VALUES (?, 'increase', ?, ?, ?)
            ");
            $transactionDescription = "Fee payment received: {$studentFeeData['student_name']} - {$studentFeeData['fee_name']} (Receipt: {$studentFeeData['receipt_number']})";
            $transactionStmt->execute([$cashAccount['id'], $paymentAmount, $transactionDescription, $userId]);

            // Update cash account balance
            $updateBalanceStmt = $pdo->prepare("UPDATE cash_accounts SET current_balance = current_balance + ? WHERE id = ?");
            $updateBalanceStmt->execute([$paymentAmount, $cashAccount['id']]);
        } else {
            // Log error if cash account not found
            error_log("ERROR: Cash account '{$cashAccountName}' not found for payment method '{$paymentMethod}'");
            // Continue with journal entry creation even if cash account transaction fails
        }

        // Create journal entry for fee payment
        $journalEntryId = createFeePaymentJournalEntry($pdo, $studentFeeId, $paymentAmount, $paymentMethod, $paymentDate, $userId, $studentFeeData);

        // Record income for budget vs actual report (only for tuition fees)
        recordFeePaymentIncome($pdo, $studentFeeId, $paymentAmount, $paymentDate, $studentFeeData);

        $pdo->commit();

        // Log successful accounting
        AccountantActivityLogger::logActivity(
            $userId,
            'fee_payment_accounting_processed',
            "Fee payment accounting processed: {$paymentAmount} GHS via {$paymentMethod} to {$cashAccountName}",
            $paymentAmount
        );

        return true;

    } catch (Exception $e) {
        $pdo->rollBack();
        error_log("ERROR: Fee payment accounting failed: " . $e->getMessage());

        // Log the error
        AccountantActivityLogger::logActivity(
            $userId,
            'fee_payment_accounting_error',
            "Fee payment accounting error: " . $e->getMessage(),
            $paymentAmount
        );

        return false;
    }
}

/**
 * Create journal entry for fee payment
 *
 * @param PDO $pdo Database connection
 * @param int $studentFeeId Student fee ID
 * @param float $paymentAmount Payment amount
 * @param string $paymentMethod Payment method
 * @param string $paymentDate Payment date
 * @param int $userId User ID
 * @param array $studentFeeData Student fee data
 * @return int Journal entry ID
 */
function createFeePaymentJournalEntry($pdo, $studentFeeId, $paymentAmount, $paymentMethod, $paymentDate, $userId, $studentFeeData) {
    // Get fee type to determine receivable account
    $feeTypeStmt = $pdo->prepare("SELECT f.type FROM fees f WHERE f.id = ?");
    $feeTypeStmt->execute([$studentFeeData['fee_id'] ?? null]);
    $feeType = $feeTypeStmt->fetch();

    // Determine receivable account based on fee type
    $receivableAccountCode = '1123'; // Default for other fees
    if ($feeType) {
        $feeTypeLower = strtolower($feeType['type']);
        if ($feeTypeLower === 'tuition') {
            $receivableAccountCode = '1121'; // Tuition fees receivable
        } elseif ($feeTypeLower === 'registration') {
            $receivableAccountCode = '1122'; // Registration Fees Receivable
        } elseif ($feeTypeLower === 'examination') {
            $receivableAccountCode = '4002'; // Examination Fees
        }
    }
    $receivableStmt = $pdo->prepare("SELECT id FROM chart_of_accounts WHERE account_code = ?");
    $receivableStmt->execute([$receivableAccountCode]);
    $receivableAccount = $receivableStmt->fetch();

    if (!$receivableAccount) {
        throw new Exception('Receivable account not found in chart of accounts for code: ' . $receivableAccountCode);
    }

    // Get cash account from chart_of_accounts
    $cashAccountName = getFeePaymentCashAccount($paymentMethod);
    $cashStmt = $pdo->prepare("SELECT id FROM chart_of_accounts WHERE account_name = ?");
    $cashStmt->execute([$cashAccountName]);
    $cashAccount = $cashStmt->fetch();

    if (!$cashAccount) {
        throw new Exception("Cash account '{$cashAccountName}' not found in chart of accounts");
    }

    // Generate journal entry number
    $entryNumber = generateJournalEntryNumber($pdo, $paymentDate);

    // Create journal entry
    $entryStmt = $pdo->prepare("
        INSERT INTO journal_entries
        (entry_number, transaction_date, description, source_type, source_id, status, created_by)
        VALUES (?, ?, ?, 'fee_payment', ?, 'posted', ?)
    ");
    $entryStmt->execute([
        $entryNumber,
        $paymentDate,
        "Fee payment received: {$studentFeeData['student_name']} - {$studentFeeData['fee_name']} (Receipt: {$studentFeeData['receipt_number']})",
        $studentFeeId,
        $userId
    ]);
    $entryId = $pdo->lastInsertId();

    // Debit: Cash Account (increases asset)
    $debitStmt = $pdo->prepare("
        INSERT INTO journal_entry_lines
        (journal_entry_id, account_id, debit_amount, line_number, description)
        VALUES (?, ?, ?, 1, ?)
    ");
    $debitStmt->execute([
        $entryId,
        $cashAccount['id'],
        $paymentAmount,
        "Cash receipt: Fee payment for {$studentFeeData['student_name']} ({$cashAccountName})"
    ]);

    // Credit: Student Fees Receivable (decreases asset)
    $creditStmt = $pdo->prepare("
        INSERT INTO journal_entry_lines
        (journal_entry_id, account_id, credit_amount, line_number, description)
        VALUES (?, ?, ?, 2, ?)
    ");
    $creditStmt->execute([
        $entryId,
        $receivableAccount['id'],
        $paymentAmount,
        "Payment received for fee: {$studentFeeData['fee_name']}"
    ]);

    // Update chart_of_accounts balances
    $updateCashStmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance + ? WHERE id = ?");
    $updateCashStmt->execute([$paymentAmount, $cashAccount['id']]); // Debit increases asset

    $updateReceivableStmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance - ? WHERE id = ?");
    $updateReceivableStmt->execute([$paymentAmount, $receivableAccount['id']]); // Credit decreases asset

    return $entryId;
}

/**
 * Record income for budget vs actual report (tuition fees only)
 *
 * @param PDO $pdo Database connection
 * @param int $studentFeeId Student fee ID
 * @param float $paymentAmount Payment amount
 * @param string $paymentDate Payment date
 * @param array $studentFeeData Student fee data
 */
function recordFeePaymentIncome($pdo, $studentFeeId, $paymentAmount, $paymentDate, $studentFeeData) {
    // Only record income for tuition fees (type = 'Tuition')
    $feeTypeStmt = $pdo->prepare("SELECT f.type FROM fees f WHERE f.id = ?");
    $feeTypeStmt->execute([$studentFeeData['fee_id'] ?? null]);
    $feeType = $feeTypeStmt->fetch();

    if ($feeType && strtolower($feeType['type']) === 'tuition') {
        // Get tuition fees category ID
        $tuitionCategoryStmt = $pdo->prepare("SELECT id FROM budget_categories WHERE name LIKE '%tuition%' LIMIT 1");
        $tuitionCategoryStmt->execute();
        $tuitionCategory = $tuitionCategoryStmt->fetch();

        if ($tuitionCategory) {
            // Record the payment as income
            $incomeStmt = $pdo->prepare("
                INSERT INTO incomes
                (source, category_id, amount, income_date, description)
                VALUES (?, ?, ?, ?, ?)
            ");
            $incomeDescription = "Fee payment received: {$studentFeeData['student_name']} - {$studentFeeData['fee_name']} (Receipt: {$studentFeeData['receipt_number']})";
            $incomeStmt->execute([
                'fee_payment',
                $tuitionCategory['id'],
                $paymentAmount,
                $paymentDate,
                $incomeDescription
            ]);
        }
    }
}

/**
 * Generate journal entry number for fee payments
 *
 * @param PDO $pdo Database connection
 * @param string $date Transaction date
 * @return string Entry number
 */
function generateJournalEntryNumber($pdo, $date) {
    static $counters = [];

    $year = date('Y', strtotime($date));

    if (!isset($counters[$year])) {
        $yearStmt = $pdo->prepare("SELECT COUNT(*) as count FROM journal_entries WHERE YEAR(transaction_date) = ?");
        $yearStmt->execute([$year]);
        $counters[$year] = $yearStmt->fetch()['count'] + 1;
    } else {
        $counters[$year]++;
    }

    return 'JE-' . $year . '-' . str_pad($counters[$year], 4, '0', STR_PAD_LEFT);
}

/**
 * Validate payment method and cash account availability
 *
 * @param PDO $pdo Database connection
 * @param string $paymentMethod Payment method
 * @return array Validation result with 'valid' boolean and 'message' string
 */
function validateFeePaymentMethod($pdo, $paymentMethod) {
    $cashAccountName = getFeePaymentCashAccount($paymentMethod);

    // Check if cash account exists in cash_accounts table
    $cashAccountStmt = $pdo->prepare("SELECT id FROM cash_accounts WHERE account_name = ? LIMIT 1");
    $cashAccountStmt->execute([$cashAccountName]);
    $cashAccount = $cashAccountStmt->fetch();

    // Check if cash account exists in chart_of_accounts
    $chartAccountStmt = $pdo->prepare("SELECT id FROM chart_of_accounts WHERE account_name = ? AND is_active = TRUE");
    $chartAccountStmt->execute([$cashAccountName]);
    $chartAccount = $chartAccountStmt->fetch();

    if (!$cashAccount) {
        return [
            'valid' => false,
            'message' => "Cash account '{$cashAccountName}' not found in cash_accounts table for payment method '{$paymentMethod}'"
        ];
    }

    if (!$chartAccount) {
        return [
            'valid' => false,
            'message' => "Cash account '{$cashAccountName}' not found in chart_of_accounts for payment method '{$paymentMethod}'"
        ];
    }

    return [
        'valid' => true,
        'message' => "Payment method '{$paymentMethod}' maps to cash account '{$cashAccountName}'",
        'cash_account' => $cashAccountName
    ];
}
?>
