<?php
require_once '../config.php';

$message = '';
$error = '';

// Get filter parameters
$as_of_date = $_GET['as_of_date'] ?? date('Y-m-t');
$account_type = $_GET['account_type'] ?? 'all';
$format = $_GET['format'] ?? 'csv';

// Calculate account balances (same logic as main report)
try {
    $query = "
        SELECT
            coa.id,
            coa.account_code,
            coa.account_name,
            coa.account_type,
            COALESCE(SUM(jel.debit_amount - jel.credit_amount), 0) as balance
        FROM chart_of_accounts coa
        LEFT JOIN journal_entry_lines jel ON coa.id = jel.account_id
        LEFT JOIN journal_entries je ON jel.journal_entry_id = je.id AND je.transaction_date <= ? AND je.status = 'posted'
        WHERE 1=1
    ";

    $params = [$as_of_date];

    if ($account_type !== 'all') {
        $query .= " AND coa.account_type = ?";
        $params[] = $account_type;
    }

    $query .= " GROUP BY coa.id, coa.account_code, coa.account_name, coa.account_type ORDER BY coa.account_code ASC";

    $stmt = $pdo->prepare($query);
    $stmt->execute($params);
    $account_balances = $stmt->fetchAll();

    // Separate debit and credit balances
    $trial_balance_entries = [];
    $total_debits = 0;
    $total_credits = 0;

    foreach ($account_balances as $account) {
        $balance = $account['balance'];

        // Determine if this is a debit or credit balance based on account type
        $is_debit_balance = in_array($account['account_type'], ['asset', 'expense']);

        if ($is_debit_balance) {
            $debit_balance = abs($balance);
            $credit_balance = 0;
        } else {
            $debit_balance = 0;
            $credit_balance = abs($balance);
        }

        // If balance is negative, it means the opposite side
        if ($balance < 0) {
            if ($is_debit_balance) {
                $debit_balance = 0;
                $credit_balance = abs($balance);
            } else {
                $debit_balance = abs($balance);
                $credit_balance = 0;
            }
        }

        $trial_balance_entries[] = [
            'account_code' => $account['account_code'],
            'account_name' => $account['account_name'],
            'account_type' => $account['account_type'],
            'debit_balance' => $debit_balance,
            'credit_balance' => $credit_balance,
            'net_balance' => $balance
        ];

        $total_debits += $debit_balance;
        $total_credits += $credit_balance;
    }

    $is_balanced = abs($total_debits - $total_credits) < 0.01;

} catch (PDOException $e) {
    $error = "Error fetching trial balance data: " . $e->getMessage();
    $trial_balance_entries = [];
    $total_debits = $total_credits = 0;
    $is_balanced = false;
}

// Export based on format
if ($format === 'excel') {
    // Set headers for Excel download
    header('Content-Type: application/vnd.ms-excel');
    header('Content-Disposition: attachment; filename="trial_balance_' . date('Y-m-d') . '.xls"');
    header('Cache-Control: max-age=0');

    echo "<html><body>";
    echo "<h1>Trial Balance Report</h1>";
    echo "<p>As of: " . date('M d, Y', strtotime($as_of_date)) . "</p>";
    echo "<p>Status: " . ($is_balanced ? 'Balanced' : 'Unbalanced') . "</p>";
    echo "<table border='1'>";

    // Table headers
    echo "<tr>";
    echo "<th>Account Code</th>";
    echo "<th>Account Name</th>";
    echo "<th>Account Type</th>";
    echo "<th>Debit Balance (GHC)</th>";
    echo "<th>Credit Balance (GHC)</th>";
    echo "</tr>";

    // Data rows
    foreach ($trial_balance_entries as $entry) {
        echo "<tr>";
        echo "<td>" . htmlspecialchars($entry['account_code']) . "</td>";
        echo "<td>" . htmlspecialchars($entry['account_name']) . "</td>";
        echo "<td>" . ucfirst($entry['account_type']) . "</td>";
        echo "<td>" . ($entry['debit_balance'] > 0 ? number_format($entry['debit_balance'], 2) : '-') . "</td>";
        echo "<td>" . ($entry['credit_balance'] > 0 ? number_format($entry['credit_balance'], 2) : '-') . "</td>";
        echo "</tr>";
    }

    // Total row
    echo "<tr>";
    echo "<td colspan='3'><strong>TOTAL</strong></td>";
    echo "<td><strong>" . number_format($total_debits, 2) . "</strong></td>";
    echo "<td><strong>" . number_format($total_credits, 2) . "</strong></td>";
    echo "</tr>";

    echo "</table></body></html>";
    exit;

} elseif ($format === 'pdf') {
    // For PDF, we'll use a simple HTML-based approach that can be printed to PDF
    header('Content-Type: application/pdf');
    header('Content-Disposition: attachment; filename="trial_balance_' . date('Y-m-d') . '.pdf"');

    echo "<html><head><title>Trial Balance Report</title></head><body>";
    echo "<h1>Trial Balance Report</h1>";
    echo "<p>As of: " . date('M d, Y', strtotime($as_of_date)) . "</p>";
    echo "<p>Status: " . ($is_balanced ? 'Balanced' : 'Unbalanced') . "</p>";
    echo "<style>table { border-collapse: collapse; width: 100%; } th, td { border: 1px solid #ddd; padding: 8px; text-align: left; } th { background-color: #f2f2f2; }</style>";

    echo "<table>";
    echo "<tr><th>Account Code</th><th>Account Name</th><th>Account Type</th><th>Debit Balance (GHC)</th><th>Credit Balance (GHC)</th></tr>";

    foreach ($trial_balance_entries as $entry) {
        echo "<tr>";
        echo "<td>" . htmlspecialchars($entry['account_code']) . "</td>";
        echo "<td>" . htmlspecialchars($entry['account_name']) . "</td>";
        echo "<td>" . ucfirst($entry['account_type']) . "</td>";
        echo "<td>" . ($entry['debit_balance'] > 0 ? number_format($entry['debit_balance'], 2) : '-') . "</td>";
        echo "<td>" . ($entry['credit_balance'] > 0 ? number_format($entry['credit_balance'], 2) : '-') . "</td>";
        echo "</tr>";
    }

    // Total row
    echo "<tr><td colspan='3'><strong>TOTAL</strong></td><td><strong>" . number_format($total_debits, 2) . "</strong></td><td><strong>" . number_format($total_credits, 2) . "</strong></td></tr>";
    echo "</table>";

    echo "</body></html>";
    exit;

} else {
    // Default CSV export
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="trial_balance_' . date('Y-m-d') . '.csv"');

    $output = fopen('php://output', 'w');

    // CSV headers
    fputcsv($output, ['Account Code', 'Account Name', 'Account Type', 'Debit Balance (GHC)', 'Credit Balance (GHC)']);

    // Data rows
    foreach ($trial_balance_entries as $entry) {
        fputcsv($output, [
            $entry['account_code'],
            $entry['account_name'],
            ucfirst($entry['account_type']),
            $entry['debit_balance'] > 0 ? number_format($entry['debit_balance'], 2) : '',
            $entry['credit_balance'] > 0 ? number_format($entry['credit_balance'], 2) : ''
        ]);
    }

    // Total row
    fputcsv($output, ['TOTAL', '', '', number_format($total_debits, 2), number_format($total_credits, 2)]);

    fclose($output);
    exit;
}
?>
