<?php
require_once '../config.php';
require_once 'accounting_functions.php';
require_once 'create_asset_acquisition_function.php';

echo "Backfilling Asset Acquisition Journal Entries\n";
echo "=============================================\n\n";

try {
    // Get all assets that don't have journal entries yet
    $stmt = $pdo->query("
        SELECT a.id, a.name, a.purchase_cost, a.purchase_date
        FROM assets a
        LEFT JOIN journal_entries je ON je.source_type = 'asset' AND je.source_id = a.id
        WHERE je.id IS NULL
        ORDER BY a.purchase_date
    ");
    $assets = $stmt->fetchAll(PDO::FETCH_ASSOC);

    if (empty($assets)) {
        echo "No assets found that need journal entries.\n";
        exit;
    }

    echo "Found " . count($assets) . " assets without journal entries:\n\n";

    $successCount = 0;
    $errorCount = 0;

    foreach ($assets as $asset) {
        echo "Processing: {$asset['name']} (ID: {$asset['id']})\n";
        echo "Purchase Cost: " . number_format($asset['purchase_cost'], 2) . " GHC\n";
        echo "Purchase Date: {$asset['purchase_date']}\n";

        try {
            // Create journal entry for this asset
            $entryId = createAssetAcquisitionEntry(
                $pdo,
                $asset['id'],
                $asset['purchase_cost'],
                'cash', // Default to cash payment
                $asset['purchase_date'],
                1 // Default user ID
            );

            echo "✅ Journal entry created (ID: $entryId)\n\n";
            $successCount++;

        } catch (Exception $e) {
            echo "❌ Error: " . $e->getMessage() . "\n\n";
            $errorCount++;
        }
    }

    echo "Summary:\n";
    echo "--------\n";
    echo "Successful: $successCount\n";
    echo "Errors: $errorCount\n\n";

    // Check final PPE account balance
    $stmt = $pdo->prepare("SELECT current_balance FROM chart_of_accounts WHERE account_code = '1210'");
    $stmt->execute();
    $balance = $stmt->fetch()['current_balance'];

    echo "Property, Plant & Equipment Account Final Balance: " . number_format($balance, 2) . " GHC\n";

} catch (Exception $e) {
    echo "Fatal error: " . $e->getMessage() . "\n";
}
?>
