<?php
require_once '../config.php';
require_once 'audit_functions.php';

// requireFeatureAccess('accounts_management');

// Get filter parameters
$start_date = $_GET['start_date'] ?? date('Y-m-d', strtotime('-30 days'));
$end_date = $_GET['end_date'] ?? date('Y-m-d');
$user_filter = $_GET['user'] ?? '';
$action_filter = $_GET['action'] ?? '';
$module_filter = $_GET['module'] ?? '';
$page = intval($_GET['page'] ?? 1);
$per_page = 50;

// Build query conditions
$conditions = [];
$params = [];

if ($start_date) {
    $conditions[] = "al.timestamp >= ?";
    $params[] = $start_date . ' 00:00:00';
}
if ($end_date) {
    $conditions[] = "al.timestamp <= ?";
    $params[] = $end_date . ' 23:59:59';
}
if ($user_filter) {
    $conditions[] = "al.user_name LIKE ?";
    $params[] = '%' . $user_filter . '%';
}
if ($action_filter) {
    $conditions[] = "al.action = ?";
    $params[] = $action_filter;
}
if ($module_filter) {
    $conditions[] = "al.module = ?";
    $params[] = $module_filter;
}

$where_clause = !empty($conditions) ? 'WHERE ' . implode(' AND ', $conditions) : '';

// Get total count for pagination
try {
    $count_stmt = $pdo->prepare("SELECT COUNT(*) as total FROM audit_log al $where_clause");
    $count_stmt->execute($params);
    $total_records = $count_stmt->fetch()['total'];
    $total_pages = ceil($total_records / $per_page);
    $offset = ($page - 1) * $per_page;

    // Get audit trail data with pagination
    $stmt = $pdo->prepare("
        SELECT
            al.*,
            CASE
                WHEN al.action = 'INSERT' THEN 'Created'
                WHEN al.action = 'UPDATE' THEN 'Modified'
                WHEN al.action = 'DELETE' THEN 'Deleted'
                ELSE al.action
            END as action_display,
            CASE
                WHEN al.action = 'INSERT' THEN 'success'
                WHEN al.action = 'UPDATE' THEN 'warning'
                WHEN al.action = 'DELETE' THEN 'danger'
                ELSE 'info'
            END as action_color
        FROM audit_log al
        $where_clause
        ORDER BY al.timestamp DESC
        LIMIT ? OFFSET ?
    ");
    $stmt->execute(array_merge($params, [$per_page, $offset]));
    $audit_events = $stmt->fetchAll();

    // Get audit statistics
    $stats = getAuditStats(30);

} catch (PDOException $e) {
    $error = "Error loading audit data: " . $e->getMessage();
    $audit_events = [];
    $stats = null;
    $total_records = 0;
    $total_pages = 0;
}
?>

<?php include 'includes/accounting_header.php'; ?>

<style>
    .audit-dashboard {
        padding: 20px;
    }

    .dashboard-header {
        background: white;
        padding: 30px;
        border-radius: 10px;
        margin-bottom: 30px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        text-align: center;
    }

    .dashboard-header h1 {
        color: #2c3e50;
        margin: 0 0 10px 0;
        font-size: 2.5em;
    }

    .dashboard-header p {
        color: #7f8c8d;
        margin: 0;
        font-size: 1.1em;
    }

    .filters-section {
        background: white;
        padding: 25px;
        border-radius: 10px;
        margin-bottom: 30px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    }

    .filters-form {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
        gap: 20px;
        align-items: end;
    }

    .filter-group {
        display: flex;
        flex-direction: column;
    }

    .filter-group label {
        font-weight: 600;
        color: #2c3e50;
        margin-bottom: 8px;
    }

    .filter-group input,
    .filter-group select {
        padding: 10px;
        border: 1px solid #ddd;
        border-radius: 5px;
        font-size: 14px;
    }

    .filter-group input:focus,
    .filter-group select:focus {
        outline: none;
        border-color: #3498db;
        box-shadow: 0 0 5px rgba(52, 152, 219, 0.3);
    }

    .filter-btn {
        padding: 10px 20px;
        background: #3498db;
        color: white;
        border: none;
        border-radius: 5px;
        cursor: pointer;
        font-weight: 600;
        transition: background 0.2s;
    }

    .filter-btn:hover {
        background: #2980b9;
    }

    .stats-cards {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
        gap: 20px;
        margin-bottom: 30px;
    }

    .stat-card {
        background: white;
        padding: 20px;
        border-radius: 10px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        text-align: center;
    }

    .stat-value {
        font-size: 2em;
        font-weight: 600;
        color: #2c3e50;
        margin: 10px 0;
    }

    .stat-label {
        color: #7f8c8d;
        font-size: 0.9em;
    }

    .timeline-container {
        background: white;
        padding: 25px;
        border-radius: 10px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    }

    .timeline {
        position: relative;
        padding-left: 30px;
    }

    .timeline::before {
        content: '';
        position: absolute;
        left: 15px;
        top: 0;
        bottom: 0;
        width: 2px;
        background: #e9ecef;
    }

    .timeline-item {
        position: relative;
        margin-bottom: 30px;
        padding-left: 40px;
    }

    .timeline-item::before {
        content: '';
        position: absolute;
        left: -25px;
        top: 15px;
        width: 12px;
        height: 12px;
        border-radius: 50%;
        background: #3498db;
        border: 3px solid white;
        box-shadow: 0 0 0 2px #3498db;
    }

    .timeline-content {
        background: #f8f9fa;
        padding: 20px;
        border-radius: 8px;
        border-left: 4px solid #3498db;
    }

    .timeline-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 10px;
    }

    .timeline-title {
        font-weight: 600;
        color: #2c3e50;
        margin: 0;
    }

    .timeline-meta {
        font-size: 0.9em;
        color: #7f8c8d;
    }

    .timeline-description {
        margin: 10px 0;
        color: #495057;
    }

    .timeline-details {
        font-size: 0.9em;
        color: #6c757d;
    }

    .action-badge {
        display: inline-block;
        padding: 4px 8px;
        border-radius: 12px;
        font-size: 0.8em;
        font-weight: bold;
        text-transform: uppercase;
    }

    .action-success { background: #d4edda; color: #155724; }
    .action-warning { background: #fff3cd; color: #856404; }
    .action-danger { background: #f8d7da; color: #721c24; }
    .action-info { background: #d1ecf1; color: #0c5460; }

    .pagination {
        display: flex;
        justify-content: center;
        margin-top: 30px;
        gap: 10px;
    }

    .page-btn {
        padding: 8px 12px;
        border: 1px solid #ddd;
        background: white;
        color: #3498db;
        text-decoration: none;
        border-radius: 5px;
        transition: all 0.2s;
    }

    .page-btn:hover,
    .page-btn.active {
        background: #3498db;
        color: white;
    }

    .no-events {
        text-align: center;
        color: #7f8c8d;
        font-style: italic;
        padding: 40px;
    }

    @media (max-width: 768px) {
        .filters-form {
            grid-template-columns: 1fr;
        }

        .stats-cards {
            grid-template-columns: 1fr;
        }

        .timeline-header {
            flex-direction: column;
            align-items: flex-start;
            gap: 5px;
        }
    }
</style>

<div class="audit-dashboard">
    <div class="dashboard-header">
        <h1>🔍 Audit Trail Dashboard</h1>
        <p>Track and monitor all accounting system activities</p>
    </div>

    <!-- Filters Section -->
    <div class="filters-section">
        <form method="GET" class="filters-form">
            <div class="filter-group">
                <label for="start_date">Start Date</label>
                <input type="date" id="start_date" name="start_date" value="<?php echo htmlspecialchars($start_date); ?>">
            </div>
            <div class="filter-group">
                <label for="end_date">End Date</label>
                <input type="date" id="end_date" name="end_date" value="<?php echo htmlspecialchars($end_date); ?>">
            </div>
            <div class="filter-group">
                <label for="user">User</label>
                <input type="text" id="user" name="user" placeholder="Search by user..." value="<?php echo htmlspecialchars($user_filter); ?>">
            </div>
            <div class="filter-group">
                <label for="action">Action</label>
                <select id="action" name="action">
                    <option value="">All Actions</option>
                    <option value="INSERT" <?php echo $action_filter === 'INSERT' ? 'selected' : ''; ?>>Created</option>
                    <option value="UPDATE" <?php echo $action_filter === 'UPDATE' ? 'selected' : ''; ?>>Modified</option>
                    <option value="DELETE" <?php echo $action_filter === 'DELETE' ? 'selected' : ''; ?>>Deleted</option>
                </select>
            </div>
            <div class="filter-group">
                <label for="module">Module</label>
                <select id="module" name="module">
                    <option value="">All Modules</option>
                    <option value="budgets" <?php echo $module_filter === 'budgets' ? 'selected' : ''; ?>>Budgets</option>
                    <option value="expenses" <?php echo $module_filter === 'expenses' ? 'selected' : ''; ?>>Expenses</option>
                    <option value="payroll" <?php echo $module_filter === 'payroll' ? 'selected' : ''; ?>>Payroll</option>
                </select>
            </div>
            <div class="filter-group">
                <button type="submit" class="filter-btn">🔍 Filter</button>
            </div>
        </form>
    </div>

    <!-- Statistics Cards -->
    <?php if ($stats): ?>
    <div class="stats-cards">
        <div class="stat-card">
            <div class="stat-value"><?php echo number_format($stats['total_events']); ?></div>
            <div class="stat-label">Total Events (30 days)</div>
        </div>
        <div class="stat-card">
            <div class="stat-value"><?php echo number_format($stats['unique_users']); ?></div>
            <div class="stat-label">Active Users</div>
        </div>
        <div class="stat-card">
            <div class="stat-value"><?php echo number_format($stats['inserts']); ?></div>
            <div class="stat-label">Created Records</div>
        </div>
        <div class="stat-card">
            <div class="stat-value"><?php echo number_format($stats['updates']); ?></div>
            <div class="stat-label">Modified Records</div>
        </div>
        <div class="stat-card">
            <div class="stat-value"><?php echo number_format($stats['deletes']); ?></div>
            <div class="stat-label">Deleted Records</div>
        </div>
    </div>
    <?php endif; ?>

    <!-- Timeline -->
    <div class="timeline-container">
        <h2 style="color: #2c3e50; margin-bottom: 20px;">📅 Activity Timeline</h2>

        <?php if (empty($audit_events)): ?>
            <div class="no-events">
                <p>No audit events found for the selected filters.</p>
            </div>
        <?php else: ?>
            <div class="timeline">
                <?php foreach ($audit_events as $event): ?>
                    <div class="timeline-item">
                        <div class="timeline-content">
                            <div class="timeline-header">
                                <h3 class="timeline-title">
                                    <?php echo htmlspecialchars($event['action_display']); ?> -
                                    <?php echo htmlspecialchars($event['table_name']); ?>
                                    <?php if ($event['record_id']): ?>
                                        (ID: <?php echo htmlspecialchars($event['record_id']); ?>)
                                    <?php endif; ?>
                                </h3>
                                <span class="action-badge action-<?php echo $event['action_color']; ?>">
                                    <?php echo htmlspecialchars($event['action_display']); ?>
                                </span>
                            </div>

                            <?php if ($event['description']): ?>
                                <div class="timeline-description">
                                    <?php echo htmlspecialchars($event['description']); ?>
                                </div>
                            <?php endif; ?>

                            <div class="timeline-details">
                                <strong>User:</strong> <?php echo htmlspecialchars($event['user_name'] ?: 'System'); ?> |
                                <strong>Module:</strong> <?php echo htmlspecialchars($event['module'] ?: 'N/A'); ?> |
                                <strong>Time:</strong> <?php echo date('M d, Y H:i:s', strtotime($event['timestamp'])); ?>
                                <?php if ($event['ip_address']): ?>
                                    | <strong>IP:</strong> <?php echo htmlspecialchars($event['ip_address']); ?>
                                <?php endif; ?>
                            </div>

                            <?php if ($event['changed_fields']): ?>
                                <div class="timeline-details" style="margin-top: 10px;">
                                    <strong>Changed Fields:</strong> <?php echo htmlspecialchars($event['changed_fields']); ?>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>

            <!-- Pagination -->
            <?php if ($total_pages > 1): ?>
            <div class="pagination">
                <?php
                $query_string = http_build_query(array_merge($_GET, ['page' => '']));
                $base_url = '?' . rtrim($query_string, '=');

                // Previous button
                if ($page > 1): ?>
                    <a href="<?php echo $base_url . ($page - 1); ?>" class="page-btn">« Previous</a>
                <?php endif;

                // Page numbers
                $start_page = max(1, $page - 2);
                $end_page = min($total_pages, $page + 2);

                if ($start_page > 1): ?>
                    <a href="<?php echo $base_url; ?>1" class="page-btn">1</a>
                    <?php if ($start_page > 2): ?>
                        <span class="page-btn">...</span>
                    <?php endif; ?>
                <?php endif;

                for ($i = $start_page; $i <= $end_page; $i++): ?>
                    <a href="<?php echo $base_url . $i; ?>" class="page-btn <?php echo $i === $page ? 'active' : ''; ?>">
                        <?php echo $i; ?>
                    </a>
                <?php endfor;

                if ($end_page < $total_pages): ?>
                    <?php if ($end_page < $total_pages - 1): ?>
                        <span class="page-btn">...</span>
                    <?php endif; ?>
                    <a href="<?php echo $base_url . $total_pages; ?>" class="page-btn"><?php echo $total_pages; ?></a>
                <?php endif;

                // Next button
                if ($page < $total_pages): ?>
                    <a href="<?php echo $base_url . ($page + 1); ?>" class="page-btn">Next »</a>
                <?php endif; ?>
            </div>
            <?php endif; ?>
        <?php endif; ?>
    </div>
</div>

<?php include '../includes/admin_footer.php'; ?>
