<?php
require_once '../config.php';
include 'accounting_functions.php';

$message = '';

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['assign_fee'])) {
        $fee_id = $_POST['fee_id'];
        $student_ids = $_POST['student_ids'] ?? [];
        $due_date = $_POST['due_date'];

        try {
            // Get fee details first
            $stmt = $pdo->prepare("SELECT * FROM fees WHERE id = ?");
            $stmt->execute([$fee_id]);
            $fee = $stmt->fetch();

            if (!$fee) {
                $message = 'Error: Selected fee not found.';
            } else {
                // Check which students already have this fee assigned
                $already_assigned_students = [];
                $students_to_assign = [];

                foreach ($student_ids as $student_id) {
                    $stmt = $pdo->prepare("SELECT id FROM student_fees WHERE student_id = ? AND fee_id = ?");
                    $stmt->execute([$student_id, $fee_id]);
                    $existing = $stmt->fetch();

                    if ($existing) {
                        // Get student name for error message
                        $stmt = $pdo->prepare("SELECT CONCAT(first_name, ' ', COALESCE(middle_name, ''), ' ', last_name) as name FROM students WHERE id = ?");
                        $stmt->execute([$student_id]);
                        $student = $stmt->fetch();
                        $already_assigned_students[] = $student['name'];
                    } else {
                        $students_to_assign[] = $student_id;
                    }
                }

                // If some students already have the fee, show error
                if (!empty($already_assigned_students)) {
                    $student_names = implode(', ', $already_assigned_students);
                    $message = "Error: The following student(s) already have this fee assigned: {$student_names}. Fee assignment cancelled.";
                } else {
                    // Proceed with assignment for students who don't have it
                    $assigned_count = 0;
                    foreach ($students_to_assign as $student_id) {
                        $stmt = $pdo->prepare("INSERT INTO student_fees (student_id, fee_id, amount, due_date) VALUES (?, ?, ?, ?)");
                        $stmt->execute([$student_id, $fee_id, $fee['amount'], $due_date]);
                        $assigned_count++;

                        // --- New code to create journal entry for receivable ---
                        // Get fee type to determine accounting treatment
                        $stmt = $pdo->prepare("SELECT f.type FROM fees f WHERE f.id = ?");
                        $stmt->execute([$fee_id]);
                        $feeType = $stmt->fetch();

                        // Determine receivable and revenue accounts based on fee type
                        $receivableAccountCode = '1123'; // Default for other fees
                        $revenueAccountCode = '4120'; // Default for other fees revenue
                        if ($feeType) {
                            $feeTypeLower = strtolower($feeType['type']);
                            if ($feeTypeLower === 'tuition') {
                                $receivableAccountCode = '1121'; // Tuition fees receivable
                                $revenueAccountCode = '4110'; // Tuition revenue
                            } elseif ($feeTypeLower === 'registration') {
                                $receivableAccountCode = '1122'; // Registration Fees Receivable
                                $revenueAccountCode = '4001'; // Registration Fees (revenue)
                            } elseif ($feeTypeLower === 'examination') {
                                $receivableAccountCode = '4002'; // Examination Fees
                                $revenueAccountCode = '4002'; // Examination Fees (revenue)
                            }
                        }
                        $stmt = $pdo->prepare("SELECT id FROM chart_of_accounts WHERE account_code = ?");
                        $stmt->execute([$receivableAccountCode]);
                        $receivableAccount = $stmt->fetch();

                        if (!$receivableAccount) {
                            throw new Exception('Receivable account not found in chart of accounts for code: ' . $receivableAccountCode);
                        }

                        // Get appropriate revenue account based on fee type
                        $stmt = $pdo->prepare("SELECT id FROM chart_of_accounts WHERE account_code = ?");
                        $stmt->execute([$revenueAccountCode]);
                        $revenueAccount = $stmt->fetch();

                        if (!$revenueAccount) {
                            throw new Exception('Revenue account not found in chart of accounts for code: ' . $revenueAccountCode);
                        }

                        // Generate journal entry number
                        $entryNumber = generateEntryNumber($pdo, date('Y-m-d'));

                        // Create journal entry
                        $stmt = $pdo->prepare("
                            INSERT INTO journal_entries
                            (entry_number, transaction_date, description, source_type, source_id, status, created_by)
                            VALUES (?, ?, ?, 'fee_assignment', ?, 'posted', ?)
                        ");
                        $stmt->execute([
                            $entryNumber,
                            date('Y-m-d'),
                            "Fee assigned to student ID {$student_id}: {$fee['name']} - GHC {$fee['amount']}",
                            $student_id,
                            $_SESSION['user_id'] ?? null
                        ]);
                        $entryId = $pdo->lastInsertId();

                        // Create debit line (Student Fees Receivable)
                        $stmt = $pdo->prepare("
                            INSERT INTO journal_entry_lines
                            (journal_entry_id, account_id, debit_amount, line_number, description)
                            VALUES (?, ?, ?, 1, ?)
                        ");
                        $stmt->execute([
                            $entryId,
                            $receivableAccount['id'],
                            $fee['amount'],
                            "Receivable for fee assignment: {$fee['name']}"
                        ]);

                        // Create credit line (Tuition Fees revenue)
                        $stmt = $pdo->prepare("
                            INSERT INTO journal_entry_lines
                            (journal_entry_id, account_id, credit_amount, line_number, description)
                            VALUES (?, ?, ?, 2, ?)
                        ");
                        $stmt->execute([
                            $entryId,
                            $revenueAccount['id'],
                            $fee['amount'],
                            "Revenue for fee assignment: {$fee['name']}"
                        ]);

                        // Update chart_of_accounts balances
                        $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance + ? WHERE id = ?");
                        $stmt->execute([$fee['amount'], $receivableAccount['id']]); // Debit receivable (increase)

                        $stmt = $pdo->prepare("UPDATE chart_of_accounts SET current_balance = current_balance + ? WHERE id = ?");
                        $stmt->execute([$fee['amount'], $revenueAccount['id']]); // Credit revenue (increase)
                        // --- End new code ---
                    }
                    $message = "Fee assigned to $assigned_count student(s) successfully.";
                }
            }
        } catch (PDOException $e) {
            $message = 'Error assigning fee: ' . $e->getMessage();
        }
    }
}

// Get all fees
$stmt = $pdo->query("SELECT f.*, ay.year as academic_year FROM fees f LEFT JOIN academic_years ay ON f.academic_year_id = ay.id ORDER BY f.created_at DESC");
$fees = $stmt->fetchAll();

// Get filter values
$filter_class = $_GET['class'] ?? '';
$filter_academic_year = $_GET['academic_year'] ?? '';

// Get students with optional filters
$query = "SELECT s.*, CONCAT(s.first_name, ' ', COALESCE(s.middle_name, ''), ' ', s.last_name) as name, su.username FROM students s JOIN student_users su ON s.user_id = su.id WHERE 1=1";
$params = [];

if ($filter_class) {
    $query .= " AND s.class = ?";
    $params[] = $filter_class;
}
if ($filter_academic_year) {
    $query .= " AND s.academic_year = ?";
    $params[] = $filter_academic_year;
}

$stmt = $pdo->prepare($query);
$stmt->execute($params);
$students = $stmt->fetchAll();

// Get distinct classes and academic years for filter dropdowns
$classes_stmt = $pdo->query("SELECT name FROM classes ORDER BY name");
$classes = $classes_stmt->fetchAll(PDO::FETCH_COLUMN);

$years_stmt = $pdo->query("SELECT year FROM academic_years ORDER BY year DESC");
$academic_years = $years_stmt->fetchAll(PDO::FETCH_COLUMN);

include 'includes/accounting_header.php';
?>

<div class="page-container">
    <h2>Assign Fees to Students</h2>

    <?php if ($message): ?>
        <div class="message <?php echo strpos($message, 'Error:') === 0 ? 'error' : 'success'; ?>"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <form method="get" action="assign_fees.php" class="filter-form">
        <div class="form-group">
            <label for="class">Filter Students by Class:</label>
            <select id="class" name="class">
                <option value="">All</option>
                <?php foreach ($classes as $class): ?>
                    <option value="<?php echo htmlspecialchars($class); ?>" <?php if ($class == $filter_class) echo 'selected'; ?>>
                        <?php echo htmlspecialchars($class); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>

        <div class="form-group">
            <label for="academic_year">Filter by Academic Year:</label>
            <select id="academic_year" name="academic_year">
                <option value="">All</option>
                <?php foreach ($academic_years as $year): ?>
                    <option value="<?php echo htmlspecialchars($year); ?>" <?php if ($year == $filter_academic_year) echo 'selected'; ?>>
                        <?php echo htmlspecialchars($year); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>

        <div class="form-actions">
            <button type="submit" class="btn btn-secondary">Filter</button>
            <a href="assign_fees.php" class="btn btn-secondary">Clear Filters</a>
        </div>
    </form>

    <form method="post" action="assign_fees.php" class="assignment-form">
        <div class="form-row">
            <div class="form-group">
                <label for="fee_id">Select Fee to Assign:</label>
                <select id="fee_id" name="fee_id" required>
                    <option value="">Select Fee</option>
                    <?php foreach ($fees as $fee): ?>
                        <option value="<?php echo $fee['id']; ?>">
                            <?php echo htmlspecialchars($fee['name']); ?> - GHC <?php echo number_format($fee['amount'], 2); ?>
                            <?php if ($fee['class']): ?> (<?php echo htmlspecialchars($fee['class']); ?>)<?php endif; ?>
                            <?php if ($fee['academic_year']): ?> - <?php echo htmlspecialchars($fee['academic_year']); ?><?php endif; ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div class="form-group">
                <label for="due_date">Due Date:</label>
                <input type="date" id="due_date" name="due_date" required>
            </div>
        </div>

        <div class="table-container">
            <h3>Select Students to Assign Fee:</h3>
            <div class="table-wrapper">
                <table class="students-table">
                    <thead>
                        <tr>
                            <th>
                                <label>
                                    <input type="checkbox" id="select_all"> Select All
                                </label>
                            </th>
                            <th>Student ID</th>
                            <th>Full Name</th>
                            <th>Class</th>
                            <th>Academic Year</th>
                            <th>Username</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($students as $student): ?>
                            <tr>
                                <td>
                                    <input type="checkbox" name="student_ids[]" value="<?php echo $student['id']; ?>">
                                </td>
                                <td><?php echo htmlspecialchars($student['student_id'] ?? 'N/A'); ?></td>
                                <td><?php echo htmlspecialchars($student['name']); ?></td>
                                <td><?php echo htmlspecialchars($student['class']); ?></td>
                                <td><?php echo htmlspecialchars($student['academic_year']); ?></td>
                                <td><?php echo htmlspecialchars($student['username']); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>

        <button type="submit" name="assign_fee" class="btn btn-primary">Assign Fee</button>
    </form>
</div>

<script>
document.getElementById('select_all').addEventListener('change', function() {
    const checkboxes = document.querySelectorAll('input[name="student_ids[]"]');
    checkboxes.forEach(cb => cb.checked = this.checked);
});
</script>

<style>
.page-container {
    background: white;
    border-radius: 10px;
    padding: 25px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.05);
    margin-bottom: 20px;
}

.page-container h2 {
    color: #2c3e50;
    margin-bottom: 20px;
    font-size: 1.8em;
    font-weight: 600;
}

.page-container h3 {
    color: #2c3e50;
    margin: 20px 0 15px 0;
    font-size: 1.4em;
    font-weight: 500;
}

.message {
    padding: 15px;
    border-radius: 5px;
    margin-bottom: 20px;
    border: 1px solid;
}

.message.success {
    background: #d4edda;
    color: #155724;
    border-color: #c3e6cb;
}

.message.error {
    background: #f8d7da;
    color: #721c24;
    border-color: #f5c6cb;
}

.filter-form {
    display: flex;
    gap: 20px;
    align-items: end;
    margin-bottom: 30px;
    padding: 20px;
    background: #f8f9fa;
    border-radius: 10px;
    border: 1px solid #dee2e6;
}

.filter-form .form-group {
    flex: 1;
}

.filter-form label {
    display: block;
    margin-bottom: 8px;
    font-weight: 600;
    color: #2c3e50;
}

.filter-form select {
    width: 100%;
    padding: 10px 15px;
    border: 2px solid #2E7D32;
    border-radius: 5px;
    background: #fff;
    color: #2c3e50;
    font-size: 0.9em;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.3s ease;
}

.filter-form select:focus {
    outline: none;
    border-color: #4CAF50;
    box-shadow: 0 0 0 3px rgba(76, 175, 80, 0.1);
}

.filter-form select:hover {
    border-color: #4CAF50;
}

.form-actions {
    display: flex;
    gap: 10px;
}

.assignment-form {
    margin-bottom: 20px;
}

.form-row {
    display: flex;
    gap: 20px;
    align-items: end;
}

.form-row .form-group {
    flex: 1;
    margin-bottom: 20px;
}

.assignment-form label {
    display: block;
    margin-bottom: 8px;
    font-weight: 600;
    color: #2c3e50;
}

.assignment-form select,
.assignment-form input[type="date"] {
    width: 100%;
    padding: 10px 15px;
    border: 2px solid #2E7D32;
    border-radius: 5px;
    background: #fff;
    color: #2c3e50;
    font-size: 0.9em;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.3s ease;
}

.assignment-form select:focus,
.assignment-form input[type="date"]:focus {
    outline: none;
    border-color: #4CAF50;
    box-shadow: 0 0 0 3px rgba(76, 175, 80, 0.1);
}

.assignment-form select:hover,
.assignment-form input[type="date"]:hover {
    border-color: #4CAF50;
}

.students-table {
    width: 100%;
    border-collapse: collapse;
    background: #fff;
    border-radius: 8px;
    overflow: hidden;
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
}

.students-table th {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: white;
    padding: 15px 12px;
    text-align: left;
    font-weight: 600;
    font-size: 0.9em;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    border: none;
}

.students-table th:first-child {
    width: 50px;
    text-align: center;
}

.students-table th:nth-child(2) {
    width: 120px;
}

.students-table th:nth-child(6) {
    width: 120px;
}

.students-table td {
    padding: 12px;
    border-bottom: 1px solid #e9ecef;
    color: #2c3e50;
    font-size: 0.9em;
}

.students-table tbody tr:hover {
    background: #f8f9fa;
    transition: background-color 0.2s ease;
}

.students-table tbody tr:last-child td {
    border-bottom: none;
}

.students-table input[type="checkbox"] {
    transform: scale(1.2);
    accent-color: #2E7D32;
    cursor: pointer;
}

.students-table td:first-child {
    text-align: center;
}

.students-table td:nth-child(2) {
    font-weight: 600;
    color: #2E7D32;
}

.students-table td:nth-child(3) {
    font-weight: 500;
}

.btn {
    padding: 10px 20px;
    border: none;
    border-radius: 5px;
    cursor: pointer;
    font-size: 0.9em;
    font-weight: 500;
    text-decoration: none;
    display: inline-block;
    transition: all 0.3s ease;
}

.btn-primary {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: #fff;
}

.btn-primary:hover {
    background: linear-gradient(135deg, #4CAF50 0%, #2E7D32 100%);
    transform: translateY(-2px);
    box-shadow: 0 4px 8px rgba(0,0,0,0.2);
}

.btn-secondary {
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: #fff;
}

.btn-secondary:hover {
    background: linear-gradient(135deg, #4CAF50 0%, #2E7D32 100%);
    transform: translateY(-2px);
    box-shadow: 0 4px 8px rgba(0,0,0,0.2);
}

.btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 8px rgba(0,0,0,0.2);
}

/* Responsive */
@media (max-width: 768px) {
    .filter-form {
        flex-direction: column;
        gap: 15px;
    }

    .form-actions {
        justify-content: center;
    }

    .form-row {
        flex-direction: column;
        gap: 15px;
    }

    .students-table {
        font-size: 0.8em;
    }

    .students-table th,
    .students-table td {
        padding: 8px 6px;
    }
}
</style>

<?php
include 'includes/accounting_footer.php';
?>
