<?php
require_once '../../config.php';
require_once '../asset_audit_functions.php';

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit;
}

$asset_id = $_POST['asset_id'] ?? null;
$name = trim($_POST['name'] ?? '');
$asset_type = $_POST['asset_type'] ?? '';
$purchase_cost = $_POST['purchase_cost'] ?? '';
$depreciation_method = $_POST['depreciation_method'] ?? '';
$depreciation_rate = $_POST['depreciation_rate'] ?? '';
$salvage_value = $_POST['salvage_value'] ?? '';
$useful_life_years = $_POST['useful_life_years'] ?? '';
$useful_life_months = $_POST['useful_life_months'] ?? '';
$depreciation_frequency = $_POST['depreciation_frequency'] ?? '';
$appreciation_method = $_POST['appreciation_method'] ?? '';
$appreciation_rate = $_POST['appreciation_rate'] ?? '';
$insured_value = $_POST['insured_value'] ?? '';
$insurance_policy_number = $_POST['insurance_policy_number'] ?? '';
$insurance_expiry_date = $_POST['insurance_expiry_date'] ?? '';
$current_value = $_POST['current_value'] ?? null;
$purchase_date = $_POST['purchase_date'] ?? '';
$category_id = $_POST['category_id'] ?? '';
$status = $_POST['status'] ?? '';
$serial_number = trim($_POST['serial_number'] ?? '');
$location = trim($_POST['location'] ?? '');
$description = trim($_POST['description'] ?? '');

// Validation
if (!$asset_id || !is_numeric($asset_id)) {
    echo json_encode(['success' => false, 'message' => 'Invalid asset ID']);
    exit;
}

if (empty($name)) {
    echo json_encode(['success' => false, 'message' => 'Asset name is required']);
    exit;
}

if (empty($asset_type)) {
    echo json_encode(['success' => false, 'message' => 'Asset type is required']);
    exit;
}

if (!is_numeric($purchase_cost) || $purchase_cost < 0) {
    echo json_encode(['success' => false, 'message' => 'Valid purchase cost is required']);
    exit;
}

if (empty($purchase_date)) {
    echo json_encode(['success' => false, 'message' => 'Purchase date is required']);
    exit;
}

if (!$category_id || !is_numeric($category_id)) {
    echo json_encode(['success' => false, 'message' => 'Valid category is required']);
    exit;
}

if (empty($status)) {
    echo json_encode(['success' => false, 'message' => 'Status is required']);
    exit;
}

try {
    // Get current asset data for audit logging
    $stmt = $pdo->prepare("SELECT * FROM assets WHERE id = ?");
    $stmt->execute([$asset_id]);
    $old_asset = $stmt->fetch(PDO::FETCH_ASSOC);

    // Prepare new asset data
    $new_asset_data = [
        'name' => $name,
        'asset_type' => $asset_type,
        'purchase_cost' => $purchase_cost,
        'current_value' => $current_value ?: null,
        'purchase_date' => $purchase_date,
        'category_id' => $category_id,
        'status' => $status,
        'serial_number' => $serial_number ?: null,
        'location' => $location ?: null,
        'description' => $description ?: null,
        'depreciation_method' => $depreciation_method ?: null,
        'depreciation_rate' => $depreciation_rate ?: null,
        'salvage_value' => $salvage_value ?: null,
        'useful_life_years' => $useful_life_years ?: null,
        'useful_life_months' => $useful_life_months ?: null,
        'depreciation_frequency' => $depreciation_frequency ?: null,
        'appreciation_method' => $appreciation_method ?: null,
        'appreciation_rate' => $appreciation_rate ?: null,
        'insured_value' => $insured_value ?: null,
        'insurance_policy_number' => $insurance_policy_number ?: null,
        'insurance_expiry_date' => $insurance_expiry_date ?: null,
        'updated_at' => date('Y-m-d H:i:s')
    ];

    // Calculate changes for audit log
    $changes = [];
    foreach ($new_asset_data as $key => $value) {
        if (isset($old_asset[$key]) && $old_asset[$key] != $value) {
            $changes[$key] = [
                'old' => $old_asset[$key],
                'new' => $value
            ];
        }
    }

    $stmt = $pdo->prepare("
        UPDATE assets SET
            name = ?,
            asset_type = ?,
            purchase_cost = ?,
            current_value = ?,
            purchase_date = ?,
            category_id = ?,
            status = ?,
            serial_number = ?,
            location = ?,
            description = ?,
            depreciation_method = ?,
            depreciation_rate = ?,
            salvage_value = ?,
            useful_life_years = ?,
            useful_life_months = ?,
            depreciation_frequency = ?,
            appreciation_method = ?,
            appreciation_rate = ?,
            insured_value = ?,
            insurance_policy_number = ?,
            insurance_expiry_date = ?,
            updated_at = NOW()
        WHERE id = ?
    ");

    $result = $stmt->execute([
        $name,
        $asset_type,
        $purchase_cost,
        $current_value ?: null,
        $purchase_date,
        $category_id,
        $status,
        $serial_number ?: null,
        $location ?: null,
        $description ?: null,
        $depreciation_method ?: null,
        $depreciation_rate ?: null,
        $salvage_value ?: null,
        $useful_life_years ?: null,
        $useful_life_months ?: null,
        $depreciation_frequency ?: null,
        $appreciation_method ?: null,
        $appreciation_rate ?: null,
        $insured_value ?: null,
        $insurance_policy_number ?: null,
        $insurance_expiry_date ?: null,
        $asset_id
    ]);

    if ($result) {
        // Log the update action
        logAssetAction($pdo, $asset_id, $_SESSION['user_id'] ?? 1, 'update', $old_asset, $new_asset_data, $changes);

        echo json_encode(['success' => true, 'message' => 'Asset updated successfully']);
    } else {
        echo json_encode(['success' => false, 'message' => 'Failed to update asset']);
    }
} catch (PDOException $e) {
    echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
}
?>
