<?php
require_once '../config.php';
require_once '../includes/accountant_activity_logger.php';

if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// Access control: ensure user is logged in and has accounting permissions
require_once '../includes/enhanced_access_control_fixed.php';
requirePermission('fee_management', '../login.php');

$user_id = $_SESSION['user_id'];

$filter_action = $_GET['action'] ?? '';
$filter_date_from = $_GET['date_from'] ?? '';
$filter_date_to = $_GET['date_to'] ?? '';
$page = max(1, intval($_GET['page'] ?? 1));
$limit = 50;
$offset = ($page - 1) * $limit;

// Initialize logger
AccountantActivityLogger::init($pdo);

// Fetch logs with filters and pagination
$logs = AccountantActivityLogger::getAccountantLogs($user_id, $limit, $offset, $filter_date_from, $filter_date_to, $filter_action);

// Fetch total count for pagination
try {
    $where_conditions = [];
    $params = [];

    if ($user_id) {
        $where_conditions[] = "user_id = ?";
        $params[] = $user_id;
    }
    if ($filter_date_from) {
        $where_conditions[] = "created_at >= ?";
        $params[] = $filter_date_from . ' 00:00:00';
    }
    if ($filter_date_to) {
        $where_conditions[] = "created_at <= ?";
        $params[] = $filter_date_to . ' 23:59:59';
    }
    if ($filter_action) {
        $where_conditions[] = "action = ?";
        $params[] = $filter_action;
    }

    $where_clause = !empty($where_conditions) ? "WHERE " . implode(" AND ", $where_conditions) : "";

    $count_sql = "SELECT COUNT(*) as total FROM accountants_logs {$where_clause}";
    $stmt = $pdo->prepare($count_sql);
    $stmt->execute($params);
    $total_count = $stmt->fetchColumn();
} catch (PDOException $e) {
    $total_count = 0;
}

$total_pages = ceil($total_count / $limit);

include 'includes/accounting_header.php';
?>

<div class="page-container">
    <h2>Accountant Activity Logs</h2>

    <form method="get" action="activity_logs.php" class="filter-form">
        <div class="form-group">
            <label for="action">Filter by Action:</label>
            <select id="action" name="action">
                <option value="">All Actions</option>
                <?php
                // Fetch distinct actions for filter dropdown
                $actions_stmt = $pdo->query("SELECT DISTINCT action FROM accountants_logs ORDER BY action");
                $actions = $actions_stmt->fetchAll(PDO::FETCH_COLUMN);
                foreach ($actions as $action):
                ?>
                <option value="<?php echo htmlspecialchars($action); ?>" <?php if ($action == $filter_action) echo 'selected'; ?>>
                    <?php echo htmlspecialchars($action); ?>
                </option>
                <?php endforeach; ?>
            </select>
        </div>

        <div class="form-group">
            <label for="date_from">Date From:</label>
            <input type="date" id="date_from" name="date_from" value="<?php echo htmlspecialchars($filter_date_from); ?>">
        </div>

        <div class="form-group">
            <label for="date_to">Date To:</label>
            <input type="date" id="date_to" name="date_to" value="<?php echo htmlspecialchars($filter_date_to); ?>">
        </div>

        <div class="form-actions">
            <button type="submit" class="btn btn-secondary">Filter</button>
            <a href="activity_logs.php" class="btn btn-secondary">Clear Filters</a>
        </div>
    </form>

    <div class="table-container">
        <table class="logs-table">
            <thead>
                <tr>
                    <th>User</th>
                    <th>Action</th>
                    <th>Description</th>
                    <th>Amount (GHC)</th>
                    <th>Date</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($logs)): ?>
                <tr>
                    <td colspan="5" style="text-align:center;">No activity logs found.</td>
                </tr>
                <?php else: ?>
                <?php foreach ($logs as $log): ?>
                <tr>
                    <td><?php echo htmlspecialchars($log['username']); ?></td>
                    <td><?php echo htmlspecialchars($log['action']); ?></td>
                    <td><?php echo htmlspecialchars($log['description']); ?></td>
                    <td><?php echo number_format($log['amount'], 2); ?></td>
                    <td><?php echo htmlspecialchars($log['created_at']); ?></td>
                </tr>
                <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>

    <?php if ($total_pages > 1): ?>
    <div class="pagination">
        <?php for ($p = 1; $p <= $total_pages; $p++): ?>
            <?php if ($p == $page): ?>
                <span class="current-page"><?php echo $p; ?></span>
            <?php else: ?>
                <a href="?<?php echo http_build_query(array_merge($_GET, ['page' => $p])); ?>"><?php echo $p; ?></a>
            <?php endif; ?>
        <?php endfor; ?>
    </div>
    <?php endif; ?>
</div>

<style>
.page-container {
    background: white;
    border-radius: 10px;
    padding: 25px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.05);
    margin-bottom: 20px;
}

.page-container h2 {
    color: #2c3e50;
    margin-bottom: 20px;
    font-size: 1.8em;
    font-weight: 600;
}

.filter-form {
    display: flex;
    gap: 20px;
    align-items: flex-end;
    margin-bottom: 30px;
    padding: 20px;
    background: #f8f9fa;
    border-radius: 10px;
    border: 1px solid #dee2e6;
    flex-wrap: wrap;
}

.filter-form .form-group {
    flex: 1;
    min-width: 150px;
}

.filter-form label {
    display: block;
    margin-bottom: 8px;
    font-weight: 600;
    color: #2c3e50;
}

.filter-form select,
.filter-form input[type="date"] {
    width: 100%;
    padding: 10px 15px;
    border: 2px solid #2E7D32;
    border-radius: 5px;
    background: #fff;
    color: #2c3e50;
    font-size: 0.9em;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.3s ease;
}

.filter-form select:hover,
.filter-form input[type="date"]:hover {
    border-color: #4CAF50;
}

.filter-form select:focus,
.filter-form input[type="date"]:focus {
    outline: none;
    border-color: #4CAF50;
    box-shadow: 0 0 0 3px rgba(76, 175, 80, 0.1);
}

.form-actions {
    display: flex;
    gap: 10px;
    align-self: flex-end;
}

.btn {
    padding: 10px 20px;
    border: none;
    border-radius: 5px;
    cursor: pointer;
    font-size: 0.9em;
    font-weight: 500;
    text-decoration: none;
    display: inline-block;
    transition: all 0.3s ease;
    background: linear-gradient(135deg, #2E7D32 0%, #4CAF50 100%);
    color: #fff;
}

.btn:hover {
    background: linear-gradient(135deg, #4CAF50 0%, #2E7D32 100%);
    transform: translateY(-2px);
    box-shadow: 0 4px 8px rgba(0,0,0,0.2);
}

.table-container {
    overflow-x: auto;
}

.logs-table {
    width: 100%;
    border-collapse: collapse;
    margin-top: 10px;
}

.logs-table th {
    background: #2E7D32;
    color: #fff;
    padding: 12px;
    text-align: left;
    border-bottom: 2px solid #4CAF50;
    font-weight: 600;
}

.logs-table td {
    padding: 12px;
    border-bottom: 1px solid #dee2e6;
}

.logs-table tr:hover {
    background: #f8f9fa;
}

.pagination {
    margin-top: 20px;
    text-align: center;
}

.pagination a,
.pagination span {
    display: inline-block;
    margin: 0 5px;
    padding: 8px 12px;
    border-radius: 5px;
    text-decoration: none;
    font-weight: 600;
    color: #2E7D32;
    border: 1px solid #2E7D32;
    transition: all 0.3s ease;
}

.pagination a:hover {
    background: #4CAF50;
    color: white;
    border-color: #4CAF50;
}

.pagination .current-page {
    background: #2E7D32;
    color: white;
    border-color: #2E7D32;
}
</style>

<?php include 'includes/accounting_footer.php'; ?>
