<?php
require_once '../config.php';

if (!in_array($_SESSION['role'], ['admin', 'teacher'])) {
    header('Location: ../login.php');
    exit;
}

$student_id = $_GET['student_id'] ?? null;
$class_name = $_GET['class'] ?? null;
$academic_year_id = $_GET['academic_year'] ?? null;
$term_id = $_GET['term'] ?? null;

if (!$student_id || !$class_name || !$academic_year_id || !$term_id) {
    die("Missing required parameters.");
}

// Fetch school settings
$school_name = 'Professional School Name';
$school_logo_url = '';
try {
    $stmt = $pdo->prepare("SELECT school_name, logo_path FROM school_settings LIMIT 1");
    $stmt->execute();
    $school_info = $stmt->fetch();
    if ($school_info) {
        $school_name = htmlspecialchars($school_info['school_name']);
        $logo_path = $school_info['logo_path'];
        if ($logo_path && !preg_match('/^https?:\/\//', $logo_path) && !preg_match('/^\//', $logo_path)) {
            $school_logo_url = '../' . $logo_path;
        } else {
            $school_logo_url = $logo_path;
        }
        $school_logo_url = htmlspecialchars($school_logo_url);
    }
} catch (Exception $e) {
    // Use default if error
}

$stmt = $pdo->prepare("SELECT CONCAT(last_name, ', ', first_name, ' ', COALESCE(middle_name, '')) AS name FROM students WHERE id = ?");
$stmt->execute([$student_id]);
$student = $stmt->fetch();
if (!$student) {
    die("Student not found.");
}
$student_name = trim($student['name']);

$stmt = $pdo->prepare("SELECT id FROM classes WHERE name = ?");
$stmt->execute([$class_name]);
$class_info = $stmt->fetch();
if (!$class_info) {
    die("Class not found.");
}
$class_id = $class_info['id'];

$stmt = $pdo->prepare("SELECT year FROM academic_years WHERE id = ?");
$stmt->execute([$academic_year_id]);
$academic_year = $stmt->fetch();
$academic_year_name = $academic_year ? $academic_year['year'] : 'N/A';

$stmt = $pdo->prepare("
    SELECT e.id AS enrollment_id, c.id AS course_id, c.code, c.name AS course_name
    FROM enrollments e
    JOIN courses c ON e.course_id = c.id
    JOIN class_courses cc ON cc.course_id = c.id AND cc.class_id = ?
    WHERE e.student_id = ? AND e.academic_year_id = ? AND e.term_id = ?
    ORDER BY c.name
");
$stmt->execute([$class_id, $student_id, $academic_year_id, $term_id]);
$enrollments = $stmt->fetchAll();

$mock_exams_data = [];
foreach ($enrollments as $enrollment) {
    $stmt = $pdo->prepare("SELECT mock_exam_number, score, grade, comments FROM mock_exams WHERE enrollment_id = ? AND term_id = ? AND academic_year_id = ? ORDER BY mock_exam_number");
    $stmt->execute([$enrollment['enrollment_id'], $term_id, $academic_year_id]);
    $mock_exams = $stmt->fetchAll();
    
    $mock_exams_data[$enrollment['course_id']] = [
        'course_name' => $enrollment['course_name'],
        'course_code' => $enrollment['code'],
        'mock_exams' => $mock_exams
    ];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Mock Exam Results - <?php echo htmlspecialchars($student_name); ?></title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: white;
            padding: 20px;
        }

        .report-container {
            max-width: 1200px;
            margin: 0 auto;
            background: white;
            padding: 40px;
            border: 2px solid #333;
        }

        .header {
            text-align: center;
            margin-bottom: 30px;
            border-bottom: 3px solid #9b59b6;
            padding-bottom: 20px;
        }

        .school-logo {
            max-height: 80px;
            margin-bottom: 10px;
        }

        .school-name {
            font-size: 2.5em;
            font-weight: 700;
            color: #9b59b6;
            margin-bottom: 10px;
        }

        .report-title {
            font-size: 1.8em;
            color: #333;
            font-weight: 600;
            margin-bottom: 5px;
        }

        .report-subtitle {
            font-size: 1.2em;
            color: #666;
            font-style: italic;
        }

        .info-section {
            display: grid;
            grid-template-columns: repeat(3, 1fr);
            gap: 15px;
            margin-bottom: 30px;
            padding: 20px;
            background: #f8f9fa;
            border-radius: 8px;
        }

        .info-item {
            font-size: 1.1em;
            color: #333;
        }

        .info-item strong {
            color: #9b59b6;
            font-weight: 600;
        }

        .results-table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 30px;
        }

        .results-table th,
        .results-table td {
            border: 1px solid #333;
            padding: 12px;
            text-align: center;
        }

        .results-table th {
            background: linear-gradient(135deg, #9b59b6 0%, #8e44ad 100%);
            color: white;
            font-weight: 600;
            font-size: 0.9em;
            text-transform: uppercase;
        }

        .results-table .course-name {
            text-align: left;
            font-weight: 600;
            background: #f8f9fa;
        }

        .results-table tr:nth-child(even) {
            background: #fafbfc;
        }

        .no-data {
            color: #999;
            font-style: italic;
        }

        .grade-cell {
            font-weight: 600;
            font-size: 1.05em;
        }

        .footer {
            margin-top: 40px;
            padding-top: 20px;
            border-top: 2px solid #333;
            text-align: center;
            font-size: 0.9em;
            color: #666;
        }

        .print-btn {
            position: fixed;
            top: 20px;
            right: 20px;
            padding: 12px 24px;
            background: linear-gradient(135deg, #9b59b6 0%, #8e44ad 100%);
            color: white;
            border: none;
            border-radius: 8px;
            font-weight: 600;
            cursor: pointer;
            font-size: 1em;
            box-shadow: 0 4px 15px rgba(155, 89, 182, 0.3);
        }

        .print-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(155, 89, 182, 0.4);
        }

        @media print {
            .print-btn {
                display: none;
            }

            body {
                padding: 0;
            }

            .report-container {
                border: none;
                padding: 20px;
            }
        }
    </style>
</head>
<body>
    <button class="print-btn" onclick="window.print()">🖨️ Print Report</button>

    <div class="report-container">
        <div class="header">
            <?php if ($school_logo_url): ?>
                <img src="<?php echo $school_logo_url; ?>" alt="School Logo" class="school-logo" />
            <?php endif; ?>
            <div class="school-name"><?php echo $school_name; ?></div>
            <div class="report-title">Mock Exam Results Report</div>
            <div class="report-subtitle">Final Year Student Performance</div>
        </div>

        <div class="info-section">
            <div class="info-item"><strong>Student Name:</strong> <?php echo htmlspecialchars($student_name); ?></div>
            <div class="info-item"><strong>Class:</strong> <?php echo htmlspecialchars($class_name); ?></div>
            <div class="info-item"><strong>Academic Year:</strong> <?php echo htmlspecialchars($academic_year_name); ?></div>
        </div>

        <table class="results-table">
            <thead>
                <tr>
                    <th>Course</th>
                    <?php for ($i = 1; $i <= 10; $i++): ?>
                        <th>Mock #<?php echo $i; ?></th>
                    <?php endfor; ?>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($mock_exams_data as $course_data): ?>
                    <tr>
                        <td class="course-name"><?php echo htmlspecialchars($course_data['course_name']); ?></td>
                        <?php 
                        $mock_exams_by_number = [];
                        foreach ($course_data['mock_exams'] as $mock) {
                            $mock_exams_by_number[$mock['mock_exam_number']] = $mock;
                        }
                        
                        for ($i = 1; $i <= 10; $i++): 
                            if (isset($mock_exams_by_number[$i])): 
                                $mock = $mock_exams_by_number[$i];
                        ?>
                            <td class="grade-cell" title="<?php echo htmlspecialchars($mock['comments'] ?? ''); ?>">
                                <?php echo htmlspecialchars($mock['score'] ?? '-'); ?><br>
                                <small>(<?php echo htmlspecialchars($mock['grade'] ?? '-'); ?>)</small>
                            </td>
                        <?php else: ?>
                            <td class="no-data">-</td>
                        <?php endif; endfor; ?>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>

        <div class="footer">
            <p><strong>Generated on:</strong> <?php echo date('F d, Y'); ?></p>
            <p style="margin-top: 10px;">This is an official document from the School Management System</p>
        </div>
    </div>
</body>
</html>
