<?php
require_once '../config.php';

if (!in_array($_SESSION['role'], ['admin', 'teacher'])) {
    header('Location: ../login.php');
    exit;
}

$user_id = $_SESSION['user_id'];
// Try to get teacher ID - handle both user_id in staff table or direct id match
$stmt = $pdo->prepare("SELECT id FROM staff WHERE user_id = ? AND role = 'teaching'");
$stmt->execute([$user_id]);
$teacher = $stmt->fetch();

// If not found, try direct ID match (some systems use staff.id = users.id)
if (!$teacher) {
    $stmt = $pdo->prepare("SELECT id FROM staff WHERE id = ? AND role = 'teaching'");
    $stmt->execute([$user_id]);
    $teacher = $stmt->fetch();
}

if (!$teacher) {
    die("Teacher not found. User ID: " . $user_id);
}

$teacher_id = $teacher['id'];

// Get filter values
$selected_class = $_GET['class'] ?? null;
$selected_student = $_GET['student_id'] ?? null;

// Get final year classes
$stmt = $pdo->prepare("SELECT DISTINCT c.name FROM classes c JOIN teacher_classes tc ON c.id = tc.class_id WHERE tc.teacher_id = ? AND c.level_id = 5 ORDER BY c.name");
$stmt->execute([$teacher_id]);
$classes = $stmt->fetchAll();

// Get students in selected class
$students = [];
if ($selected_class) {
    $stmt = $pdo->prepare("SELECT id, CONCAT(last_name, ', ', first_name) as name FROM students WHERE class = ? ORDER BY last_name, first_name");
    $stmt->execute([$selected_class]);
    $students = $stmt->fetchAll();
}

// Get mock exam data
$mock_data = [];
$progress_data = [];
if ($selected_student && $selected_class) {
    $stmt = $pdo->prepare("SELECT id FROM classes WHERE name = ?");
    $stmt->execute([$selected_class]);
    $class_info = $stmt->fetch();
    
    if ($class_info) {
        $stmt = $pdo->prepare("
            SELECT c.name as course_name, me.mock_exam_number, me.score, me.grade
            FROM mock_exams me
            JOIN enrollments e ON me.enrollment_id = e.id
            JOIN courses c ON e.course_id = c.id
            WHERE e.student_id = ?
            ORDER BY c.name, me.mock_exam_number
        ");
        $stmt->execute([$selected_student]);
        $results = $stmt->fetchAll();
        
        foreach ($results as $result) {
            $course = $result['course_name'];
            if (!isset($mock_data[$course])) {
                $mock_data[$course] = array_fill(1, 10, null);
            }
            $mock_data[$course][$result['mock_exam_number']] = $result['score'];
        }
        
        // Calculate progress (average improvement)
        foreach ($mock_data as $course => $scores) {
            $valid_scores = array_filter($scores, fn($s) => $s !== null);
            if (count($valid_scores) >= 2) {
                $first = reset($valid_scores);
                $last = end($valid_scores);
                $progress_data[$course] = $last - $first;
            }
        }
    }
}

// Class-wide statistics
$class_stats = [];
if ($selected_class) {
    $stmt = $pdo->prepare("
        SELECT me.mock_exam_number, AVG(me.score) as avg_score, COUNT(DISTINCT e.student_id) as student_count
        FROM mock_exams me
        JOIN enrollments e ON me.enrollment_id = e.id
        JOIN students s ON e.student_id = s.id
        WHERE s.class = ?
        GROUP BY me.mock_exam_number
        ORDER BY me.mock_exam_number
    ");
    $stmt->execute([$selected_class]);
    $class_stats = $stmt->fetchAll();
}

$role = $_SESSION['role'];
if ($role == 'admin') {
    include '../includes/admin_header.php';
} else {
    include '../includes/teacher_header.php';
}
?>

<style>
    .analytics-container {
        padding: 20px;
        background: #f8f9fa;
        min-height: 100vh;
    }

    .page-header {
        background: linear-gradient(135deg, #9b59b6 0%, #8e44ad 100%);
        color: white;
        padding: 30px;
        border-radius: 15px;
        margin-bottom: 30px;
        text-align: center;
        box-shadow: 0 10px 30px rgba(0,0,0,0.1);
    }

    .page-header h2 {
        margin: 0;
        font-size: 2.5em;
        font-weight: 700;
    }

    .filter-section {
        background: white;
        padding: 25px;
        border-radius: 15px;
        box-shadow: 0 5px 15px rgba(0,0,0,0.08);
        margin-bottom: 30px;
    }

    .filter-form {
        display: flex;
        gap: 20px;
        flex-wrap: wrap;
        align-items: end;
    }

    .filter-group {
        flex: 1;
        min-width: 200px;
    }

    .filter-group label {
        display: block;
        font-weight: 600;
        color: #333;
        margin-bottom: 8px;
    }

    .filter-group select {
        width: 100%;
        padding: 10px;
        border: 2px solid #ddd;
        border-radius: 8px;
        font-size: 1em;
    }

    .filter-btn {
        padding: 12px 24px;
        background: linear-gradient(135deg, #9b59b6 0%, #8e44ad 100%);
        color: white;
        border: none;
        border-radius: 8px;
        font-weight: 600;
        cursor: pointer;
    }

    .chart-container {
        background: white;
        padding: 25px;
        border-radius: 15px;
        box-shadow: 0 5px 15px rgba(0,0,0,0.08);
        margin-bottom: 30px;
    }

    .chart-container h3 {
        color: #2c3e50;
        margin-bottom: 20px;
        font-size: 1.5em;
    }

    .no-data {
        text-align: center;
        padding: 60px;
        color: #666;
    }

    canvas {
        max-height: 400px;
    }
</style>

<div class="analytics-container">
    <div class="page-header">
        <h2>📊 Mock Exam Analytics</h2>
        <p>Track student progress across mock examinations</p>
    </div>

    <div class="filter-section">
        <form method="GET" class="filter-form">
            <div class="filter-group">
                <label>Class:</label>
                <select name="class" required onchange="this.form.submit()">
                    <option value="">Select Class</option>
                    <?php foreach ($classes as $class): ?>
                        <option value="<?php echo htmlspecialchars($class['name']); ?>" <?php echo $selected_class == $class['name'] ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($class['name']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <?php if ($selected_class): ?>
                <div class="filter-group">
                    <label>Student:</label>
                    <select name="student_id" onchange="this.form.submit()">
                        <option value="">Select Student</option>
                        <?php foreach ($students as $student): ?>
                            <option value="<?php echo $student['id']; ?>" <?php echo $selected_student == $student['id'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($student['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
            <?php endif; ?>
        </form>
    </div>

    <?php if ($selected_class && !empty($class_stats)): ?>
        <div class="chart-container">
            <h3>Class Average Progress</h3>
            <canvas id="classProgressChart"></canvas>
        </div>
    <?php endif; ?>

    <?php if ($selected_student && !empty($mock_data)): ?>
        <div class="chart-container">
            <h3>Student Performance by Course</h3>
            <canvas id="studentProgressChart"></canvas>
        </div>

        <?php if (!empty($progress_data)): ?>
            <div class="chart-container">
                <h3>Improvement Analysis</h3>
                <canvas id="improvementChart"></canvas>
            </div>
        <?php endif; ?>
    <?php elseif ($selected_class): ?>
        <div class="no-data">
            <h3>Select a student to view detailed analytics</h3>
        </div>
    <?php else: ?>
        <div class="no-data">
            <h3>Select a class to begin</h3>
        </div>
    <?php endif; ?>
</div>

<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
<?php if ($selected_class && !empty($class_stats)): ?>
// Class Progress Chart
const classData = <?php echo json_encode($class_stats); ?>;
new Chart(document.getElementById('classProgressChart'), {
    type: 'line',
    data: {
        labels: classData.map(d => 'Mock ' + d.mock_exam_number),
        datasets: [{
            label: 'Class Average Score',
            data: classData.map(d => parseFloat(d.avg_score)),
            borderColor: 'rgba(155, 89, 182, 1)',
            backgroundColor: 'rgba(155, 89, 182, 0.1)',
            borderWidth: 3,
            tension: 0.4,
            fill: true
        }]
    },
    options: {
        responsive: true,
        scales: {
            y: {
                beginAtZero: true,
                max: 100
            }
        }
    }
});
<?php endif; ?>

<?php if ($selected_student && !empty($mock_data)): ?>
// Student Progress Chart
const mockData = <?php echo json_encode($mock_data); ?>;
const colors = ['#667eea', '#f093fb', '#4facfe', '#43e97b', '#fa709a', '#fee140', '#30cfd0', '#a8edea', '#fed6e3', '#c471ed'];
const datasets = Object.keys(mockData).map((course, index) => ({
    label: course,
    data: Object.values(mockData[course]),
    borderColor: colors[index % colors.length],
    backgroundColor: colors[index % colors.length] + '20',
    borderWidth: 2,
    tension: 0.4
}));

new Chart(document.getElementById('studentProgressChart'), {
    type: 'line',
    data: {
        labels: ['Mock 1', 'Mock 2', 'Mock 3', 'Mock 4', 'Mock 5', 'Mock 6', 'Mock 7', 'Mock 8', 'Mock 9', 'Mock 10'],
        datasets: datasets
    },
    options: {
        responsive: true,
        scales: {
            y: {
                beginAtZero: true,
                max: 100
            }
        }
    }
});

<?php if (!empty($progress_data)): ?>
// Improvement Chart
const progressData = <?php echo json_encode($progress_data); ?>;
new Chart(document.getElementById('improvementChart'), {
    type: 'bar',
    data: {
        labels: Object.keys(progressData),
        datasets: [{
            label: 'Score Change (First to Last Mock)',
            data: Object.values(progressData),
            backgroundColor: Object.values(progressData).map(v => v >= 0 ? 'rgba(67, 233, 123, 0.6)' : 'rgba(250, 112, 154, 0.6)'),
            borderColor: Object.values(progressData).map(v => v >= 0 ? 'rgba(67, 233, 123, 1)' : 'rgba(250, 112, 154, 1)'),
            borderWidth: 2
        }]
    },
    options: {
        responsive: true,
        scales: {
            y: {
                beginAtZero: true
            }
        }
    }
});
<?php endif; ?>
<?php endif; ?>
</script>

<?php
if ($role == 'admin') {
    include '../includes/admin_footer.php';
} else {
    include '../includes/teacher_footer.php';
}
?>
