<?php
require_once '../config.php';

if (!isset($_SESSION['user_id']) || $_SESSION['role'] != 'teacher') {
    header('Location: ../login.php');
    exit;
}

$user_id = $_SESSION['user_id'];

// Fetch teacher information
$stmt = $pdo->prepare("SELECT *, CONCAT(first_name, ' ', last_name) as name FROM staff WHERE user_id = ? AND role = 'teaching'");
$stmt->execute([$user_id]);
$teacher = $stmt->fetch();

if (!$teacher) {
    die("Teacher not found.");
}

$teacher_id = $teacher['id'];

// Handle leave application submission
$message = '';
$message_type = '';

if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['submit_leave'])) {
    $leave_type = $_POST['leave_type'];
    $start_date = $_POST['start_date'];
    $end_date = $_POST['end_date'];
    $reason = $_POST['reason'];
    $emergency_contact = $_POST['emergency_contact'];

    // Validate input
    if (empty($leave_type) || empty($start_date) || empty($end_date) || empty($reason)) {
        $message = 'All fields are required.';
        $message_type = 'error';
    } elseif (strtotime($end_date) < strtotime($start_date)) {
        $message = 'End date cannot be earlier than start date.';
        $message_type = 'error';
    } else {
        // Insert leave application
        $insert_stmt = $pdo->prepare("
            INSERT INTO staff_leave_applications
            (user_id, staff_role, leave_type, start_date, end_date, reason, emergency_contact, status, created_at)
            VALUES (?, 'teaching', ?, ?, ?, ?, ?, 'pending', NOW())
        ");

        if ($insert_stmt->execute([$user_id, $leave_type, $start_date, $end_date, $reason, $emergency_contact])) {
            $message = 'Leave application submitted successfully!';
            $message_type = 'success';
        } else {
            $message = 'Failed to submit leave application. Please try again.';
            $message_type = 'error';
        }
    }
}

try {
    $table_check_stmt = $pdo->prepare("SHOW TABLES LIKE 'staff_leave_applications'");
    $table_check_stmt->execute();
    $table_exists = $table_check_stmt->rowCount() > 0;

    if ($table_exists) {
        // Fetch leave history
        $leave_history_stmt = $pdo->prepare("
            SELECT *, DATEDIFF(end_date, start_date) + 1 as duration_days
            FROM staff_leave_applications
            WHERE user_id = ? AND staff_role = 'teaching'
            ORDER BY created_at DESC
        ");
        $leave_history_stmt->execute([$user_id]);
        $leave_history = $leave_history_stmt->fetchAll();
    } else {
        $leave_history = [];
    }
} catch (PDOException $e) {
    $table_exists = false;
    $leave_history = [];
}

include '../includes/teacher_header.php';
?>

<div class="leave-container">
    <div class="leave-header">
        <h1>📋 Leave Application</h1>
        <p class="header-subtitle">Manage your leave requests and view application history</p>
    </div>

    <!-- Apply Leave Button -->
    <div class="apply-leave-section">
        <button id="applyLeaveBtn" class="apply-leave-btn">
            <span class="btn-icon">➕</span>
            Apply for Leave
        </button>
    </div>

    <!-- Leave History -->
    <div class="leave-history-section">
        <div class="section-header">
            <h2>📜 Leave History</h2>
            <div class="history-stats">
                <span class="stat-item">
                    <strong>Total Applications:</strong> <?php echo count($leave_history); ?>
                </span>
                <span class="stat-item">
                    <strong>Pending:</strong>
                    <?php
                    $pending_count = 0;
                    foreach ($leave_history as $leave) {
                        if ($leave['status'] == 'pending') $pending_count++;
                    }
                    echo $pending_count;
                    ?>
                </span>
            </div>
        </div>

        <?php if (!$table_exists): ?>
            <div class="setup-required">
                <div class="setup-icon">⚙️</div>
                <h3>Database Setup Required</h3>
                <p>The leave application system needs to be set up first. Click the button below to create the necessary database tables.</p>
        <button onclick="window.location.href='../create_staff_leave_table.php'" class="setup-btn">
            <span class="btn-icon">🔧</span>
            Setup Database
        </button>
            </div>
        <?php elseif (empty($leave_history)): ?>
            <div class="no-history">
                <div class="no-history-icon">📅</div>
                <h3>No Leave Applications Yet</h3>
                <p>You haven't submitted any leave applications. Click the button above to apply for leave.</p>
            </div>
        <?php else: ?>
            <div class="history-grid">
                <?php foreach ($leave_history as $leave): ?>
                    <div class="leave-card <?php echo 'status-' . $leave['status']; ?>">
                        <div class="leave-card-header">
                            <div class="leave-type"><?php echo htmlspecialchars($leave['leave_type']); ?></div>
                            <div class="leave-status status-<?php echo $leave['status']; ?>">
                                <?php echo ucfirst($leave['status']); ?>
                            </div>
                        </div>

                        <div class="leave-card-body">
                            <div class="leave-dates">
                                <div class="date-item">
                                    <span class="date-label">From:</span>
                                    <span class="date-value"><?php echo date('M d, Y', strtotime($leave['start_date'])); ?></span>
                                </div>
                                <div class="date-item">
                                    <span class="date-label">To:</span>
                                    <span class="date-value"><?php echo date('M d, Y', strtotime($leave['end_date'])); ?></span>
                                </div>
                                <div class="date-item">
                                    <span class="date-label">Duration:</span>
                                    <span class="date-value"><?php echo $leave['duration_days']; ?> day(s)</span>
                                </div>
                            </div>

                            <div class="leave-reason">
                                <strong>Reason:</strong>
                                <p><?php echo nl2br(htmlspecialchars($leave['reason'])); ?></p>
                            </div>

                            <?php if (!empty($leave['emergency_contact'])): ?>
                                <div class="emergency-contact">
                                    <strong>Emergency Contact:</strong>
                                    <span><?php echo htmlspecialchars($leave['emergency_contact']); ?></span>
                                </div>
                            <?php endif; ?>

                            <?php if (($leave['status'] == 'approved' || $leave['status'] == 'rejected') && !empty($leave['admin_comments'])): ?>
                                <div class="admin-comments">
                                    <strong>Admin Comments:</strong>
                                    <p><?php echo nl2br(htmlspecialchars($leave['admin_comments'])); ?></p>
                                </div>
                            <?php endif; ?>

                            <div class="leave-meta">
                                <small>Applied on: <?php echo date('M d, Y \a\t h:i A', strtotime($leave['created_at'])); ?></small>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- Leave Application Modal -->
<div id="leaveModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h2>📝 Apply for Leave</h2>
            <span class="close-modal" id="closeModal">&times;</span>
        </div>

        <?php if ($message): ?>
            <div class="message <?php echo $message_type; ?>">
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>

        <form method="POST" action="leave_application.php" class="leave-form">
            <div class="form-row">
                <div class="form-group">
                    <label for="leave_type">Leave Type *</label>
                    <select id="leave_type" name="leave_type" required>
                        <option value="">Select Leave Type</option>
                        <option value="sick">Sick Leave</option>
                        <option value="casual">Casual Leave</option>
                        <option value="maternity">Maternity Leave</option>
                        <option value="paternity">Paternity Leave</option>
                        <option value="emergency">Emergency Leave</option>
                        <option value="vacation">Vacation Leave</option>
                        <option value="other">Other</option>
                    </select>
                </div>
            </div>

            <div class="form-row">
                <div class="form-group">
                    <label for="start_date">Start Date *</label>
                    <input type="date" id="start_date" name="start_date" required min="<?php echo date('Y-m-d'); ?>">
                </div>
                <div class="form-group">
                    <label for="end_date">End Date *</label>
                    <input type="date" id="end_date" name="end_date" required min="<?php echo date('Y-m-d'); ?>">
                </div>
            </div>

            <div class="form-group">
                <label for="reason">Reason *</label>
                <textarea id="reason" name="reason" rows="4" placeholder="Please provide detailed reason for your leave request..." required></textarea>
            </div>

            <div class="form-group">
                <label for="emergency_contact">Emergency Contact</label>
                <input type="text" id="emergency_contact" name="emergency_contact" placeholder="Phone number or email for emergency contact">
            </div>

            <div class="form-actions">
                <button type="button" id="cancelBtn" class="cancel-btn">Cancel</button>
                <button type="submit" name="submit_leave" class="submit-btn">Submit Application</button>
            </div>
        </form>
    </div>
</div>

<style>
/* Main Container */
.leave-container {
    max-width: 1200px;
    margin: 30px auto;
    padding: 30px;
    background: linear-gradient(135deg, #3498db 0%, #2980b9 100%);
    border-radius: 20px;
    box-shadow: 0 20px 40px rgba(0,0,0,0.1);
    color: white;
}

/* Header */
.leave-header {
    text-align: center;
    margin-bottom: 40px;
    padding-bottom: 30px;
    border-bottom: 2px solid rgba(255,255,255,0.2);
}

.leave-header h1 {
    margin: 0 0 10px 0;
    font-size: 2.5em;
    font-weight: 700;
    text-shadow: 2px 2px 4px rgba(0,0,0,0.3);
}

.header-subtitle {
    margin: 0;
    font-size: 1.2em;
    opacity: 0.9;
    font-weight: 300;
}

/* Apply Leave Section */
.apply-leave-section {
    display: flex;
    justify-content: center;
    margin-bottom: 40px;
}

.apply-leave-btn {
    padding: 15px 30px;
    background: linear-gradient(135deg, #27ae60, #2ecc71);
    color: white;
    border: none;
    border-radius: 15px;
    font-size: 1.1em;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    gap: 10px;
    box-shadow: 0 8px 25px rgba(39, 174, 96, 0.3);
}

.apply-leave-btn:hover {
    transform: translateY(-3px);
    box-shadow: 0 12px 35px rgba(39, 174, 96, 0.4);
}

.btn-icon {
    font-size: 1.2em;
}

/* Leave History Section */
.leave-history-section {
    background: rgba(255,255,255,0.95);
    border-radius: 15px;
    padding: 30px;
    box-shadow: 0 10px 30px rgba(0,0,0,0.1);
    backdrop-filter: blur(10px);
}

.section-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 25px;
    padding-bottom: 15px;
    border-bottom: 2px solid #e9ecef;
}

.section-header h2 {
    margin: 0;
    color: #2c3e50;
    font-size: 1.8em;
    font-weight: 600;
}

.history-stats {
    display: flex;
    gap: 20px;
}

.stat-item {
    color: #6c757d;
    font-size: 0.95em;
}

.stat-item strong {
    color: #495057;
}

/* Setup Required State */
.setup-required {
    text-align: center;
    padding: 60px 20px;
    background: linear-gradient(135deg, #fff3cd, #ffeaa7);
    border-radius: 12px;
    border: 2px solid #ffc107;
    margin: 20px 0;
}

.setup-icon {
    font-size: 4em;
    margin-bottom: 20px;
    opacity: 0.7;
}

.setup-required h3 {
    margin: 0 0 15px 0;
    color: #856404;
    font-size: 1.5em;
    font-weight: 600;
}

.setup-required p {
    margin: 0 0 25px 0;
    font-size: 1.1em;
    color: #856404;
    line-height: 1.6;
}

.setup-btn {
    padding: 15px 30px;
    background: linear-gradient(135deg, #ffc107, #fdcb6e);
    color: #212529;
    border: none;
    border-radius: 10px;
    font-size: 1.1em;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    display: inline-flex;
    align-items: center;
    gap: 10px;
    box-shadow: 0 4px 15px rgba(255, 193, 7, 0.3);
}

.setup-btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(255, 193, 7, 0.4);
    background: linear-gradient(135deg, #e0a800, #ffc107);
}

/* No History State */
.no-history {
    text-align: center;
    padding: 60px 20px;
    color: #6c757d;
}

.no-history-icon {
    font-size: 4em;
    margin-bottom: 20px;
    opacity: 0.5;
}

.no-history h3 {
    margin: 0 0 15px 0;
    color: #495057;
    font-size: 1.5em;
}

.no-history p {
    margin: 0;
    font-size: 1.1em;
}

/* History Grid */
.history-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
    gap: 20px;
}

/* Leave Cards */
.leave-card {
    background: #fff;
    border-radius: 12px;
    border-left: 5px solid;
    box-shadow: 0 4px 15px rgba(0,0,0,0.08);
    transition: transform 0.3s ease, box-shadow 0.3s ease;
    overflow: hidden;
}

.leave-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(0,0,0,0.12);
}

.leave-card.status-pending {
    border-left-color: #ffc107;
}

.leave-card.status-approved {
    border-left-color: #28a745;
}

.leave-card.status-rejected {
    border-left-color: #dc3545;
}

.leave-card-header {
    padding: 15px 20px;
    background: #f8f9fa;
    display: flex;
    justify-content: space-between;
    align-items: center;
    border-bottom: 1px solid #e9ecef;
}

.leave-type {
    font-weight: 600;
    color: #495057;
    font-size: 1.1em;
}

.leave-status {
    padding: 5px 12px;
    border-radius: 20px;
    font-size: 0.85em;
    font-weight: 600;
    text-transform: uppercase;
}

.leave-status.status-pending {
    background: #fff3cd;
    color: #856404;
}

.leave-status.status-approved {
    background: #d4edda;
    color: #155724;
}

.leave-status.status-rejected {
    background: #f8d7da;
    color: #721c24;
}

.leave-card-body {
    padding: 20px;
}

.leave-dates {
    margin-bottom: 15px;
}

.date-item {
    display: flex;
    justify-content: space-between;
    margin-bottom: 8px;
}

.date-label {
    font-weight: 600;
    color: #6c757d;
}

.date-value {
    color: #495057;
    font-weight: 500;
}

.leave-reason {
    margin-bottom: 15px;
}

.leave-reason strong {
    color: #495057;
    display: block;
    margin-bottom: 5px;
}

.leave-reason p {
    margin: 0;
    color: #6c757d;
    line-height: 1.5;
}

.emergency-contact {
    margin-bottom: 15px;
}

.emergency-contact strong {
    color: #495057;
    display: block;
    margin-bottom: 5px;
}

.emergency-contact span {
    color: #6c757d;
}

.admin-comments {
    margin-bottom: 15px;
}

.admin-comments strong {
    color: #495057;
    display: block;
    margin-bottom: 5px;
}

.admin-comments p {
    margin: 0;
    color: #6c757d;
    line-height: 1.5;
    font-style: italic;
}

.leave-meta {
    text-align: right;
    color: #adb5bd;
    font-size: 0.85em;
}

/* Modal Styles */
.modal {
    display: none;
    position: fixed;
    z-index: 1000;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0,0,0,0.5);
    backdrop-filter: blur(5px);
}

.modal.show {
    display: flex;
    align-items: center;
    justify-content: center;
}

.modal-content {
    background: #fff;
    border-radius: 15px;
    width: 90%;
    max-width: 600px;
    max-height: 90vh;
    overflow-y: auto;
    box-shadow: 0 20px 60px rgba(0,0,0,0.3);
    position: relative;
}

.modal-header {
    padding: 25px 30px;
    border-bottom: 2px solid #e9ecef;
    display: flex;
    justify-content: space-between;
    align-items: center;
    background: linear-gradient(135deg, #3498db, #2980b9);
    color: white;
    border-radius: 15px 15px 0 0;
}

.modal-header h2 {
    margin: 0;
    font-size: 1.8em;
    font-weight: 600;
}

.close-modal {
    font-size: 28px;
    font-weight: bold;
    cursor: pointer;
    color: white;
    opacity: 0.8;
    transition: opacity 0.3s ease;
}

.close-modal:hover {
    opacity: 1;
}

/* Messages */
.message {
    padding: 15px 20px;
    border-radius: 8px;
    margin: 20px 30px;
    font-weight: 500;
    border-left: 4px solid;
}

.message.success {
    background-color: #d4edda;
    color: #155724;
    border-left-color: #28a745;
}

.message.error {
    background-color: #f8d7da;
    color: #721c24;
    border-left-color: #dc3545;
}

/* Leave Form */
.leave-form {
    padding: 30px;
}

.form-row {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 20px;
    margin-bottom: 20px;
}

.form-group {
    display: flex;
    flex-direction: column;
}

.form-group label {
    font-weight: 600;
    color: #495057;
    margin-bottom: 8px;
    font-size: 1em;
}

.form-group input,
.form-group select,
.form-group textarea {
    padding: 12px 15px;
    border: 2px solid #e9ecef;
    border-radius: 8px;
    font-size: 1em;
    transition: all 0.3s ease;
    background: #f8f9fa;
}

.form-group input:focus,
.form-group select:focus,
.form-group textarea:focus {
    outline: none;
    border-color: #3498db;
    box-shadow: 0 0 0 3px rgba(52, 152, 219, 0.1);
    background: #fff;
}

.form-group textarea {
    resize: vertical;
    min-height: 100px;
}

.form-actions {
    display: flex;
    justify-content: flex-end;
    gap: 15px;
    margin-top: 30px;
    padding-top: 20px;
    border-top: 2px solid #e9ecef;
}

.cancel-btn {
    padding: 12px 25px;
    background: #6c757d;
    color: white;
    border: none;
    border-radius: 8px;
    font-size: 1em;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
}

.cancel-btn:hover {
    background: #5a6268;
    transform: translateY(-1px);
}

.submit-btn {
    padding: 12px 25px;
    background: linear-gradient(135deg, #3498db, #2980b9);
    color: white;
    border: none;
    border-radius: 8px;
    font-size: 1em;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    box-shadow: 0 4px 15px rgba(52, 152, 219, 0.3);
}

.submit-btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(52, 152, 219, 0.4);
}

/* Responsive Design */
@media (max-width: 768px) {
    .leave-container {
        margin: 20px;
        padding: 20px;
    }

    .leave-header h1 {
        font-size: 2em;
    }

    .section-header {
        flex-direction: column;
        align-items: flex-start;
        gap: 15px;
    }

    .history-stats {
        width: 100%;
        justify-content: space-between;
    }

    .history-grid {
        grid-template-columns: 1fr;
    }

    .form-row {
        grid-template-columns: 1fr;
        gap: 15px;
    }

    .form-actions {
        flex-direction: column;
    }

    .cancel-btn,
    .submit-btn {
        width: 100%;
    }

    .modal-content {
        width: 95%;
        margin: 20px;
    }

    .modal-header {
        padding: 20px;
    }

    .modal-header h2 {
        font-size: 1.5em;
    }

    .leave-form {
        padding: 20px;
    }
}

@media (max-width: 480px) {
    .leave-container {
        margin: 10px;
        padding: 15px;
    }

    .leave-header h1 {
        font-size: 1.8em;
    }

    .section-header h2 {
        font-size: 1.5em;
    }

    .leave-card-header,
    .leave-card-body {
        padding: 15px;
    }
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const applyLeaveBtn = document.getElementById('applyLeaveBtn');
    const leaveModal = document.getElementById('leaveModal');
    const closeModal = document.getElementById('closeModal');
    const cancelBtn = document.getElementById('cancelBtn');

    // Open modal
    applyLeaveBtn.addEventListener('click', function() {
        leaveModal.classList.add('show');
        document.body.style.overflow = 'hidden';
    });

    // Close modal functions
    function closeLeaveModal() {
        leaveModal.classList.remove('show');
        document.body.style.overflow = 'auto';
    }

    closeModal.addEventListener('click', closeLeaveModal);
    cancelBtn.addEventListener('click', closeLeaveModal);

    // Close modal when clicking outside
    leaveModal.addEventListener('click', function(e) {
        if (e.target === leaveModal) {
            closeLeaveModal();
        }
    });

    // Close modal with Escape key
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape' && leaveModal.classList.contains('show')) {
            closeLeaveModal();
        }
    });

    // Auto-calculate end date minimum based on start date
    const startDateInput = document.getElementById('start_date');
    const endDateInput = document.getElementById('end_date');

    startDateInput.addEventListener('change', function() {
        const startDate = new Date(this.value);
        const minEndDate = startDate.toISOString().split('T')[0];
        endDateInput.min = minEndDate;

        // If end date is before start date, clear it
        if (endDateInput.value && endDateInput.value < minEndDate) {
            endDateInput.value = '';
        }
    });
});
</script>

<?php
include '../includes/teacher_footer.php';
?>
