<?php
require_once '../config.php';

if (!in_array($_SESSION['role'], ['admin', 'teacher'])) {
    header('Location: ../login.php');
    exit;
}

$user_id = $_SESSION['user_id'];
// Try to get teacher ID - handle both user_id in staff table or direct id match
$stmt = $pdo->prepare("SELECT id FROM staff WHERE user_id = ? AND role = 'teaching'");
$stmt->execute([$user_id]);
$teacher = $stmt->fetch();

// If not found, try direct ID match (some systems use staff.id = users.id)
if (!$teacher) {
    $stmt = $pdo->prepare("SELECT id FROM staff WHERE id = ? AND role = 'teaching'");
    $stmt->execute([$user_id]);
    $teacher = $stmt->fetch();
}

if (!$teacher) {
    die("Teacher not found. User ID: " . $user_id);
}

$teacher_id = $teacher['id'];

// Get active academic year and term
$active_year = $pdo->query("SELECT id, year FROM academic_years WHERE is_active = 1 LIMIT 1")->fetch();
$active_term = $pdo->query("SELECT id, name FROM terms WHERE is_active = 1 LIMIT 1")->fetch();

// Total students taught
$stmt = $pdo->prepare("SELECT COUNT(DISTINCT e.student_id) as total FROM enrollments e JOIN teacher_courses tc ON e.course_id = tc.course_id WHERE tc.teacher_id = ?");
$stmt->execute([$teacher_id]);
$total_students = $stmt->fetch()['total'];

// Total courses
$stmt = $pdo->prepare("SELECT COUNT(*) as total FROM teacher_courses WHERE teacher_id = ?");
$stmt->execute([$teacher_id]);
$total_courses = $stmt->fetch()['total'];

// Students with grades this term
$stmt = $pdo->prepare("SELECT COUNT(DISTINCT e.student_id) as total FROM enrollments e JOIN teacher_courses tc ON e.course_id = tc.course_id JOIN grades g ON e.id = g.enrollment_id WHERE tc.teacher_id = ? AND g.term_id = ? AND g.academic_year_id = ?");
$stmt->execute([$teacher_id, $active_term['id'] ?? 0, $active_year['id'] ?? 0]);
$graded_students = $stmt->fetch()['total'];

// Average class performance
$stmt = $pdo->prepare("SELECT AVG(g.final_score) as avg_score FROM enrollments e JOIN teacher_courses tc ON e.course_id = tc.course_id JOIN grades g ON e.id = g.enrollment_id WHERE tc.teacher_id = ? AND g.term_id = ? AND g.academic_year_id = ?");
$stmt->execute([$teacher_id, $active_term['id'] ?? 0, $active_year['id'] ?? 0]);
$avg_performance = round($stmt->fetch()['avg_score'] ?? 0, 2);

// Grade distribution
$stmt = $pdo->prepare("SELECT g.grade, COUNT(*) as count FROM enrollments e JOIN teacher_courses tc ON e.course_id = tc.course_id JOIN grades g ON e.id = g.enrollment_id WHERE tc.teacher_id = ? AND g.term_id = ? AND g.academic_year_id = ? GROUP BY g.grade");
$stmt->execute([$teacher_id, $active_term['id'] ?? 0, $active_year['id'] ?? 0]);
$grade_distribution = $stmt->fetchAll();

// Performance by course
$stmt = $pdo->prepare("SELECT c.name, AVG(g.final_score) as avg_score, COUNT(DISTINCT e.student_id) as student_count FROM enrollments e JOIN courses c ON e.course_id = c.id JOIN teacher_courses tc ON c.id = tc.course_id JOIN grades g ON e.id = g.enrollment_id WHERE tc.teacher_id = ? AND g.term_id = ? AND g.academic_year_id = ? GROUP BY c.id ORDER BY avg_score DESC LIMIT 10");
$stmt->execute([$teacher_id, $active_term['id'] ?? 0, $active_year['id'] ?? 0]);
$course_performance = $stmt->fetchAll();

// Attendance rate
$stmt = $pdo->prepare("SELECT COUNT(CASE WHEN status = 'present' THEN 1 END) * 100.0 / COUNT(*) as rate FROM attendance a JOIN students s ON a.student_id = s.id JOIN enrollments e ON s.id = e.student_id JOIN teacher_courses tc ON e.course_id = tc.course_id WHERE tc.teacher_id = ?");
$stmt->execute([$teacher_id]);
$attendance_rate = round($stmt->fetch()['rate'] ?? 0, 2);

$role = $_SESSION['role'];
if ($role == 'admin') {
    include '../includes/admin_header.php';
} else {
    include '../includes/teacher_header.php';
}
?>

<style>
    .analytics-container {
        padding: 20px;
        background: #f8f9fa;
        min-height: 100vh;
    }

    .page-header {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        padding: 30px;
        border-radius: 15px;
        margin-bottom: 30px;
        text-align: center;
        box-shadow: 0 10px 30px rgba(0,0,0,0.1);
    }

    .page-header h2 {
        margin: 0;
        font-size: 2.5em;
        font-weight: 700;
    }

    .stats-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 20px;
        margin-bottom: 30px;
    }

    .stat-card {
        background: white;
        padding: 25px;
        border-radius: 15px;
        box-shadow: 0 5px 15px rgba(0,0,0,0.08);
        text-align: center;
    }

    .stat-card .icon {
        font-size: 3em;
        margin-bottom: 10px;
    }

    .stat-card .value {
        font-size: 2.5em;
        font-weight: 700;
        color: #667eea;
        margin: 10px 0;
    }

    .stat-card .label {
        font-size: 1.1em;
        color: #666;
        font-weight: 500;
    }

    .chart-container {
        background: white;
        padding: 25px;
        border-radius: 15px;
        box-shadow: 0 5px 15px rgba(0,0,0,0.08);
        margin-bottom: 30px;
    }

    .chart-container h3 {
        color: #2c3e50;
        margin-bottom: 20px;
        font-size: 1.5em;
    }

    canvas {
        max-height: 400px;
    }
</style>

<div class="analytics-container">
    <div class="page-header">
        <h2>📊 Dashboard Analytics</h2>
        <p><?php echo htmlspecialchars($active_year['year'] ?? 'N/A'); ?> - <?php echo htmlspecialchars($active_term['name'] ?? 'N/A'); ?></p>
    </div>

    <div class="stats-grid">
        <div class="stat-card">
            <div class="icon">👨‍🎓</div>
            <div class="value"><?php echo $total_students; ?></div>
            <div class="label">Total Students</div>
        </div>
        <div class="stat-card">
            <div class="icon">📚</div>
            <div class="value"><?php echo $total_courses; ?></div>
            <div class="label">Courses Teaching</div>
        </div>
        <div class="stat-card">
            <div class="icon">✅</div>
            <div class="value"><?php echo $graded_students; ?></div>
            <div class="label">Students Graded</div>
        </div>
        <div class="stat-card">
            <div class="icon">📈</div>
            <div class="value"><?php echo $avg_performance; ?>%</div>
            <div class="label">Average Performance</div>
        </div>
        <div class="stat-card">
            <div class="icon">📅</div>
            <div class="value"><?php echo $attendance_rate; ?>%</div>
            <div class="label">Attendance Rate</div>
        </div>
    </div>

    <div class="chart-container">
        <h3>Grade Distribution</h3>
        <canvas id="gradeChart"></canvas>
    </div>

    <div class="chart-container">
        <h3>Performance by Course</h3>
        <canvas id="courseChart"></canvas>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
// Grade Distribution Chart
const gradeData = <?php echo json_encode($grade_distribution); ?>;
const gradeLabels = gradeData.map(item => item.grade || 'N/A');
const gradeCounts = gradeData.map(item => item.count);

new Chart(document.getElementById('gradeChart'), {
    type: 'bar',
    data: {
        labels: gradeLabels,
        datasets: [{
            label: 'Number of Students',
            data: gradeCounts,
            backgroundColor: 'rgba(102, 126, 234, 0.6)',
            borderColor: 'rgba(102, 126, 234, 1)',
            borderWidth: 2
        }]
    },
    options: {
        responsive: true,
        maintainAspectRatio: true,
        scales: {
            y: {
                beginAtZero: true,
                ticks: {
                    stepSize: 1
                }
            }
        }
    }
});

// Course Performance Chart
const courseData = <?php echo json_encode($course_performance); ?>;
const courseLabels = courseData.map(item => item.name);
const courseScores = courseData.map(item => parseFloat(item.avg_score));

new Chart(document.getElementById('courseChart'), {
    type: 'bar',
    data: {
        labels: courseLabels,
        datasets: [{
            label: 'Average Score (%)',
            data: courseScores,
            backgroundColor: 'rgba(155, 89, 182, 0.6)',
            borderColor: 'rgba(155, 89, 182, 1)',
            borderWidth: 2
        }]
    },
    options: {
        indexAxis: 'y',
        responsive: true,
        maintainAspectRatio: true,
        scales: {
            x: {
                beginAtZero: true,
                max: 100
            }
        }
    }
});
</script>

<?php
if ($role == 'admin') {
    include '../includes/admin_footer.php';
} else {
    include '../includes/teacher_footer.php';
}
?>
