<?php
require_once '../config.php';

if (!in_array($_SESSION['role'], ['admin', 'teacher'])) {
    header('Location: ../dashboard.php');
    exit;
}

$message = '';

// Get parameters
$student_id = $_GET['student_id'] ?? null;
$class_name = $_GET['class'] ?? null;
$academic_year_id = $_GET['academic_year'] ?? null;
$term_id = $_GET['term'] ?? null;

if (!$student_id || !$class_name || !$academic_year_id || !$term_id) {
    die("Missing required parameters.");
}

// Get student name
$stmt = $pdo->prepare("SELECT CONCAT(last_name, ', ', first_name, ' ', COALESCE(middle_name, '')) AS name FROM students WHERE id = ?");
$stmt->execute([$student_id]);
$student = $stmt->fetch();
if (!$student) {
    die("Student not found.");
}
$student_name = trim($student['name']);

// Get class info
$stmt = $pdo->prepare("SELECT id, level_id FROM classes WHERE name = ?");
$stmt->execute([$class_name]);
$class_info = $stmt->fetch();

if (!$class_info) {
    die("Class not found.");
}

$class_id = $class_info['id'];
$class_level_id = $class_info['level_id'];

// Load grading system for final year (SHS level, assuming level_id = 5)
$grading_system = $pdo->prepare("SELECT grade, min_score, max_score, description FROM grading_systems WHERE level_id = ? ORDER BY min_score DESC");
$grading_system->execute([5]);
$grading_rules = $grading_system->fetchAll(PDO::FETCH_ASSOC);

// Get enrollments for this student in the term and academic year, only for courses assigned to the class
$stmt = $pdo->prepare("
    SELECT e.id AS enrollment_id, c.id AS course_id, c.code, c.name AS course_name
    FROM enrollments e
    JOIN courses c ON e.course_id = c.id
    JOIN class_courses cc ON cc.course_id = c.id AND cc.class_id = ?
    WHERE e.student_id = ? AND e.academic_year_id = ? AND e.term_id = ?
    ORDER BY c.name
");
$stmt->execute([$class_id, $student_id, $academic_year_id, $term_id]);
$enrollments = $stmt->fetchAll();

if (empty($enrollments)) {
    die("No enrollments found for this student, academic year, and term.");
}

// Get existing final year grades for each enrollment
$existing_grades = [];
foreach ($enrollments as $enrollment) {
    $stmt = $pdo->prepare("SELECT mock_exam_score, grade, comments FROM final_year_grades WHERE enrollment_id = ? AND term_id = ? AND academic_year_id = ?");
    $stmt->execute([$enrollment['enrollment_id'], $term_id, $academic_year_id]);
    $grade = $stmt->fetch();
    $existing_grades[$enrollment['enrollment_id']] = $grade ?: ['mock_exam_score' => '0', 'grade' => 'F', 'comments' => 'Fail'];
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $enrollment_ids = $_POST['enrollment_id'] ?? [];
    $mock_exam_scores = $_POST['mock_exam_score'] ?? [];
    $grades = $_POST['grade'] ?? [];
    $comments_list = $_POST['comments'] ?? [];

    $saved_count = 0;
    $warnings = [];

    foreach ($enrollment_ids as $index => $enrollment_id) {
        $mock_exam_score = $mock_exam_scores[$index] ?? null;
        $grade = $grades[$index] ?? '';
        $comments = $comments_list[$index] ?? '';

        // Auto-assign grade and comments based on score if not manually selected
        if ($mock_exam_score !== null && empty($grade)) {
            foreach ($grading_rules as $rule) {
                if ($mock_exam_score >= $rule['min_score'] && $mock_exam_score <= $rule['max_score']) {
                    $grade = $rule['grade'];
                    $comments = $rule['description'] ?: '';
                    break;
                }
            }
        } elseif ($mock_exam_score !== null && !empty($grade)) {
            // Validate that the selected grade matches the score range
            $valid = false;
            foreach ($grading_rules as $rule) {
                if ($grade == $rule['grade'] && $mock_exam_score >= $rule['min_score'] && $mock_exam_score <= $rule['max_score']) {
                    $valid = true;
                    break;
                }
            }
            if (!$valid) {
                $course_name = '';
                foreach ($enrollments as $enrollment) {
                    if ($enrollment['enrollment_id'] == $enrollment_id) {
                        $course_name = $enrollment['course_name'];
                        break;
                    }
                }
                $warnings[] = "Warning for {$course_name}: The selected grade does not match the score range for this grading system.";
            }
        }

        // Check if grade already exists
        $stmt = $pdo->prepare("SELECT id FROM final_year_grades WHERE enrollment_id = ? AND term_id = ? AND academic_year_id = ?");
        $stmt->execute([$enrollment_id, $term_id, $academic_year_id]);
        $existing = $stmt->fetch();

        if ($existing) {
            // Update
            $stmt = $pdo->prepare("UPDATE final_year_grades SET mock_exam_score = ?, grade = ?, comments = ? WHERE enrollment_id = ? AND term_id = ? AND academic_year_id = ?");
            $stmt->execute([$mock_exam_score, $grade, $comments, $enrollment_id, $term_id, $academic_year_id]);
        } else {
            // Insert
            $stmt = $pdo->prepare("INSERT INTO final_year_grades (enrollment_id, mock_exam_score, grade, comments, term_id, academic_year_id) VALUES (?, ?, ?, ?, ?, ?)");
            $stmt->execute([$enrollment_id, $mock_exam_score, $grade, $comments, $term_id, $academic_year_id]);
        }
        $saved_count++;
    }

    $message = "Final year grades saved successfully for {$saved_count} courses.";
    if (!empty($warnings)) {
        $message .= "<br>" . implode("<br>", $warnings);
    }
}

$role = $_SESSION['role'];

if ($role == 'admin') {
    include '../includes/admin_header.php';
} else {
    include '../includes/teacher_header.php';
}
?>

<style>
    .add-final-grades-container {
        padding: 20px;
        background-color: #f8f9fa;
        min-height: 100vh;
    }

    .page-header {
        background: linear-gradient(135deg, #27ae60 0%, #2ecc71 100%);
        color: white;
        padding: 30px;
        border-radius: 15px;
        margin-bottom: 30px;
        text-align: center;
        box-shadow: 0 10px 30px rgba(0,0,0,0.1);
    }

    .page-header h2 {
        margin: 0;
        font-size: 2.5em;
        font-weight: 700;
    }

    .message {
        padding: 15px 20px;
        border-radius: 8px;
        margin-bottom: 20px;
        font-weight: 500;
        background: #d4edda;
        color: #155724;
        border: 1px solid #c3e6cb;
    }

    .warning {
        padding: 15px 20px;
        border-radius: 8px;
        margin-bottom: 20px;
        font-weight: 500;
        background: #fff3cd;
        color: #856404;
        border: 1px solid #ffeaa7;
    }

    .form-container {
        background: white;
        border-radius: 15px;
        padding: 25px;
        box-shadow: 0 5px 15px rgba(0,0,0,0.08);
        margin-bottom: 30px;
    }

    .form-container h3 {
        color: #2c3e50;
        margin-bottom: 20px;
        font-size: 1.8em;
    }

    .back-link {
        display: inline-block;
        padding: 10px 20px;
        background: #6c757d;
        color: white;
        text-decoration: none;
        border-radius: 8px;
        font-weight: 600;
        margin-bottom: 20px;
        transition: background-color 0.3s ease;
    }

    .back-link:hover {
        background: #5a6268;
        color: white;
        text-decoration: none;
    }

    .form-group {
        margin-bottom: 20px;
    }

    .form-group label {
        display: block;
        margin-bottom: 8px;
        font-weight: 600;
        color: #2c3e50;
    }

    .form-group input,
    .form-group textarea,
    .form-group select {
        width: 100%;
        padding: 12px 15px;
        border: 2px solid #ddd;
        border-radius: 8px;
        font-size: 1em;
        transition: border-color 0.3s ease;
        box-sizing: border-box;
        background-color: #fff;
    }

    .form-group input:focus,
    .form-group textarea:focus,
    .form-group select:focus {
        outline: none;
        border-color: #27ae60;
        box-shadow: 0 0 0 3px rgba(39, 174, 96, 0.1);
    }

    .form-group textarea {
        resize: vertical;
        min-height: 80px;
    }

    .grades-table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 20px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        border-radius: 8px;
        overflow: hidden;
    }

    .grades-table th,
    .grades-table td {
        padding: 12px 15px;
        text-align: left;
        border-bottom: 1px solid #ddd;
    }

    .grades-table th {
        background: linear-gradient(135deg, #27ae60 0%, #2ecc71 100%);
        color: white;
        font-weight: 600;
        font-size: 0.9em;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    .grades-table tr:nth-child(even) {
        background-color: #f8f9fa;
    }

    .grades-table tr:hover {
        background-color: #e8f5e8;
    }

    .grades-table .course-name {
        font-weight: 600;
        color: #2c3e50;
        min-width: 200px;
    }

    .grades-table input,
    .grades-table select,
    .grades-table textarea {
        width: 90px;
        min-width: 90px;
        padding: 8px 10px;
        border: 1px solid #ddd;
        border-radius: 4px;
        font-size: 0.9em;
        transition: border-color 0.3s ease;
        box-sizing: border-box;
    }

    .grades-table select#grade {
        width: 150px;
        min-width: 150px;
    }

    .grades-table input:focus,
    .grades-table select:focus,
    .grades-table textarea:focus {
        outline: none;
        border-color: #27ae60;
        box-shadow: 0 0 0 2px rgba(39, 174, 96, 0.1);
    }

    .grades-table input[readonly] {
        background-color: #f8f9fa;
        cursor: not-allowed;
    }

    .grades-table input.invalid {
        border-color: red;
        background-color: #ffe6e6;
    }

    .grades-table textarea {
        resize: vertical;
        min-height: 60px;
    }

    .grades-table textarea#comments {
        width: 200px;
        min-width: 200px;
    }

    .btn-primary {
        padding: 12px 24px;
        background: linear-gradient(135deg, #27ae60 0%, #2ecc71 100%);
        color: white;
        border: none;
        border-radius: 8px;
        font-size: 1em;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.3s ease;
    }

    .btn-primary:hover {
        transform: translateY(-2px);
        box-shadow: 0 5px 15px rgba(39, 174, 96, 0.3);
    }
</style>

<div class="add-final-grades-container">
    <div class="page-header">
        <h2>🎓 Add Final Year Grades</h2>
    </div>

    <?php if ($message): ?>
        <div class="message"><?php echo htmlspecialchars($message); ?></div>
    <?php endif; ?>

    <div class="form-container">
        <a href="final_year_grades.php?class=<?php echo urlencode($class_name); ?>&academic_year=<?php echo $academic_year_id; ?>&term=<?php echo $term_id; ?>" class="back-link">← Back to Students</a>

        <h3>Grades for <?php echo htmlspecialchars($student_name); ?> in <?php echo htmlspecialchars($class_name); ?></h3>

        <form method="POST">
            <table class="grades-table">
                <thead>
                    <tr>
                        <th>Course</th>
                        <th>Mock Exam Score (100 marks)</th>
                        <th>Grade</th>
                        <th>Comments</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($enrollments as $enrollment): ?>
                        <tr>
                            <td class="course-name">
                                <?php echo htmlspecialchars($enrollment['course_name']); ?>
                                <input type="hidden" name="enrollment_id[]" value="<?php echo $enrollment['enrollment_id']; ?>">
                            </td>
                            <td><input type="number" name="mock_exam_score[]" min="0" max="100" step="0.01" value="<?php echo htmlspecialchars($existing_grades[$enrollment['enrollment_id']]['mock_exam_score'] ?? ''); ?>" required></td>
                            <td><input type="text" name="grade[]" readonly value="<?php echo htmlspecialchars($existing_grades[$enrollment['enrollment_id']]['grade'] ?? ''); ?>" required></td>
                            <td><textarea name="comments[]" readonly placeholder="Auto-filled based on grade"><?php echo htmlspecialchars($existing_grades[$enrollment['enrollment_id']]['comments'] ?? ''); ?></textarea></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>

            <div style="margin-top: 20px; text-align: center;">
                <button type="submit" class="btn-primary">💾 Save Grades</button>
            </div>
        </form>
    </div>
</div>

<script>
    const gradingRules = <?php echo json_encode($grading_rules); ?>;

    document.addEventListener('DOMContentLoaded', function() {
        const scoreInputs = document.querySelectorAll('input[name="mock_exam_score[]"]');
        const gradeInputs = document.querySelectorAll('input[name="grade[]"]');
        const commentInputs = document.querySelectorAll('textarea[name="comments[]"]');

        scoreInputs.forEach((scoreInput, index) => {
            scoreInput.addEventListener('input', function() {
                const score = parseFloat(this.value);
                if (!isNaN(score)) {
                    // Find the appropriate grade
                    let assignedGrade = '';
                    let assignedComment = '';
                    for (const rule of gradingRules) {
                        if (score >= rule.min_score && score <= rule.max_score) {
                            assignedGrade = rule.grade;
                            assignedComment = rule.description || '';
                            break;
                        }
                    }
                    // Always update grade and comments based on score
                    if (gradeInputs[index]) {
                        gradeInputs[index].value = assignedGrade;
                    }
                    if (commentInputs[index]) {
                        commentInputs[index].value = assignedComment;
                    }
                }
            });
        });
    });
</script>

<?php
if ($role == 'admin') {
    include '../includes/admin_footer.php';
} else {
    include '../includes/teacher_footer.php';
}
?>
