<?php
require_once '../config.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: index.php');
    exit;
}

$user_id = $_SESSION['user_id'];
$username = $_SESSION['username'];

$message = '';
$error = '';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['add_package'])) {
        // Add new package
        $name = trim($_POST['name']);
        $code = trim($_POST['code']);
        $description = trim($_POST['description']);
        $max_students = $_POST['max_students'] ?: null;
        $max_teachers = $_POST['max_teachers'] ?: null;
        $price_ghs = $_POST['price_ghs'] ?: null;
        $features = isset($_POST['features']) ? array_fill_keys($_POST['features'], true) : [];

        if (empty($name) || empty($code)) {
            $error = 'Package name and code are required.';
        } else {
            try {
                $stmt = $pdo->prepare("INSERT INTO subscription_packages (name, code, description, max_students, max_teachers, features, price_ghs) VALUES (?, ?, ?, ?, ?, ?, ?)");
                $stmt->execute([$name, $code, $description, $max_students, $max_teachers, json_encode($features), $price_ghs]);
                $message = 'Package added successfully!';
            } catch (PDOException $e) {
                $error = 'Error adding package: ' . $e->getMessage();
            }
        }
    } elseif (isset($_POST['update_package'])) {
        // Update package
        $package_id = $_POST['package_id'];
        $name = trim($_POST['name']);
        $code = trim($_POST['code']);
        $description = trim($_POST['description']);
        $max_students = $_POST['max_students'] ?: null;
        $max_teachers = $_POST['max_teachers'] ?: null;
        $price_ghs = $_POST['price_ghs'] ?: null;
        $features = isset($_POST['features']) ? array_fill_keys($_POST['features'], true) : [];

        // Handle pricing details
        $pricing_details = [];
        if (!empty($_POST['one_time_license'])) {
            $pricing_details['one_time_license'] = $_POST['one_time_license'] === 'custom_quote' ? 'custom_quote' : (float)$_POST['one_time_license'];
        }
        if (!empty($_POST['annual_subscription'])) {
            $pricing_details['annual_subscription'] = (float)$_POST['annual_subscription'];
        }
        if (!empty($_POST['monthly_subscription'])) {
            $pricing_details['monthly_subscription'] = (float)$_POST['monthly_subscription'];
        }
        if (!empty($_POST['implementation_fee'])) {
            $pricing_details['implementation_fee'] = (float)$_POST['implementation_fee'];
        }
        if (!empty($_POST['training_support'])) {
            $pricing_details['training_support'] = (float)$_POST['training_support'];
        }

        if (!empty($pricing_details)) {
            $features['pricing_details'] = $pricing_details;
        }

        try {
            $stmt = $pdo->prepare("UPDATE subscription_packages SET name = ?, code = ?, description = ?, max_students = ?, max_teachers = ?, features = ?, price_ghs = ? WHERE id = ?");
            $stmt->execute([$name, $code, $description, $max_students, $max_teachers, json_encode($features), $price_ghs, $package_id]);
            $message = 'Package updated successfully!';
        } catch (PDOException $e) {
            $error = 'Error updating package: ' . $e->getMessage();
        }
    } elseif (isset($_POST['delete_package'])) {
        // Delete package
        $package_id = $_POST['package_id'];

        try {
            $stmt = $pdo->prepare("DELETE FROM subscription_packages WHERE id = ?");
            $stmt->execute([$package_id]);
            $message = 'Package deleted successfully!';
        } catch (PDOException $e) {
            $error = 'Error deleting package: ' . $e->getMessage();
        }
    }
}

// Get all packages
$stmt = $pdo->query("SELECT * FROM subscription_packages ORDER BY price_ghs ASC");
$packages = $stmt->fetchAll();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Packages - School Management System</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <link rel="stylesheet" href="../css/admin_custom.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            color: #333;
            background-color: #f4f4f4;
        }

        .header {
            background: linear-gradient(135deg, #8e44ad 0%, #9b59b6 100%);
            color: #fff;
            padding: 20px 0;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }

        .header .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .header .logo {
            font-size: 1.5em;
            font-weight: bold;
            display: flex;
            align-items: center;
        }

        .header .logo i {
            margin-right: 10px;
            font-size: 1.8em;
        }

        .header .user-info {
            display: flex;
            align-items: center;
            gap: 20px;
        }

        .header .user-info span {
            font-weight: 500;
        }

        .header .manage-btn {
            background: rgba(255,255,255,0.2);
            color: #fff;
            padding: 8px 16px;
            border-radius: 5px;
            text-decoration: none;
            font-weight: 500;
            transition: background 0.3s ease;
        }

        .header .manage-btn:hover {
            background: rgba(255,255,255,0.3);
        }

        .header .logout-btn {
            background: rgba(255,255,255,0.2);
            color: #fff;
            padding: 8px 16px;
            border-radius: 5px;
            text-decoration: none;
            font-weight: 500;
            transition: background 0.3s ease;
        }

        .header .logout-btn:hover {
            background: rgba(255,255,255,0.3);
        }

        .main-content {
            max-width: 1200px;
            margin: 30px auto;
            padding: 0 20px;
        }

        .package-form {
            background: #fff;
            padding: 20px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            margin-bottom: 30px;
        }

        .package-form h3 {
            margin-bottom: 20px;
            color: #8e44ad;
        }

        .form-row {
            display: flex;
            gap: 15px;
            margin-bottom: 15px;
        }

        .form-row .form-group {
            flex: 1;
        }

        .form-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: 500;
            color: #555;
        }

        .form-group input, .form-group textarea {
            width: 100%;
            padding: 8px 12px;
            border: 1px solid #ddd;
            border-radius: 4px;
            font-size: 14px;
        }

        .form-group textarea {
            resize: vertical;
            min-height: 80px;
        }

        .btn-submit {
            background: #8e44ad;
            color: #fff;
            padding: 10px 20px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 14px;
        }

        .btn-submit:hover {
            background: #7d3c98;
        }

        .packages-table {
            background: #fff;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            overflow: hidden;
        }

        .packages-table table {
            width: 100%;
            border-collapse: collapse;
        }

        .packages-table th, .packages-table td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #eee;
        }

        .packages-table th {
            background: #f8f9fa;
            font-weight: 600;
            color: #2c3e50;
        }

        .pricing-details {
            font-size: 12px;
            color: #666;
            line-height: 1.4;
        }

        .pricing-details strong {
            color: #2c3e50;
        }

        .action-btn {
            padding: 6px 12px;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            font-size: 12px;
            margin-right: 5px;
        }

        .btn-edit {
            background: #ffc107;
            color: #212529;
        }

        .btn-delete {
            background: #dc3545;
            color: #fff;
        }

        .message {
            padding: 10px;
            margin-bottom: 20px;
            border-radius: 4px;
        }

        .message.success {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }

        .message.error {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }

        .modal {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0,0,0,0.5);
        }

        .modal-content {
            background-color: #fff;
            margin: 15% auto;
            padding: 20px;
            border-radius: 8px;
            width: 80%;
            max-width: 500px;
        }

        .close {
            color: #aaa;
            float: right;
            font-size: 28px;
            font-weight: bold;
            cursor: pointer;
        }

        .close:hover {
            color: #000;
        }

        .features-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 10px;
        }

        .features-grid label {
            display: flex;
            align-items: center;
            margin: 0;
        }

        .features-grid input[type="checkbox"] {
            margin-right: 8px;
        }

        .feature-section {
            margin-bottom: 25px;
            padding: 15px;
            border: 1px solid #e1e8ed;
            border-radius: 8px;
            background: #fafbfc;
        }

        .feature-header {
            display: flex;
            align-items: center;
            margin-bottom: 15px;
            padding-bottom: 10px;
            border-bottom: 2px solid #8e44ad;
        }

        .feature-header i {
            color: #8e44ad;
            font-size: 18px;
            margin-right: 10px;
        }

        .feature-header span {
            font-weight: 600;
            color: #2c3e50;
            font-size: 16px;
        }

        .features-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
            gap: 12px;
        }

        .features-grid label {
            display: flex;
            align-items: center;
            padding: 8px 12px;
            background: #fff;
            border: 1px solid #e1e8ed;
            border-radius: 6px;
            transition: all 0.3s ease;
            cursor: pointer;
            font-size: 14px;
        }

        .features-grid label:hover {
            background: #f8f9fa;
            border-color: #8e44ad;
        }

        .features-grid label i {
            color: #6c757d;
            margin-right: 8px;
            width: 16px;
        }

        .features-grid input[type="checkbox"] {
            margin-right: 8px;
            accent-color: #8e44ad;
        }

        .features-grid input[type="checkbox"]:checked + i {
            color: #8e44ad;
        }

        @media (max-width: 768px) {
            .form-row {
                flex-direction: column;
            }

            .packages-table {
                overflow-x: auto;
            }

            .features-grid {
                grid-template-columns: 1fr;
            }

            .feature-section {
                margin-bottom: 20px;
                padding: 12px;
            }

            .feature-header {
                margin-bottom: 12px;
                padding-bottom: 8px;
            }
        }
    </style>
</head>
<body>
    <!-- Header -->
    <div class="header">
        <div class="container">
            <div class="logo">
                <i class="fas fa-crown"></i>
                <span>Subscription Dashboard</span>
            </div>
            <div class="user-info">
                <span>Welcome, <?php echo htmlspecialchars($username); ?></span>
                <a href="dashboard.php" class="manage-btn">
                    <i class="fas fa-tachometer-alt"></i> Dashboard
                </a>
                <a href="logout.php" class="logout-btn">
                    <i class="fas fa-sign-out-alt"></i> Logout
                </a>
            </div>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <div class="container" style="max-width: 1200px; margin: 0 auto;">
            <h2 style="color: #8e44ad; margin-bottom: 20px;"><i class="fas fa-box"></i> Manage Subscription Packages</h2>

            <?php if ($message): ?>
                <div class="message success"><?php echo htmlspecialchars($message); ?></div>
            <?php endif; ?>

            <?php if ($error): ?>
                <div class="message error"><?php echo htmlspecialchars($error); ?></div>
            <?php endif; ?>

            <!-- Add New Package Form -->
            <div class="package-form">
                <h3><i class="fas fa-plus"></i> Add New Package</h3>
                <form method="post" action="manage_packages.php">
                    <div class="form-row">
                        <div class="form-group">
                            <label for="name">Package Name *</label>
                            <input type="text" id="name" name="name" required>
                        </div>
                        <div class="form-group">
                            <label for="code">Package Code *</label>
                            <input type="text" id="code" name="code" required>
                        </div>
                    </div>
                    <div class="form-group">
                        <label for="description">Description</label>
                        <textarea id="description" name="description"></textarea>
                    </div>
                    <div class="form-row">
                        <div class="form-group">
                            <label for="max_students">Max Students</label>
                            <input type="number" id="max_students" name="max_students" placeholder="Unlimited">
                        </div>
                        <div class="form-group">
                            <label for="max_teachers">Max Teachers</label>
                            <input type="number" id="max_teachers" name="max_teachers" placeholder="Unlimited">
                        </div>
                        <div class="form-group">
                            <label for="price_ghs">Price (GHS/Year)</label>
                            <input type="number" step="0.01" id="price_ghs" name="price_ghs" placeholder="Custom pricing">
                        </div>
                    </div>

                    <div class="form-group">
                        <label style="font-weight: 600; color: #2c3e50; margin-bottom: 15px; display: block;">Package Features:</label>

                        <!-- Administrative Features -->
                        <div class="feature-section">
                            <div class="feature-header">
                                <i class="fas fa-cog"></i>
                                <span>Administrative Features</span>
                            </div>
                            <div class="features-grid">
                                <label><input type="checkbox" name="features[]" value="student_management"> <i class="fas fa-users"></i> Student Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="teacher_management"> <i class="fas fa-chalkboard-teacher"></i> Teacher Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="course_management"> <i class="fas fa-book"></i> Course Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="enrollment_management"> <i class="fas fa-user-plus"></i> Enrollment Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="class_management"> <i class="fas fa-school"></i> Class Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="subject_management"> <i class="fas fa-book-open"></i> Subject Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="fee_management"> <i class="fas fa-money-bill-wave"></i> Fee Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="payment_management"> <i class="fas fa-credit-card"></i> Payment Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="assign_fees"> <i class="fas fa-hand-holding-usd"></i> Assign Fees <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="fee_reports"> <i class="fas fa-chart-bar"></i> Fee Reports <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="term_management"> <i class="fas fa-calendar-alt"></i> Term Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="academic_management"> <i class="fas fa-cog"></i> Academic Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="academic_year_management"> <i class="fas fa-calendar-check"></i> Academic Year Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="level_management"> <i class="fas fa-layer-group"></i> Level Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="grading_system"> <i class="fas fa-graduation-cap"></i> Grading System <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="student_promotion"> <i class="fas fa-arrow-up"></i> Student Promotion <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="director_management"> <i class="fas fa-user-tie"></i> Director Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="head_teacher_management"> <i class="fas fa-user-graduate"></i> Head Teacher Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="librarian_management"> <i class="fas fa-book-reader"></i> Librarian Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="non_teaching_management"> <i class="fas fa-users-cog"></i> Non-teaching Staff Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="inventory_management"> <i class="fas fa-boxes"></i> Inventory Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="facilities_management"> <i class="fas fa-building"></i> Facilities Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="transportation_management"> <i class="fas fa-bus"></i> Transportation Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="reports_analytics"> <i class="fas fa-chart-line"></i> Reports & Analytics <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="subscription_management"> <i class="fas fa-crown"></i> Subscription Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="package_management"> <i class="fas fa-box"></i> Package Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            </div>
                        </div>

                        <!-- Teacher Panel Features -->
                        <div class="feature-section">
                            <div class="feature-header">
                                <i class="fas fa-chalkboard-teacher"></i>
                                <span>Teacher Panel Features</span>
                            </div>
                            <div class="features-grid">
                                <label><input type="checkbox" name="features[]" value="grade_assignment"> <i class="fas fa-graduation-cap"></i> Grade Assignment <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="attendance_management"> <i class="fas fa-clipboard-check"></i> Attendance Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="student_performance"> <i class="fas fa-chart-bar"></i> Student Performance <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="student_ranking"> <i class="fas fa-trophy"></i> Student Ranking <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="student_results"> <i class="fas fa-file-pdf"></i> Student Results <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            </div>
                        </div>

                        <!-- Student Panel Features -->
                        <div class="feature-section">
                            <div class="feature-header">
                                <i class="fas fa-user-graduate"></i>
                                <span>Student Panel Features</span>
                            </div>
                            <div class="features-grid">
                                <label><input type="checkbox" name="features[]" value="view_grades"> <i class="fas fa-eye"></i> View Grades <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="view_courses"> <i class="fas fa-book"></i> View Courses <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="fee_payment_status"> <i class="fas fa-money-check"></i> Fee Payment Status <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="print_results"> <i class="fas fa-print"></i> Print Results <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                                <label><input type="checkbox" name="features[]" value="academic_records"> <i class="fas fa-file-contract"></i> Academic Records <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            </div>
                        </div>

                        <!-- Library Management Features -->
                        <div class="feature-section">
                            <div class="feature-header">
                                <i class="fas fa-book-reader"></i>
                                <span>Library Management Features</span>
                            </div>
                            <div class="features-grid">
                                <label><input type="checkbox" name="features[]" value="library_management"> <i class="fas fa-book"></i> Library Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            </div>
                        </div>
                    </div>

                    <button type="submit" name="add_package" class="btn-submit">
                        <i class="fas fa-plus"></i> Add Package
                    </button>
                </form>
            </div>

            <!-- Packages List -->
            <div class="packages-table">
                <table>
                    <thead>
                        <tr>
                            <th>Name</th>
                            <th>Code</th>
                            <th>Description</th>
                            <th>Max Students</th>
                            <th>Max Teachers</th>
                            <th>Price (GHS)</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($packages as $package): ?>
                            <?php
                            $features = json_decode($package['features'], true);
                            $pricing_details = $features['pricing_details'] ?? null;
                            ?>
                            <tr>
                                <td><?php echo htmlspecialchars($package['name']); ?></td>
                                <td><?php echo htmlspecialchars($package['code']); ?></td>
                                <td><?php echo htmlspecialchars(substr($package['description'], 0, 50)) . (strlen($package['description']) > 50 ? '...' : ''); ?></td>
                                <td><?php echo $package['max_students'] ? $package['max_students'] : 'Unlimited'; ?></td>
                                <td><?php echo $package['max_teachers'] ? $package['max_teachers'] : 'Unlimited'; ?></td>
                                <td>
                                    <?php if ($pricing_details): ?>
                                        <div class="pricing-details">
                                            <?php if ($pricing_details['one_time_license'] && $pricing_details['one_time_license'] !== 'custom_quote'): ?>
                                                <strong>One-time:</strong> ₵<?php echo number_format($pricing_details['one_time_license'], 0); ?><br>
                                            <?php endif; ?>
                                            <?php if ($pricing_details['annual_subscription'] && $pricing_details['annual_subscription'] !== 'custom_quote'): ?>
                                                <strong>Annual:</strong> ₵<?php echo number_format($pricing_details['annual_subscription'], 0); ?>/year<br>
                                            <?php endif; ?>
                                            <?php if ($pricing_details['monthly_subscription'] && $pricing_details['monthly_subscription'] !== 'custom_quote'): ?>
                                                <strong>Monthly:</strong> ₵<?php echo number_format($pricing_details['monthly_subscription'], 0); ?>/month<br>
                                            <?php endif; ?>
                                            <?php if ($pricing_details['implementation_fee'] && $pricing_details['implementation_fee'] !== 'custom_quote'): ?>
                                                <strong>Implementation:</strong> ₵<?php echo number_format($pricing_details['implementation_fee'], 0); ?><br>
                                            <?php endif; ?>
                                            <?php if ($pricing_details['training_support'] && $pricing_details['training_support'] !== 'custom_quote'): ?>
                                                <strong>Training:</strong> ₵<?php echo number_format($pricing_details['training_support'], 0); ?>
                                            <?php endif; ?>
                                            <?php if ($pricing_details['one_time_license'] === 'custom_quote'): ?>
                                                <strong>Custom Quote</strong>
                                            <?php endif; ?>
                                        </div>
                                    <?php else: ?>
                                        <?php echo $package['price_ghs'] ? '₵' . number_format($package['price_ghs'], 2) : 'Custom'; ?>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <button onclick='editPackage(<?php echo $package['id']; ?>, <?php echo json_encode($package['name']); ?>, <?php echo json_encode($package['code']); ?>, <?php echo json_encode($package['description']); ?>, <?php echo $package['max_students'] ?: 'null'; ?>, <?php echo $package['max_teachers'] ?: 'null'; ?>, <?php echo $package['price_ghs'] ?: 'null'; ?>, JSON.stringify(<?php echo json_encode($features); ?>))' class="action-btn btn-edit">
                                        <i class="fas fa-edit"></i> Edit
                                    </button>
                                    <form method="post" action="manage_packages.php" style="display: inline;" onsubmit="return confirm('Are you sure you want to delete this package?')">
                                        <input type="hidden" name="package_id" value="<?php echo $package['id']; ?>">
                                        <button type="submit" name="delete_package" class="action-btn btn-delete">
                                            <i class="fas fa-trash"></i> Delete
                                        </button>
                                    </form>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- Edit Modal -->
    <div id="editModal" class="modal" style="z-index: 1050;">
        <div class="modal-content" style="width: 90%; max-width: 600px; margin: 10% auto; max-height: 80vh; overflow-y: auto;">
            <span class="close">&times;</span>
            <h3 style="color: #8e44ad;">Edit Package</h3>
            <form method="post" action="manage_packages.php">
                <input type="hidden" id="edit_package_id" name="package_id">
                <div class="form-row" style="display: flex; gap: 15px; margin-bottom: 15px;">
                    <div class="form-group" style="flex: 1;">
                        <label for="edit_name" style="font-weight: 500; color: #555;">Package Name *</label>
                        <input type="text" id="edit_name" name="name" required style="width: 100%; padding: 8px 12px; border: 1px solid #ddd; border-radius: 4px; font-size: 14px;">
                    </div>
                    <div class="form-group" style="flex: 1;">
                        <label for="edit_code" style="font-weight: 500; color: #555;">Package Code *</label>
                        <input type="text" id="edit_code" name="code" required style="width: 100%; padding: 8px 12px; border: 1px solid #ddd; border-radius: 4px; font-size: 14px;">
                    </div>
                </div>
                <div class="form-group" style="margin-bottom: 15px;">
                    <label for="edit_description" style="font-weight: 500; color: #555;">Description</label>
                    <textarea id="edit_description" name="description" style="width: 100%; padding: 8px 12px; border: 1px solid #ddd; border-radius: 4px; font-size: 14px; resize: vertical; min-height: 80px;"></textarea>
                </div>
                <div class="form-row" style="display: flex; gap: 15px; margin-bottom: 15px;">
                    <div class="form-group" style="flex: 1;">
                        <label for="edit_max_students" style="font-weight: 500; color: #555;">Max Students</label>
                        <input type="number" id="edit_max_students" name="max_students" style="width: 100%; padding: 8px 12px; border: 1px solid #ddd; border-radius: 4px; font-size: 14px;">
                    </div>
                    <div class="form-group" style="flex: 1;">
                        <label for="edit_max_teachers" style="font-weight: 500; color: #555;">Max Teachers</label>
                        <input type="number" id="edit_max_teachers" name="max_teachers" style="width: 100%; padding: 8px 12px; border: 1px solid #ddd; border-radius: 4px; font-size: 14px;">
                    </div>
                    <div class="form-group" style="flex: 1;">
                        <label for="edit_price_ghs" style="font-weight: 500; color: #555;">Price (GHS/Year)</label>
                        <input type="number" step="0.01" id="edit_price_ghs" name="price_ghs" style="width: 100%; padding: 8px 12px; border: 1px solid #ddd; border-radius: 4px; font-size: 14px;">
                    </div>
                </div>

                <!-- Pricing Details Section -->
                <div class="form-group" style="margin-bottom: 15px;">
                    <label style="font-weight: 600; color: #2c3e50; margin-bottom: 10px; display: block;">Pricing Details (for custom pricing packages)</label>
                    <div class="form-row" style="display: flex; gap: 15px; margin-bottom: 10px;">
                        <div class="form-group" style="flex: 1;">
                            <label for="edit_one_time_license" style="font-weight: 500; color: #555;">One-time License (GHS)</label>
                            <input type="number" step="0.01" id="edit_one_time_license" name="one_time_license" style="width: 100%; padding: 8px 12px; border: 1px solid #ddd; border-radius: 4px; font-size: 14px;" placeholder="Custom quote">
                        </div>
                        <div class="form-group" style="flex: 1;">
                            <label for="edit_annual_subscription" style="font-weight: 500; color: #555;">Annual Subscription (GHS)</label>
                            <input type="number" step="0.01" id="edit_annual_subscription" name="annual_subscription" style="width: 100%; padding: 8px 12px; border: 1px solid #ddd; border-radius: 4px; font-size: 14px;">
                        </div>
                        <div class="form-group" style="flex: 1;">
                            <label for="edit_monthly_subscription" style="font-weight: 500; color: #555;">Monthly Subscription (GHS)</label>
                            <input type="number" step="0.01" id="edit_monthly_subscription" name="monthly_subscription" style="width: 100%; padding: 8px 12px; border: 1px solid #ddd; border-radius: 4px; font-size: 14px;">
                        </div>
                    </div>
                    <div class="form-row" style="display: flex; gap: 15px;">
                        <div class="form-group" style="flex: 1;">
                            <label for="edit_implementation_fee" style="font-weight: 500; color: #555;">Implementation Fee (GHS)</label>
                            <input type="number" step="0.01" id="edit_implementation_fee" name="implementation_fee" style="width: 100%; padding: 8px 12px; border: 1px solid #ddd; border-radius: 4px; font-size: 14px;">
                        </div>
                        <div class="form-group" style="flex: 1;">
                            <label for="edit_training_support" style="font-weight: 500; color: #555;">Training Support (GHS)</label>
                            <input type="number" step="0.01" id="edit_training_support" name="training_support" style="width: 100%; padding: 8px 12px; border: 1px solid #ddd; border-radius: 4px; font-size: 14px;">
                        </div>
                    </div>
                </div>

                <div class="form-group" style="margin-bottom: 15px;">
                    <label style="font-weight: 600; color: #2c3e50; margin-bottom: 15px; display: block;">Package Features:</label>

                    <!-- Administrative Features -->
                    <div class="feature-section">
                        <div class="feature-header">
                            <i class="fas fa-cog"></i>
                            <span>Administrative Features</span>
                        </div>
                        <div class="features-grid">
                            <label><input type="checkbox" id="edit_student_management" name="features[]" value="student_management"> <i class="fas fa-users"></i> Student Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_teacher_management" name="features[]" value="teacher_management"> <i class="fas fa-chalkboard-teacher"></i> Teacher Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_course_management" name="features[]" value="course_management"> <i class="fas fa-book"></i> Course Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_enrollment_management" name="features[]" value="enrollment_management"> <i class="fas fa-user-plus"></i> Enrollment Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_class_management" name="features[]" value="class_management"> <i class="fas fa-school"></i> Class Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_subject_management" name="features[]" value="subject_management"> <i class="fas fa-book-open"></i> Subject Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_fee_management" name="features[]" value="fee_management"> <i class="fas fa-money-bill-wave"></i> Fee Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_payment_management" name="features[]" value="payment_management"> <i class="fas fa-credit-card"></i> Payment Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_assign_fees" name="features[]" value="assign_fees"> <i class="fas fa-hand-holding-usd"></i> Assign Fees <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_fee_reports" name="features[]" value="fee_reports"> <i class="fas fa-chart-bar"></i> Fee Reports <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_term_management" name="features[]" value="term_management"> <i class="fas fa-calendar-alt"></i> Term Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_academic_management" name="features[]" value="academic_management"> <i class="fas fa-cog"></i> Academic Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_academic_year_management" name="features[]" value="academic_year_management"> <i class="fas fa-calendar-check"></i> Academic Year Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_level_management" name="features[]" value="level_management"> <i class="fas fa-layer-group"></i> Level Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_grading_system" name="features[]" value="grading_system"> <i class="fas fa-graduation-cap"></i> Grading System <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_student_promotion" name="features[]" value="student_promotion"> <i class="fas fa-arrow-up"></i> Student Promotion <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_director_management" name="features[]" value="director_management"> <i class="fas fa-user-tie"></i> Director Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_head_teacher_management" name="features[]" value="head_teacher_management"> <i class="fas fa-user-graduate"></i> Head Teacher Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_librarian_management" name="features[]" value="librarian_management"> <i class="fas fa-book-reader"></i> Librarian Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_non_teaching_management" name="features[]" value="non_teaching_management"> <i class="fas fa-users-cog"></i> Non-teaching Staff Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_inventory_management" name="features[]" value="inventory_management"> <i class="fas fa-boxes"></i> Inventory Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_facilities_management" name="features[]" value="facilities_management"> <i class="fas fa-building"></i> Facilities Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_transportation_management" name="features[]" value="transportation_management"> <i class="fas fa-bus"></i> Transportation Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_reports_analytics" name="features[]" value="reports_analytics"> <i class="fas fa-chart-line"></i> Reports & Analytics <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_subscription_management" name="features[]" value="subscription_management"> <i class="fas fa-crown"></i> Subscription Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_package_management" name="features[]" value="package_management"> <i class="fas fa-box"></i> Package Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                        </div>
                    </div>

                    <!-- Teacher Panel Features -->
                    <div class="feature-section">
                        <div class="feature-header">
                            <i class="fas fa-chalkboard-teacher"></i>
                            <span>Teacher Panel Features</span>
                        </div>
                        <div class="features-grid">
                            <label><input type="checkbox" id="edit_grade_assignment" name="features[]" value="grade_assignment"> <i class="fas fa-graduation-cap"></i> Grade Assignment <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_attendance_management" name="features[]" value="attendance_management"> <i class="fas fa-clipboard-check"></i> Attendance Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_course_materials" name="features[]" value="course_materials"> <i class="fas fa-file-alt"></i> Course Materials <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_student_progress_tracking" name="features[]" value="student_progress_tracking"> <i class="fas fa-chart-line"></i> Student Progress Tracking <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_report_generation" name="features[]" value="report_generation"> <i class="fas fa-file-pdf"></i> Report Generation <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_timetable_management" name="features[]" value="timetable_management"> <i class="fas fa-clock"></i> Timetable Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_assignment_management" name="features[]" value="assignment_management"> <i class="fas fa-tasks"></i> Assignment Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_class_communication" name="features[]" value="class_communication"> <i class="fas fa-comments"></i> Class Communication <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                        </div>
                    </div>

                    <!-- Student Panel Features -->
                    <div class="feature-section">
                        <div class="feature-header">
                            <i class="fas fa-user-graduate"></i>
                            <span>Student Panel Features</span>
                        </div>
                        <div class="features-grid">
                            <label><input type="checkbox" id="edit_view_grades" name="features[]" value="view_grades"> <i class="fas fa-eye"></i> View Grades <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_view_courses" name="features[]" value="view_courses"> <i class="fas fa-book"></i> View Courses <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_fee_payment_status" name="features[]" value="fee_payment_status"> <i class="fas fa-money-check"></i> Fee Payment Status <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_print_results" name="features[]" value="print_results"> <i class="fas fa-print"></i> Print Results <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                            <label><input type="checkbox" id="edit_academic_records" name="features[]" value="academic_records"> <i class="fas fa-file-contract"></i> Academic Records <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                        </div>
                    </div>

                    <!-- Library Management Features -->
                    <div class="feature-section">
                        <div class="feature-header">
                            <i class="fas fa-book-reader"></i>
                            <span>Library Management Features</span>
                        </div>
                        <div class="features-grid">
                            <label><input type="checkbox" id="edit_library_management" name="features[]" value="library_management"> <i class="fas fa-book"></i> Library Management <i class="fas fa-crown" style="color: #ffd700; margin-left: 5px;" title="Premium Feature"></i></label>
                        </div>
                    </div>
                </div>

                <button type="submit" name="update_package" class="btn-submit" style="background: #8e44ad; color: #fff; padding: 10px 20px; border: none; border-radius: 4px; cursor: pointer; font-size: 14px;">
                    <i class="fas fa-save"></i> Update Package
                </button>
            </form>
        </div>
    </div>

    <script>
        // Modal functionality
        var modal = document.getElementById('editModal');
        var span = document.getElementsByClassName('close')[0];

        function editPackage(id, name, code, description, maxStudents, maxTeachers, price, featuresJson) {
            document.getElementById('edit_package_id').value = id;
            document.getElementById('edit_name').value = name;
            document.getElementById('edit_code').value = code;
            document.getElementById('edit_description').value = description;
            document.getElementById('edit_max_students').value = maxStudents;
            document.getElementById('edit_max_teachers').value = maxTeachers;
            document.getElementById('edit_price_ghs').value = price;

            // Clear all checkboxes first
            var checkboxes = document.querySelectorAll('#editModal input[type="checkbox"]');
            checkboxes.forEach(function(checkbox) {
                checkbox.checked = false;
            });

            // Clear pricing details fields
            document.getElementById('edit_one_time_license').value = '';
            document.getElementById('edit_annual_subscription').value = '';
            document.getElementById('edit_monthly_subscription').value = '';
            document.getElementById('edit_implementation_fee').value = '';
            document.getElementById('edit_training_support').value = '';

            // Parse features and check corresponding checkboxes
            if (featuresJson) {
                var features = JSON.parse(featuresJson);
                for (var feature in features) {
                    if (features.hasOwnProperty(feature) && features[feature]) {
                        var checkbox = document.getElementById('edit_' + feature);
                        if (checkbox) {
                            checkbox.checked = true;
                        }
                    }
                }

                // Handle pricing details
                if (features.pricing_details) {
                    var pricing = features.pricing_details;
                    if (pricing.one_time_license) {
                        document.getElementById('edit_one_time_license').value = pricing.one_time_license === 'custom_quote' ? '' : pricing.one_time_license;
                    }
                    if (pricing.annual_subscription) {
                        document.getElementById('edit_annual_subscription').value = pricing.annual_subscription;
                    }
                    if (pricing.monthly_subscription) {
                        document.getElementById('edit_monthly_subscription').value = pricing.monthly_subscription;
                    }
                    if (pricing.implementation_fee) {
                        document.getElementById('edit_implementation_fee').value = pricing.implementation_fee;
                    }
                    if (pricing.training_support) {
                        document.getElementById('edit_training_support').value = pricing.training_support;
                    }
                }
            }

            modal.style.display = 'block';
        }

        span.onclick = function() {
            modal.style.display = 'none';
        }

        window.onclick = function(event) {
            if (event.target == modal) {
                modal.style.display = 'none';
            }
        }
    </script>

    <script src="../js/script.js"></script>
</body>
</html>
