<?php
require_once '../config.php';

if (!isset($_SESSION['user_id']) || $_SESSION['role'] != 'student') {
    header('Location: ../login.php');
    exit;
}

$user_id = $_SESSION['user_id'];

// Handle password change
$message = '';
$message_type = '';

if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['change_password'])) {
    $current_password = $_POST['current_password'];
    $new_password = $_POST['new_password'];
    $confirm_password = $_POST['confirm_password'];

    // Validate input
    if (empty($current_password) || empty($new_password) || empty($confirm_password)) {
        $message = 'All fields are required.';
        $message_type = 'error';
    } elseif ($new_password !== $confirm_password) {
        $message = 'New passwords do not match.';
        $message_type = 'error';
    } elseif (strlen($new_password) < 6) {
        $message = 'New password must be at least 6 characters long.';
        $message_type = 'error';
    } else {
        // Fetch current password from users table
        $stmt = $pdo->prepare("SELECT password FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        $user = $stmt->fetch();

        if ($user && password_verify($current_password, $user['password'])) {
            // Update password
            $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
            $update_stmt = $pdo->prepare("UPDATE users SET password = ? WHERE id = ?");
            if ($update_stmt->execute([$hashed_password, $user_id])) {
                $message = 'Password changed successfully!';
                $message_type = 'success';
            } else {
                $message = 'Failed to update password. Please try again.';
                $message_type = 'error';
            }
        } else {
            $message = 'Current password is incorrect.';
            $message_type = 'error';
        }
    }
}

// Fetch student information with parent data
$stmt = $pdo->prepare("SELECT s.*, u.created_at as enrollment_date, p.full_name as parent_name, p.mobile_phone as parent_phone, p.email as parent_email, p.relationship as parent_relationship FROM students s JOIN users u ON s.user_id = u.id LEFT JOIN parents p ON s.id = p.student_id WHERE s.user_id = ?");
$stmt->execute([$user_id]);
$student = $stmt->fetch();

if (!$student) {
    die("Student not found.");
}

$student_id = $student['id']; // Use students.id for further queries

include '../includes/student_header.php';
?>

<div class="profile-container">
    <div class="profile-header">
        <div class="profile-avatar">
            <?php if (!empty($student['photo_path']) && file_exists('../' . $student['photo_path'])): ?>
                <img src="<?php echo htmlspecialchars('../' . $student['photo_path']); ?>" alt="Student Photo" class="student-photo">
            <?php else: ?>
                <div class="avatar-circle">
                    <?php echo strtoupper(substr(trim($student['first_name'] . ' ' . ($student['middle_name'] ?? '') . ' ' . $student['last_name']), 0, 1)); ?>
                </div>
            <?php endif; ?>
        </div>
        <div class="profile-info">
            <h1><?php echo htmlspecialchars(trim($student['first_name'] . ' ' . ($student['middle_name'] ?? '') . ' ' . $student['last_name'])); ?></h1>
            <p class="profile-role">Student</p>
        </div>
    </div>

    <div class="profile-grid">
        <div class="profile-card">
            <div class="card-header">
                <i class="icon">📚</i>
                <h3>Academic Information</h3>
            </div>
            <div class="card-content">
                <div class="info-item">
                    <span class="label">Student ID:</span>
                    <span class="value"><?php echo htmlspecialchars($student['student_id'] ?? $student['id']); ?></span>
                </div>
                <div class="info-item">
                    <span class="label">Class:</span>
                    <span class="value"><?php echo htmlspecialchars($student['class']); ?></span>
                </div>
                <div class="info-item">
                    <span class="label">Academic Year:</span>
                    <span class="value"><?php echo htmlspecialchars($student['academic_year']); ?></span>
                </div>
                <div class="info-item">
                    <span class="label">Enrollment Date:</span>
                    <span class="value"><?php echo htmlspecialchars($student['enrollment_date'] ?? 'N/A'); ?></span>
                </div>
            </div>
        </div>

        <div class="profile-card">
            <div class="card-header">
                <i class="icon">👤</i>
                <h3>Personal Information</h3>
            </div>
            <div class="card-content">
                <div class="info-item">
                    <span class="label">Full Name:</span>
                    <span class="value"><?php echo htmlspecialchars(trim($student['first_name'] . ' ' . ($student['middle_name'] ?? '') . ' ' . $student['last_name'])); ?></span>
                </div>
                <div class="info-item">
                    <span class="label">Date of Birth:</span>
                    <span class="value"><?php echo htmlspecialchars($student['dob']); ?></span>
                </div>
                <div class="info-item">
                    <span class="label">Gender:</span>
                    <span class="value"><?php echo htmlspecialchars($student['gender'] ?? 'N/A'); ?></span>
                </div>
                <div class="info-item">
                    <span class="label">Nationality:</span>
                    <span class="value"><?php echo htmlspecialchars($student['nationality'] ?? 'N/A'); ?></span>
                </div>
                <div class="info-item">
                    <span class="label">Phone:</span>
                    <span class="value"><?php echo htmlspecialchars($student['phone']); ?></span>
                </div>
                <div class="info-item">
                    <span class="label">Email:</span>
                    <span class="value"><?php echo htmlspecialchars($student['email'] ?? 'N/A'); ?></span>
                </div>
                <div class="info-item">
                    <span class="label">Address:</span>
                    <span class="value"><?php
                        $address_parts = [];
                        if (!empty($student['home_street'])) $address_parts[] = $student['home_street'];
                        if (!empty($student['home_city'])) $address_parts[] = $student['home_city'];
                        if (!empty($student['home_state'])) $address_parts[] = $student['home_state'];
                        if (!empty($student['home_postal'])) $address_parts[] = $student['home_postal'];
                        if (!empty($student['home_country'])) $address_parts[] = $student['home_country'];
                        echo nl2br(htmlspecialchars(implode(', ', $address_parts)));
                    ?></span>
                </div>
            </div>
        </div>

        <div class="profile-card">
            <div class="card-header">
                <i class="icon">👨‍👩‍👧‍👦</i>
                <h3>Parent/Guardian Information</h3>
            </div>
            <div class="card-content">
                <div class="info-item">
                    <span class="label">Name:</span>
                    <span class="value"><?php echo htmlspecialchars($student['parent_name'] ?? 'N/A'); ?></span>
                </div>
                <div class="info-item">
                    <span class="label">Phone:</span>
                    <span class="value"><?php echo htmlspecialchars($student['parent_phone'] ?? 'N/A'); ?></span>
                </div>
                <div class="info-item">
                    <span class="label">Email:</span>
                    <span class="value"><?php echo htmlspecialchars($student['parent_email'] ?? 'N/A'); ?></span>
                </div>
                <div class="info-item">
                    <span class="label">Relationship:</span>
                    <span class="value"><?php echo htmlspecialchars($student['parent_relationship'] ?? 'N/A'); ?></span>
                </div>
            </div>
        </div>

        <div class="profile-card">
            <div class="card-header">
                <i class="icon">🚨</i>
                <h3>Emergency Contact</h3>
            </div>
            <div class="card-content">
                <div class="info-item">
                    <span class="label">Name:</span>
                    <span class="value"><?php echo htmlspecialchars($student['emergency_contact_name'] ?? 'N/A'); ?></span>
                </div>
                <div class="info-item">
                    <span class="label">Phone:</span>
                    <span class="value"><?php echo htmlspecialchars($student['emergency_contact_phone'] ?? 'N/A'); ?></span>
                </div>
                <div class="info-item">
                    <span class="label">Relationship:</span>
                    <span class="value"><?php echo htmlspecialchars($student['emergency_contact_relationship'] ?? 'N/A'); ?></span>
                </div>
            </div>
        </div>

        <div class="profile-card">
            <div class="card-header">
                <i class="icon">🏥</i>
                <h3>Health Information</h3>
            </div>
            <div class="card-content">
                <div class="info-item">
                    <span class="label">Medical Conditions:</span>
                    <span class="value"><?php echo htmlspecialchars($student['medical_conditions'] ?? 'None'); ?></span>
                </div>
                <div class="info-item">
                    <span class="label">Allergies:</span>
                    <span class="value"><?php echo htmlspecialchars($student['allergies'] ?? 'None'); ?></span>
                </div>
                <div class="info-item">
                    <span class="label">Blood Type:</span>
                    <span class="value"><?php echo htmlspecialchars($student['blood_type'] ?? 'N/A'); ?></span>
                </div>
                <div class="info-item">
                    <span class="label">Insurance Provider:</span>
                    <span class="value"><?php echo htmlspecialchars($student['insurance_provider'] ?? 'N/A'); ?></span>
                </div>
            </div>
        </div>
    </div>
</div>

<div class="password-change-container">
    <h2>Change Login Password</h2>

    <?php if ($message): ?>
        <div class="message <?php echo $message_type; ?>">
            <?php echo htmlspecialchars($message); ?>
        </div>
    <?php endif; ?>

    <form method="POST" action="view_profile.php" class="password-form">
        <div class="form-group">
            <label for="current_password">Current Password:</label>
            <input type="password" id="current_password" name="current_password" required>
        </div>

        <div class="form-group">
            <label for="new_password">New Password:</label>
            <input type="password" id="new_password" name="new_password" required>
            <div id="password-requirements" class="password-requirements">
                <div id="req-length" class="requirement">At least 8 characters <span class="check">✗</span></div>
                <div id="req-uppercase" class="requirement">One uppercase letter <span class="check">✗</span></div>
                <div id="req-lowercase" class="requirement">One lowercase letter <span class="check">✗</span></div>
                <div id="req-number" class="requirement">One number <span class="check">✗</span></div>
                <div id="req-special" class="requirement">One special character <span class="check">✗</span></div>
            </div>
        </div>

        <div class="form-group">
            <label for="confirm_password">Confirm New Password:</label>
            <input type="password" id="confirm_password" name="confirm_password" required>
            <div id="confirm-password-match" class="password-requirements">
                <div id="req-match" class="requirement">Passwords match <span class="check">✗</span></div>
            </div>
        </div>

        <button type="submit" name="change_password" class="change-password-btn">Change Password</button>
    </form>
</div>

<style>
/* Profile Container */
.profile-container {
    max-width: 1200px;
    margin: 30px auto;
    padding: 30px;
    background: linear-gradient(135deg, #3498db 0%, #2980b9 100%);
    border-radius: 20px;
    box-shadow: 0 20px 40px rgba(0,0,0,0.1);
    color: white;
}

/* Profile Header */
.profile-header {
    display: flex;
    align-items: center;
    margin-bottom: 40px;
    padding-bottom: 30px;
    border-bottom: 2px solid rgba(255,255,255,0.2);
}

.profile-avatar {
    margin-right: 25px;
}

.avatar-circle {
    width: 100px;
    height: 100px;
    border-radius: 50%;
    background: linear-gradient(135deg, #ff6b6b, #ffa500);
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 36px;
    font-weight: bold;
    color: white;
    box-shadow: 0 10px 30px rgba(0,0,0,0.2);
    border: 4px solid rgba(255,255,255,0.3);
}

.student-photo {
    width: 100px;
    height: 100px;
    border-radius: 50%;
    object-fit: cover;
    box-shadow: 0 10px 30px rgba(0,0,0,0.2);
    border: 4px solid rgba(255,255,255,0.3);
}

.profile-info h1 {
    margin: 0 0 5px 0;
    font-size: 2.5em;
    font-weight: 700;
    text-shadow: 2px 2px 4px rgba(0,0,0,0.3);
}

.profile-role {
    margin: 0;
    font-size: 1.2em;
    opacity: 0.9;
    font-weight: 300;
}

/* Profile Grid */
.profile-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(350px, 1fr));
    gap: 25px;
    margin-top: 30px;
}

/* Profile Cards */
.profile-card {
    background: rgba(255,255,255,0.95);
    border-radius: 15px;
    overflow: hidden;
    box-shadow: 0 10px 30px rgba(0,0,0,0.1);
    transition: transform 0.3s ease, box-shadow 0.3s ease;
    backdrop-filter: blur(10px);
}

.profile-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 20px 40px rgba(0,0,0,0.15);
}

.card-header {
    background: linear-gradient(135deg, #ff6b6b, #ffa500);
    padding: 20px;
    display: flex;
    align-items: center;
    color: white;
}

.card-header .icon {
    font-size: 24px;
    margin-right: 15px;
}

.card-header h3 {
    margin: 0;
    font-size: 1.3em;
    font-weight: 600;
}

.card-content {
    padding: 25px;
}

.info-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 12px 0;
    border-bottom: 1px solid #f0f0f0;
}

.info-item:last-child {
    border-bottom: none;
}

.info-item .label {
    font-weight: 600;
    color: #2c3e50;
    font-size: 0.95em;
}

.info-item .value {
    color: #34495e;
    font-weight: 500;
    text-align: right;
    max-width: 60%;
    word-wrap: break-word;
}

/* Password Change Container */
.password-change-container {
    max-width: 600px;
    margin: 30px auto;
    padding: 30px;
    background: #fff;
    border-radius: 15px;
    box-shadow: 0 10px 30px rgba(0,0,0,0.1);
    border: 1px solid #e9ecef;
}

.password-change-container h2 {
    margin: 0 0 25px 0;
    color: #2c3e50;
    font-size: 1.8em;
    font-weight: 700;
    text-align: center;
    position: relative;
}

.password-change-container h2::after {
    content: '';
    position: absolute;
    bottom: -10px;
    left: 50%;
    transform: translateX(-50%);
    width: 50px;
    height: 3px;
    background: linear-gradient(135deg, #ff6b6b, #ffa500);
    border-radius: 2px;
}

/* Messages */
.message {
    padding: 15px 20px;
    border-radius: 10px;
    margin-bottom: 25px;
    font-weight: 500;
    text-align: center;
    border-left: 4px solid;
}

.message.success {
    background-color: #d4edda;
    color: #155724;
    border-left-color: #28a745;
}

.message.error {
    background-color: #f8d7da;
    color: #721c24;
    border-left-color: #dc3545;
}

/* Password Form */
.password-form {
    display: flex;
    flex-direction: column;
    gap: 20px;
}

.form-group {
    display: flex;
    flex-direction: column;
}

.form-group label {
    font-weight: 600;
    color: #2c3e50;
    margin-bottom: 8px;
    font-size: 1em;
}

.form-group input {
    padding: 15px 18px;
    border: 2px solid #e9ecef;
    border-radius: 10px;
    font-size: 1em;
    transition: all 0.3s ease;
    background: #f8f9fa;
}

.form-group input:focus {
    outline: none;
    border-color: #667eea;
    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
    background: #fff;
}

.change-password-btn {
    padding: 15px 30px;
    background: linear-gradient(135deg, #ff6b6b, #ffa500);
    color: white;
    border: none;
    border-radius: 10px;
    font-size: 1.1em;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    align-self: center;
    box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
}

.change-password-btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(102, 126, 234, 0.4);
}

.change-password-btn:active {
    transform: translateY(0);
}

/* Responsive Design */
@media (max-width: 768px) {
    .profile-container {
        margin: 120px 20px 20px 20px;
        padding: 20px;
    }

    .profile-header {
        flex-direction: column;
        text-align: center;
        margin-bottom: 30px;
    }

    .profile-avatar {
        margin-right: 0;
        margin-bottom: 20px;
    }

    .profile-info h1 {
        font-size: 2em;
    }

    .profile-grid {
        grid-template-columns: 1fr;
        gap: 20px;
    }

    .info-item {
        flex-direction: column;
        align-items: flex-start;
        gap: 5px;
    }

    .info-item .value {
        text-align: left;
        max-width: 100%;
    }

    .password-change-container {
        margin: 20px;
        padding: 20px;
    }
}

@media (max-width: 480px) {
    .profile-container {
        margin: 120px 10px 10px 10px;
        padding: 15px;
    }

    .profile-info h1 {
        font-size: 1.8em;
    }

    .avatar-circle {
        width: 80px;
        height: 80px;
        font-size: 28px;
    }

    .card-content {
        padding: 20px;
    }

    .password-form {
        gap: 15px;
    }

    .form-group input {
        padding: 12px 15px;
    }
}

.password-requirements {
    margin-top: 5px;
    font-size: 14px;
}

.requirement {
    display: flex;
    justify-content: space-between;
    margin-bottom: 2px;
}

.requirement .check {
    font-weight: bold;
}

.requirement.met .check {
    color: green;
}

.requirement.not-met .check {
    color: red;
}
</style>

<script>
    // Password strength validation
    function checkPasswordStrength(password) {
        const requirements = {
            length: password.length >= 8,
            uppercase: /[A-Z]/.test(password),
            lowercase: /[a-z]/.test(password),
            number: /\d/.test(password),
            special: /[!@#$%^&*()_+\-=\[\]{};':"\\|,.<>\/?]/.test(password)
        };
        return requirements;
    }

    function updatePasswordRequirements(password) {
        const requirements = checkPasswordStrength(password);
        document.getElementById('req-length').className = requirements.length ? 'requirement met' : 'requirement not-met';
        document.getElementById('req-length').querySelector('.check').textContent = requirements.length ? '✓' : '✗';
        document.getElementById('req-uppercase').className = requirements.uppercase ? 'requirement met' : 'requirement not-met';
        document.getElementById('req-uppercase').querySelector('.check').textContent = requirements.uppercase ? '✓' : '✗';
        document.getElementById('req-lowercase').className = requirements.lowercase ? 'requirement met' : 'requirement not-met';
        document.getElementById('req-lowercase').querySelector('.check').textContent = requirements.lowercase ? '✓' : '✗';
        document.getElementById('req-number').className = requirements.number ? 'requirement met' : 'requirement not-met';
        document.getElementById('req-number').querySelector('.check').textContent = requirements.number ? '✓' : '✗';
        document.getElementById('req-special').className = requirements.special ? 'requirement met' : 'requirement not-met';
        document.getElementById('req-special').querySelector('.check').textContent = requirements.special ? '✓' : '✗';
    }

    function updateConfirmPasswordMatch() {
        const password = document.getElementById('new_password').value;
        const confirmPassword = document.getElementById('confirm_password').value;
        const match = password === confirmPassword && password !== '';
        document.getElementById('req-match').className = match ? 'requirement met' : 'requirement not-met';
        document.getElementById('req-match').querySelector('.check').textContent = match ? '✓' : '✗';
    }

    // Password input event listener
    document.getElementById('new_password').addEventListener('input', function() {
        updatePasswordRequirements(this.value);
        updateConfirmPasswordMatch();
    });

    // Password confirmation validation
    document.getElementById('confirm_password').addEventListener('input', function() {
        const password = document.getElementById('new_password').value;
        const confirmPassword = this.value;

        if (password !== confirmPassword) {
            this.setCustomValidity('Passwords do not match');
        } else {
            this.setCustomValidity('');
        }
        updateConfirmPasswordMatch();
    });

    // Form validation enhancement
    document.querySelector('form').addEventListener('submit', function(e) {
        const password = document.getElementById('new_password').value;
        const confirmPassword = document.getElementById('confirm_password').value;

        if (password !== confirmPassword) {
            e.preventDefault();
            alert('Passwords do not match. Please try again.');
        }
    });
</script>

<?php
include '../includes/student_footer.php';
?>
