<?php
require_once '../config.php';

if (!isset($_SESSION['user_id']) || $_SESSION['role'] != 'student') {
    header('Location: ../login.php');
    exit;
}

$user_id = $_SESSION['user_id'];

// Fetch student information
$student_info = $pdo->prepare("SELECT * FROM students WHERE user_id = ?");
$student_info->execute([$user_id]);
$student = $student_info->fetch();

if (!$student) {
    die("Student not found.");
}

$student_id = $student['id'];

// Get all academic years for selection
$academic_years = $pdo->query("SELECT id, year FROM academic_years ORDER BY start_date DESC")->fetchAll();

// Get all terms for selection
$terms = $pdo->query("SELECT id, name FROM terms ORDER BY start_date DESC")->fetchAll();

// Get current active academic year and term as defaults
$current_academic_year = $pdo->query("SELECT id, year FROM academic_years WHERE is_active = 1 LIMIT 1")->fetch();
$current_term = $pdo->query("SELECT id, name FROM terms WHERE is_active = 1 LIMIT 1")->fetch();

// Use current active ones as defaults if not provided
$selected_academic_year_id = $_GET['academic_year_id'] ?? ($current_academic_year ? $current_academic_year['id'] : null);
$selected_term_id = $_GET['term_id'] ?? ($current_term ? $current_term['id'] : null);

$attendance_records = [];

if ($selected_academic_year_id && $selected_term_id) {
    // Fetch attendance records for the student, filtered by academic year and term
    $attendance_stmt = $pdo->prepare("
        SELECT a.date, a.status, c.name as class_name, t.name as term_name, ay.year as academic_year
        FROM attendance a
        JOIN classes c ON a.class_id = c.id
        JOIN terms t ON a.term_id = t.id
        JOIN academic_years ay ON a.academic_year_id = ay.id
        WHERE a.student_id = ? AND a.academic_year_id = ? AND a.term_id = ?
        ORDER BY a.date DESC
    ");
    $attendance_stmt->execute([$student_id, $selected_academic_year_id, $selected_term_id]);
    $attendance_records = $attendance_stmt->fetchAll();
}

include '../includes/student_header.php';
?>

<div class="attendance-container">
    <h1>📋 My Attendance Records</h1>

    <form method="GET" class="filter-form">
        <label for="academic_year_id">Academic Year:</label>
        <select name="academic_year_id" id="academic_year_id" required onchange="this.form.submit()">
            <option value="">-- Select Academic Year --</option>
            <?php foreach ($academic_years as $year): ?>
                <option value="<?php echo $year['id']; ?>" <?php if ($year['id'] == $selected_academic_year_id) echo 'selected'; ?>>
                    <?php echo htmlspecialchars($year['year']); ?>
                </option>
            <?php endforeach; ?>
        </select>

        <label for="term_id">Term/Semester:</label>
        <select name="term_id" id="term_id" required onchange="this.form.submit()">
            <option value="">-- Select Term --</option>
            <?php foreach ($terms as $term): ?>
                <option value="<?php echo $term['id']; ?>" <?php if ($term['id'] == $selected_term_id) echo 'selected'; ?>>
                    <?php echo htmlspecialchars($term['name']); ?>
                </option>
            <?php endforeach; ?>
        </select>
    </form>

    <?php if ($selected_academic_year_id && $selected_term_id): ?>
        <?php if (count($attendance_records) > 0): ?>
            <div class="attendance-summary">
                <h3>Attendance Summary</h3>
                <?php
                $total_days = count($attendance_records);
                $present_days = count(array_filter($attendance_records, function($record) { return $record['status'] === 'present'; }));
                $absent_days = count(array_filter($attendance_records, function($record) { return $record['status'] === 'absent'; }));
                $late_days = count(array_filter($attendance_records, function($record) { return $record['status'] === 'late'; }));
                $attendance_percentage = $total_days > 0 ? round(($present_days / $total_days) * 100, 1) : 0;
                ?>
                <div class="summary-stats">
                    <div class="stat-item">
                        <span class="stat-label">Total Days:</span>
                        <span class="stat-value"><?php echo $total_days; ?></span>
                    </div>
                    <div class="stat-item">
                        <span class="stat-label">Present:</span>
                        <span class="stat-value present"><?php echo $present_days; ?></span>
                    </div>
                    <div class="stat-item">
                        <span class="stat-label">Absent:</span>
                        <span class="stat-value absent"><?php echo $absent_days; ?></span>
                    </div>
                    <div class="stat-item">
                        <span class="stat-label">Late:</span>
                        <span class="stat-value late"><?php echo $late_days; ?></span>
                    </div>
                    <div class="stat-item">
                        <span class="stat-label">Attendance Rate:</span>
                        <span class="stat-value <?php echo $attendance_percentage >= 75 ? 'good' : 'poor'; ?>"><?php echo $attendance_percentage; ?>%</span>
                    </div>
                </div>
            </div>

            <table class="attendance-table">
                <thead>
                    <tr>
                        <th>Date</th>
                        <th>Class</th>
                        <th>Status</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($attendance_records as $record): ?>
                        <tr>
                            <td><?php echo htmlspecialchars(date('M d, Y', strtotime($record['date']))); ?></td>
                            <td><?php echo htmlspecialchars($record['class_name']); ?></td>
                            <td>
                                <span class="status-badge status-<?php echo $record['status']; ?>">
                                    <?php echo htmlspecialchars(ucfirst($record['status'])); ?>
                                </span>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php else: ?>
            <div class="no-records">
                <p>No attendance records found for the selected academic year and term.</p>
            </div>
        <?php endif; ?>
    <?php else: ?>
        <div class="no-records">
            <p>Please select both academic year and term to view your attendance records.</p>
        </div>
    <?php endif; ?>
</div>

<style>
.attendance-container {
    max-width: 1000px;
    margin: 20px auto;
    padding: 20px;
    background: white;
    border-radius: 12px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.05);
}

.filter-form {
    display: flex;
    gap: 20px;
    align-items: center;
    margin-bottom: 30px;
    padding: 20px;
    background: #f8f9fa;
    border-radius: 8px;
    border: 1px solid #dee2e6;
}

.filter-form label {
    font-weight: 600;
    color: #2c3e50;
    white-space: nowrap;
}

.filter-form select {
    padding: 10px 12px;
    border-radius: 6px;
    border: 1px solid #ccc;
    font-size: 1em;
    min-width: 200px;
    background-color: #fff;
}

.attendance-summary {
    margin-bottom: 30px;
    padding: 20px;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border-radius: 12px;
    box-shadow: 0 4px 15px rgba(0,0,0,0.1);
}

.attendance-summary h3 {
    margin: 0 0 20px 0;
    font-size: 1.4em;
    text-align: center;
}

.summary-stats {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
    gap: 15px;
}

.stat-item {
    text-align: center;
    padding: 10px;
    background: rgba(255,255,255,0.1);
    border-radius: 8px;
    backdrop-filter: blur(10px);
}

.stat-label {
    display: block;
    font-size: 0.9em;
    opacity: 0.9;
    margin-bottom: 5px;
}

.stat-value {
    display: block;
    font-size: 1.5em;
    font-weight: 700;
}

.stat-value.present { color: #27ae60; }
.stat-value.absent { color: #e74c3c; }
.stat-value.late { color: #f39c12; }
.stat-value.good { color: #27ae60; }
.stat-value.poor { color: #e74c3c; }

.attendance-table {
    width: 100%;
    border-collapse: collapse;
    margin-top: 20px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    border-radius: 8px;
    overflow: hidden;
}

.attendance-table th,
.attendance-table td {
    border: 1px solid #ddd;
    padding: 15px;
    text-align: center;
}

.attendance-table th {
    background: linear-gradient(135deg, #27ae60 0%, #2ecc71 100%);
    color: white;
    font-weight: 700;
    font-size: 0.95em;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.attendance-table tr:nth-child(even) {
    background-color: #f8f9fa;
}

.attendance-table tr:hover {
    background-color: #e8f5e8;
}

.status-badge {
    display: inline-block;
    padding: 6px 12px;
    border-radius: 20px;
    font-size: 0.85em;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.status-present {
    background-color: #d4edda;
    color: #155724;
    border: 1px solid #c3e6cb;
}

.status-absent {
    background-color: #f8d7da;
    color: #721c24;
    border: 1px solid #f5c6cb;
}

.status-late {
    background-color: #fff3cd;
    color: #856404;
    border: 1px solid #ffeaa7;
}

.no-records {
    text-align: center;
    color: #6c757d;
    padding: 60px 20px;
    background: #f8f9fa;
    border-radius: 8px;
    border: 2px dashed #dee2e6;
}

.no-records p {
    margin: 0;
    font-size: 1.1em;
}

@media (max-width: 768px) {
    .filter-form {
        flex-direction: column;
        align-items: stretch;
    }

    .filter-form select {
        min-width: auto;
    }

    .summary-stats {
        grid-template-columns: repeat(2, 1fr);
    }

    .attendance-table {
        font-size: 0.9em;
    }

    .attendance-table th,
    .attendance-table td {
        padding: 10px 8px;
    }
}
</style>

<?php
include '../includes/student_footer.php';
?>
