<?php
require_once '../config.php';

if (!isset($_SESSION['user_id']) || $_SESSION['role'] != 'student') {
    header('Location: ../login.php');
    exit;
}

$user_id = $_SESSION['user_id'];

// Fetch student information
$student_info = $pdo->prepare("SELECT * FROM students WHERE user_id = ?");
$student_info->execute([$user_id]);
$student = $student_info->fetch();

if (!$student) {
    echo "Student not found.";
    exit;
}

$student_id = $student['id'];

// Check if fee_id is provided
if (!isset($_GET['fee']) || !is_numeric($_GET['fee'])) {
    echo "Invalid fee ID.";
    exit;
}

$fee_id = $_GET['fee'];

// Fetch the specific fee for this student
$fee_query = $pdo->prepare("
    SELECT
        sf.*,
        f.name as fee_name,
        f.amount as fee_amount,
        f.description as fee_description
    FROM student_fees sf
    JOIN fees f ON sf.fee_id = f.id
    WHERE sf.id = ? AND sf.student_id = ?
");
$fee_query->execute([$fee_id, $student_id]);
$fee = $fee_query->fetch();

if (!$fee) {
    echo "Fee not found or access denied.";
    exit;
}

// Only allow printing if there's a payment (paid or partial)
if (!$fee['paid'] && $fee['paid_amount'] == 0) {
    echo "No payment found for this fee.";
    exit;
}

// Prepare receipt data
$receipt = [
    'receipt_number' => $fee['receipt_number'] ?? 'N/A',
    'student_name' => $student['name'],
    'fee_name' => $fee['fee_name'],
    'amount' => $fee['paid_amount'] ?? 0,
    'total_paid' => $fee['paid_amount'] ?? 0,
    'remaining_balance' => $fee['fee_amount'] - ($fee['paid_amount'] ?? 0),
    'payment_date' => $fee['paid_at'] ?? date('Y-m-d'),
    'payment_method' => $fee['payment_method'] ?? 'N/A',
    'status' => $fee['paid'] ? 'Fully Paid' : 'Partially Paid',
    'accountant_name' => 'School Accountant' // Since we don't have individual accountant info for students
];

// Fetch school settings for print
$school_settings = $pdo->query("SELECT school_name, address, phone, email FROM school_settings LIMIT 1")->fetch(PDO::FETCH_ASSOC);
$schoolName = htmlspecialchars($school_settings['school_name'] ?? 'School Name');
$schoolAddress = htmlspecialchars($school_settings['address'] ?? 'School Address');
$schoolContact = htmlspecialchars(($school_settings['phone'] ?? '') . ' | ' . ($school_settings['email'] ?? ''));

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Payment Receipt - <?php echo htmlspecialchars($receipt['receipt_number']); ?></title>
    <style>
        body {
            font-family: Arial, sans-serif;
            margin: 20px;
            color: #333;
        }
        h1, h2, h3 {
            text-align: center;
            margin-bottom: 5px;
        }
        p {
            text-align: center;
            margin: 2px 0;
            font-size: 14px;
        }
        hr {
            border: 0;
            border-top: 1px solid #ccc;
            margin: 15px 0;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
        }
        th, td {
            border: 1px solid #ddd;
            padding: 10px;
            text-align: left;
            font-size: 14px;
        }
        th {
            background-color: #f4f4f4;
        }
        .total-row td {
            font-weight: bold;
        }
        @media print {
            body {
                margin: 0;
            }
        }
    </style>
</head>
<body>
    <h1><?php echo $schoolName; ?></h1>
    <p><?php echo $schoolAddress; ?></p>
    <p>Contact: <?php echo $schoolContact; ?></p>
    <hr>
    <h2>Student Fees Receipt</h2>

    <table>
        <tr>
            <th>Receipt Number</th>
            <td><?php echo htmlspecialchars($receipt['receipt_number']); ?></td>
        </tr>
        <tr>
            <th>Student Name</th>
            <td><?php echo htmlspecialchars($receipt['student_name']); ?></td>
        </tr>
        <tr>
            <th>Fee Description</th>
            <td><?php echo htmlspecialchars($receipt['fee_name']); ?></td>
        </tr>
        <tr>
            <th>Payment Date</th>
            <td><?php echo htmlspecialchars($receipt['payment_date']); ?></td>
        </tr>
        <tr>
            <th>Payment Method</th>
            <td><?php echo htmlspecialchars($receipt['payment_method']); ?></td>
        </tr>
        <tr>
            <th>Amount Paid</th>
            <td>GHC <?php echo number_format($receipt['amount'], 2); ?></td>
        </tr>
        <tr>
            <th>Outstanding Balance</th>
            <td>GHC <?php echo number_format($receipt['remaining_balance'], 2); ?></td>
        </tr>
        <tr class="total-row">
            <th>Total Fee Amount</th>
            <td>GHC <?php echo number_format($fee['fee_amount'], 2); ?></td>
        </tr>
    </table>

    <p style="margin-top: 30px; text-align: center; font-style: italic;">Thank you for your payment. This is a computer-generated receipt.</p>
    <p style="text-align: center;">Generated on: <?php echo date('F j, Y \at g:i A'); ?></p>

    <div style="margin-top: 50px; text-align: center;">
        <button onclick="window.print()" style="padding: 10px 20px; margin: 0 10px; background: #007bff; color: white; border: none; border-radius: 4px; cursor: pointer;">Print Receipt</button>
        <button onclick="window.close()" style="padding: 10px 20px; margin: 0 10px; background: #6c757d; color: white; border: none; border-radius: 4px; cursor: pointer;">Close</button>
    </div>
</body>
</html>
