<?php
require_once '../config.php';

if (!isset($_SESSION['user_id']) || $_SESSION['role'] != 'student') {
    header('Location: ../login.php');
    exit;
}

$username = $_SESSION['username'];
$user_id = $_SESSION['user_id'];

// Fetch student information
$student_info = $pdo->prepare("SELECT * FROM students WHERE user_id = ?");
$student_info->execute([$user_id]);
$student = $student_info->fetch();

if (!$student) {
    echo "Student not found.";
    exit;
}

$student_id = $student['id'];

// Fetch student's fees with payment details
$fees_query = $pdo->prepare("
    SELECT
        sf.*,
        f.name as fee_name,
        f.amount as fee_amount,
        f.description as fee_description
    FROM student_fees sf
    JOIN fees f ON sf.fee_id = f.id
    WHERE sf.student_id = ?
    ORDER BY sf.due_date DESC
");
$fees_query->execute([$student_id]);
$fees = $fees_query->fetchAll();

// Calculate summary statistics
$total_fees = count($fees);
$paid_fees = 0;
$partial_fees = 0;
$unpaid_fees = 0;
$total_amount = 0;
$paid_amount = 0;
$outstanding_amount = 0;

foreach ($fees as $fee) {
    $total_amount += $fee['fee_amount'];
    $paid_amount_value = floatval($fee['paid_amount'] ?? 0);
    $paid_amount += $paid_amount_value;
    $outstanding_amount += max(0, $fee['fee_amount'] - $paid_amount_value);

    if ($fee['paid']) {
        $paid_fees++;
    } elseif ($paid_amount_value > 0) {
        $partial_fees++;
    } else {
        $unpaid_fees++;
    }
}

include '../includes/student_header.php';
?>

<div class="fees-container">
    <div class="fees-header">
        <h1>💰 Fees & Payments</h1>
        <p class="fees-subtitle">View all your assigned fees and payment status</p>
    </div>

    <!-- Summary Cards -->
    <div class="summary-section">
        <div class="summary-header">
            <h2>📊 Payment Overview</h2>
            <p>Your fee payment statistics at a glance</p>
        </div>

        <div class="stats-container">
            <!-- Primary Stats -->
            <div class="primary-stats">
                <div class="stat-card total">
                    <div class="stat-icon">📋</div>
                    <div class="stat-info">
                        <div class="stat-number"><?php echo $total_fees; ?></div>
                        <div class="stat-label">Total Fees</div>
                    </div>
                </div>

                <div class="stat-card paid">
                    <div class="stat-icon">✅</div>
                    <div class="stat-info">
                        <div class="stat-number"><?php echo $paid_fees; ?></div>
                        <div class="stat-label">Fully Paid</div>
                    </div>
                </div>

                <div class="stat-card pending">
                    <div class="stat-icon">⏳</div>
                    <div class="stat-info">
                        <div class="stat-number"><?php echo $unpaid_fees; ?></div>
                        <div class="stat-label">Pending</div>
                    </div>
                </div>
            </div>

            <!-- Financial Summary -->
            <div class="financial-summary">
                <div class="finance-card">
                    <div class="finance-header">
                        <span class="finance-icon">💰</span>
                        <span class="finance-title">Financial Summary</span>
                    </div>
                    <div class="finance-details">
                        <div class="finance-item">
                            <span class="finance-label">Total Amount:</span>
                            <span class="finance-value">GHC <?php echo number_format($total_amount, 2); ?></span>
                        </div>
                        <div class="finance-item">
                            <span class="finance-label">Paid:</span>
                            <span class="finance-value paid-amount">GHC <?php echo number_format($paid_amount, 2); ?></span>
                        </div>
                        <div class="finance-item">
                            <span class="finance-label">Outstanding:</span>
                            <span class="finance-value outstanding">GHC <?php echo number_format($outstanding_amount, 2); ?></span>
                        </div>
                    </div>
                </div>

                <!-- Progress Bar -->
                <div class="progress-card">
                    <div class="progress-header">
                        <span>Payment Progress</span>
                        <span><?php echo $total_fees > 0 ? round(($paid_fees / $total_fees) * 100) : 0; ?>%</span>
                    </div>
                    <div class="progress-bar">
                        <div class="progress-fill" style="width: <?php echo $total_fees > 0 ? ($paid_fees / $total_fees) * 100 : 0; ?>%"></div>
                    </div>
                    <div class="progress-legend">
                        <div class="legend-item">
                            <span class="legend-color paid"></span>
                            <span>Paid (<?php echo $paid_fees; ?>)</span>
                        </div>
                        <div class="legend-item">
                            <span class="legend-color partial"></span>
                            <span>Partial (<?php echo $partial_fees; ?>)</span>
                        </div>
                        <div class="legend-item">
                            <span class="legend-color unpaid"></span>
                            <span>Unpaid (<?php echo $unpaid_fees; ?>)</span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Fees Table -->
    <div class="fees-table-container">
        <h2>Your Fees</h2>
        <?php if (count($fees) > 0): ?>
            <div class="table-responsive">
                <table class="fees-table">
                    <thead>
                        <tr>
                            <th>Fee Name</th>
                            <th>Description</th>
                            <th>Amount</th>
                            <th>Due Date</th>
                            <th>Payment Date</th>
                            <th>Payment Method</th>
                            <th>Receipt Number</th>
                            <th>Status</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($fees as $fee): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($fee['fee_name']); ?></td>
                                <td><?php echo htmlspecialchars($fee['fee_description'] ?? 'N/A'); ?></td>
                                <td>GHC <?php echo number_format($fee['fee_amount'], 2); ?></td>
                                <td><?php echo htmlspecialchars($fee['due_date']); ?></td>
                                <td><?php echo $fee['paid_at'] ? htmlspecialchars($fee['paid_at']) : '-'; ?></td>
                                <td><?php echo htmlspecialchars($fee['payment_method'] ?: '-'); ?></td>
                                <td><?php echo htmlspecialchars($fee['receipt_number'] ?: '-'); ?></td>
                                <td>
                                    <?php
                                    $status_class = 'unpaid';
                                    $status_text = 'Unpaid';

                                    if ($fee['paid']) {
                                        $status_class = 'paid';
                                        $status_text = 'Paid';
                                    } elseif ($fee['paid_amount'] > 0 && $fee['paid_amount'] < $fee['fee_amount']) {
                                        $status_class = 'partial';
                                        $status_text = 'Partial';
                                    }
                                    ?>
                                    <span class="status-badge <?php echo $status_class; ?>">
                                        <?php echo $status_text; ?>
                                    </span>
                                </td>
                                <td>
                                    <?php if ($fee['paid'] || $fee['paid_amount'] > 0): ?>
                                        <a href="print_receipt.php?fee=<?php echo $fee['id']; ?>" target="_blank" class="print-receipt-btn">Print Receipt</a>
                                    <?php else: ?>
                                        -
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php else: ?>
            <div class="no-fees">
                <p>No fees assigned yet.</p>
            </div>
        <?php endif; ?>
    </div>
</div>

<style>
.fees-container {
    padding: 20px;
    max-width: 1200px;
    margin: 0 auto;
}

.fees-header h1 {
    margin-bottom: 5px;
    color: #2c3e50;
    font-size: 2.5em;
}

.fees-subtitle {
    color: #7f8c8d;
    margin-bottom: 30px;
    font-size: 1.1em;
}

.summary-section {
    margin-bottom: 40px;
}

.summary-header {
    text-align: center;
    margin-bottom: 30px;
}

.summary-header h2 {
    color: #2c3e50;
    font-size: 2em;
    margin-bottom: 5px;
}

.summary-header p {
    color: #7f8c8d;
    font-size: 1.1em;
}

.stats-container {
    display: grid;
    grid-template-columns: 1fr 2fr;
    gap: 30px;
    align-items: start;
}

.primary-stats {
    display: flex;
    flex-direction: column;
    gap: 20px;
}

.stat-card {
    background: white;
    border-radius: 16px;
    padding: 25px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.08);
    border: 1px solid #e1e8ed;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    gap: 20px;
}

.stat-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 8px 30px rgba(0,0,0,0.12);
}

.stat-card.total {
    border-left: 4px solid #3498db;
}

.stat-card.paid {
    border-left: 4px solid #2ecc71;
}

.stat-card.pending {
    border-left: 4px solid #e74c3c;
}

.stat-icon {
    font-size: 2.5em;
    opacity: 0.8;
}

.stat-info {
    flex: 1;
}

.stat-number {
    font-size: 2.5em;
    font-weight: 700;
    color: #2c3e50;
    margin-bottom: 5px;
    letter-spacing: -0.5px;
}

.stat-label {
    color: #7f8c8d;
    font-weight: 600;
    font-size: 1em;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.financial-summary {
    display: flex;
    flex-direction: column;
    gap: 20px;
}

.finance-card {
    background: white;
    border-radius: 16px;
    padding: 25px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.08);
    border: 1px solid #e1e8ed;
}

.finance-header {
    display: flex;
    align-items: center;
    gap: 10px;
    margin-bottom: 20px;
    padding-bottom: 15px;
    border-bottom: 1px solid #e9ecef;
}

.finance-icon {
    font-size: 1.5em;
}

.finance-title {
    font-size: 1.3em;
    font-weight: 600;
    color: #2c3e50;
}

.finance-details {
    display: flex;
    flex-direction: column;
    gap: 15px;
}

.finance-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.finance-label {
    color: #7f8c8d;
    font-weight: 500;
}

.finance-value {
    font-weight: 600;
    color: #2c3e50;
}

.finance-value.paid-amount {
    color: #27ae60;
}

.finance-value.outstanding {
    color: #e74c3c;
}

.progress-card {
    background: white;
    border-radius: 16px;
    padding: 25px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.08);
    border: 1px solid #e1e8ed;
}

.progress-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 20px;
    font-weight: 600;
    color: #2c3e50;
}

.progress-bar {
    width: 100%;
    height: 12px;
    background: #e9ecef;
    border-radius: 6px;
    overflow: hidden;
    margin-bottom: 20px;
}

.progress-fill {
    height: 100%;
    background: linear-gradient(90deg, #3498db, #2ecc71);
    border-radius: 6px;
    transition: width 0.3s ease;
}

.progress-legend {
    display: flex;
    justify-content: space-between;
    gap: 15px;
}

.legend-item {
    display: flex;
    align-items: center;
    gap: 8px;
    font-size: 0.9em;
    color: #7f8c8d;
}

.legend-color {
    width: 12px;
    height: 12px;
    border-radius: 2px;
}

.legend-color.paid {
    background: #2ecc71;
}

.legend-color.partial {
    background: #f39c12;
}

.legend-color.unpaid {
    background: #e74c3c;
}

.fees-table-container {
    background: #fff;
    border-radius: 12px;
    padding: 25px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.05);
}

.fees-table-container h2 {
    margin: 0 0 20px 0;
    color: #2c3e50;
    font-size: 1.8em;
}

.table-responsive {
    overflow-x: auto;
}

.fees-table {
    width: 100%;
    border-collapse: collapse;
    margin-top: 20px;
}

.fees-table th,
.fees-table td {
    padding: 12px 15px;
    text-align: left;
    border-bottom: 1px solid #e9ecef;
}

.fees-table th {
    background: white;
    font-weight: 600;
    color: #2c3e50;
    font-size: 0.95em;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.fees-table tbody tr:hover {
    background-color: #f8f9fa;
}

.fees-table tbody tr:last-child td {
    border-bottom: none;
}

.status-badge {
    padding: 6px 12px;
    border-radius: 20px;
    font-size: 0.85em;
    font-weight: bold;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.status-badge.paid {
    background-color: #d4edda;
    color: #155724;
}

.status-badge.partial {
    background-color: #fff3cd;
    color: #856404;
}

.status-badge.unpaid {
    background-color: #f8d7da;
    color: #721c24;
}

.print-receipt-btn {
    background-color: #007bff;
    color: white;
    padding: 6px 12px;
    border-radius: 4px;
    text-decoration: none;
    font-size: 0.85em;
    font-weight: 500;
    display: inline-block;
    transition: background-color 0.3s ease;
}

.print-receipt-btn:hover {
    background-color: #0056b3;
    color: white;
}

.no-fees {
    text-align: center;
    padding: 60px 20px;
    color: #7f8c8d;
    font-style: italic;
}

.no-fees p {
    font-size: 1.2em;
    margin: 0;
}

/* Mobile Responsive */
@media (max-width: 768px) {
    .fees-container {
        padding: 15px;
    }

    .fees-header h1 {
        font-size: 2em;
    }

    .summary-grid {
        grid-template-columns: repeat(2, 1fr);
        gap: 15px;
        margin-bottom: 30px;
    }

    .summary-card {
        padding: 20px;
    }

    .summary-icon {
        font-size: 2.5em;
    }

    .summary-content h3 {
        font-size: 2em;
    }

    .fees-table-container {
        padding: 20px;
    }

    .fees-table-container h2 {
        font-size: 1.5em;
    }

    .fees-table th,
    .fees-table td {
        padding: 10px 8px;
        font-size: 0.9em;
    }

    .status-badge {
        padding: 4px 8px;
        font-size: 0.8em;
    }
}

@media (max-width: 480px) {
    .fees-container {
        padding: 10px;
    }

    .fees-header h1 {
        font-size: 1.8em;
    }

    .summary-grid {
        grid-template-columns: 1fr;
        gap: 10px;
    }

    .summary-card {
        padding: 15px;
    }

    .summary-icon {
        font-size: 2em;
    }

    .summary-content h3 {
        font-size: 1.8em;
    }

    .fees-table-container {
        padding: 15px;
    }

    .fees-table-container h2 {
        font-size: 1.3em;
    }

    .fees-table th,
    .fees-table td {
        padding: 8px 6px;
        font-size: 0.85em;
    }

    .status-badge {
        padding: 3px 6px;
        font-size: 0.75em;
    }
}
</style>

<?php
include '../includes/student_footer.php';
?>
