<?php
require_once '../config.php';

session_start();

if (!isset($_SESSION['user_id'])) {
    header('Location: ../login.php');
    exit;
}

$user_id = $_SESSION['user_id'];
$user_role = $_SESSION['role'];

// Check if user is staff
$staff_roles = ['teacher', 'librarian', 'non_teaching', 'director', 'head_teacher'];
if (!in_array($user_role, $staff_roles)) {
    header('Location: ../index.php');
    exit;
}

// Get staff information based on role
$staff_info = null;
$table_name = '';
$staff_name = '';

switch ($user_role) {
    case 'teacher':
        $table_name = 'teachers';
        break;
    case 'librarian':
        $table_name = 'librarians';
        break;
    case 'non_teaching':
        $table_name = 'non_teaching';
        break;
    case 'director':
        $table_name = 'directors';
        break;
    case 'head_teacher':
        $table_name = 'head_teachers';
        break;
}

if ($table_name) {
    $stmt = $pdo->prepare("SELECT * FROM $table_name WHERE user_id = ?");
    $stmt->execute([$user_id]);
    $staff_info = $stmt->fetch();

    if ($staff_info) {
        $staff_name = $staff_info['first_name'] . ' ' . $staff_info['last_name'];
    }
}

// Handle form submission
$message = '';
$message_type = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $leave_type = $_POST['leave_type'] ?? '';
    $start_date = $_POST['start_date'] ?? '';
    $end_date = $_POST['end_date'] ?? '';
    $reason = trim($_POST['reason'] ?? '');
    $emergency_contact = trim($_POST['emergency_contact'] ?? '');

    // Validation
    $errors = [];

    if (empty($leave_type)) {
        $errors[] = 'Please select a leave type.';
    }

    if (empty($start_date)) {
        $errors[] = 'Please select a start date.';
    }

    if (empty($end_date)) {
        $errors[] = 'Please select an end date.';
    }

    if (strtotime($start_date) > strtotime($end_date)) {
        $errors[] = 'Start date cannot be after end date.';
    }

    if (empty($reason)) {
        $errors[] = 'Please provide a reason for the leave.';
    }

    if (empty($errors)) {
        try {
            $stmt = $pdo->prepare("
                INSERT INTO staff_leave_applications
                (user_id, staff_role, leave_type, start_date, end_date, reason, emergency_contact)
                VALUES (?, ?, ?, ?, ?, ?, ?)
            ");

            $success = $stmt->execute([
                $user_id,
                $user_role,
                $leave_type,
                $start_date,
                $end_date,
                $reason,
                $emergency_contact
            ]);

            if ($success) {
                $message = 'Leave application submitted successfully!';
                $message_type = 'success';
            } else {
                $errorInfo = $stmt->errorInfo();
                $message = 'Error submitting leave application: ' . $errorInfo[2];
                $message_type = 'error';
            }

        } catch(PDOException $e) {
            $message = 'Exception submitting leave application: ' . $e->getMessage();
            $message_type = 'error';
        }
    } else {
        $message = implode('<br>', $errors);
        $message_type = 'error';
    }
}

// Get user's leave applications
$stmt = $pdo->prepare("
    SELECT * FROM staff_leave_applications
    WHERE user_id = ?
    ORDER BY created_at DESC
");
$stmt->execute([$user_id]);
$leave_applications = $stmt->fetchAll();

$page_title = 'Leave Application - ' . ucfirst($user_role);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?> - School Management System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .leave-form-card {
            box-shadow: 0 0 15px rgba(0,0,0,0.1);
            border: none;
            border-radius: 10px;
        }

        .leave-history-card {
            box-shadow: 0 0 15px rgba(0,0,0,0.1);
            border: none;
            border-radius: 10px;
        }

        .status-badge {
            padding: 5px 10px;
            border-radius: 20px;
            font-size: 0.8em;
            font-weight: 500;
        }

        .status-pending { background-color: #fff3cd; color: #856404; }
        .status-approved { background-color: #d4edda; color: #155724; }
        .status-rejected { background-color: #f8d7da; color: #721c24; }

        .leave-type-icon {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: bold;
        }

        .leave-type-sick { background-color: #e74c3c; }
        .leave-type-casual { background-color: #3498db; }
        .leave-type-maternity { background-color: #9b59b6; }
        .leave-type-paternity { background-color: #1abc9c; }
        .leave-type-emergency { background-color: #e67e22; }
        .leave-type-vacation { background-color: #27ae60; }
        .leave-type-other { background-color: #95a5a6; }

        .form-section {
            background-color: #f8f9fa;
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 20px;
        }

        .staff-info {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
    </style>
</head>
<body>
    <?php include '../includes/staff_header.php'; ?>

    <div class="container-fluid">
        <div class="row">
            <?php include '../includes/staff_sidebar.php'; ?>

            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2"><i class="fas fa-calendar-times"></i> Leave Application</h1>
                </div>

                <?php if ($message): ?>
                    <div class="alert alert-<?php echo $message_type === 'success' ? 'success' : 'danger'; ?> alert-dismissible fade show" role="alert">
                        <?php echo $message; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <!-- Staff Information -->
                <div class="staff-info">
                    <div class="row">
                        <div class="col-md-6">
                            <h4><i class="fas fa-user"></i> <?php echo htmlspecialchars($staff_name); ?></h4>
                            <p class="mb-0"><strong>Role:</strong> <?php echo ucfirst(str_replace('_', ' ', $user_role)); ?></p>
                        </div>
                        <div class="col-md-6 text-end">
                            <p class="mb-0"><strong>Staff ID:</strong> <?php echo htmlspecialchars($staff_info['staff_id'] ?? 'N/A'); ?></p>
                            <p class="mb-0"><strong>Department:</strong> <?php echo htmlspecialchars($staff_info['job_position'] ?? 'N/A'); ?></p>
                        </div>
                    </div>
                </div>

                <div class="row">
                    <!-- Leave Application Form -->
                    <div class="col-lg-5">
                        <div class="card leave-form-card">
                            <div class="card-header bg-primary text-white">
                                <h5 class="mb-0"><i class="fas fa-plus-circle"></i> Apply for Leave</h5>
                            </div>
                            <div class="card-body">
                                <form method="POST">
                                    <div class="mb-3">
                                        <label for="leave_type" class="form-label">Leave Type <span class="text-danger">*</span></label>
                                        <select class="form-select" id="leave_type" name="leave_type" required>
                                            <option value="">Select leave type</option>
                                            <option value="sick">Sick Leave</option>
                                            <option value="casual">Casual Leave</option>
                                            <option value="maternity">Maternity Leave</option>
                                            <option value="paternity">Paternity Leave</option>
                                            <option value="emergency">Emergency Leave</option>
                                            <option value="vacation">Vacation Leave</option>
                                            <option value="other">Other</option>
                                        </select>
                                    </div>

                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label for="start_date" class="form-label">Start Date <span class="text-danger">*</span></label>
                                                <input type="date" class="form-control" id="start_date" name="start_date"
                                                       min="<?php echo date('Y-m-d'); ?>" required>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label for="end_date" class="form-label">End Date <span class="text-danger">*</span></label>
                                                <input type="date" class="form-control" id="end_date" name="end_date"
                                                       min="<?php echo date('Y-m-d'); ?>" required>
                                            </div>
                                        </div>
                                    </div>

                                    <div class="mb-3">
                                        <label for="reason" class="form-label">Reason <span class="text-danger">*</span></label>
                                        <textarea class="form-control" id="reason" name="reason" rows="4"
                                                  placeholder="Please provide detailed reason for your leave application..." required></textarea>
                                    </div>

                                    <div class="mb-3">
                                        <label for="emergency_contact" class="form-label">Emergency Contact</label>
                                        <input type="text" class="form-control" id="emergency_contact" name="emergency_contact"
                                               placeholder="Phone number or contact details">
                                        <div class="form-text">Optional: Provide contact information in case of emergency</div>
                                    </div>

                                    <button type="submit" class="btn btn-primary w-100">
                                        <i class="fas fa-paper-plane"></i> Submit Application
                                    </button>
                                </form>
                            </div>
                        </div>
                    </div>

                    <!-- Leave History -->
                    <div class="col-lg-7">
                        <div class="card leave-history-card">
                            <div class="card-header bg-success text-white">
                                <h5 class="mb-0"><i class="fas fa-history"></i> Leave History</h5>
                            </div>
                            <div class="card-body">
                                <?php if (empty($leave_applications)): ?>
                                    <div class="text-center py-5">
                                        <i class="fas fa-calendar-times fa-3x text-muted mb-3"></i>
                                        <h5 class="text-muted">No Leave Applications</h5>
                                        <p class="text-muted">You haven't submitted any leave applications yet.</p>
                                    </div>
                                <?php else: ?>
                                    <div class="table-responsive">
                                        <table class="table table-hover">
                                            <thead>
                                                <tr>
                                                    <th>Type</th>
                                                    <th>Duration</th>
                                                    <th>Status</th>
                                                    <th>Applied</th>
                                                    <th>Action</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php foreach ($leave_applications as $application): ?>
                                                    <tr>
                                                        <td>
                                                            <div class="d-flex align-items-center">
                                                                <div class="leave-type-icon leave-type-<?php echo $application['leave_type']; ?> me-2">
                                                                    <?php echo strtoupper(substr($application['leave_type'], 0, 1)); ?>
                                                                </div>
                                                                <div>
                                                                    <div class="fw-bold"><?php echo ucfirst($application['leave_type']); ?></div>
                                                                    <small class="text-muted">
                                                                        <?php echo date('M j', strtotime($application['start_date'])); ?> -
                                                                        <?php echo date('M j, Y', strtotime($application['end_date'])); ?>
                                                                    </small>
                                                                </div>
                                                            </div>
                                                        </td>
                                                        <td>
                                                            <?php
                                                            $start = strtotime($application['start_date']);
                                                            $end = strtotime($application['end_date']);
                                                            $days = ceil(($end - $start) / (60 * 60 * 24)) + 1;
                                                            echo $days . ' day' . ($days > 1 ? 's' : '');
                                                            ?>
                                                        </td>
                                                        <td>
                                                            <span class="status-badge status-<?php echo $application['status']; ?>">
                                                                <?php echo ucfirst($application['status']); ?>
                                                            </span>
                                                        </td>
                                                        <td><?php echo date('M j, Y', strtotime($application['created_at'])); ?></td>
                                                        <td>
                                                            <button class="btn btn-sm btn-outline-primary" onclick="viewDetails(<?php echo $application['id']; ?>)">
                                                                <i class="fas fa-eye"></i>
                                                            </button>
                                                        </td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <!-- Leave Details Modal -->
    <div class="modal fade" id="leaveDetailsModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Leave Application Details</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body" id="leaveDetailsContent">
                    <!-- Content will be loaded here -->
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function viewDetails(leaveId) {
            // Load leave details via AJAX
            fetch(`../api/get_leave_details.php?id=${leaveId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        const leave = data.leave;
                        const content = `
                            <div class="row">
                                <div class="col-md-6">
                                    <h6>Leave Information</h6>
                                    <p><strong>Type:</strong> ${leave.leave_type.charAt(0).toUpperCase() + leave.leave_type.slice(1)} Leave</p>
                                    <p><strong>Start Date:</strong> ${new Date(leave.start_date).toLocaleDateString()}</p>
                                    <p><strong>End Date:</strong> ${new Date(leave.end_date).toLocaleDateString()}</p>
                                    <p><strong>Status:</strong>
                                        <span class="badge bg-${leave.status === 'approved' ? 'success' : leave.status === 'rejected' ? 'danger' : 'warning'}">
                                            ${leave.status.charAt(0).toUpperCase() + leave.status.slice(1)}
                                        </span>
                                    </p>
                                </div>
                                <div class="col-md-6">
                                    <h6>Additional Details</h6>
                                    <p><strong>Reason:</strong> ${leave.reason}</p>
                                    ${leave.emergency_contact ? `<p><strong>Emergency Contact:</strong> ${leave.emergency_contact}</p>` : ''}
                                    <p><strong>Applied On:</strong> ${new Date(leave.created_at).toLocaleDateString()}</p>
                                    ${leave.admin_comments ? `<p><strong>Admin Comments:</strong> ${leave.admin_comments}</p>` : ''}
                                </div>
                            </div>
                        `;
                        document.getElementById('leaveDetailsContent').innerHTML = content;
                        new bootstrap.Modal(document.getElementById('leaveDetailsModal')).show();
                    } else {
                        alert('Error loading leave details');
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Error loading leave details');
                });
        }

        // Set minimum date for date inputs
        document.addEventListener('DOMContentLoaded', function() {
            const today = new Date().toISOString().split('T')[0];
            document.getElementById('start_date').setAttribute('min', today);
            document.getElementById('end_date').setAttribute('min', today);

            // Update end date min when start date changes
            document.getElementById('start_date').addEventListener('change', function() {
                document.getElementById('end_date').setAttribute('min', this.value);
            });
        });
    </script>
</body>
</html>
