<?php
require_once '../config.php';

// Check if user is logged in and is admin or librarian
if (!isset($_SESSION['user_id']) || !in_array($_SESSION['role'], ['admin', 'librarian'])) {
    header('Location: ../login.php');
    exit();
}

$message = '';
$message_type = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['issue_book'])) {
        $book_id = intval($_POST['book_id']);
        $user_id = intval($_POST['user_id']);
        $due_date = $_POST['due_date'];

        try {
            // Check if book is available
            $check_stmt = $pdo->prepare("SELECT available_copies FROM library_books WHERE id = ?");
            $check_stmt->execute([$book_id]);
            $book = $check_stmt->fetch(PDO::FETCH_ASSOC);

            if ($book && $book['available_copies'] > 0) {
                // Issue the book
                $stmt = $pdo->prepare("INSERT INTO library_borrowings (book_id, user_id, due_date, issued_by) VALUES (?, ?, ?, ?)");
                $stmt->execute([$book_id, $user_id, $due_date, $_SESSION['user_id']]);

                // Update available copies
                $update_stmt = $pdo->prepare("UPDATE library_books SET available_copies = available_copies - 1 WHERE id = ?");
                $update_stmt->execute([$book_id]);

                $message = "Book issued successfully!";
                $message_type = "success";
            } else {
                $message = "Book is not available for borrowing.";
                $message_type = "error";
            }
        } catch (PDOException $e) {
            $message = "Error issuing book: " . $e->getMessage();
            $message_type = "error";
        }
    } elseif (isset($_POST['return_book'])) {
        $borrowing_id = intval($_POST['borrowing_id']);

        try {
            // Get borrowing details
            $borrow_stmt = $pdo->prepare("SELECT * FROM library_borrowings WHERE id = ?");
            $borrow_stmt->execute([$borrowing_id]);
            $borrowing = $borrow_stmt->fetch(PDO::FETCH_ASSOC);

            if ($borrowing) {
                // Update borrowing record
                $return_stmt = $pdo->prepare("UPDATE library_borrowings SET returned_at = NOW(), status = 'returned' WHERE id = ?");
                $return_stmt->execute([$borrowing_id]);

                // Update available copies
                $update_stmt = $pdo->prepare("UPDATE library_books SET available_copies = available_copies + 1 WHERE id = ?");
                $update_stmt->execute([$borrowing['book_id']]);

                $message = "Book returned successfully!";
                $message_type = "success";
            }
        } catch (PDOException $e) {
            $message = "Error returning book: " . $e->getMessage();
            $message_type = "error";
        }
    } elseif (isset($_POST['extend_due_date'])) {
        $borrowing_id = intval($_POST['borrowing_id']);
        $new_due_date = $_POST['new_due_date'];

        try {
            // Get current borrowing details
            $borrow_stmt = $pdo->prepare("SELECT * FROM library_borrowings WHERE id = ?");
            $borrow_stmt->execute([$borrowing_id]);
            $borrowing = $borrow_stmt->fetch(PDO::FETCH_ASSOC);

            if ($borrowing) {
                // Validate new due date is after current due date
                if (strtotime($new_due_date) > strtotime($borrowing['due_date'])) {
                    // Update due date
                    $update_stmt = $pdo->prepare("UPDATE library_borrowings SET due_date = ? WHERE id = ?");
                    $update_stmt->execute([$new_due_date, $borrowing_id]);

                    $message = "Due date extended successfully!";
                    $message_type = "success";
                } else {
                    $message = "New due date must be after the current due date.";
                    $message_type = "error";
                }
            } else {
                $message = "Borrowing record not found.";
                $message_type = "error";
            }
        } catch (PDOException $e) {
            $message = "Error extending due date: " . $e->getMessage();
            $message_type = "error";
        }
    }
}

// Get all current borrowings
try {
    $borrowings_stmt = $pdo->prepare("
        SELECT lb.*, bk.title, bk.author, u.username, u.role,
               CASE
                   WHEN u.role = 'student' THEN s.name
                   WHEN u.role = 'teacher' THEN t.name
                   ELSE u.username
               END as user_name
        FROM library_borrowings lb
        JOIN library_books bk ON lb.book_id = bk.id
        JOIN users u ON lb.user_id = u.id
        LEFT JOIN students s ON (u.role = 'student' AND s.user_id = u.id)
        LEFT JOIN teachers t ON (u.role = 'teacher' AND t.user_id = u.id)
        WHERE lb.status = 'borrowed'
        ORDER BY lb.due_date ASC
    ");
    $borrowings_stmt->execute();
    $borrowings = $borrowings_stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $borrowings = [];
}

// Get available books
try {
    $books_stmt = $pdo->query("SELECT id, title, author, available_copies FROM library_books WHERE available_copies > 0 ORDER BY title");
    $available_books = $books_stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $available_books = [];
}

// Get all users (students and teachers)
try {
    $users_stmt = $pdo->prepare("
        SELECT u.id, u.username, u.role,
               CASE
                   WHEN u.role = 'student' THEN s.name
                   WHEN u.role = 'teacher' THEN t.name
                   ELSE u.username
               END as display_name
        FROM users u
        LEFT JOIN students s ON (u.role = 'student' AND s.user_id = u.id)
        LEFT JOIN teachers t ON (u.role = 'teacher' AND t.user_id = u.id)
        WHERE u.role IN ('student', 'teacher')
        ORDER BY display_name
    ");
    $users_stmt->execute();
    $users = $users_stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $users = [];
}

$page_title = "Library Management - Borrowings";
include '../includes/library_header.php';
?>
<link rel="stylesheet" href="../css/library_custom.css">

<style>
/* Page Header */
.container-fluid h2 {
    color: #8e44ad;
    font-weight: 700;
    margin-bottom: 30px;
    font-size: 2em;
}

/* Buttons */
.btn-issue-book, .btn-manage-books {
    border-radius: 25px;
    padding: 10px 20px;
    font-weight: 600;
    transition: all 0.3s ease;
    box-shadow: 0 4px 15px rgba(142, 68, 173, 0.3);
    border: none;
    color: white;
    cursor: pointer;
}

.btn-issue-book {
    background: linear-gradient(135deg, #27ae60 0%, #2ecc71 100%);
    margin-right: 10px;
}

.btn-issue-book:hover {
    background: linear-gradient(135deg, #2ecc71 0%, #27ae60 100%);
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(46, 204, 113, 0.4);
}

.btn-manage-books {
    background: linear-gradient(135deg, #8e44ad 0%, #9b59b6 100%);
}

.btn-manage-books:hover {
    background: linear-gradient(135deg, #9b59b6 0%, #8e44ad 100%);
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(155, 89, 182, 0.4);
}

/* Alert Messages */
.alert {
    border-radius: 10px;
    border: none;
    box-shadow: 0 4px 6px rgba(0,0,0,0.1);
}

.alert-success {
    background-color: #d4edda;
    color: #155724;
}

.alert-danger {
    background-color: #f8d7da;
    color: #721c24;
}

/* Card Styles */
.card {
    border-radius: 15px;
    box-shadow: 0 20px 40px rgba(0,0,0,0.1);
    margin-bottom: 30px;
    border: none;
}

.card-header {
    background: linear-gradient(135deg, #8e44ad 0%, #9b59b6 100%);
    color: white;
    border-radius: 15px 15px 0 0;
    padding: 20px 25px;
    font-weight: 700;
    font-size: 1.2em;
}

.card-body {
    padding: 25px;
}

/* Table Styles */
.table-responsive {
    border-radius: 10px;
    overflow: hidden;
    box-shadow: 0 4px 6px rgba(0,0,0,0.1);
}

.library-borrowings-table {
    margin-bottom: 0;
    background: white;
}

.library-borrowings-table thead th {
    background: linear-gradient(135deg, #8e44ad 0%, #9b59b6 100%);
    color: white;
    font-weight: 600;
    border: none;
    padding: 15px;
    text-transform: uppercase;
    font-size: 0.85em;
    letter-spacing: 0.5px;
}

.library-borrowings-table tbody tr:hover {
    background-color: #f8f9ff;
    transform: scale(1.01);
    transition: all 0.3s ease;
}

.library-borrowings-table tbody tr {
    transition: all 0.3s ease;
}

.library-borrowings-table tbody td {
    padding: 15px;
    border-bottom: 1px solid #f0f0f0;
    vertical-align: middle;
}

.library-borrowings-table tbody tr:last-child td {
    border-bottom: none;
}

.badge {
    font-size: 0.85em;
    padding: 5px 10px;
    border-radius: 15px;
    font-weight: 600;
}

.badge-info {
    background-color: #17a2b8;
    color: white;
}

.badge-secondary {
    background-color: #6c757d;
    color: white;
}

/* Buttons in table */
.btn-sm {
    padding: 5px 10px;
    border-radius: 15px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
}

.btn-sm:hover {
    opacity: 0.85;
}

.btn-sm i {
    margin-right: 5px;
}

/* Return Button */
.btn-sm[style*="background-color: #17a2b8"] {
    background-color: #17a2b8 !important;
    color: white !important;
    border: none !important;
    margin-right: 5px !important;
}

/* Extend Button */
.btn-sm[style*="background-color: #ffc107"] {
    background-color: #dc3545 !important;
    color: white !important;
    border: none !important;
}

/* Form Styles */
.form-control {
    border-radius: 8px;
    border: 2px solid #e9ecef;
    padding: 10px 15px;
    font-size: 14px;
    transition: all 0.3s ease;
    background-color: #fff;
    color: #495057;
}

.form-control:focus {
    border-color: #8e44ad;
    box-shadow: 0 0 0 0.2rem rgba(142, 68, 173, 0.25);
    outline: none;
}

.form-control::placeholder {
    color: #6c757d;
    opacity: 1;
}

.form-control option {
    padding: 10px;
    background-color: #fff;
    color: #495057;
}

.form-control option:first-child {
    color: #6c757d;
    font-style: italic;
}

/* Select specific styles */
select.form-control {
    background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 20 20'%3e%3cpath stroke='%236c757d' stroke-linecap='round' stroke-linejoin='round' stroke-width='1.5' d='m6 8 4 4 4-4'/%3e%3c/svg%3e");
    background-position: right 0.75rem center;
    background-repeat: no-repeat;
    background-size: 1.5em 1.5em;
    padding-right: 2.5rem;
    appearance: none;
    -webkit-appearance: none;
    -moz-appearance: none;
    line-height: 1.5;
    height: auto;
    display: block;
}

select.form-control:focus {
    background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' fill='none' viewBox='0 0 20 20'%3e%3cpath stroke='%238e44ad' stroke-linecap='round' stroke-linejoin='round' stroke-width='1.5' d='m6 8 4 4 4-4'/%3e%3c/svg%3e");
}

/* Modal Styles */
.modal-content {
    border-radius: 15px;
    border: none;
    box-shadow: 0 20px 60px rgba(0,0,0,0.15);
}

.modal-header {
    background: linear-gradient(135deg, #8e44ad 0%, #9b59b6 100%);
    color: white;
    border-radius: 15px 15px 0 0;
    border-bottom: none;
    padding: 20px 25px;
}

.modal-header .close {
    color: white;
    opacity: 0.8;
    font-size: 1.5em;
}

.modal-header .close:hover {
    color: white;
    opacity: 1;
}

.modal-body {
    padding: 25px;
}

.modal-footer {
    border-top: 1px solid #e9ecef;
    padding: 20px 25px;
    border-radius: 0 0 15px 15px;
}

/* Responsive Design */
@media (max-width: 768px) {
    .container-fluid h2 {
        font-size: 1.5em;
        margin-bottom: 20px;
    }

    .table-responsive {
        margin-top: 15px;
    }

    .card-body {
        padding: 15px;
    }

    .table-hover tbody tr td {
        padding: 8px;
        font-size: 0.85em;
    }

    .modal-dialog {
        margin: 10px;
    }

    .modal-content {
        border-radius: 10px;
    }

    .modal-header, .modal-footer {
        padding: 15px 20px;
    }

    .modal-body {
        padding: 20px;
    }

    .form-control {
        font-size: 16px; /* Prevents zoom on iOS */
        padding: 12px 15px;
    }
}

@media (max-width: 480px) {
    .container-fluid {
        padding: 15px;
    }

    .btn-issue-book, .btn-manage-books {
        width: 100%;
        margin-bottom: 15px;
    }

    .modal-dialog {
        margin: 5px;
    }
}
</style>
<div class="container-fluid">
    <div class="row">
        <div class="col-md-12">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h2><i class="fas fa-book"></i> Library Management - Borrowings</h2>
                <div>
                    <button class="btn btn-issue-book" data-toggle="modal" data-target="#issueBookModal">
                        <i class="fas fa-plus"></i> Issue Book
                    </button>
                    <button class="btn btn-manage-books" onclick="window.location.href='manage_books.php'">
                        <i class="fas fa-book"></i> Manage Books
                    </button>
                </div>
            </div>

            <?php if ($message): ?>
                <div class="alert alert-<?php echo $message_type === 'success' ? 'success' : 'danger'; ?> alert-dismissible fade show" role="alert">
                    <?php echo htmlspecialchars($message); ?>
                    <button type="button" class="close" data-dismiss="alert">
                        <span>&times;</span>
                    </button>
                </div>
            <?php endif; ?>

            <!-- Current Borrowings -->
            <div class="card">
                <div class="card-header">
                    <h5><i class="fas fa-book-open"></i> Current Borrowings</h5>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-hover library-borrowings-table">
                            <thead>
                                <tr>
                                    <th style="border: none; padding: 15px;">Book Title</th>
                                    <th style="border: none; padding: 15px;">Author</th>
                                    <th style="border: none; padding: 15px;">Borrowed By</th>
                                    <th style="border: none; padding: 15px;">Role</th>
                                    <th style="border: none; padding: 15px;">Borrowed Date</th>
                                    <th style="border: none; padding: 15px;">Due Date</th>
                                    <th style="border: none; padding: 15px;">Status</th>
                                    <th style="border: none; padding: 15px;">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($borrowings as $borrowing): ?>
                                    <tr style="border-bottom: 1px solid #e9ecef; <?php echo (strtotime($borrowing['due_date']) < time()) ? 'background-color: #fff5f5;' : ''; ?>">
                                        <td style="padding: 12px; font-weight: 500;"><?php echo htmlspecialchars($borrowing['title']); ?></td>
                                        <td style="padding: 12px; color: #6c757d;"><?php echo htmlspecialchars($borrowing['author']); ?></td>
                                        <td style="padding: 12px;"><?php echo htmlspecialchars($borrowing['user_name']); ?></td>
                                        <td style="padding: 12px;">
                                            <span class="badge badge-<?php echo $borrowing['role'] == 'student' ? 'info' : 'secondary'; ?>">
                                                <?php echo ucfirst($borrowing['role']); ?>
                                            </span>
                                        </td>
                                        <td style="padding: 12px;"><?php echo date('M d, Y', strtotime($borrowing['borrowed_at'])); ?></td>
                                        <td style="padding: 12px;"><?php echo date('M d, Y', strtotime($borrowing['due_date'])); ?></td>
                                        <td style="padding: 12px;">
                                            <?php if (strtotime($borrowing['due_date']) < time()): ?>
                                                <span class="badge" style="background-color: #dc3545; color: white;">Overdue</span>
                                            <?php else: ?>
                                                <span class="badge" style="background-color: #28a745; color: white;">Active</span>
                                            <?php endif; ?>
                                        </td>
                                        <td style="padding: 12px;">
                                            <button class="btn btn-sm" style="background-color: #17a2b8; color: white; border: none; margin-right: 5px;" onclick="returnBook(<?php echo $borrowing['id']; ?>)">
                                                <i class="fas fa-undo"></i> Return
                                            </button>
                                            <button class="btn btn-sm" style="background-color: #ffc107; color: #212529; border: none;" onclick="extendDueDate(<?php echo $borrowing['id']; ?>)">
                                                <i class="fas fa-calendar-plus"></i> Extend
                                            </button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                                <?php if (empty($borrowings)): ?>
                                    <tr>
                                        <td colspan="8" class="text-center" style="padding: 40px; color: #6c757d; font-style: italic;">No active borrowings found.</td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Issue Book Modal -->
<div class="modal fade" id="issueBookModal" tabindex="-1" role="dialog">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Issue Book</h5>
                <button type="button" class="close" data-dismiss="modal">
                    <span>&times;</span>
                </button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <div class="form-group">
                        <label for="book_id">Book *</label>
<select class="form-control" id="book_id" name="book_id" required style="text-align-last: center;">
    <option value="">Select Book</option>
    <?php foreach ($available_books as $book): ?>
        <option value="<?php echo $book['id']; ?>">
            <?php echo htmlspecialchars($book['title']); ?> by <?php echo htmlspecialchars($book['author']); ?> (<?php echo $book['available_copies']; ?> available)
        </option>
    <?php endforeach; ?>
</select>
                    </div>
                    <div class="form-group">
                        <label for="user_id">Borrower *</label>
<select class="form-control" id="user_id" name="user_id" required style="text-align-last: center;">
    <option value="">Select Borrower</option>
    <?php foreach ($users as $user): ?>
        <option value="<?php echo $user['id']; ?>">
            <?php echo htmlspecialchars($user['display_name']); ?> (<?php echo ucfirst($user['role']); ?>)
        </option>
    <?php endforeach; ?>
</select>
                    </div>
                    <div class="form-group">
                        <label for="due_date">Due Date *</label>
                        <input type="date" class="form-control" id="due_date" name="due_date" required min="<?php echo date('Y-m-d'); ?>">
                        <small class="form-text text-muted">Books are typically borrowed for 14 days</small>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn" style="background-color: #6c757d; color: white; border: none;" data-dismiss="modal">Cancel</button>
                    <button type="submit" name="issue_book" class="btn" style="background-color: #28a745; color: white; border: none;">Issue Book</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Return Book Modal -->
<div class="modal fade" id="returnBookModal" tabindex="-1" role="dialog">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Return Book</h5>
                <button type="button" class="close" data-dismiss="modal">
                    <span>&times;</span>
                </button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <p>Are you sure you want to return this book?</p>
                    <input type="hidden" id="return_borrowing_id" name="borrowing_id">
                    <div id="return-book-details"></div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn" style="background-color: #6c757d; color: white; border: none;" data-dismiss="modal">Cancel</button>
                    <button type="submit" name="return_book" class="btn" style="background-color: #dc3545; color: white; border: none;">Confirm Return</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Extend Due Date Modal -->
<div class="modal fade" id="extendDueDateModal" tabindex="-1" role="dialog">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Extend Due Date</h5>
                <button type="button" class="close" data-dismiss="modal">
                    <span>&times;</span>
                </button>
            </div>
            <form method="POST">
                <div class="modal-body">
                    <div class="form-group">
                        <label for="extend_due_date">New Due Date *</label>
                        <input type="date" class="form-control" id="extend_due_date" name="new_due_date" required>
                        <small class="form-text text-muted">Select a new due date (must be after current due date)</small>
                    </div>
                    <input type="hidden" id="extend_borrowing_id" name="borrowing_id">
                    <div id="extend-book-details"></div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn" style="background-color: #6c757d; color: white; border: none;" data-dismiss="modal">Cancel</button>
                    <button type="submit" name="extend_due_date" class="btn" style="background-color: #dc3545; color: white; border: none;">Extend Due Date</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function returnBook(borrowingId) {
    document.getElementById('return_borrowing_id').value = borrowingId;
    // You could add AJAX call here to get borrowing details
    $('#returnBookModal').modal('show');
}

function extendDueDate(borrowingId) {
    document.getElementById('extend_borrowing_id').value = borrowingId;
    // Set minimum date to current due date + 1 day
    const currentDueDate = new Date();
    currentDueDate.setDate(currentDueDate.getDate() + 1);
    document.getElementById('extend_due_date').min = currentDueDate.toISOString().split('T')[0];
    $('#extendDueDateModal').modal('show');
}

// Set default due date to 14 days from now
document.getElementById('due_date').value = new Date(Date.now() + 14 * 24 * 60 * 60 * 1000).toISOString().split('T')[0];
</script>

<?php include '../includes/library_footer.php'; ?>
