<?php
require_once '../config.php';

// Check if user is logged in and is librarian or admin
if (!isset($_SESSION['user_id']) || !in_array($_SESSION['role'], ['librarian', 'admin'])) {
    header('Location: ../login.php');
    exit();
}

// Fetch statistics
$stats = [
    'total_books' => 0,
    'available_books' => 0,
    'borrowed_books' => 0,
    'overdue_books' => 0,
    'total_borrowings' => 0,
    'active_users' => 0
];

try {
    // Total books
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM library_books");
    $stats['total_books'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'];

    // Available books
    $stmt = $pdo->query("SELECT SUM(available_copies) as total FROM library_books");
    $stats['available_books'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'] ?? 0;

    // Borrowed books
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM library_borrowings WHERE status IN ('borrowed', 'overdue')");
    $stats['borrowed_books'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'];

    // Overdue books
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM library_borrowings WHERE status IN ('borrowed', 'overdue') AND due_date < CURDATE()");
    $stats['overdue_books'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'];

    // Total borrowings
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM library_borrowings");
    $stats['total_borrowings'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'];

    // Active users (users who have borrowed books)
    $stmt = $pdo->query("SELECT COUNT(DISTINCT user_id) as total FROM library_borrowings");
    $stats['active_users'] = $stmt->fetch(PDO::FETCH_ASSOC)['total'];

} catch (PDOException $e) {
    // Handle error
}

$page_title = "Library Reports";
include '../includes/library_header.php';
?>

<div class="container-fluid">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h2><i class="fas fa-chart-bar"></i> Library Reports</h2>
        <div class="text-muted">
            Report generated on <?php echo date('M d, Y H:i'); ?>
        </div>
    </div>

    <!-- Statistics Cards -->
    <div class="row mb-4">
        <div class="col-md-2">
            <div class="card text-center">
                <div class="card-body">
                    <i class="fas fa-book fa-2x text-primary"></i>
                    <h5 class="card-title mt-2">Total Books</h5>
                    <p class="card-text h4"><?php echo number_format($stats['total_books']); ?></p>
                </div>
            </div>
        </div>
        <div class="col-md-2">
            <div class="card text-center">
                <div class="card-body">
                    <i class="fas fa-check-circle fa-2x text-success"></i>
                    <h5 class="card-title mt-2">Available</h5>
                    <p class="card-text h4"><?php echo number_format($stats['available_books']); ?></p>
                </div>
            </div>
        </div>
        <div class="col-md-2">
            <div class="card text-center">
                <div class="card-body">
                    <i class="fas fa-hand-holding fa-2x text-info"></i>
                    <h5 class="card-title mt-2">Borrowed</h5>
                    <p class="card-text h4"><?php echo number_format($stats['borrowed_books']); ?></p>
                </div>
            </div>
        </div>
        <div class="col-md-2">
            <div class="card text-center">
                <div class="card-body">
                    <i class="fas fa-exclamation-triangle fa-2x text-warning"></i>
                    <h5 class="card-title mt-2">Overdue</h5>
                    <p class="card-text h4"><?php echo number_format($stats['overdue_books']); ?></p>
                </div>
            </div>
        </div>
        <div class="col-md-2">
            <div class="card text-center">
                <div class="card-body">
                    <i class="fas fa-exchange-alt fa-2x text-secondary"></i>
                    <h5 class="card-title mt-2">Total Borrowings</h5>
                    <p class="card-text h4"><?php echo number_format($stats['total_borrowings']); ?></p>
                </div>
            </div>
        </div>
        <div class="col-md-2">
            <div class="card text-center">
                <div class="card-body">
                    <i class="fas fa-users fa-2x text-dark"></i>
                    <h5 class="card-title mt-2">Active Users</h5>
                    <p class="card-text h4"><?php echo number_format($stats['active_users']); ?></p>
                </div>
            </div>
        </div>
    </div>

    <!-- Detailed Reports -->
    <div class="row">
        <div class="col-md-6">
            <div class="card">
                <div class="card-header">
                    <h5><i class="fas fa-list"></i> Books by Category</h5>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-sm">
                            <thead>
                                <tr>
                                    <th>Category</th>
                                    <th>Count</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php
                                try {
                                    $stmt = $pdo->query("
                                        SELECT lc.name, COUNT(lb.id) as count
                                        FROM library_categories lc
                                        LEFT JOIN library_books lb ON lc.id = lb.category_id
                                        GROUP BY lc.id, lc.name
                                        ORDER BY count DESC
                                    ");
                                    $categories = $stmt->fetchAll(PDO::FETCH_ASSOC);
                                    foreach ($categories as $cat) {
                                        echo "<tr><td>{$cat['name']}</td><td>{$cat['count']}</td></tr>";
                                    }
                                } catch (PDOException $e) {
                                    echo "<tr><td colspan='2'>Error loading data</td></tr>";
                                }
                                ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-md-6">
            <div class="card">
                <div class="card-header">
                    <h5><i class="fas fa-calendar-alt"></i> Recent Activity</h5>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-sm">
                            <thead>
                                <tr>
                                    <th>Book</th>
                                    <th>User</th>
                                    <th>Action</th>
                                    <th>Date</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php
                                try {
                                    $stmt = $pdo->query("
                                        SELECT bk.title, u.username, lb.status, lb.issued_date
                                        FROM library_borrowings lb
                                        JOIN library_books bk ON lb.book_id = bk.id
                                        JOIN users u ON lb.user_id = u.id
                                        ORDER BY lb.issued_date DESC
                                        LIMIT 10
                                    ");
                                    $activities = $stmt->fetchAll(PDO::FETCH_ASSOC);
                                    foreach ($activities as $activity) {
                                        $action = $activity['status'] === 'returned' ? 'Returned' : 'Borrowed';
                                        $date = date('M d', strtotime($activity['issued_date']));
                                        echo "<tr><td>{$activity['title']}</td><td>{$activity['username']}</td><td>{$action}</td><td>{$date}</td></tr>";
                                    }
                                } catch (PDOException $e) {
                                    echo "<tr><td colspan='4'>Error loading data</td></tr>";
                                }
                                ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<?php include '../includes/library_footer.php'; ?>
