<?php
require_once 'config.php';
require_once 'includes/password_reset.php';
require_once 'includes/csrf_protection.php';
require_once 'includes/input_validation.php';

$message = '';
$messageType = '';

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    // Validate CSRF token
    if (!CSRFProtection::validateRequest()) {
        $message = 'Security validation failed. Please refresh the page and try again.';
        $messageType = 'error';
    } else {
        // Sanitize and validate input
        $username = InputValidation::getPostData('username', 'string');

        // Validate required fields
        $validationErrors = InputValidation::validateRequired($_POST, ['username']);
        if (!empty($validationErrors)) {
            $message = implode('<br>', $validationErrors);
            $messageType = 'error';
        } elseif (empty($username)) {
            $message = 'Please enter your username.';
            $messageType = 'error';
        } else {
            // Process password reset request
            $passwordReset = new PasswordReset();
            $result = $passwordReset->createResetToken($username);

            if ($result['success']) {
                // Send reset email
                $emailSent = $passwordReset->sendResetEmail($username, $result['token'], $result['user']);

                if ($emailSent) {
                    $message = 'Password reset instructions have been sent to your email address. Please check your inbox and follow the link to reset your password.';
                    $messageType = 'success';
                } else {
                    $message = 'Failed to send reset email. Please try again later or contact support.';
                    $messageType = 'error';
                }
            } else {
                $message = $result['message'];
                $messageType = 'error';
            }
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Forgot Password - School Management System</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: #ffffff;
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }

        .top-bar {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            padding: 15px 0;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
            z-index: 1000;
        }

        .top-bar .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 20px;
            display: flex;
            justify-content: center;
            align-items: center;
        }

        .top-bar .logo {
            font-size: 1.8rem;
            font-weight: 700;
            color: #667eea;
            display: flex;
            align-items: center;
            text-shadow: 0 2px 4px rgba(102, 126, 234, 0.3);
            letter-spacing: 0.5px;
        }

        .top-bar .logo i {
            margin-right: 12px;
            font-size: 2rem;
            color: #764ba2;
            filter: drop-shadow(0 2px 4px rgba(118, 75, 162, 0.3));
        }

        .forgot-container {
            background: rgba(255, 255, 255, 0.98);
            backdrop-filter: blur(20px);
            padding: 40px;
            border-radius: 20px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.15);
            width: 100%;
            max-width: 400px;
            border: 2px solid #667eea;
            margin-top: 80px;
        }

        .forgot-header {
            text-align: center;
            margin-bottom: 30px;
        }

        .forgot-header .icon {
            font-size: 3.5rem;
            color: #667eea;
            margin-bottom: 15px;
        }

        .forgot-header h1 {
            color: #2d3748;
            margin-bottom: 8px;
            font-size: 2rem;
            font-weight: 600;
        }

        .forgot-header p {
            color: #718096;
            font-size: 0.95rem;
            line-height: 1.5;
        }

        .form-group {
            margin-bottom: 20px;
            position: relative;
        }

        .form-group input {
            width: 100%;
            padding: 15px 15px 15px 45px;
            border: 2px solid #e2e8f0;
            border-radius: 12px;
            font-size: 16px;
            transition: all 0.3s ease;
            background: #f8fafc;
            outline: none;
        }

        .form-group input:focus {
            border-color: #667eea;
            background: white;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
            transform: translateY(-1px);
        }

        .form-group .input-icon {
            position: absolute;
            left: 15px;
            top: 50%;
            transform: translateY(-50%);
            color: #a0aec0;
            font-size: 1.1rem;
            transition: color 0.3s ease;
        }

        .form-group input:focus + .input-icon {
            color: #667eea;
        }

        .btn {
            width: 100%;
            padding: 15px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            border-radius: 12px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
        }

        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(102, 126, 234, 0.4);
        }

        .btn:active {
            transform: translateY(0);
        }

        .message {
            padding: 12px 15px;
            margin-bottom: 20px;
            border-radius: 8px;
            font-size: 14px;
        }

        .message.success {
            background: #c6f6d5;
            color: #22543d;
            border: 1px solid #9ae6b4;
        }

        .message.error {
            background: #fed7d7;
            color: #c53030;
            border: 1px solid #fbb6ce;
        }

        .links-section {
            margin-top: 25px;
            text-align: center;
        }

        .back-link {
            color: #718096;
            text-decoration: none;
            font-size: 14px;
            transition: color 0.3s ease;
        }

        .back-link:hover {
            color: #667eea;
        }

        /* Responsive */
        @media (max-width: 768px) {
            .forgot-container {
                margin: 20px;
                padding: 30px 25px;
            }

            .forgot-header h1 {
                font-size: 1.8rem;
            }

            .forgot-header .icon {
                font-size: 3rem;
            }

            .top-bar .container {
                padding: 0 15px;
            }

            .top-bar .logo {
                font-size: 1.3rem;
            }

            .top-bar .logo i {
                font-size: 1.5rem;
            }
        }

        @media (max-width: 480px) {
            body {
                padding: 10px;
            }

            .forgot-container {
                padding: 25px 20px;
            }

            .forgot-header h1 {
                font-size: 1.6rem;
            }

            .forgot-header .icon {
                font-size: 2.5rem;
            }

            .top-bar {
                padding: 12px 0;
            }

            .top-bar .logo {
                font-size: 1.2rem;
            }

            .top-bar .logo i {
                font-size: 1.4rem;
            }
        }
    </style>
</head>
<body>
    <!-- Top Navigation Bar -->
    <div class="top-bar">
        <div class="container">
            <div class="logo">
                <i class="fas fa-graduation-cap"></i>
                <span>Elitek School Management System</span>
            </div>
        </div>
    </div>

    <!-- Forgot Password Container -->
    <div class="forgot-container">
        <div class="forgot-header">
            <div class="icon"><i class="fas fa-key"></i></div>
            <h1>Forgot Password</h1>
            <p>Enter your username and we'll send you a link to reset your password.</p>
        </div>

        <?php if ($message): ?>
            <div class="message <?php echo $messageType; ?>"><?php echo htmlspecialchars($message); ?></div>
        <?php endif; ?>

        <form method="post" action="forgot_password.php">
            <?php echo CSRFProtection::getHiddenInput(); ?>
            <div class="form-group">
                <input type="text" id="username" name="username" placeholder="Enter your username" required autofocus>
                <i class="fas fa-user input-icon"></i>
            </div>

            <button type="submit" class="btn"><i class="fas fa-paper-plane"></i> Send Reset Link</button>
        </form>

        <div class="links-section">
            <a href="login.php" class="back-link"><i class="fas fa-arrow-left"></i> Back to Login</a>
        </div>
    </div>

    <script src="/school/js/script.js"></script>
</body>
</html>
