<?php
require_once 'config.php';

/**
 * Auto-enroll all students in a given class into all courses assigned to that class.
 * This ensures that every student in the class has an enrollment record for each course.
 *
 * @param string $class_name The name of the class (e.g., 'Grade 1A')
 * @return array ['success' => bool, 'message' => string]
 */
function autoEnrollClass($class_name) {
    global $pdo;

    try {
        // Get the class ID from the class name
        $stmt = $pdo->prepare("SELECT id FROM classes WHERE name = ?");
        $stmt->execute([$class_name]);
        $class = $stmt->fetch();

        if (!$class) {
            return ['success' => false, 'message' => "Class '$class_name' not found."];
        }

        $class_id = $class['id'];

        // Get all courses assigned to this class
        $stmt = $pdo->prepare("SELECT course_id FROM class_courses WHERE class_id = ?");
        $stmt->execute([$class_id]);
        $courses = $stmt->fetchAll(PDO::FETCH_COLUMN);

        if (empty($courses)) {
            return ['success' => false, 'message' => "No courses assigned to class '$class_name'."];
        }

        // Get all students in this class
        $stmt = $pdo->prepare("SELECT id FROM students WHERE class = ?");
        $stmt->execute([$class_name]);
        $students = $stmt->fetchAll(PDO::FETCH_COLUMN);

        if (empty($students)) {
            return ['success' => false, 'message' => "No students found in class '$class_name'."];
        }

        // Get active academic year and term
        $active_year = $pdo->query("SELECT id FROM academic_years WHERE is_active = 1 LIMIT 1")->fetch();
        $academic_year_id = $active_year ? $active_year['id'] : null;

        $active_term = $pdo->query("SELECT id FROM terms WHERE is_active = 1 LIMIT 1")->fetch();
        $term_id = $active_term ? $active_term['id'] : null;

        $enrolled_count = 0;

        // For each student, check and enroll in each course if not already enrolled
        foreach ($students as $student_id) {
            foreach ($courses as $course_id) {
                // Check if enrollment already exists
                $stmt = $pdo->prepare("SELECT id FROM enrollments WHERE student_id = ? AND course_id = ? AND academic_year_id = ? AND term_id = ?");
                $stmt->execute([$student_id, $course_id, $academic_year_id, $term_id]);
                $existing = $stmt->fetch();

                if (!$existing) {
                    // Insert new enrollment with academic year and term
                    $stmt = $pdo->prepare("INSERT INTO enrollments (student_id, course_id, academic_year_id, term_id, enrolled_at) VALUES (?, ?, ?, ?, NOW())");
                    $stmt->execute([$student_id, $course_id, $academic_year_id, $term_id]);
                    $enrolled_count++;
                }
            }
        }

        return ['success' => true, 'message' => "Auto-enrollment completed. $enrolled_count new enrollments created."];

    } catch (Exception $e) {
        return ['success' => false, 'message' => "Error during auto-enrollment: " . $e->getMessage()];
    }
}

/**
 * Auto-enroll a single student into all courses assigned to their class.
 *
 * @param int $student_id The ID of the student
 * @param string $class_name The name of the class (optional, will be fetched if not provided)
 * @return array ['success' => bool, 'message' => string]
 */
function autoEnrollStudent($student_id, $class_name = null) {
    global $pdo;

    try {
        // If class_name is not provided, get it from the student record
        if ($class_name === null) {
            $stmt = $pdo->prepare("SELECT class FROM students WHERE id = ?");
            $stmt->execute([$student_id]);
            $student = $stmt->fetch();

            if (!$student) {
                return ['success' => false, 'message' => "Student with ID $student_id not found."];
            }

            $class_name = $student['class'];
        }

        // Get the class ID from the class name
        $stmt = $pdo->prepare("SELECT id FROM classes WHERE name = ?");
        $stmt->execute([$class_name]);
        $class = $stmt->fetch();

        if (!$class) {
            return ['success' => false, 'message' => "Class '$class_name' not found."];
        }

        $class_id = $class['id'];

        // Get all courses assigned to this class
        $stmt = $pdo->prepare("SELECT course_id FROM class_courses WHERE class_id = ?");
        $stmt->execute([$class_id]);
        $courses = $stmt->fetchAll(PDO::FETCH_COLUMN);

        if (empty($courses)) {
            return ['success' => false, 'message' => "No courses assigned to class '$class_name'."];
        }

        // Get active academic year and term
        $active_year = $pdo->query("SELECT id FROM academic_years WHERE is_active = 1 LIMIT 1")->fetch();
        $academic_year_id = $active_year ? $active_year['id'] : null;

        $active_term = $pdo->query("SELECT id FROM terms WHERE is_active = 1 LIMIT 1")->fetch();
        $term_id = $active_term ? $active_term['id'] : null;

        $enrolled_count = 0;

        // For this student, check and enroll in each course if not already enrolled
        foreach ($courses as $course_id) {
            // Check if enrollment already exists
            $stmt = $pdo->prepare("SELECT id FROM enrollments WHERE student_id = ? AND course_id = ? AND academic_year_id = ? AND term_id = ?");
            $stmt->execute([$student_id, $course_id, $academic_year_id, $term_id]);
            $existing = $stmt->fetch();

            if (!$existing) {
                // Insert new enrollment with academic year and term
                $stmt = $pdo->prepare("INSERT INTO enrollments (student_id, course_id, academic_year_id, term_id, enrolled_at) VALUES (?, ?, ?, ?, NOW())");
                $stmt->execute([$student_id, $course_id, $academic_year_id, $term_id]);
                $enrolled_count++;
            }
        }

        return ['success' => true, 'message' => "Auto-enrollment completed for student. $enrolled_count new enrollments created."];

    } catch (Exception $e) {
        return ['success' => false, 'message' => "Error during auto-enrollment: " . $e->getMessage()];
    }
}
?>
